/***************************************************************************
 *            burn-dialog.c
 *
 *  mer jun 29 13:05:45 2005
 *  Copyright  2005  Philippe Rouquier
 *  bonfire-app@wanadoo.fr
 ***************************************************************************/

/*
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <stdlib.h>
#include <string.h>

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <glib.h>
#include <glib/gi18n-lib.h>
#include <glib/gstdio.h>

#include <gdk/gdk.h>

#include <gtk/gtkwidget.h>
#include <gtk/gtkdialog.h>
#include <gtk/gtkmessagedialog.h>
#include <gtk/gtkbox.h>
#include <gtk/gtktogglebutton.h>
#include <gtk/gtkcheckbutton.h>
#include <gtk/gtkstock.h>
#include <gtk/gtklabel.h>

#ifdef HAVE_LIBNOTIFY
#include <libnotify/notify.h>
#endif

#include "utils.h"
#include "tray.h"
#include "burn-basics.h"
#include "burn-dialog.h"
#include "recorder-selection.h"
#include "progress.h"

extern gint debug;

static void bonfire_burn_dialog_class_init (BonfireBurnDialogClass *klass);
static void bonfire_burn_dialog_init (BonfireBurnDialog *obj);
static void bonfire_burn_dialog_finalize (GObject *object);
static void bonfire_burn_dialog_destroy (GtkObject *object);

static gboolean
bonfire_burn_dialog_delete (GtkWidget *widget,
			    GdkEventAny *event);
static void
bonfire_burn_dialog_cancel_clicked_cb (GtkWidget *button,
				       BonfireBurnDialog *dialog);

static void
bonfire_burn_dialog_tray_cancel_cb (BonfireTrayIcon *tray,
				    BonfireBurnDialog *dialog);
static void
bonfire_burn_dialog_tray_show_dialog_cb (BonfireTrayIcon *tray,
					 gboolean show,
					 GtkWidget *dialog);
static void
bonfire_burn_dialog_tray_close_after_cb (BonfireTrayIcon *tray,
					 gboolean close,
					 BonfireBurnDialog *dialog);

/*static void
bonfire_burn_dialog_device_changed_cb (BonfireRecorderSelection *selection,
				       gboolean can_record,
				       BonfireBurnDialog *dialog);
*/
struct BonfireBurnDialogPrivate {
	BonfireBurn *burn;
	BonfireTrackSourceType track_type;

	GtkWidget *waiting_disc_dialog;

	GtkWidget *close_check;
	GtkWidget *header;
	GtkWidget *image;
	GtkWidget *tray;

	gint on_the_fly:1;
};


static GObjectClass *parent_class = NULL;

GType
bonfire_burn_dialog_get_type ()
{
	static GType type = 0;

	if (type == 0) {
		static const GTypeInfo our_info = {
			sizeof (BonfireBurnDialogClass),
			NULL,
			NULL,
			(GClassInitFunc) bonfire_burn_dialog_class_init,
			NULL,
			NULL,
			sizeof (BonfireBurnDialog),
			0,
			(GInstanceInitFunc) bonfire_burn_dialog_init,
		};

		type = g_type_register_static (GTK_TYPE_DIALOG,
					       "BonfireBurnDialog",
					       &our_info, 0);
	}

	return type;
}

static void
bonfire_burn_dialog_class_init (BonfireBurnDialogClass * klass)
{
	GObjectClass *object_class = G_OBJECT_CLASS (klass);
	GtkObjectClass *gtk_object_class = GTK_OBJECT_CLASS (klass);
	GtkWidgetClass *widget_class = GTK_WIDGET_CLASS (klass);

	parent_class = g_type_class_peek_parent (klass);
	object_class->finalize = bonfire_burn_dialog_finalize;
	gtk_object_class->destroy = bonfire_burn_dialog_destroy;
	widget_class->delete_event = bonfire_burn_dialog_delete;
}

static char *
bonfire_burn_dialog_get_media_type_string (BonfireBurn *burn,
					   BonfireMediaType type,
					   gboolean insert)
{
	char *message = NULL;

	if (type & BONFIRE_MEDIA_WITH_DATA) {
		if (!insert) {
			if (type & BONFIRE_MEDIA_REWRITABLE)
				message = g_strdup (_("replace the disc with a rewritable disc holding data."));
			else
				message = g_strdup (_("replace the disc with a disc holding data."));
		}
		else {
			if (type & BONFIRE_MEDIA_REWRITABLE)
				message = g_strdup (_("insert a rewritable disc holding data."));
			else
				message = g_strdup (_("insert a disc holding data."));
		}
	}
	else if (type & BONFIRE_MEDIA_WRITABLE) {
		gint64 isosize = 0;
	
		bonfire_burn_status (burn,
				     &isosize,
				     NULL,
				     NULL,
				     NULL);

		if ((type & BONFIRE_MEDIA_TYPE_CD) && !(type & BONFIRE_MEDIA_TYPE_DVD)) {
			if (!insert) {
				if (isosize)
					message = g_strdup_printf (_("replace the disc with a recordable CD with a least %i Mb free."), 
								   (int) isosize / 1048576);
				else
					message = g_strdup_printf (_("replace the disc with a recordable CD."));
			}
			else {
				if (isosize)
					message = g_strdup_printf (_("insert a recordable CD with a least %i Mb free."), 
								   (int) isosize / 1048576);
				else
					message = g_strdup_printf (_("insert a recordable CD."));
			}
		}
		else if (!(type & BONFIRE_MEDIA_TYPE_CD) && (type & BONFIRE_MEDIA_TYPE_DVD)) {
			if (!insert) {
				if (isosize)
					message = g_strdup_printf (_("replace the disc with a recordable DVD with a least %i Mb free."), 
								   (int) isosize / 1048576);
				else
					message = g_strdup_printf (_("replace the disc with a recordable DVD."));
			}
			else {
				if (isosize)
					message = g_strdup_printf (_("insert a recordable DVD with a least %i Mb free."), 
								   (int) isosize / 1048576);
				else
					message = g_strdup_printf (_("insert a recordable DVD."));
			}
		}
		else if (!insert) {
			if (isosize)
				message = g_strdup_printf (_("replace the disc with a recordable CD or DVD with a least %i Mb free."), 
							   (int) isosize / 1048576);
			else
				message = g_strdup_printf (_("replace the disc with a recordable CD or DVD."));
		}
		else {
			if (isosize)
				message = g_strdup_printf (_("insert a recordable CD or DVD with a least %i Mb free."), 
							   (int) isosize / 1048576);
			else
				message = g_strdup_printf (_("insert a recordable CD or DVD."));
		}
	}

	return message;
}

static BonfireBurnResult
bonfire_burn_dialog_insert_disc_cb (BonfireBurn *burn,
				    BonfireBurnError error,
				    BonfireMediaType type,
				    BonfireBurnDialog *dialog)
{
	gint result;
	GtkWindow *window;
	GtkWidget *message;
	gboolean hide = FALSE;
	char *main_message = NULL, *secondary_message = NULL;

	if (!GTK_WIDGET_VISIBLE (dialog)) {
		gtk_widget_show (GTK_WIDGET (dialog));
		hide = TRUE;
	}

	if (error == BONFIRE_BURN_ERROR_MEDIA_BUSY) {
		main_message = g_strdup (_("The disc in the drive is busy:"));
		secondary_message = g_strdup (_("make sure another application is not using it and click OK"));
	} 
	else if (error == BONFIRE_BURN_ERROR_MEDIA_NONE) {
		main_message = g_strdup (_("There is no disc in the drive:"));
		secondary_message = bonfire_burn_dialog_get_media_type_string (burn, type, TRUE);
	}
	else if (error == BONFIRE_BURN_ERROR_MEDIA_NOT_REWRITABLE) {
		main_message = g_strdup (_("The disc in the drive is not rewritable:"));
		secondary_message = bonfire_burn_dialog_get_media_type_string (burn, type, FALSE);
	}
	else if (error == BONFIRE_BURN_ERROR_MEDIA_BLANK) {
		main_message = g_strdup (_("The disc in the drive is empty:"));
		secondary_message = bonfire_burn_dialog_get_media_type_string (burn, type, FALSE);
	}
	else if (error == BONFIRE_BURN_ERROR_MEDIA_NOT_WRITABLE) {
		main_message = g_strdup (_("The disc in the drive is not writable:"));
		secondary_message = bonfire_burn_dialog_get_media_type_string (burn, type, FALSE);
	}
	else if (error == BONFIRE_BURN_ERROR_DVD_NOT_SUPPORTED) {
		main_message = g_strdup (_("The disc in the drive is a DVD:"));
		secondary_message = bonfire_burn_dialog_get_media_type_string (burn, type, FALSE);
	}
	else if (error == BONFIRE_BURN_ERROR_CD_NOT_SUPPORTED) {
		main_message = g_strdup (_("The disc in the drive is a CD:"));
		secondary_message = bonfire_burn_dialog_get_media_type_string (burn, type, FALSE);
	}
	else if (error == BONFIRE_BURN_ERROR_MEDIA_SPACE) {
		main_message = g_strdup (_("The disc in the drive is not big enough:"));
		secondary_message = bonfire_burn_dialog_get_media_type_string (burn, type, FALSE);
	}
	else if (error == BONFIRE_BURN_ERROR_NONE) {
		main_message = g_strdup_printf ("<b><big>%s</big></b>",
						bonfire_burn_dialog_get_media_type_string (burn, type, FALSE));
	}
	else if (error == BONFIRE_BURN_ERROR_RELOAD_MEDIA) {
		main_message = g_strdup (_("The disc in the drive needs to be reloaded:"));
		secondary_message = g_strdup (_("eject the disc and reload it."));
	}

	if (gtk_window_get_transient_for (GTK_WINDOW (dialog)))
		window = gtk_window_get_transient_for (GTK_WINDOW (dialog));
	else
		window = GTK_WINDOW (dialog);

	if (secondary_message) {
		message = gtk_message_dialog_new (window,
						  GTK_DIALOG_DESTROY_WITH_PARENT|
						  GTK_DIALOG_MODAL,
						  GTK_MESSAGE_QUESTION,
						  GTK_BUTTONS_CANCEL,
						  main_message);

		if (secondary_message) {
			gtk_message_dialog_format_secondary_text (GTK_MESSAGE_DIALOG (message),
								  secondary_message);
			g_free (secondary_message);
		}
	}
	else
		message = gtk_message_dialog_new_with_markup (window,
							      GTK_DIALOG_DESTROY_WITH_PARENT|
							      GTK_DIALOG_MODAL,
							      GTK_MESSAGE_QUESTION,
							      GTK_BUTTONS_CANCEL,
							      main_message);

	g_free (main_message);

	if (error == BONFIRE_BURN_ERROR_MEDIA_NONE)
		gtk_window_set_title (GTK_WINDOW (message), _("Waiting for disc insertion"));
	else
		gtk_window_set_title (GTK_WINDOW (message), _("Waiting for disc replacement"));

	dialog->priv->waiting_disc_dialog = message;

	result = gtk_dialog_run (GTK_DIALOG (message));
	gtk_widget_destroy (message);

	dialog->priv->waiting_disc_dialog = NULL;

	if (hide)
		gtk_widget_hide (GTK_WIDGET (dialog));

	if (result != GTK_RESPONSE_OK)
		return BONFIRE_BURN_CANCEL;

	return BONFIRE_BURN_OK;
}

static BonfireBurnResult
bonfire_burn_dialog_data_loss_cb (BonfireBurn *burn,
				  GtkDialog *dialog)
{
	gint result;
	GtkWindow *window;
	GtkWidget *button;
	GtkWidget *message;
	gboolean hide = FALSE;

	if (!GTK_WIDGET_VISIBLE (dialog)) {
		gtk_widget_show (GTK_WIDGET (dialog));
		hide = TRUE;
	}

	if (gtk_window_get_transient_for (GTK_WINDOW (dialog)))
		window = gtk_window_get_transient_for (GTK_WINDOW (dialog));
	else
		window = GTK_WINDOW (dialog);

	message = gtk_message_dialog_new (window,
					  GTK_DIALOG_DESTROY_WITH_PARENT|
					  GTK_DIALOG_MODAL,
					  GTK_MESSAGE_QUESTION,
					  GTK_BUTTONS_NONE,
					  _("The disc in the drive holds data:"));

	gtk_window_set_title (GTK_WINDOW (message), _("Erasing data"));

	gtk_message_dialog_format_secondary_text (GTK_MESSAGE_DIALOG (message),
						  _("Do you want to erase (you will lose all data previously written)\nor replace it ?"));

	button = bonfire_utils_make_button (_("Replace the disc"), GTK_STOCK_REFRESH);
	gtk_widget_show_all (button);
	gtk_dialog_add_action_widget (GTK_DIALOG (message),
				      button, GTK_RESPONSE_ACCEPT);

	gtk_dialog_add_buttons (GTK_DIALOG (message),
				GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL,
				GTK_STOCK_CLEAR, GTK_RESPONSE_OK,
				NULL);

	result = gtk_dialog_run (GTK_DIALOG (message));
	gtk_widget_destroy (message);

	if (hide)
		gtk_widget_hide (GTK_WIDGET (dialog));

	if (result == GTK_RESPONSE_ACCEPT)
		return BONFIRE_BURN_NEED_RELOAD;

	if (result != GTK_RESPONSE_OK)
		return BONFIRE_BURN_CANCEL;

	return BONFIRE_BURN_OK;
}

static BonfireBurnResult
bonfire_burn_dialog_rewritable_cb (BonfireBurn *burn,
				   GtkDialog *dialog)
{
	gint result;
	GtkWindow *window;
	GtkWidget *button;
	GtkWidget *message;
	gboolean hide = FALSE;

	if (!GTK_WIDGET_VISIBLE (dialog)) {
		gtk_widget_show (GTK_WIDGET (dialog));
		hide = TRUE;
	}

	if (gtk_window_get_transient_for (GTK_WINDOW (dialog)))
		window = gtk_window_get_transient_for (GTK_WINDOW (dialog));
	else
		window = GTK_WINDOW (dialog);

	message = gtk_message_dialog_new (window,
					  GTK_DIALOG_DESTROY_WITH_PARENT|
					  GTK_DIALOG_MODAL,
					  GTK_MESSAGE_QUESTION,
					  GTK_BUTTONS_NONE,
					  _("Recording audio tracks on a rewritable disc is not advised:"));

	gtk_window_set_title (GTK_WINDOW (message), _("Rewritable disc"));

	gtk_message_dialog_format_secondary_text (GTK_MESSAGE_DIALOG (message),
						  _("you might not be able to listen to it with stereos.\nDo you want to continue anyway?"));

	button = bonfire_utils_make_button (_("Replace the disc"), GTK_STOCK_REFRESH);
	gtk_widget_show_all (button);
	gtk_dialog_add_action_widget (GTK_DIALOG (message),
				      button, GTK_RESPONSE_ACCEPT);

	gtk_dialog_add_buttons (GTK_DIALOG (message),
				GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL,
				NULL);

	button = bonfire_utils_make_button (_("Continue"), GTK_STOCK_OK);
	gtk_widget_show_all (button);
	gtk_dialog_add_action_widget (GTK_DIALOG (message),
				      button, GTK_RESPONSE_OK);

	result = gtk_dialog_run (GTK_DIALOG (message));
	gtk_widget_destroy (message);

	if (hide)
		gtk_widget_hide (GTK_WIDGET (dialog));

	if (result == GTK_RESPONSE_ACCEPT)
		return BONFIRE_BURN_NEED_RELOAD;

	if (result != GTK_RESPONSE_OK)
		return BONFIRE_BURN_CANCEL;

	return BONFIRE_BURN_OK;
}

static BonfireBurnResult
bonfire_burn_dialog_disable_joliet_cb (BonfireBurn *burn,
				       GtkDialog *dialog)
{
	gint result;
	GtkWindow *window;
	GtkWidget *button;
	GtkWidget *message;
	gboolean hide = FALSE;

	if (!GTK_WIDGET_VISIBLE (dialog)) {
		gtk_widget_show (GTK_WIDGET (dialog));
		hide = TRUE;
	}

	if (gtk_window_get_transient_for (GTK_WINDOW (dialog)))
		window = gtk_window_get_transient_for (GTK_WINDOW (dialog));
	else
		window = GTK_WINDOW (dialog);

	message = gtk_message_dialog_new (window,
					  GTK_DIALOG_DESTROY_WITH_PARENT|
					  GTK_DIALOG_MODAL,
					  GTK_MESSAGE_QUESTION,
					  GTK_BUTTONS_NONE,
					  _("Some files don't have a suitable name for a Windows-compatible CD:"));

	gtk_message_dialog_format_secondary_text (GTK_MESSAGE_DIALOG (message),
						  _("Do you want to continue with Windows compatibility disabled?"));

	gtk_window_set_title (GTK_WINDOW (message), _("Windows compatibility"));
	gtk_dialog_add_buttons (GTK_DIALOG (message),
				GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL,
				NULL);

	button = bonfire_utils_make_button (_("Continue"), GTK_STOCK_OK);
	gtk_widget_show_all (button);
	gtk_dialog_add_action_widget (GTK_DIALOG (message),
				      button, GTK_RESPONSE_OK);

	result = gtk_dialog_run (GTK_DIALOG (message));
	gtk_widget_destroy (message);

	if (hide)
		gtk_widget_hide (GTK_WIDGET (dialog));

	if (result != GTK_RESPONSE_OK)
		return BONFIRE_BURN_CANCEL;

	return BONFIRE_BURN_OK;
}

static void
bonfire_burn_dialog_progress_changed_cb (BonfireBurn *burn, 
					 gdouble fraction,
					 long remaining,
					 BonfireBurnDialog *dialog)
{
	bonfire_tray_icon_set_progress (BONFIRE_TRAYICON (dialog->priv->tray),
					dialog->priv->on_the_fly ? fraction: (fraction - 0.5) * 2.0,
					remaining);
}

static void
bonfire_burn_dialog_action_changed_cb (BonfireBurn *burn, 
				       BonfireBurnAction action,
				       BonfireBurnDialog *dialog)
{
	bonfire_tray_icon_set_action (BONFIRE_TRAYICON (dialog->priv->tray),
				      action);
}

static void
bonfire_burn_dialog_init (BonfireBurnDialog * obj)
{
	GtkWidget *box;
	GtkWidget *cancel;
	GtkWidget *progress;

	obj->priv = g_new0 (BonfireBurnDialogPrivate, 1);
	gtk_dialog_set_has_separator (GTK_DIALOG (obj), FALSE);
	gtk_window_set_skip_taskbar_hint (GTK_WINDOW (obj), FALSE);
	gtk_window_set_skip_pager_hint (GTK_WINDOW (obj), FALSE);
	gtk_window_set_type_hint (GTK_WINDOW (obj), GDK_WINDOW_TYPE_HINT_NORMAL);
	gtk_window_set_position (GTK_WINDOW (obj),GTK_WIN_POS_CENTER);

	obj->priv->tray = bonfire_tray_icon_new ();
	g_signal_connect (obj->priv->tray,
			  "cancel",
			  G_CALLBACK (bonfire_burn_dialog_tray_cancel_cb),
			  obj);
	g_signal_connect (obj->priv->tray,
			  "show-dialog",
			  G_CALLBACK (bonfire_burn_dialog_tray_show_dialog_cb),
			  obj);
	g_signal_connect (obj->priv->tray,
			  "close-after",
			  G_CALLBACK (bonfire_burn_dialog_tray_close_after_cb),
			  obj);

	box = gtk_hbox_new (FALSE, 0);

	obj->priv->header = gtk_label_new (NULL);
	gtk_misc_set_alignment (GTK_MISC (obj->priv->header), 0.0, 0.5);
	gtk_misc_set_padding (GTK_MISC (obj->priv->header), 0, 18);
	gtk_box_pack_start (GTK_BOX (box), obj->priv->header, TRUE, TRUE, 0);

	obj->priv->image = gtk_image_new ();
	gtk_box_pack_start (GTK_BOX (box), obj->priv->image, FALSE, FALSE, 0);
	gtk_box_pack_start (GTK_BOX (GTK_DIALOG (obj)->vbox), box, TRUE, TRUE, 0);

	progress = bonfire_burn_progress_new ();
	gtk_box_pack_start (GTK_BOX (GTK_DIALOG (obj)->vbox),
			    progress,
			    FALSE,
			    FALSE,
			    0);

	obj->priv->close_check = gtk_check_button_new_with_label (_("Close the application when the disc is successfully burnt"));
	gtk_box_pack_end (GTK_BOX (progress),
			  obj->priv->close_check,
			  FALSE,
			  FALSE,
			  0);

	/* Create tracks */
	obj->priv->burn = bonfire_burn_new ();
	bonfire_burn_plug_progress_widget (obj->priv->burn,
					   BONFIRE_BURN_PROGRESS (progress));

	g_signal_connect (obj->priv->burn,
			  "insert_media",
			  G_CALLBACK (bonfire_burn_dialog_insert_disc_cb),
			  obj);
	g_signal_connect (obj->priv->burn,
			  "warn_data_loss",
			  G_CALLBACK (bonfire_burn_dialog_data_loss_cb),
			  obj);
	g_signal_connect (obj->priv->burn,
			  "warn_rewritable",
			  G_CALLBACK (bonfire_burn_dialog_rewritable_cb),
			  obj);
	g_signal_connect (obj->priv->burn,
			  "disable_joliet",
			  G_CALLBACK (bonfire_burn_dialog_disable_joliet_cb),
			  obj);
	g_signal_connect (obj->priv->burn,
			  "progress_changed",
			  G_CALLBACK (bonfire_burn_dialog_progress_changed_cb),
			  obj);
	g_signal_connect (obj->priv->burn,
			  "action_changed",
			  G_CALLBACK (bonfire_burn_dialog_action_changed_cb),
			  obj);

	/* buttons */
	cancel = gtk_button_new_from_stock (GTK_STOCK_CANCEL);
	g_signal_connect (G_OBJECT (cancel),
			  "clicked",
			  G_CALLBACK (bonfire_burn_dialog_cancel_clicked_cb),
			  obj);
	gtk_dialog_add_action_widget (GTK_DIALOG (obj), cancel, GTK_RESPONSE_CANCEL);
}

static void
bonfire_burn_dialog_destroy (GtkObject * object)
{
	BonfireBurnDialog *cobj;

	cobj = BONFIRE_BURN_DIALOG (object);

	if (cobj->priv->burn) {
		g_object_unref (cobj->priv->burn);
		cobj->priv->burn = NULL;
	}

	if (GTK_OBJECT_CLASS (parent_class)->destroy)
		GTK_OBJECT_CLASS (parent_class)->destroy (object);
}

static void
bonfire_burn_dialog_finalize (GObject * object)
{
	BonfireBurnDialog *cobj;

	cobj = BONFIRE_BURN_DIALOG (object);

	if (cobj->priv->burn) {
		g_object_unref (cobj->priv->burn);
		cobj->priv->burn = NULL;
	}

	if (cobj->priv->tray) {
		gtk_widget_destroy (cobj->priv->tray);
		cobj->priv->tray = NULL;
	}

	g_free (cobj->priv);
	G_OBJECT_CLASS (parent_class)->finalize (object);
}

GtkWidget *
bonfire_burn_dialog_new (void)
{
	BonfireBurnDialog *obj;

	obj = BONFIRE_BURN_DIALOG (g_object_new (BONFIRE_TYPE_BURN_DIALOG, NULL));

	return GTK_WIDGET (obj);
}

static void
bonfire_burn_dialog_update_info (BonfireBurnDialog *dialog,
				 NautilusBurnDrive *drive)
{
	char *title = NULL;
	char *header = NULL;
	GdkPixbuf *pixbuf = NULL;
	NautilusBurnMediaType media_type;

	char *types [] = { 	NULL,
				NULL,
				NULL,
				"gnome-dev-cdrom",
				"gnome-dev-disc-cdr",
				"gnome-dev-disc-cdrw",
				"gnome-dev-disc-dvdrom",
				"gnome-dev-disc-dvdr",
				"gnome-dev-disc-dvdrw",
				"gnome-dev-disc-dvdram",
				"gnome-dev-disc-dvdr-plus",
				"gnome-dev-disc-dvdrw", /* FIXME */
				"gnome-dev-disc-dvdr-plus" /* FIXME */,
				NULL };

	media_type = nautilus_burn_drive_get_media_type (drive);
	if (media_type > NAUTILUS_BURN_MEDIA_TYPE_CDRW) {
		if (dialog->priv->track_type == BONFIRE_TRACK_SOURCE_DATA
		||  dialog->priv->track_type == BONFIRE_TRACK_SOURCE_ISO
		||  dialog->priv->track_type == BONFIRE_TRACK_SOURCE_ISO_JOLIET
		||  dialog->priv->track_type == BONFIRE_TRACK_SOURCE_GRAFTS) {
			title = g_strdup (_("Burning DVD"));
			header = g_strdup_printf ("<big><b>Burning data DVD</b></big>");
		}
		else if (dialog->priv->track_type == BONFIRE_TRACK_SOURCE_DISC) {
			title = g_strdup (_("Copying DVD"));
			header = g_strdup_printf ("<big><b>Copying data DVD</b></big>");
		}
	}
	else if (dialog->priv->track_type == BONFIRE_TRACK_SOURCE_AUDIO
	      ||  dialog->priv->track_type == BONFIRE_TRACK_SOURCE_SONG) {
		title = g_strdup (_("Burning CD"));
		header = g_strdup_printf ("<big><b>Burning audio CD</b></big>");
		pixbuf = bonfire_utils_get_icon ("gnome-dev-cdrom-audio", 48);
	}
	else if (dialog->priv->track_type == BONFIRE_TRACK_SOURCE_DATA
	      ||  dialog->priv->track_type == BONFIRE_TRACK_SOURCE_ISO
	      ||  dialog->priv->track_type == BONFIRE_TRACK_SOURCE_ISO_JOLIET
	      ||  dialog->priv->track_type == BONFIRE_TRACK_SOURCE_GRAFTS) {
		title = g_strdup (_("Burning CD"));
		header = g_strdup_printf ("<big><b>Burning data CD</b></big>");
	}
	else if (dialog->priv->track_type == BONFIRE_TRACK_SOURCE_DISC) {
		title = g_strdup (_("Copying CD"));
		header = g_strdup_printf ("<big><b>Copying CD</b></big>");
	}
	else if (dialog->priv->track_type == BONFIRE_TRACK_SOURCE_CUE
	      ||  dialog->priv->track_type == BONFIRE_TRACK_SOURCE_RAW) {
		title = g_strdup (_("Burning CD"));
		header = g_strdup_printf ("<big><b>Burning disc image</b></big>");
	}

	gtk_window_set_title (GTK_WINDOW (dialog), title);
	g_free (title);

	if (!pixbuf) {
		pixbuf = bonfire_utils_get_icon (types [media_type], 48);
		if (!pixbuf)
			pixbuf = bonfire_utils_get_icon ("gnome-dev-removable", 48);
	}

	gtk_image_set_from_pixbuf (GTK_IMAGE (dialog->priv->image), pixbuf);
	g_object_unref (pixbuf);

	gtk_label_set_markup (GTK_LABEL (dialog->priv->header), header);
	g_free (header);
}

static void
bonfire_burn_dialog_media_added_cb (NautilusBurnDrive *drive,
				    BonfireBurnDialog *dialog)
{
	bonfire_burn_dialog_update_info (dialog, drive);

	/* we might have a dialog waiting for the 
	 * insertion of a disc if so close it */
	if (dialog->priv->waiting_disc_dialog) {
		gtk_dialog_response (GTK_DIALOG (dialog->priv->waiting_disc_dialog),
				     GTK_RESPONSE_OK);
	}
}

static void
bonfire_burn_dialog_media_removed_cb (NautilusBurnDrive *drive,
				      BonfireBurnDialog *dialog)
{
	GdkPixbuf *pixbuf;

	pixbuf = bonfire_utils_get_icon ("gnome-dev-removable", 48);
	gtk_image_set_from_pixbuf (GTK_IMAGE (dialog->priv->image), pixbuf);
	g_object_unref (pixbuf);
}

#ifdef HAVE_LIBNOTIFY
static void
bonfire_burn_dialog_notify_daemon_close (NotifyNotification *notification,
					 BonfireBurnDialog *dialog)
{
	GtkWindow *window;

	window = gtk_window_get_transient_for (GTK_WINDOW (dialog));
	if (!gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (dialog->priv->close_check))) {
		gtk_widget_destroy (GTK_WIDGET (dialog));
		gtk_widget_show (GTK_WIDGET (window));
	}
	else
		gtk_widget_destroy (GTK_WIDGET (window));
}

gboolean
bonfire_burn_dialog_notify_daemon (BonfireBurnDialog *dialog,
				   const char *primary,
				   const char *secondary)
{
	NotifyNotification *notification;
	gboolean result;

	notification = notify_notification_new (primary,
						secondary,
						GTK_STOCK_CDROM,
						dialog->priv->tray);
	if (!notification)
		return FALSE;

	g_signal_connect (notification,
			  "closed",
			  G_CALLBACK (bonfire_burn_dialog_notify_daemon_close),
			  dialog);

	notify_notification_set_timeout (notification, 10000);
	notify_notification_set_urgency (notification, NOTIFY_URGENCY_NORMAL);
	result = notify_notification_show (notification, NULL);

	return result;
}
#endif

static void
bonfire_burn_dialog_notify_dialog (BonfireBurnDialog *dialog,
				   const char *primary,
				   const char *secondary,
				   GtkMessageType type)
{
	GtkWidget *message;

	if (!GTK_WIDGET_VISIBLE (dialog))
		gtk_widget_show (GTK_WIDGET (dialog));

	message = gtk_message_dialog_new (GTK_WINDOW (dialog),
					  GTK_DIALOG_DESTROY_WITH_PARENT |
					  GTK_DIALOG_MODAL,
					  type,
					  GTK_BUTTONS_CLOSE,
					  primary);

	gtk_message_dialog_format_secondary_text (GTK_MESSAGE_DIALOG (message),
						  "%s.",
						  secondary);
	gtk_dialog_run (GTK_DIALOG (message));
	gtk_widget_destroy (message);
}

static void
bonfire_burn_dialog_notify_error (BonfireBurnDialog *dialog,
				  GError *error)
{
	char *secondary;

	if (error) {
		secondary =  g_strdup (error->message);
		g_error_free (error);
	}
	else
		secondary = g_strdup (_("An unknown error occured. Check your disc"));

	bonfire_burn_dialog_notify_dialog (dialog,
					   _("Error while burning:"),
					   secondary,
					   GTK_MESSAGE_ERROR);

	g_free (secondary);
}

static void
bonfire_burn_dialog_notify_success (BonfireBurnDialog *dialog,
				    const BonfireTrackSource *track,
				    gboolean to_disc)
{
	char *primary = NULL;
	char *secondary = NULL;

	switch (track->type) {
	case BONFIRE_TRACK_SOURCE_SONG:
		primary = g_strdup (_("Audio disc successfully burnt:"));
		secondary = g_strdup_printf (_("\"%s\" is now ready for use"), track->contents.songs.album);
		break;
	case BONFIRE_TRACK_SOURCE_DISC:
		if (to_disc) {
			primary = g_strdup (_("Disc successfully copied:"));
			secondary = g_strdup_printf (_("disc is now ready for use"));
		}
		else {
			primary = g_strdup (_("Disc successfully copied to disc:"));
			secondary = g_strdup_printf (_("Image is now ready for use"));
		}
		break;
	case BONFIRE_TRACK_SOURCE_ISO:
	case BONFIRE_TRACK_SOURCE_RAW:
	case BONFIRE_TRACK_SOURCE_ISO_JOLIET:
	case BONFIRE_TRACK_SOURCE_CUE:
		primary = g_strdup (_("Image successfully burnt to disc:"));
		secondary = g_strdup_printf (_("Image was successfully burnt"));
		break;
	default:
		if (to_disc) {
			primary = g_strdup (_("Data disc successfully burnt:"));
			secondary = g_strdup_printf (_("\"%s\" is now ready for use"), track->contents.data.label);
		}
		else {
			primary = g_strdup (_("Image successfully created on disc:"));
			secondary = g_strdup_printf (_("\"%s\" is now ready for use"), track->contents.data.label);
		}
		break;
	}

#ifdef HAVE_LIBNOTIFY
	if (GTK_WIDGET_VISIBLE (dialog))
		bonfire_burn_dialog_notify_dialog (dialog,
						   primary,
						   secondary,
						   GTK_MESSAGE_INFO);
	else if (!bonfire_burn_dialog_notify_daemon (dialog,
						     primary,
						     secondary))
#endif
		bonfire_burn_dialog_notify_dialog (dialog,
						   primary,
						   secondary,
						   GTK_MESSAGE_INFO);
	g_free (primary);
	g_free (secondary);
}

void
bonfire_burn_dialog_run (BonfireBurnDialog *dialog,
			 NautilusBurnDrive *drive,
			 gint speed,
			 const char *output,
			 const BonfireTrackSource *track,
			 BonfireBurnFlag flags)
{
	int added_id;
	int removed_id;
	GdkCursor *cursor;
	GError *error = NULL;
	BonfireBurnResult result;

	gtk_widget_show_all (dialog->priv->tray);

	dialog->priv->track_type = track->type;
	dialog->priv->on_the_fly = ((flags & BONFIRE_BURN_FLAG_ON_THE_FLY) != 0);

	if (debug)
		flags |= BONFIRE_BURN_FLAG_DEBUG;

	nautilus_burn_drive_ref (drive);
	added_id = g_signal_connect (drive,
				     "media-added",
				     G_CALLBACK (bonfire_burn_dialog_media_added_cb),
				     dialog);
	removed_id = g_signal_connect (drive,
				       "media-removed",
				       G_CALLBACK (bonfire_burn_dialog_media_removed_cb),
				       dialog);
	bonfire_burn_dialog_update_info (dialog, drive);

	cursor = gdk_cursor_new (GDK_WATCH);
	gdk_window_set_cursor (GTK_WIDGET (dialog)->window, cursor);
	gdk_cursor_unref (cursor);

	result = bonfire_burn_record (dialog->priv->burn,
				      flags,
				      drive,
				      speed,
				      track,
				      output,
				      &error);

	/* tell the user the result of the operation */
	if (error || result == BONFIRE_BURN_ERR)
		bonfire_burn_dialog_notify_error (dialog, error);
	else if (result == BONFIRE_BURN_OK)
		bonfire_burn_dialog_notify_success (dialog,
						    track,
						    (drive->type != NAUTILUS_BURN_DRIVE_TYPE_FILE));

	if (added_id) {
		g_signal_handler_disconnect (drive, added_id);
		added_id = 0;
	}

	if (removed_id) {
		g_signal_handler_disconnect (drive, removed_id);
		removed_id = 0;
	}

	nautilus_burn_drive_unref (drive);

	if (result != BONFIRE_BURN_OK
	||  GTK_WIDGET_VISIBLE (GTK_WIDGET (dialog))) {
		GtkWindow *window;

		window = gtk_window_get_transient_for (GTK_WINDOW (dialog));

		if (!gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (dialog->priv->close_check))
		||  result != BONFIRE_BURN_OK) {
			gtk_widget_destroy (GTK_WIDGET (dialog));
			gtk_widget_show (GTK_WIDGET (window));
		}
		else
			gtk_widget_destroy (GTK_WIDGET (window));
	}
}

static gboolean
bonfire_burn_dialog_cancel_dialog (GtkWidget *toplevel)
{
	gint result;
	GtkWidget *button;
	GtkWidget *message;

	message = gtk_message_dialog_new (GTK_WINDOW (toplevel),
					  GTK_DIALOG_DESTROY_WITH_PARENT |
					  GTK_DIALOG_MODAL,
					  GTK_MESSAGE_QUESTION,
					  GTK_BUTTONS_NONE,
					  _("Do you really want to quit?"));

	gtk_window_set_title (GTK_WINDOW (message), _("Confirm"));

	gtk_message_dialog_format_secondary_text (GTK_MESSAGE_DIALOG
						  (message),
						  _("Interrupting the process may make disc unusable."));

	gtk_dialog_add_buttons (GTK_DIALOG (message),
				GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL,
				NULL);

	button = bonfire_utils_make_button (_("Continue"), GTK_STOCK_OK);
	gtk_widget_show_all (button);
	gtk_dialog_add_action_widget (GTK_DIALOG (message),
				      button, GTK_RESPONSE_OK);

	result = gtk_dialog_run (GTK_DIALOG (message));
	gtk_widget_destroy (message);

	return (result != GTK_RESPONSE_OK);
}

static gboolean
bonfire_burn_dialog_cancel (BonfireBurnDialog *dialog)
{
	if (dialog->priv->burn) {
		BonfireBurnResult result;

		result = bonfire_burn_cancel (dialog->priv->burn, TRUE);

		if (result == BONFIRE_BURN_DANGEROUS) {
			if (bonfire_burn_dialog_cancel_dialog (GTK_WIDGET (dialog)))
				bonfire_burn_cancel (dialog->priv->burn, FALSE);
			else
				return FALSE;
		}
	}
	else
		gtk_widget_destroy (GTK_WIDGET (dialog));

	return TRUE;
}

static gboolean
bonfire_burn_dialog_delete (GtkWidget *widget, 
			    GdkEventAny *event)
{
	BonfireBurnDialog *dialog;

	dialog = BONFIRE_BURN_DIALOG (widget);

	bonfire_tray_icon_set_show_dialog (BONFIRE_TRAYICON (dialog->priv->tray), FALSE);
 	return TRUE;
}

static void
bonfire_burn_dialog_cancel_clicked_cb (GtkWidget *button,
				       BonfireBurnDialog *dialog)
{
	bonfire_burn_dialog_cancel (dialog);
}

static void
bonfire_burn_dialog_tray_cancel_cb (BonfireTrayIcon *tray,
				    BonfireBurnDialog *dialog)
{
	bonfire_burn_dialog_cancel (dialog);
}

static void
bonfire_burn_dialog_tray_show_dialog_cb (BonfireTrayIcon *tray,
					 gboolean show,
					 GtkWidget *dialog)
{
	if (show)
		gtk_widget_show (dialog);
	else
		gtk_widget_hide (dialog);
}

static void
bonfire_burn_dialog_tray_close_after_cb (BonfireTrayIcon *tray,
					 gboolean close,
					 BonfireBurnDialog *dialog)
{
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (dialog->priv->close_check), close);
}
