/***************************************************************************
 *            burn-options-dialog.c
 *
 *  mer mar 29 13:47:56 2006
 *  Copyright  2006  Rouquier Philippe
 *  bonfire-app@wanadoo.fr
 ***************************************************************************/

/*
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <glib.h>
#include <glib/gi18n-lib.h>
#include <glib-object.h>

#include <gtk/gtkdialog.h>
#include <gtk/gtkwindow.h>
#include <gtk/gtkvbox.h>
#include <gtk/gtkbutton.h>

#include "utils.h"
#include "burn-basics.h"
#include "burn-options-dialog.h"
#include "recorder-selection.h"
#include "burn-dialog.h"

static void bonfire_burn_option_dialog_class_init (BonfireBurnOptionDialogClass *klass);
static void bonfire_burn_option_dialog_init (BonfireBurnOptionDialog *sp);
static void bonfire_burn_option_dialog_finalize (GObject *object);

static void
bonfire_burn_option_dialog_dont_close_toggled_cb (GtkToggleButton *toggle,
						  BonfireBurnOptionDialog *dialog);
static void
bonfire_burn_option_dialog_append_toggled_cb (GtkToggleButton *toggle,
					      BonfireBurnOptionDialog *dialog);
static void
bonfire_burn_option_dialog_dao_toggled_cb (GtkToggleButton *toggle,
					   BonfireBurnOptionDialog *dialog);
static void
bonfire_burn_option_dialog_cancel_clicked_cb (GtkButton *button,
					      GtkWidget *dialog);
static void
bonfire_burn_option_dialog_burn_clicked_cb (GtkWidget *button,
					    BonfireBurnOptionDialog *dialog);

struct BonfireBurnOptionDialogPrivate {
	GtkWidget *selection;
	GtkWidget *label;

	BonfireTrackSource *track;
	BonfireBurnFlag flags;
};

static GObjectClass *parent_class = NULL;

GType
bonfire_burn_option_dialog_get_type ()
{
	static GType type = 0;

	if(type == 0) {
		static const GTypeInfo our_info = {
			sizeof (BonfireBurnOptionDialogClass),
			NULL,
			NULL,
			(GClassInitFunc)bonfire_burn_option_dialog_class_init,
			NULL,
			NULL,
			sizeof (BonfireBurnOptionDialog),
			0,
			(GInstanceInitFunc)bonfire_burn_option_dialog_init,
		};

		type = g_type_register_static(GTK_TYPE_DIALOG,
					      "BonfireBurnOptionDialog",
					      &our_info,
					      0);
	}

	return type;
}

static void
bonfire_burn_option_dialog_class_init (BonfireBurnOptionDialogClass *klass)
{
	GObjectClass *object_class = G_OBJECT_CLASS (klass);

	parent_class = g_type_class_peek_parent (klass);
	object_class->finalize = bonfire_burn_option_dialog_finalize;
}

static void
bonfire_burn_option_dialog_init (BonfireBurnOptionDialog *obj)
{
	GtkWidget *burn;
	GtkWidget *cancel;

	obj->priv = g_new0 (BonfireBurnOptionDialogPrivate, 1);
	gtk_dialog_set_has_separator (GTK_DIALOG (obj), FALSE);

	/* first box */
	obj->priv->selection = bonfire_recorder_selection_new ();
	g_object_set (G_OBJECT (obj->priv->selection),
		      "file-image", TRUE,
		      "show-properties", TRUE,
		      "show-recorders-only", TRUE,
		      NULL);

	gtk_box_pack_start (GTK_BOX (GTK_DIALOG (obj)->vbox),
			    bonfire_utils_pack_properties (_("<b>Select a drive to write to</b>"),
							   obj->priv->selection,
							   NULL),
			    FALSE,
			    FALSE,
			    0);

	bonfire_recorder_selection_select_default_drive (BONFIRE_RECORDER_SELECTION (obj->priv->selection),
							 BONFIRE_MEDIA_WRITABLE);

	/* buttons */
	cancel = gtk_button_new_from_stock (GTK_STOCK_CANCEL);
	g_signal_connect (G_OBJECT (cancel),
			  "clicked",
			  G_CALLBACK (bonfire_burn_option_dialog_cancel_clicked_cb),
			  obj);
	gtk_dialog_add_action_widget (GTK_DIALOG (obj), cancel, GTK_RESPONSE_CANCEL);

	burn = bonfire_utils_make_button (_("Burn"), GTK_STOCK_CDROM);
	g_signal_connect (G_OBJECT (burn),
			  "clicked",
			  G_CALLBACK (bonfire_burn_option_dialog_burn_clicked_cb),
			  obj);
	gtk_dialog_add_action_widget (GTK_DIALOG (obj), burn, GTK_RESPONSE_OK);
}

static void
bonfire_burn_option_dialog_finalize (GObject *object)
{
	BonfireBurnOptionDialog *cobj;

	cobj = BONFIRE_BURN_OPTION_DIALOG (object);

	if (cobj->priv->track) {
		bonfire_track_source_free (cobj->priv->track);
		cobj->priv->track = NULL;
	}

	g_free (cobj->priv);
	G_OBJECT_CLASS (parent_class)->finalize (object);
}

static void
bonfire_burn_option_set_title_widget (BonfireBurnOptionDialog *dialog,
				      gboolean is_data)
{
	GtkWidget *widget;
	char buffer [128];
	char *title_str;
	char *label;
	time_t t;

	dialog->priv->label = gtk_entry_new ();
	gtk_entry_set_max_length(GTK_ENTRY (dialog->priv->label), 32);

	/* Header : This must be less that 32 characters long */
	t = time (NULL);
	strftime (buffer, sizeof (buffer), "%d %b %y", localtime (&t));

	if (is_data)
		title_str = g_strdup_printf (_("Data disc (%s)"), buffer);
	else
		title_str = g_strdup_printf (_("Audio disc (%s)"), buffer);

	gtk_entry_set_text (GTK_ENTRY (dialog->priv->label), title_str);
	g_free (title_str);

	if (is_data)
		label = g_strdup (_("<b>Label of the disc</b>"));
	else
		label = g_strdup (_("<b>Title</b>"));
	widget = bonfire_utils_pack_properties (label,
						dialog->priv->label,
						NULL);
	g_free (label);

	gtk_box_pack_start (GTK_BOX (GTK_DIALOG (dialog)->vbox),
			    widget,
			    TRUE,
			    TRUE,
			    0);
	gtk_box_reorder_child (GTK_BOX (GTK_DIALOG (dialog)->vbox), widget, 0);
}

GtkWidget *
bonfire_burn_option_dialog_new (const BonfireTrackSource *track,
				gboolean overburn)
{
	BonfireBurnOptionDialog *obj;
	GtkWidget *widget = NULL;
	
	obj = BONFIRE_BURN_OPTION_DIALOG (g_object_new (BONFIRE_TYPE_BURN_OPTION_DIALOG,
							"title", _("Disc burning setup"),
							NULL));

	bonfire_recorder_selection_set_track_type (BONFIRE_RECORDER_SELECTION (obj->priv->selection), track->type);

	/* keep a copy of the track for later */
	obj->priv->track = bonfire_track_source_copy (track);
	if (track->type == BONFIRE_TRACK_SOURCE_SONG) {
		GtkWidget *toggle;

		bonfire_burn_option_set_title_widget (obj, FALSE);

		toggle = gtk_check_button_new_with_label (_("Leave a two second pause between tracks"));
		g_signal_connect (G_OBJECT (toggle),
				  "toggled",
				  G_CALLBACK (bonfire_burn_option_dialog_dao_toggled_cb),
				  obj);

		widget = bonfire_utils_pack_properties (_("<b>Option</b>"),
							toggle,
							NULL);
		g_object_set (obj->priv->selection, "file-image", FALSE, NULL);
	}
	else if (track->type == BONFIRE_TRACK_SOURCE_DATA) {
		GtkWidget *dont_close_check;
		GtkWidget *append_check;

		bonfire_burn_option_set_title_widget (obj, TRUE);

		append_check = gtk_check_button_new_with_label (_("Append the files to those already on the disc"));
		g_signal_connect (G_OBJECT (append_check),
				  "toggled",
				  G_CALLBACK (bonfire_burn_option_dialog_append_toggled_cb),
				  obj);

		dont_close_check = gtk_check_button_new_with_label (_("Leave the disc open to add other files later"));
		g_signal_connect (G_OBJECT (dont_close_check),
				  "toggled",
				  G_CALLBACK (bonfire_burn_option_dialog_dont_close_toggled_cb),
				  obj);

		widget = bonfire_utils_pack_properties (_("<b>Options</b>"),
							gtk_label_new (NULL),
							append_check,
							dont_close_check,
							NULL);

		g_object_set (obj->priv->selection, "file-image", TRUE, NULL);
	}
	else
		g_object_set (obj->priv->selection, "file-image", FALSE, NULL);


	if (widget) {
		gtk_box_pack_end (GTK_BOX (GTK_DIALOG (obj)->vbox),
				  widget,
				  FALSE,
				  FALSE,
				  0);
		gtk_widget_show_all (widget);
	}

	if (overburn)
		obj->priv->flags = BONFIRE_BURN_FLAG_OVERBURN;

	return GTK_WIDGET (obj);
}

static void
bonfire_burn_option_dialog_dao_toggled_cb (GtkToggleButton *toggle,
					   BonfireBurnOptionDialog *dialog)
{
	GSList *iter;
	gboolean value;

	g_return_if_fail (dialog->priv->track != NULL);

	value = gtk_toggle_button_get_active (toggle);
	for (iter = dialog->priv->track->contents.songs.files; iter; iter = iter->next) {
		BonfireSongFile *song;

		song = iter->data;
		song->gap = value;
	}
}

static void
bonfire_burn_option_dialog_append_toggled_cb (GtkToggleButton *toggle,
					      BonfireBurnOptionDialog *dialog)
{
	if (gtk_toggle_button_get_active (toggle))
		dialog->priv->flags |= BONFIRE_BURN_FLAG_MERGE;
	else
		dialog->priv->flags &= ~BONFIRE_BURN_FLAG_MERGE;
}

static void
bonfire_burn_option_dialog_dont_close_toggled_cb (GtkToggleButton *toggle,
						  BonfireBurnOptionDialog *dialog)
{
	if (gtk_toggle_button_get_active (toggle))
		dialog->priv->flags |= BONFIRE_BURN_FLAG_DONT_CLOSE;
	else
		dialog->priv->flags &= ~BONFIRE_BURN_FLAG_DONT_CLOSE;
}

static void
bonfire_burn_option_dialog_burn_clicked_cb (GtkWidget *button, BonfireBurnOptionDialog *dialog)
{
	BonfireBurnFlag burn_flags;
	BonfireTrackSource *track;
	NautilusBurnDrive *drive;
	BonfireDriveProp props;
	GtkWidget *burn_dialog;
	GtkWindow *window;
	char *output;
	int speed;

	bonfire_recorder_selection_get_drive (BONFIRE_RECORDER_SELECTION (dialog->priv->selection),
					      &drive,
					      &props);

	burn_flags = BONFIRE_BURN_FLAG_DONT_OVERWRITE;

	if (drive->type == NAUTILUS_BURN_DRIVE_TYPE_FILE) {
		output = props.output_path;
		speed = 0;
	}
	else {
		burn_flags |= props.flags | dialog->priv->flags;
		speed = props.props.drive_speed;
		output = NULL;
	}

	if (dialog->priv->track->type == BONFIRE_TRACK_SOURCE_DATA)
		dialog->priv->track->contents.data.label = g_strdup (gtk_entry_get_text (GTK_ENTRY (dialog->priv->label)));
	else if (dialog->priv->track->type == BONFIRE_TRACK_SOURCE_SONG)
		dialog->priv->track->contents.songs.album = g_strdup (gtk_entry_get_text (GTK_ENTRY (dialog->priv->label)));

	burn_dialog = bonfire_burn_dialog_new ();

	window = gtk_window_get_transient_for (GTK_WINDOW (dialog));
	track = dialog->priv->track;
	dialog->priv->track = NULL;

	gtk_widget_hide (GTK_WIDGET (window));
	gtk_widget_destroy (GTK_WIDGET (dialog));

	gtk_window_set_transient_for (GTK_WINDOW (burn_dialog), window);
	gtk_window_set_modal (GTK_WINDOW (burn_dialog), TRUE);
	gtk_widget_show_all (burn_dialog);

	bonfire_burn_dialog_run (BONFIRE_BURN_DIALOG (burn_dialog),
				 drive,
				 speed,
				 output,
				 track,
				 burn_flags);

	/* NOTE: burn dialog self destroys itself */
	/* gtk_widget_destroy (GTK_WIDGET (burn_dialog)); */

	g_free (output);
	nautilus_burn_drive_unref (drive);
}

static void
bonfire_burn_option_dialog_cancel_clicked_cb (GtkButton *cancel, GtkWidget *dialog)
{
	gtk_widget_destroy (dialog);
}
