/***************************************************************************
 *            disc-copy-dialog.c
 *
 *  ven jui 15 16:02:10 2005
 *  Copyright  2005  Philippe Rouquier
 *  bonfire-app@wanadoo.fr
 ***************************************************************************/

/*
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */


#include <stdlib.h>
#include <unistd.h>
#include <string.h>

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <glib.h>
#include <glib/gi18n-lib.h>
#include <glib/gstdio.h>

#include <gdk/gdkcursor.h>

#include <gtk/gtkvbox.h>
#include <gtk/gtkwindow.h>

#include <nautilus-burn-drive.h>

#include "burn-basics.h"
#include "utils.h"
#include "burn-dialog.h"
#include "disc-copy-dialog.h"
#include "recorder-selection.h"

static void bonfire_disc_copy_dialog_class_init (BonfireDiscCopyDialogClass *klass);
static void bonfire_disc_copy_dialog_init (BonfireDiscCopyDialog *sp);
static void bonfire_disc_copy_dialog_finalize (GObject *object);

static void
bonfire_disc_copy_dialog_src_media_changed (BonfireRecorderSelection *src_selection,
					    NautilusBurnMediaType media_type,
					    BonfireDiscCopyDialog *dialog);
static void
bonfire_disc_copy_dialog_burn_clicked_cb (GtkWidget *button,
					  BonfireDiscCopyDialog *dialog);
static void
bonfire_disc_copy_dialog_cancel_clicked_cb (GtkWidget *button,
					    BonfireDiscCopyDialog *dialog);

struct BonfireDiscCopyDialogPrivate {
	GtkWidget *selection;
	GtkWidget *source;
};

static GObjectClass *parent_class = NULL;

GType
bonfire_disc_copy_dialog_get_type ()
{
	static GType type = 0;

	if (type == 0) {
		static const GTypeInfo our_info = {
			sizeof (BonfireDiscCopyDialogClass),
			NULL,
			NULL,
			(GClassInitFunc) bonfire_disc_copy_dialog_class_init,
			NULL,
			NULL,
			sizeof (BonfireDiscCopyDialog),
			0,
			(GInstanceInitFunc) bonfire_disc_copy_dialog_init,
		};

		type = g_type_register_static (GTK_TYPE_DIALOG,
					       "BonfireDiscCopyDialog",
					       &our_info,
					       0);
	}

	return type;
}

static void
bonfire_disc_copy_dialog_class_init (BonfireDiscCopyDialogClass * klass)
{
	GObjectClass *object_class = G_OBJECT_CLASS (klass);

	parent_class = g_type_class_peek_parent (klass);
	object_class->finalize = bonfire_disc_copy_dialog_finalize;
}

static void
bonfire_disc_copy_dialog_init (BonfireDiscCopyDialog * obj)
{
	GtkWidget *burn;
	GtkWidget *cancel;
	NautilusBurnDrive *drive = NULL;

	obj->priv = g_new0 (BonfireDiscCopyDialogPrivate, 1);
	gtk_dialog_set_has_separator (GTK_DIALOG (obj), FALSE);

	gtk_window_set_title (GTK_WINDOW (obj), _("CD/DVD copy options"));

	/* take care of source media */
	obj->priv->source = bonfire_recorder_selection_new ();
	g_object_set (G_OBJECT (obj->priv->source),
		      "show-properties", FALSE,
		      "show-recorders-only", FALSE,
		      NULL);
	gtk_box_pack_start (GTK_BOX (GTK_DIALOG (obj)->vbox),
			    bonfire_utils_pack_properties (_("<b>Select source drive to copy</b>"),
							   obj->priv->source,
							   NULL),
			    FALSE,
			    FALSE,
			    6);

	bonfire_recorder_selection_select_default_drive (BONFIRE_RECORDER_SELECTION (obj->priv->source),
							 BONFIRE_MEDIA_WITH_DATA);

	/* destination drive */
	obj->priv->selection = bonfire_recorder_selection_new ();
	bonfire_recorder_selection_set_track_type (BONFIRE_RECORDER_SELECTION (obj->priv->selection), BONFIRE_TRACK_SOURCE_DISC);
	g_object_set (G_OBJECT (obj->priv->selection),
		      "file-image", TRUE,
		      "show-recorders-only", TRUE,
		      NULL);

	gtk_box_pack_start (GTK_BOX (GTK_DIALOG (obj)->vbox),
			    bonfire_utils_pack_properties (_("<b>Select a drive to write to</b>"),
							   obj->priv->selection,
							   NULL),
			    FALSE,
			    FALSE,
			    6);

	bonfire_recorder_selection_select_default_drive (BONFIRE_RECORDER_SELECTION (obj->priv->selection),
							 BONFIRE_MEDIA_WRITABLE);

	/* tell the destination what type of media to expect from source */
	g_signal_connect (G_OBJECT (obj->priv->source),
			  "media-changed",
			  G_CALLBACK (bonfire_disc_copy_dialog_src_media_changed),
			  obj);

	bonfire_recorder_selection_get_drive (BONFIRE_RECORDER_SELECTION (obj->priv->source),
					      &drive,
					      NULL);
	if (drive && drive->type != NAUTILUS_BURN_DRIVE_TYPE_FILE) {
		NautilusBurnMediaType src_media_type;

		src_media_type = nautilus_burn_drive_get_media_type (drive);
		bonfire_recorder_selection_set_src_media_type (BONFIRE_RECORDER_SELECTION (obj->priv->selection),
							       src_media_type);
		nautilus_burn_drive_unref (drive);
	}

	/* buttons */
	cancel = gtk_button_new_from_stock (GTK_STOCK_CANCEL);
	g_signal_connect (G_OBJECT (cancel),
			  "clicked",
			  G_CALLBACK (bonfire_disc_copy_dialog_cancel_clicked_cb),
			  obj);
	gtk_dialog_add_action_widget (GTK_DIALOG (obj), cancel, GTK_RESPONSE_CANCEL);

	burn = bonfire_utils_make_button (_("Copy"), GTK_STOCK_CDROM);
	g_signal_connect (G_OBJECT (burn),
			  "clicked",
			  G_CALLBACK (bonfire_disc_copy_dialog_burn_clicked_cb),
			  obj);
	gtk_dialog_add_action_widget (GTK_DIALOG (obj), burn, GTK_RESPONSE_OK);
}

static void
bonfire_disc_copy_dialog_finalize (GObject *object)
{
	BonfireDiscCopyDialog *cobj;

	cobj = BONFIRE_DISC_COPY_DIALOG (object);

	g_free (cobj->priv);
	G_OBJECT_CLASS (parent_class)->finalize (object);
}

GtkWidget *
bonfire_disc_copy_dialog_new ()
{
	BonfireDiscCopyDialog *obj;

	obj = BONFIRE_DISC_COPY_DIALOG (g_object_new (BONFIRE_TYPE_DISC_COPY_DIALOG,
				       NULL));

	return GTK_WIDGET (obj);
}

static void
bonfire_disc_copy_dialog_src_media_changed (BonfireRecorderSelection *src_selection,
					    NautilusBurnMediaType media_type,
					    BonfireDiscCopyDialog *dialog)
{
	bonfire_recorder_selection_set_src_media_type (BONFIRE_RECORDER_SELECTION (dialog->priv->selection),
						       media_type);
}

static void
bonfire_disc_copy_dialog_cancel_clicked_cb (GtkWidget *button,
					    BonfireDiscCopyDialog *dialog)
{
	gtk_widget_destroy (GTK_WIDGET (dialog));
}

static void
bonfire_disc_copy_dialog_burn_clicked_cb (GtkWidget *button,
					  BonfireDiscCopyDialog *dialog)
{
	gint speed;
	GtkWindow *window;
	char *output = NULL;
	GtkWidget *burn_dialog;
	BonfireBurnFlag flags;
	BonfireDriveProp props;
	BonfireTrackSource *track;
	NautilusBurnDrive *drive_dest = NULL;
	NautilusBurnDrive *drive_source = NULL;

	/* get drives and flags */
	bonfire_recorder_selection_get_drive (BONFIRE_RECORDER_SELECTION (dialog->priv->source),
					      &drive_source,
					      NULL);
	bonfire_recorder_selection_get_drive (BONFIRE_RECORDER_SELECTION (dialog->priv->selection),
					      &drive_dest,
					      &props);

	flags = BONFIRE_BURN_FLAG_NONE;

	track = g_new0 (BonfireTrackSource, 1);
	track->type = BONFIRE_TRACK_SOURCE_DISC;
	track->contents.drive.disc = drive_source;

	if (drive_dest->type == NAUTILUS_BURN_DRIVE_TYPE_FILE) {
		output = props.output_path;
		track->contents.drive.target = props.props.image_type;
		speed = 0;
	}
	else {
		flags |= props.flags;
		speed = props.props.drive_speed;
		output = NULL;
	}

	burn_dialog = bonfire_burn_dialog_new ();
	window = gtk_window_get_transient_for (GTK_WINDOW (dialog));

	gtk_widget_hide (GTK_WIDGET (window));
	gtk_widget_destroy (GTK_WIDGET (dialog));

	gtk_window_set_transient_for (GTK_WINDOW (burn_dialog), window);
	gtk_window_set_modal (GTK_WINDOW (burn_dialog), TRUE);
	gtk_widget_show_all (burn_dialog);

	bonfire_burn_dialog_run (BONFIRE_BURN_DIALOG (burn_dialog),
				 drive_dest,
				 speed,
				 output,
				 track,
				 flags);

	/* burn dialog self destroys itself */
	/* gtk_widget_destroy (GTK_WIDGET (burn_dialog)); */

	bonfire_track_source_free (track);

	if (output)
		g_free (output);

	if (drive_dest)
		nautilus_burn_drive_unref (drive_dest);
}
