/*
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Library General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */
/***************************************************************************
 *            main.c
 *
 *  Sat Jun 11 12:00:29 2005
 *  Copyright  2005  Philippe Rouquier	
 *  <bonfire-app@wanadoo.fr>
 ****************************************************************************/

#include <string.h>

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <locale.h>

#include <glib.h>
#include <glib/gi18n-lib.h>
#include <gtk/gtk.h>

#include <gst/gst.h>

#include <libgnomevfs/gnome-vfs.h>
#include <libgnomevfs/gnome-vfs-mime-handlers.h>
#include <libgnome/libgnome.h>
#include <libgnomeui/libgnomeui.h>

#include <nautilus-burn-recorder.h>

#include <gconf/gconf-client.h>

#ifdef HAVE_LIBNOTIFY
#include <libnotify/notify.h>
#endif

#include "bonfire-app.h"
#include "menu.h"
#include "blank-dialog.h"
#include "bonfire-session.h"
#include "bonfire-project-manager.h"

static GConfClient *client;
gchar *project_name;
gchar *iso_uri;
gint audio_project;
gint data_project;
gint copy_project;
gint debug;

static const GOptionEntry options [] = {
	{ "project", 'p', G_OPTION_FLAG_FILENAME, G_OPTION_ARG_STRING, &project_name,
	  N_("Open the specified project"),
	  N_("PROJECT") },

	{ "audio", 'a', 0, G_OPTION_ARG_NONE, &audio_project,
	  N_("Open an empty audio project"),
	  NULL },

	{ "data", 'd', 0, G_OPTION_ARG_NONE, &data_project,
          N_("Open an empty data project"),
          NULL },

	{ "copy", 'c', 0, G_OPTION_ARG_NONE, &copy_project,
	  N_("Copy a disc"),
	  NULL },

	{ "iso", 'i', G_OPTION_FLAG_FILENAME, G_OPTION_ARG_STRING, &iso_uri,
          N_("Uri of an image file or a cue file to be burnt (autodetected)"),
          NULL },

	{ "debug", 'g', 0, G_OPTION_ARG_NONE, &debug,
	  N_("Display debug statements on stdout"),
	  NULL },

	{ NULL }
};

static gboolean
on_delete_cb (GtkWidget *window, GdkEvent *event, BonfireApp *app)
{
	bonfire_session_save (app);
	return FALSE;
}

static gboolean
on_destroy_cb (GtkWidget *window, GdkEvent *event, BonfireApp *app)
{
	gtk_main_quit ();
	return FALSE;
}

void
on_exit_cb (GtkAction *action, BonfireApp *app)
{
	bonfire_session_save (app);
	gtk_widget_destroy (app->mainwin);
}

void
on_erase_cb (GtkAction *action, BonfireApp *app)
{
	GtkWidget *dialog;
	GtkWidget *toplevel;

	dialog = bonfire_blank_dialog_new ();
	toplevel = gtk_widget_get_toplevel (app->mainwin);

	gtk_window_set_transient_for (GTK_WINDOW (dialog), GTK_WINDOW (toplevel));
	gtk_window_set_modal (GTK_WINDOW (dialog), TRUE);
	gtk_window_set_position (GTK_WINDOW (dialog), GTK_WIN_POS_CENTER_ON_PARENT);

	gtk_widget_show_all (dialog);
}

void
on_disc_info_cb (GtkAction *button, BonfireApp *app)
{

}

void
on_about_cb (GtkAction *action, BonfireApp *app)
{
	GtkWidget *dialog;
	const char *authors[] = { "Philippe Rouquier & Mario Danic", NULL };
	GdkPixbuf *logo;

	logo =
	    gdk_pixbuf_new_from_file (BONFIRE_DATADIR
				      "/icon-final-128x128.png", NULL);

	dialog = gtk_about_dialog_new ();
	gtk_about_dialog_set_name (GTK_ABOUT_DIALOG (dialog), "Bonfire");
	gtk_about_dialog_set_copyright (GTK_ABOUT_DIALOG (dialog),
					"Copyright (c) Philippe Rouquier & Mario Danic");
	gtk_about_dialog_set_authors (GTK_ABOUT_DIALOG (dialog), authors);
	gtk_about_dialog_set_comments (GTK_ABOUT_DIALOG (dialog),
				       "Disc burning tool");
	gtk_about_dialog_set_logo (GTK_ABOUT_DIALOG (dialog), logo);
	g_object_unref (logo);
	gtk_about_dialog_set_version (GTK_ABOUT_DIALOG (dialog), VERSION);
	gtk_about_dialog_set_translator_credits (GTK_ABOUT_DIALOG (dialog),
						 _("translator-credits"));

	gtk_window_set_transient_for (GTK_WINDOW (dialog),
				      GTK_WINDOW (app->mainwin));
	gtk_window_set_modal (GTK_WINDOW (dialog), TRUE);
	gtk_window_set_position (GTK_WINDOW (dialog),
				 GTK_WIN_POS_CENTER_ON_PARENT);

	gtk_widget_show (dialog);
	gtk_dialog_run (GTK_DIALOG (dialog));
	gtk_widget_destroy (dialog);
}

static gboolean
on_window_state_changed_cb (GtkWidget *widget,
			    GdkEventWindowState *event,
			    BonfireApp *app)
{
	if (event->new_window_state & GDK_WINDOW_STATE_MAXIMIZED)
		app->is_maximised = 1;
	else
		app->is_maximised = 0;

	return FALSE;
}

static gboolean
on_configure_event_cb (GtkWidget *widget,
		       GdkEventConfigure *event,
		       BonfireApp *app)
{
	if (!app->is_maximised) {
		app->width = event->width;
		app->height = event->height;
	}

	return FALSE;
}

BonfireApp *
bonfire_app_create_app (void)
{
	BonfireApp *app;
	GtkWidget *menubar;
	GtkActionGroup *action_group;
	GtkAccelGroup *accel_group;
	GError *error = NULL;

	/* New window */
	app = g_new0 (BonfireApp, 1);
	app->mainwin = gnome_app_new ("Bonfire", NULL);

	gtk_window_set_icon_name (GTK_WINDOW (app->mainwin),
				  "disc-burner");
	gtk_window_set_icon_from_file (GTK_WINDOW (app->mainwin),
				       BONFIRE_DATADIR G_DIR_SEPARATOR_S "icon-final-48x48.png",
				       NULL);

	g_signal_connect (G_OBJECT (app->mainwin), "delete-event",
			  G_CALLBACK (on_delete_cb), app);
	g_signal_connect (G_OBJECT (app->mainwin), "destroy",
			  G_CALLBACK (on_destroy_cb), app);

	/* status bar FIXME: what for ??? */
	app->statusbar = gtk_statusbar_new ();
	gnome_app_set_statusbar (GNOME_APP (app->mainwin), app->statusbar);

	/* window contents */
	app->contents = bonfire_project_manager_new ();
	gtk_widget_show (app->contents);
	gnome_app_set_contents (GNOME_APP (app->mainwin), app->contents);

	/* menu and toolbar */
	app->manager = gtk_ui_manager_new ();
	bonfire_project_manager_register_menu (BONFIRE_PROJECT_MANAGER (app->contents),
					       app->manager);

	action_group = gtk_action_group_new ("MenuActions");
	gtk_action_group_set_translation_domain (action_group, GETTEXT_PACKAGE);
	gtk_action_group_add_actions (action_group,
				      entries,
				      G_N_ELEMENTS (entries),
				      app);

	gtk_ui_manager_insert_action_group (app->manager, action_group, 0);

	if (!gtk_ui_manager_add_ui_from_string (app->manager, description, -1, &error)) {
		g_message ("building menus failed: %s", error->message);
		g_error_free (error);
	}

	menubar = gtk_ui_manager_get_widget (app->manager, "/menubar");
	gnome_app_set_menus (GNOME_APP (app->mainwin), GTK_MENU_BAR (menubar));

	/* add accelerators */
	accel_group = gtk_ui_manager_get_accel_group (app->manager);
	gtk_window_add_accel_group (GTK_WINDOW (app->mainwin), accel_group);

	/* set up the window geometry */
	gtk_window_set_position (GTK_WINDOW (app->mainwin), GTK_WIN_POS_CENTER);

	bonfire_session_connect (app);
	bonfire_session_load (app);

	g_signal_connect (app->mainwin,
			  "window-state-event",
			  G_CALLBACK (on_window_state_changed_cb),
			  app);
	g_signal_connect (app->mainwin,
			  "configure-event",
			  G_CALLBACK (on_configure_event_cb),
			  app);
	return app;
}

int
main (int argc, char **argv)
{
	BonfireApp *app;
	GnomeProgram *program;
	GOptionContext *context;

	context = g_option_context_new (NULL);
	g_option_context_add_main_entries (context,
					   options,
					   GETTEXT_PACKAGE);

	program = gnome_program_init (PACKAGE, VERSION, LIBGNOMEUI_MODULE,
				      argc, argv,
				      GNOME_PARAM_GOPTION_CONTEXT, context,
				      GNOME_PARAM_HUMAN_READABLE_NAME, _("CD/DVD burning"),
				      NULL);

	bonfire_utils_register_icons ();

#ifdef ENABLE_NLS
	bindtextdomain (GETTEXT_PACKAGE, PACKAGE_LOCALE_DIR);
	bind_textdomain_codeset (GETTEXT_PACKAGE, "UTF-8");
	textdomain (GETTEXT_PACKAGE);
#endif

	if (!g_thread_supported ())
		g_thread_init (NULL);

	gnome_vfs_init ();
	gst_init (&argc, &argv);

#ifdef HAVE_LIBNOTIFY
	notify_init ("Bonfire");
#endif

	client = gconf_client_get_default ();

	app = bonfire_app_create_app ();
	if (app == NULL)
		return 1;

	if (audio_project)
		bonfire_project_manager_audio (BONFIRE_PROJECT_MANAGER (app->contents));
	else if (data_project)
		bonfire_project_manager_data (BONFIRE_PROJECT_MANAGER (app->contents));
	else if (copy_project)
		bonfire_project_manager_copy (BONFIRE_PROJECT_MANAGER (app->contents));
	else if (iso_uri)
		bonfire_project_manager_iso (BONFIRE_PROJECT_MANAGER (app->contents), iso_uri);
	else if (project_name)
     		bonfire_project_manager_open (BONFIRE_PROJECT_MANAGER (app->contents), project_name);
	else
     		bonfire_project_manager_empty (BONFIRE_PROJECT_MANAGER (app->contents));

	gtk_widget_show (app->mainwin);
	gtk_main ();

	bonfire_session_disconnect (app);
	g_object_unref (program);
	g_free (app);
	gst_deinit ();

	g_object_unref (client);
	client = NULL;

	return 0;
}
