/***************************************************************************
 *            metadata.c
 *
 *  jeu jui 28 12:49:41 2005
 *  Copyright  2005  Philippe Rouquier
 *  bonfire-app@wanadoo.fr
 ***************************************************************************/

/*
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <string.h>

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <glib.h>
#include <glib/gi18n-lib.h>
#include <glib-object.h>

#include <gst/gst.h>
#include <gst/base/gstbasesink.h>
#include <gst/tag/tag.h>

#include "metadata.h"
#include "utils.h"

extern gint debug;

static void bonfire_metadata_class_init (BonfireMetadataClass *klass);
static void bonfire_metadata_init (BonfireMetadata *sp);
static void bonfire_metadata_finalize (GObject *object);

static void bonfire_metadata_get_property (GObject *obj,
					   guint prop_id,
					   GValue *value,
					   GParamSpec *pspec);
static void bonfire_metadata_set_property (GObject *obj,
					   guint prop_id,
					   const GValue *value,
					   GParamSpec *pspec);

struct BonfireMetadataPrivate {
	GstElement *pipeline;
	GstElement *source;
	GstElement *decode;
	GstElement *sink;

	GMainLoop *loop;
	GError *error;
};

typedef enum {
	COMPLETED_SIGNAL,
	LAST_SIGNAL
} BonfireMetadataSignalType;

static guint bonfire_metadata_signals[LAST_SIGNAL] = { 0 };
static GObjectClass *parent_class = NULL;

static gboolean bonfire_metadata_bus_messages (GstBus *bus,
					       GstMessage *msg,
					       BonfireMetadata *meta);

static void bonfire_metadata_new_decoded_pad_cb (GstElement *decode,
						 GstPad *pad,
						 gboolean arg2,
						 BonfireMetadata *meta);

enum {
	PROP_NONE,
	PROP_URI
};

GType
bonfire_metadata_get_type ()
{
	static GType type = 0;

	if (type == 0) {
		static const GTypeInfo our_info = {
			sizeof (BonfireMetadataClass),
			NULL,
			NULL,
			(GClassInitFunc) bonfire_metadata_class_init,
			NULL,
			NULL,
			sizeof (BonfireMetadata),
			0,
			(GInstanceInitFunc) bonfire_metadata_init,
		};

		type = g_type_register_static (G_TYPE_OBJECT,
					       "BonfireMetadata",
					       &our_info, 0);
	}

	return type;
}

static void
bonfire_metadata_class_init (BonfireMetadataClass *klass)
{
	GObjectClass *object_class = G_OBJECT_CLASS (klass);

	parent_class = g_type_class_peek_parent (klass);
	object_class->finalize = bonfire_metadata_finalize;
	object_class->set_property = bonfire_metadata_set_property;
	object_class->get_property = bonfire_metadata_get_property;

	bonfire_metadata_signals[COMPLETED_SIGNAL] =
	    g_signal_new ("completed",
			  G_TYPE_FROM_CLASS (klass),
			  G_SIGNAL_RUN_LAST,
			  G_STRUCT_OFFSET (BonfireMetadataClass,
					   completed),
			  NULL, NULL,
			  g_cclosure_marshal_VOID__POINTER,
			  G_TYPE_NONE,
			  1,
			  G_TYPE_POINTER);

	g_object_class_install_property (object_class,
					 PROP_URI,
					 g_param_spec_string ("uri",
							      "The uri of the song",
							      "The uri of the song",
							      NULL,
							      G_PARAM_READWRITE));
}

static void
bonfire_metadata_get_property (GObject *obj,
			       guint prop_id,
			       GValue *value,
			       GParamSpec *pspec)
{
	char *uri;
	BonfireMetadata *meta;

	meta = BONFIRE_METADATA (obj);
	switch (prop_id) {
	case PROP_URI:
		g_object_get (G_OBJECT (meta->priv->source), "location",
			      &uri, NULL);
		g_value_set_string (value, uri);
		g_free (uri);
		break;
	default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID (obj, prop_id, pspec);
		break;
	}
}

static void
bonfire_metadata_set_property (GObject *obj,
			       guint prop_id,
			       const GValue *value,
			       GParamSpec *pspec)
{
	const char *uri;
	BonfireMetadata *meta;

	meta = BONFIRE_METADATA (obj);
	switch (prop_id) {
	case PROP_URI:
		uri = g_value_get_string (value);
		gst_element_set_state (GST_ELEMENT (meta->priv->pipeline), GST_STATE_NULL);
		if (meta->priv->source)
			g_object_set (G_OBJECT (meta->priv->source),
				      "location", uri,
				      NULL);
		gst_element_set_state (GST_ELEMENT (meta->priv->pipeline),
				       GST_STATE_PAUSED);
		break;
	default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID (obj, prop_id, pspec);
		break;
	}
}

static void
bonfire_metadata_init (BonfireMetadata *obj)
{
	obj->priv = g_new0 (BonfireMetadataPrivate, 1);
}

static void
bonfire_metadata_finalize (GObject *object)
{
	BonfireMetadata *cobj;

	cobj = BONFIRE_METADATA (object);

	if (cobj->priv->pipeline) {
		gst_element_set_state (GST_ELEMENT (cobj->priv->pipeline), GST_STATE_NULL);
		gst_object_unref (cobj->priv->pipeline);
		cobj->priv->pipeline = NULL;
	}

	if (cobj->uri) {
		g_free (cobj->uri);
		cobj->uri = NULL;
	}

	if (cobj->type)
		g_free (cobj->type);

	if (cobj->title)
		g_free (cobj->title);

	if (cobj->artist)
		g_free (cobj->artist);

	if (cobj->album)
		g_free (cobj->album);

	if (cobj->musicbrainz_id)
		g_free (cobj->musicbrainz_id);

	if (cobj->priv->error) {
		g_error_free (cobj->priv->error);
		cobj->priv->error = NULL;
	}

	g_free (cobj->priv);
	G_OBJECT_CLASS (parent_class)->finalize (object);
}

static gboolean
bonfire_metadata_create_pipeline (BonfireMetadata *metadata)
{
	char *uri;

	metadata->priv->pipeline = gst_pipeline_new (NULL);
	gst_bus_add_watch (gst_pipeline_get_bus (GST_PIPELINE (metadata->priv->pipeline)),
			   (GstBusFunc) bonfire_metadata_bus_messages,
			   metadata);

	uri = gnome_vfs_escape_host_and_path_string (metadata->uri);
	metadata->priv->source = gst_element_make_from_uri (GST_URI_SRC,
							    uri,
							    NULL);
	g_free (uri);

	if (metadata->priv->source == NULL) {
		metadata->priv->error = g_error_new (BONFIRE_ERROR,
						     BONFIRE_ERROR_GENERAL,
						     "Can't create file source");
		return FALSE;
	}
	gst_bin_add (GST_BIN (metadata->priv->pipeline), metadata->priv->source);

	metadata->priv->decode = gst_element_factory_make ("decodebin", NULL);
	if (metadata->priv->decode == NULL) {
		metadata->priv->error = g_error_new (BONFIRE_ERROR,
						     BONFIRE_ERROR_GENERAL,
						     "Can't create decode");
		return FALSE;
	}
	g_signal_connect (G_OBJECT (metadata->priv->decode), "new-decoded-pad",
			  G_CALLBACK (bonfire_metadata_new_decoded_pad_cb),
			  metadata);
	gst_bin_add (GST_BIN (metadata->priv->pipeline), metadata->priv->decode);

	gst_element_link_many (metadata->priv->source, metadata->priv->decode, NULL);

	metadata->priv->sink = gst_element_factory_make ("fakesink", NULL);
	if (metadata->priv->sink == NULL) {
		metadata->priv->error = g_error_new (BONFIRE_ERROR,
				                BONFIRE_ERROR_GENERAL,
						"Can't create fake sink");
		return FALSE;
	}
	gst_bin_add (GST_BIN (metadata->priv->pipeline), metadata->priv->sink);

	return TRUE;
}

BonfireMetadata *
bonfire_metadata_new (const char *uri)
{
	BonfireMetadata *obj = NULL;
	char *unescaped_uri;

	unescaped_uri = gnome_vfs_unescape_string_for_display (uri);
	if (gst_uri_is_valid (unescaped_uri)) {
		obj = BONFIRE_METADATA (g_object_new (BONFIRE_TYPE_METADATA, NULL));
		obj->uri = unescaped_uri;
	}
	else
		g_free (unescaped_uri);

	return obj;
}

static void
bonfire_metadata_is_seekable (BonfireMetadata *meta)
{
	GstQuery *query;
	GstFormat format;
	gboolean seekable;

	meta->is_seekable = FALSE;
	query = gst_query_new_seeking (GST_FORMAT_DEFAULT);
	if (!gst_element_query (meta->priv->pipeline, query))
		goto end;

	gst_query_parse_seeking (query,
				 &format,
				 &seekable,
				 NULL,
				 NULL);

	meta->is_seekable = seekable;
end:
	gst_query_unref (query);
}

static gboolean
bonfire_metadata_completed (BonfireMetadata *meta)
{
	GstFormat format = GST_FORMAT_TIME;
	GstElement *typefind;
	GstCaps *caps = NULL;


	/* find the type of the file */
	typefind = gst_bin_get_by_name (GST_BIN (meta->priv->decode), "typefind");
	if (typefind == NULL) {
		meta->priv->error = g_error_new (BONFIRE_ERROR,
						 BONFIRE_ERROR_GENERAL,
						 "can't get typefind");
		goto signal;
	}

	g_object_get (typefind, "caps", &caps, NULL);
	if (!caps) {
		meta->priv->error = g_error_new (BONFIRE_ERROR,
						 BONFIRE_ERROR_GENERAL,
						_("unknown type of file"));
		goto signal;
	}

	if (caps && gst_caps_get_size (caps) > 0) {
		meta->type = g_strdup (gst_structure_get_name (gst_caps_get_structure (caps, 0)));
		if (meta->type && !strcmp (meta->type, "application/x-id3")) {
			g_free (meta->type);
			meta->type = g_strdup ("audio/mpeg");
		}
	}

	bonfire_metadata_is_seekable (meta);

	if (!gst_element_query_duration (GST_ELEMENT (meta->priv->pipeline),
					 &format,
					 &meta->len)
	&&  meta->priv->error == NULL) {
		meta->priv->error = g_error_new (BONFIRE_ERROR,
						 BONFIRE_ERROR_GENERAL,
						_("this format is not supported by gstreamer"));
		goto signal;
	}

signal:
	gst_element_set_state (GST_ELEMENT (meta->priv->pipeline), GST_STATE_NULL);

	if (meta->priv->loop
	&&  g_main_loop_is_running (meta->priv->loop))
		g_main_loop_quit (meta->priv->loop);

	g_object_ref (meta);
	g_signal_emit (G_OBJECT (meta),
		       bonfire_metadata_signals [COMPLETED_SIGNAL],
		       0,
		       meta->priv->error);
	g_object_unref (meta);

	return FALSE;
}

static void
foreach_tag (const GstTagList *list,
	     const char *tag,
	     BonfireMetadata *meta)
{
	if (!strcmp (tag, GST_TAG_TITLE)) {
		if (meta->title)
			g_free (meta->title);

		gst_tag_list_get_string (list, tag, &(meta->title));
	} else if (!strcmp (tag, GST_TAG_ARTIST)
	       ||  !strcmp (tag, GST_TAG_PERFORMER)) {
		if (meta->artist)
			g_free (meta->artist);

		gst_tag_list_get_string (list, tag, &(meta->artist));
	}
	else if (!strcmp (tag, GST_TAG_ALBUM)) {
		if (meta->album)
			g_free (meta->album);

		gst_tag_list_get_string (list, tag, &(meta->album));
	}
/*	else if (!strcmp (tag, GST_TAG_COMPOSER)) {
		if (meta->composer)
			g_free (meta->composer);

		gst_tag_list_get_string (list, tag, &(meta->composer));
	}
*/	else if (!strcmp (tag, GST_TAG_ISRC)) {
		gst_tag_list_get_int (list, tag, &(meta->isrc));
	}
	else if (!strcmp (tag, GST_TAG_MUSICBRAINZ_TRACKID)) {
		gst_tag_list_get_string (list, tag, &(meta->musicbrainz_id));
	}
}

static gboolean
bonfire_metadata_bus_messages (GstBus *bus,
			       GstMessage *msg,
			       BonfireMetadata *meta)
{
	GstStateChangeReturn result;
	GstTagList *tags = NULL;
	gboolean stop_pipeline;
	GError *error = NULL;
	gchar *debug_string;
	GstState newstate;

	stop_pipeline = FALSE;

start:

	switch (GST_MESSAGE_TYPE (msg)) {
	case GST_MESSAGE_ERROR:
		/* when stopping the pipeline we are only interested in TAGS */
		if (stop_pipeline)
			break;

		gst_message_parse_error (msg, &error, &debug_string);
		if (debug)
			g_warning ("DEBUG: %s\n", debug_string);

		g_free (debug_string);

		meta->priv->error = error;
		bonfire_metadata_completed (meta);
		return FALSE;

	case GST_MESSAGE_EOS:
		/* when stopping the pipeline we are only interested in TAGS */
		if (stop_pipeline)
			break;

		bonfire_metadata_completed (meta);
		return FALSE;

	case GST_MESSAGE_TAG:
		gst_message_parse_tag (msg, &tags);
		gst_tag_list_foreach (tags, (GstTagForeachFunc) foreach_tag, meta);
		gst_tag_list_free (tags);
		break;

	case GST_MESSAGE_STATE_CHANGED:
		/* when stopping the pipeline we are only interested in TAGS */
		if (stop_pipeline)
			break;

		result = gst_element_get_state (GST_ELEMENT (meta->priv->pipeline),
						&newstate,
						NULL,
						0);

		if (result != GST_STATE_CHANGE_SUCCESS)
			break;

		if (newstate == GST_STATE_PAUSED) {
			msg = gst_message_ref (msg);
			stop_pipeline = TRUE;
		}

		if (newstate == GST_STATE_PLAYING) {
			msg = gst_message_ref (msg);
			stop_pipeline = TRUE;
			gst_element_set_state (GST_ELEMENT (meta->priv->pipeline),
					       GST_STATE_PAUSED);
		}

		break;

	default:
		break;
	}

	if (stop_pipeline) {
		gst_message_unref (msg);

		if (gst_bus_have_pending (bus)) {
			msg = gst_bus_pop (bus);
			goto start;
		}

		bonfire_metadata_completed (meta);
		return FALSE;
	}

	return TRUE;
}

static void
bonfire_metadata_new_decoded_pad_cb (GstElement *decode,
				     GstPad *pad,
				     gboolean arg2,
				     BonfireMetadata *meta)
{
	GstPad *sink;
	GstCaps *caps;
	GstStructure *structure;

	sink = gst_element_get_pad (meta->priv->sink, "sink");
	if (GST_PAD_IS_LINKED (sink))
		return;

	/* make sure that this is audio /video */
	caps = gst_pad_get_caps (pad);
	structure = gst_caps_get_structure (caps, 0);
	if (g_strrstr (gst_structure_get_name (structure), "audio")
	||  g_strrstr (gst_structure_get_name (structure), "video"))
		gst_pad_link (pad, sink);

	gst_object_unref (sink);
	gst_caps_unref (caps);
}

gboolean
bonfire_metadata_get_sync (BonfireMetadata *meta, GError **error)
{
	if (!meta->priv->pipeline
	&&  !bonfire_metadata_create_pipeline (meta)) {
		g_propagate_error (error, meta->priv->error);
		meta->priv->error = NULL;
		return FALSE;
	}

	gst_element_set_state (GST_ELEMENT (meta->priv->pipeline), GST_STATE_PLAYING);

	meta->priv->loop = g_main_loop_new (NULL, FALSE);
	g_main_loop_run (meta->priv->loop);
	g_main_loop_unref (meta->priv->loop);
	meta->priv->loop = NULL;

	if (meta->priv->error) {
		if (error) {
			g_propagate_error (error, meta->priv->error);
			meta->priv->error = NULL;
		} 
		else {
			g_warning ("ERROR getting metadata : %s\n",
				   meta->priv->error->message);
			g_error_free (meta->priv->error);
			meta->priv->error = NULL;
		}
		return FALSE;
	}

	return TRUE;
}

gboolean
bonfire_metadata_get_async (BonfireMetadata *meta)
{
	if (!meta->priv->pipeline
	&&  !bonfire_metadata_create_pipeline (meta)) {
		g_object_ref (meta);
		g_signal_emit (G_OBJECT (meta),
			       bonfire_metadata_signals [COMPLETED_SIGNAL],
			       0,
			       meta->priv->error);
		g_object_unref (meta);

		if (meta->priv->error) {
			g_error_free (meta->priv->error);
			meta->priv->error = NULL;
		}
		return FALSE;
	}

	gst_element_set_state (GST_ELEMENT (meta->priv->pipeline), GST_STATE_PLAYING);
	return TRUE;
}

void
bonfire_metadata_cancel (BonfireMetadata *meta)
{
	if (meta->priv->pipeline) {
		gst_element_set_state (GST_ELEMENT (meta->priv->pipeline), GST_STATE_NULL);
		gst_object_unref (GST_OBJECT (meta->priv->pipeline));
		meta->priv->pipeline = NULL;
	}
}
