/***************************************************************************
 *            project.c
 *
 *  mar nov 29 09:32:17 2005
 *  Copyright  2005  Rouquier Philippe
 *  bonfire-app@wanadoo.fr
 ***************************************************************************/

/*
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */



#include <string.h>

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <glib.h>
#include <glib/gstdio.h>
#include <glib/gi18n.h>

#include <gtk/gtknotebook.h>
#include <gtk/gtkscrolledwindow.h>
#include <gtk/gtkfilechooser.h>
#include <gtk/gtkfilechooserdialog.h>
#include <gtk/gtkmessagedialog.h>
#include <gtk/gtkprogressbar.h>
#include <gtk/gtkstock.h>

#include <libxml/xmlerror.h>
#include <libxml/xmlwriter.h>
#include <libxml/parser.h>
#include <libxml/xmlstring.h>

#include <libgnomevfs/gnome-vfs.h>
#include <libgnomevfs/gnome-vfs-file-info.h>

#include "project.h"
#include "project-size.h"
#include "project-type-chooser.h"
#include "disc.h"
#include "data-disc.h"
#include "audio-disc.h"
#include "burn-options-dialog.h"
#include "utils.h"
#include "bonfire-uri-container.h"

static void bonfire_project_class_init (BonfireProjectClass *klass);
static void bonfire_project_init (BonfireProject *sp);
static void bonfire_project_iface_uri_container_init (BonfireURIContainerIFace *iface);
static void bonfire_project_finalize (GObject *object);

static void
bonfire_project_save_cb (GtkAction *action, BonfireProject *project);
static void
bonfire_project_save_as_cb (GtkAction *action, BonfireProject *project);

static void
bonfire_project_add_uris_cb (GtkAction *action, BonfireProject *project);
static void
bonfire_project_remove_selected_uris_cb (GtkAction *action, BonfireProject *project);
static void
bonfire_project_empty_cb (GtkAction *action, BonfireProject *project);

static void
bonfire_project_burn_cb (GtkAction *action, BonfireProject *project);

static void
bonfire_project_disc_changed_cb (BonfireProjectSize *size,
				 BonfireProject *project);
static void
bonfire_project_size_changed_cb (BonfireDisc *disc,
			         double size,
			         const char *string,
			         BonfireProject *project);

static void
bonfire_project_add_clicked_cb (GtkButton *button, BonfireProject *project);
static void
bonfire_project_remove_clicked_cb (GtkButton *button, BonfireProject *project);
static void
bonfire_project_burn_clicked_cb (GtkButton *button, BonfireProject *project);

static void
bonfire_project_contents_changed_cb (BonfireDisc *disc,
				     int nb_files,
				     BonfireProject *project);
static void
bonfire_project_selection_changed_cb (BonfireDisc *disc,
				      BonfireProject *project);
static char *
bonfire_project_get_selected_uri (BonfireURIContainer *container);

struct BonfireProjectPrivate {
	GtkWidget *size_display;
	GtkWidget *discs;
	GtkWidget *audio;
	GtkWidget *data;

	/* header */
	GtkWidget *image;
	GtkWidget *label;
	GtkWidget *add;
	GtkWidget *remove;
	GtkWidget *burn;

	GtkTooltips *tooltip;
	GtkActionGroup *project_group;
	GtkActionGroup *action_group;

	char *project;
	BonfireDisc *current;
	BonfireURIContainer *current_source;

	int oversized:1;
	int overburn:1;
	int ask_overburn:1;
};

static GtkActionEntry entries_project [] = {
	{"Save", GTK_STOCK_SAVE, N_("_Save"), NULL,
	 N_("Save current project"), G_CALLBACK (bonfire_project_save_cb)},
	{"SaveAs", GTK_STOCK_SAVE_AS, N_("_Save as"), NULL,
	 N_("Save current project to a different location"), G_CALLBACK (bonfire_project_save_as_cb)},
};

static const char *description_project = {
	"<ui>"
	    "<menubar name='menubar' >"
		"<menu action='ProjectMenu'>"
			"<placeholder name='ProjectPlaceholder'/>"
			    "<menuitem action='Save'/>"
			    "<menuitem action='SaveAs'/>"
			    "<separator/>"
		"</menu>"
	    "</menubar>"
	"</ui>"
};

static GtkActionEntry entries_actions [] = {
	{"Add", GTK_STOCK_ADD, N_("_Add files"), NULL,
	 N_("Add files to the project"), G_CALLBACK (bonfire_project_add_uris_cb)},
	{"Delete", GTK_STOCK_REMOVE, N_("_Remove files"), NULL,
	 N_("Remove the selected files from the project"), G_CALLBACK (bonfire_project_remove_selected_uris_cb)},
	{"DeleteAll", GTK_STOCK_DELETE, N_("E_mpty Project"), NULL,
	 N_("Delete all files from the project"), G_CALLBACK (bonfire_project_empty_cb)},

	{"Burn", GTK_STOCK_CDROM, N_("_Burn"), NULL,
	 N_("Burn the disc"), G_CALLBACK (bonfire_project_burn_cb)},
};

static const char *description_actions = {
	"<ui>"
	    "<menubar name='menubar' >"
		"<menu action='EditMenu'>"
			"<placeholder name='EditPlaceholder'/>"
			    "<menuitem action='Add'/>"
			    "<menuitem action='Delete'/>"
			    "<menuitem action='DeleteAll'/>"
			    "<separator/>"
		"</menu>"
		"<menu action='ViewMenu'>"
		"</menu>"
		"<menu action='DiscMenu'>"
			"<placeholder name='DiscPlaceholder'/>"
			"<menuitem action='Burn'/>"
		"</menu>"
		"</menubar>"
	"</ui>"
};

static GObjectClass *parent_class = NULL;

#define BONFIRE_PROJECT_VERSION "0.2"

GType
bonfire_project_get_type ()
{
	static GType type = 0;

	if(type == 0) {
		static const GTypeInfo our_info = {
			sizeof (BonfireProjectClass),
			NULL,
			NULL,
			(GClassInitFunc)bonfire_project_class_init,
			NULL,
			NULL,
			sizeof (BonfireProject),
			0,
			(GInstanceInitFunc) bonfire_project_init,
		};

		static const GInterfaceInfo uri_container_info =
		{
			(GInterfaceInitFunc) bonfire_project_iface_uri_container_init,
			NULL,
			NULL
		};

		type = g_type_register_static (GTK_TYPE_VBOX, 
					       "BonfireProject",
					       &our_info, 0);

		g_type_add_interface_static (type,
					     BONFIRE_TYPE_URI_CONTAINER,
					     &uri_container_info);
	}

	return type;
}

static void
bonfire_project_class_init (BonfireProjectClass *klass)
{
	GObjectClass *object_class = G_OBJECT_CLASS (klass);

	parent_class = g_type_class_peek_parent (klass);
	object_class->finalize = bonfire_project_finalize;
}

static void
bonfire_project_iface_uri_container_init (BonfireURIContainerIFace *iface)
{
	iface->get_selected_uri = bonfire_project_get_selected_uri;
}

static void
bonfire_project_init (BonfireProject *obj)
{
	GtkWidget *alignment;
	GtkWidget *separator;
	GtkWidget *box;

	obj->priv = g_new0 (BonfireProjectPrivate, 1);
	gtk_box_set_spacing (GTK_BOX (obj), 6);

	obj->priv->tooltip = gtk_tooltips_new ();

	/* header */
	box = gtk_hbox_new (FALSE, 8);

	obj->priv->image = gtk_image_new ();
	gtk_box_pack_start (GTK_BOX (box), obj->priv->image, FALSE, FALSE, 0);

	obj->priv->label = gtk_label_new (NULL);
	gtk_label_set_use_markup (GTK_LABEL (obj->priv->label), TRUE);
	gtk_misc_set_alignment (GTK_MISC (obj->priv->label), 0, 0.5);
	gtk_box_pack_start (GTK_BOX (box), obj->priv->label, FALSE, FALSE, 0);

	/* add button set insensitive since there are no files in the selection */
	obj->priv->add = bonfire_utils_make_button (NULL, GTK_STOCK_ADD);
	gtk_widget_set_sensitive (obj->priv->add, FALSE);

	g_signal_connect (obj->priv->add,
			  "clicked",
			  G_CALLBACK (bonfire_project_add_clicked_cb),
			  obj);
	gtk_tooltips_set_tip (obj->priv->tooltip,
			      obj->priv->add,
			      _("Add selected files"),
			      NULL);
	alignment = gtk_alignment_new (1.0, 0.5, 0.0, 0.0);
	gtk_container_add (GTK_CONTAINER (alignment), obj->priv->add);
	gtk_box_pack_start (GTK_BOX (box), alignment, TRUE, TRUE, 0);

	obj->priv->remove = bonfire_utils_make_button (NULL, GTK_STOCK_REMOVE);
	gtk_widget_set_sensitive (obj->priv->remove, FALSE);
	g_signal_connect (obj->priv->remove,
			  "clicked",
			  G_CALLBACK (bonfire_project_remove_clicked_cb),
			  obj);
	gtk_tooltips_set_tip (obj->priv->tooltip,
			      obj->priv->remove,
			      _("Remove files selected in project"),
			      NULL);
	alignment = gtk_alignment_new (1.0, 0.5, 0.0, 0.0);
	gtk_container_add (GTK_CONTAINER (alignment), obj->priv->remove);
	gtk_box_pack_start (GTK_BOX (box), alignment, FALSE, FALSE, 0);

	separator = gtk_vseparator_new ();
	gtk_box_pack_start (GTK_BOX (box), separator, FALSE, FALSE, 0);

	/* burn button set insensitive since there are no files in the selection */
	obj->priv->burn = bonfire_utils_make_button (_("Burn"), GTK_STOCK_CDROM);
	gtk_widget_set_sensitive (obj->priv->burn, FALSE);
	g_signal_connect (obj->priv->burn,
			  "clicked",
			  G_CALLBACK (bonfire_project_burn_clicked_cb),
			  obj);
	gtk_tooltips_set_tip (obj->priv->tooltip,
			      obj->priv->burn,
			      _("Start to burn the contents of the selection"),
			      NULL);
	alignment = gtk_alignment_new (1.0, 0.5, 0.0, 0.0);
	gtk_container_add (GTK_CONTAINER (alignment), obj->priv->burn);
	gtk_box_pack_start (GTK_BOX (box), alignment, FALSE, FALSE, 0);

	gtk_box_pack_start (GTK_BOX (obj), box, FALSE, FALSE, 0);

	/* size widget */
	obj->priv->size_display = bonfire_project_size_new ();
	g_signal_connect (G_OBJECT (obj->priv->size_display), 
			  "disc-changed",
			  G_CALLBACK (bonfire_project_disc_changed_cb),
			  obj);

	obj->priv->audio = bonfire_audio_disc_new ();
	g_signal_connect (G_OBJECT (obj->priv->audio),
			  "contents_changed",
			  G_CALLBACK (bonfire_project_contents_changed_cb),
			  obj);
	g_signal_connect (G_OBJECT (obj->priv->audio),
			  "size_changed",
			  G_CALLBACK (bonfire_project_size_changed_cb),
			  obj);
	g_signal_connect (G_OBJECT (obj->priv->audio),
			  "selection_changed",
			  G_CALLBACK (bonfire_project_selection_changed_cb),
			  obj);

	obj->priv->data = bonfire_data_disc_new ();
	g_signal_connect (G_OBJECT (obj->priv->data),
			  "contents_changed",
			  G_CALLBACK (bonfire_project_contents_changed_cb),
			  obj);
	g_signal_connect (G_OBJECT (obj->priv->data),
			  "size_changed",
			  G_CALLBACK (bonfire_project_size_changed_cb),
			  obj);
	g_signal_connect (G_OBJECT (obj->priv->data),
			  "selection_changed",
			  G_CALLBACK (bonfire_project_selection_changed_cb),
			  obj);

	obj->priv->discs = gtk_notebook_new ();
	gtk_notebook_set_show_border (GTK_NOTEBOOK (obj->priv->discs), FALSE);
	gtk_notebook_set_show_tabs (GTK_NOTEBOOK (obj->priv->discs), FALSE);
	gtk_notebook_prepend_page (GTK_NOTEBOOK (obj->priv->discs),
				   obj->priv->data, NULL);
	gtk_notebook_prepend_page (GTK_NOTEBOOK (obj->priv->discs),
				   obj->priv->audio, NULL);

	gtk_box_pack_end (GTK_BOX (obj), obj->priv->size_display, FALSE, FALSE, 0);
	gtk_box_pack_end (GTK_BOX (obj), obj->priv->discs, TRUE, TRUE, 0);
}

static void
bonfire_project_finalize (GObject *object)
{
	BonfireProject *cobj;
	cobj = BONFIRE_PROJECT(object);

	if (cobj->priv->project)
		g_free (cobj->priv->project);

	if (cobj->priv->tooltip) {
		gtk_object_sink (GTK_OBJECT (cobj->priv->tooltip));
		cobj->priv->tooltip = NULL;
	}

	g_free(cobj->priv);
	G_OBJECT_CLASS(parent_class)->finalize(object);
}

GtkWidget *
bonfire_project_new ()
{
	BonfireProject *obj;
	
	obj = BONFIRE_PROJECT(g_object_new(BONFIRE_TYPE_PROJECT, NULL));
	gtk_notebook_set_current_page (GTK_NOTEBOOK (obj->priv->discs), 0);

	return GTK_WIDGET (obj);
}

/********************************** size ***************************************/
static void
bonfire_project_error_size_dialog (BonfireProject *project)
{
	GtkWidget *dialog;
	GtkWidget *toplevel;

	toplevel = gtk_widget_get_toplevel (GTK_WIDGET (project));
	dialog = gtk_message_dialog_new (GTK_WINDOW (toplevel),
					 GTK_DIALOG_DESTROY_WITH_PARENT |
					 GTK_DIALOG_MODAL,
					 GTK_MESSAGE_ERROR,
					 GTK_BUTTONS_CLOSE,
					 _("The size of the project is too large for the disc even with the overburn option:"));

	gtk_window_set_title (GTK_WINDOW (dialog), _("Project size"));

	gtk_message_dialog_format_secondary_text (GTK_MESSAGE_DIALOG (dialog),
						  _("you must delete some files."));

	gtk_dialog_run (GTK_DIALOG (dialog));
	gtk_widget_destroy (dialog);
}

static void
bonfire_project_overburn_dialog (BonfireProject *project)
{
	GtkWidget *dialog, *toplevel;
	gint result;

	/* get the current CD length and make sure selection is not too long */
	toplevel = gtk_widget_get_toplevel (GTK_WIDGET (project));
	dialog = gtk_message_dialog_new (GTK_WINDOW (toplevel),
					 GTK_DIALOG_DESTROY_WITH_PARENT|
					 GTK_DIALOG_MODAL,
					 GTK_MESSAGE_WARNING,
					 GTK_BUTTONS_NONE,
					 _("The size of the project is too large for the disc:"));

	gtk_window_set_title (GTK_WINDOW (dialog), _("Project size"));

	gtk_message_dialog_format_secondary_text (GTK_MESSAGE_DIALOG (dialog),
						  _("Would you like to activate overburn (otherwise you must delete files) ?"
						    "\nNOTE: This option might cause failure."));

	gtk_dialog_add_buttons (GTK_DIALOG (dialog),
				GTK_STOCK_NO,
				GTK_RESPONSE_NO,
				GTK_STOCK_YES,
				GTK_RESPONSE_YES, NULL);

	result = gtk_dialog_run (GTK_DIALOG (dialog));
	gtk_widget_destroy (dialog);
	if (result == GTK_RESPONSE_YES) {
		project->priv->oversized = 0;
		project->priv->overburn = 1;
	}
	else {
		project->priv->oversized = 1;
		project->priv->overburn = 0;
	}
}

static void
bonfire_project_check_size (BonfireProject *project)
{
	gboolean result;
	gboolean overburn;

	result = bonfire_project_size_check_status (BONFIRE_PROJECT_SIZE (project->priv->size_display),
						    &overburn);

	if (result) {
		project->priv->ask_overburn = 0;
		project->priv->oversized = 0;
		project->priv->overburn = 0;
		goto end;
	}

	if (overburn
	&& !project->priv->ask_overburn) {
		if (!project->priv->overburn)
			bonfire_project_overburn_dialog (project);

		project->priv->ask_overburn = 1;
		goto end;
	}

	if (!overburn
	&&  project->priv->overburn)
		project->priv->overburn = 0;

	/* avoid telling the user the same thing twice */
	if (project->priv->oversized)
		goto end;

	bonfire_project_error_size_dialog (project);
	project->priv->oversized = 1;
	project->priv->overburn = 0;

end:
	if (project->priv->oversized) {
		g_object_set (G_OBJECT (project->priv->data), "reject-file", TRUE, NULL);
		g_object_set (G_OBJECT (project->priv->data), "reject-file", TRUE, NULL);
	}
	else {
		g_object_set (G_OBJECT (project->priv->data), "reject-file", FALSE, NULL);
		g_object_set (G_OBJECT (project->priv->data), "reject-file", FALSE, NULL);
	}
}

static void
bonfire_project_size_changed_cb (BonfireDisc *disc,
			         double size,
			         const char *string,
			         BonfireProject *project)
{
	bonfire_project_size_set_size (BONFIRE_PROJECT_SIZE (project->priv->size_display),
				       size,
				       string);

	bonfire_project_check_size (project);
}

static void
bonfire_project_disc_changed_cb (BonfireProjectSize *size,
				 BonfireProject *project)
{
	bonfire_project_check_size (project);
}

/***************************** URIContainer ************************************/
static void
bonfire_project_selection_changed_cb (BonfireDisc *disc,
				      BonfireProject *project)
{
	/* FIXME! from here we can control the add button state depending on the source URI */
	bonfire_uri_container_uri_selected (BONFIRE_URI_CONTAINER (project));
}

static char *
bonfire_project_get_selected_uri (BonfireURIContainer *container)
{
	BonfireProject *project;

	project = BONFIRE_PROJECT (container);
	return bonfire_disc_get_selected_uri (project->priv->current);
}

/******************** useful function to wait when burning/saving **************/
static gboolean
_wait_for_ready_state (GtkWidget *dialog) {
	GtkProgressBar *progress;
	BonfireProject *project;

	project = g_object_get_data (G_OBJECT (dialog), "Project");
	if (project->priv->oversized) {
		gtk_dialog_response (GTK_DIALOG (dialog), GTK_RESPONSE_CANCEL);
		return FALSE;
	}

	progress = g_object_get_data (G_OBJECT (dialog), "ProgressBar");

	if (bonfire_disc_get_status (project->priv->current) == BONFIRE_DISC_NOT_READY) {
		gtk_progress_bar_pulse (progress);
		return TRUE;
	}

	gtk_dialog_response (GTK_DIALOG (dialog), GTK_RESPONSE_OK);
	return FALSE;
}

static BonfireDiscResult
bonfire_project_check_status (BonfireProject *project,
			      BonfireDisc *disc)
{
	int id;
	int answer;
	GtkWidget *dialog;
	GtkWidget *progress;
	GtkWidget *toplevel;
	BonfireDiscResult result;

	if (project->priv->oversized) {
		bonfire_project_error_size_dialog (project);
		return BONFIRE_DISC_ERROR_SIZE;
	}

	result = bonfire_disc_get_status (disc);
	if (result != BONFIRE_DISC_NOT_READY)
		return result;

	/* we are not ready to create tracks presumably because
	 * data or audio has not finished to explore a directory
	 * or get the metadata of a song or a film */
	toplevel = gtk_widget_get_toplevel (GTK_WIDGET (project));
	dialog = gtk_message_dialog_new (GTK_WINDOW (toplevel),
					 GTK_DIALOG_DESTROY_WITH_PARENT |
					 GTK_DIALOG_MODAL,
					 GTK_MESSAGE_INFO,
					 GTK_BUTTONS_CLOSE,
					 _("Please wait:"));

	gtk_window_set_title (GTK_WINDOW (dialog), _("Please wait"));

	gtk_message_dialog_format_secondary_text (GTK_MESSAGE_DIALOG (dialog),
						  _("some tasks are not completed yet."));

	gtk_window_set_title (GTK_WINDOW (dialog), _("Waiting for ongoing tasks"));

	progress = gtk_progress_bar_new ();
	gtk_progress_bar_set_text (GTK_PROGRESS_BAR (progress), " ");
	gtk_box_pack_end (GTK_BOX (GTK_DIALOG (dialog)->vbox),
			  progress,
			  TRUE,
			  TRUE,
			  10);

	gtk_widget_show_all (dialog);
	gtk_progress_bar_pulse (GTK_PROGRESS_BAR (progress));

	g_object_set_data (G_OBJECT (dialog), "ProgressBar", progress);
	g_object_set_data (G_OBJECT (dialog), "Project", project);

	id = g_timeout_add (100,
			    (GSourceFunc) _wait_for_ready_state,
		            dialog);

	answer = gtk_dialog_run (GTK_DIALOG (dialog));
	g_source_remove (id);

	gtk_widget_destroy (dialog);

	if (answer != GTK_RESPONSE_OK)
		return BONFIRE_DISC_CANCELLED;
	else if (project->priv->oversized)
		return BONFIRE_DISC_ERROR_SIZE;

	return bonfire_disc_get_status (disc);
}

/******************************** burning **************************************/
static void
bonfire_project_no_song_dialog (BonfireProject *project)
{
	GtkWidget *message;
	GtkWidget *toplevel;

	toplevel = gtk_widget_get_toplevel (GTK_WIDGET (project));
	message = gtk_message_dialog_new (GTK_WINDOW (toplevel),
					  GTK_DIALOG_DESTROY_WITH_PARENT|
					  GTK_DIALOG_MODAL,
					  GTK_MESSAGE_WARNING,
					  GTK_BUTTONS_CLOSE,
					  _("No song in the project:"));

	gtk_window_set_title (GTK_WINDOW (message), _("Empty project"));

	gtk_message_dialog_format_secondary_text (GTK_MESSAGE_DIALOG (message),
						  _("please add songs to the project."));

	gtk_dialog_run (GTK_DIALOG (message));
	gtk_widget_destroy (message);
}

static void
bonfire_project_no_file_dialog (BonfireProject *project)
{
	GtkWidget *message;
	GtkWidget *toplevel;

	toplevel = gtk_widget_get_toplevel (GTK_WIDGET (project));
	message = gtk_message_dialog_new (GTK_WINDOW (toplevel),
					  GTK_DIALOG_DESTROY_WITH_PARENT|
					  GTK_DIALOG_MODAL,
					  GTK_MESSAGE_WARNING,
					  GTK_BUTTONS_CLOSE,
					  _("Project is empty:"));

	gtk_window_set_title (GTK_WINDOW (message), _("Empty project"));

	gtk_message_dialog_format_secondary_text (GTK_MESSAGE_DIALOG (message),
						  _("please add files to the project."));

	gtk_dialog_run (GTK_DIALOG (message));
	gtk_widget_destroy (message);
}

void
bonfire_project_burn (BonfireProject *project)
{
	BonfireDiscResult result;
	BonfireDiscTrack track;
	GtkWidget *toplevel;
	GtkWidget *dialog;

	result = bonfire_project_check_status (project, project->priv->current);
	if (result != BONFIRE_DISC_OK)
		return;

	bzero (&track, sizeof (track));
	result = bonfire_disc_get_track (project->priv->current, &track, TRUE);
	if (result == BONFIRE_DISC_CANCELLED)
		return;

	if (result == BONFIRE_DISC_ERROR_SIZE)
		return;

	if (!track.contents.src) {
		if (BONFIRE_IS_AUDIO_DISC (project->priv->current))
			bonfire_project_no_song_dialog (project);
		else
			bonfire_project_no_file_dialog (project);

		return;
	}

	toplevel = gtk_widget_get_toplevel (GTK_WIDGET (project));
	dialog = bonfire_burn_option_dialog_new (track.contents.src,
						 project->priv->overburn);

	gtk_window_set_transient_for (GTK_WINDOW (dialog), GTK_WINDOW (toplevel));
	gtk_window_set_modal (GTK_WINDOW (dialog), TRUE);
	gtk_window_set_position (GTK_WINDOW (dialog), GTK_WIN_POS_CENTER_ON_PARENT);

	gtk_widget_show_all (dialog);
}

/********************************     ******************************************/
static void
bonfire_project_switch (BonfireProject *project, gboolean audio)
{
	GdkPixbuf *pixbuf;
	GtkAction *action;

	bonfire_project_size_set_size (BONFIRE_PROJECT_SIZE (project->priv->size_display),
				       0.0,
				       NULL);

	gtk_action_group_set_visible (project->priv->action_group, TRUE);
	gtk_widget_set_sensitive (project->priv->add, TRUE);

	if (project->priv->current)
		bonfire_disc_reset (project->priv->current);

	if (project->priv->project) {
		g_free (project->priv->project);
		project->priv->project = NULL;
	}

	if (audio) {
		pixbuf = bonfire_utils_get_icon_for_mime ("gnome-dev-cdrom-audio", 24);
		gtk_label_set_markup (GTK_LABEL (project->priv->label),
				      _("<big><b>Audio tracks</b></big> (no track)"));

		project->priv->current = BONFIRE_DISC (project->priv->audio);
		gtk_notebook_set_current_page (GTK_NOTEBOOK (project->priv->discs), 0);
		bonfire_project_size_set_context (BONFIRE_PROJECT_SIZE (project->priv->size_display), TRUE);
	}
	else {
		pixbuf = bonfire_utils_get_icon_for_mime ("gnome-dev-cdrom", 24);
		gtk_label_set_markup (GTK_LABEL (project->priv->label),
				      _("<big><b>Data project</b></big>"));

		project->priv->current = BONFIRE_DISC (project->priv->data);
		gtk_notebook_set_current_page (GTK_NOTEBOOK (project->priv->discs), 1);
		bonfire_project_size_set_context (BONFIRE_PROJECT_SIZE (project->priv->size_display), FALSE);
	}

	/* set the title */
	gtk_image_set_from_pixbuf (GTK_IMAGE (project->priv->image), pixbuf);
	g_object_unref (pixbuf);
	
	/* update the menus */
	action = gtk_action_group_get_action (project->priv->project_group, "SaveAs");
	gtk_action_set_sensitive (action, TRUE);

	action = gtk_action_group_get_action (project->priv->project_group, "Save");
	gtk_action_set_sensitive (action, FALSE);
}

void
bonfire_project_set_audio (BonfireProject *project)
{
	bonfire_project_switch (project, TRUE);
}

void
bonfire_project_set_data (BonfireProject *project)
{
	bonfire_project_switch (project, FALSE);
}

void
bonfire_project_set_none (BonfireProject *project)
{
	GtkAction *action;

	if (project->priv->current)
		bonfire_disc_reset (project->priv->current);

	project->priv->current = NULL;

	/* update button */
	gtk_action_group_set_visible (project->priv->action_group, FALSE);
	gtk_widget_set_sensitive (project->priv->add, FALSE);

	/* update the menus */
	action = gtk_action_group_get_action (project->priv->project_group, "SaveAs");
	gtk_action_set_sensitive (action, FALSE);
	action = gtk_action_group_get_action (project->priv->project_group, "Save");
	gtk_action_set_sensitive (action, FALSE);
}

/********************* update the appearance of menus and buttons **************/
static void
bonfire_project_contents_changed_cb (BonfireDisc *disc,
				     int nb_files,
				     BonfireProject *project)
{
	GtkAction *action;
	gboolean sensitive;

	sensitive = (nb_files > 0);

	action = gtk_action_group_get_action (project->priv->action_group, "Delete");
	gtk_action_set_sensitive (action, sensitive);
	action = gtk_action_group_get_action (project->priv->action_group, "DeleteAll");
	gtk_action_set_sensitive (action, sensitive);
	action = gtk_action_group_get_action (project->priv->action_group, "Burn");
	gtk_action_set_sensitive (action, sensitive);

	action = gtk_action_group_get_action (project->priv->project_group, "Save");
	if (project->priv->project)
		gtk_action_set_sensitive (action, TRUE);
	else
		gtk_action_set_sensitive (action, FALSE);

	gtk_widget_set_sensitive (project->priv->remove, sensitive);
	gtk_widget_set_sensitive (project->priv->burn, sensitive);

	if (BONFIRE_IS_AUDIO_DISC (disc)) {
		char *string;

		if (nb_files == 1)
			string = g_strdup (_("<big><b>Audio tracks</b></big> (1 track)"));
		else if (nb_files)
			string = g_strdup_printf (_("<big><b>Audio tracks</b></big> (%i tracks)"),
						  nb_files);
		else
			string = g_strdup (_("<big><b>Audio tracks</b></big> (no track)"));

		gtk_label_set_markup (GTK_LABEL (project->priv->label), string);
		g_free (string);
	}
}

/**************************** manage the relations with the sources ************/
static void
bonfire_project_transfer_uris_from_src (BonfireProject *project)
{
	char **uris;
	char **uri;

	uris = bonfire_uri_container_get_selected_uris (project->priv->current_source);
	if (!uris)
		return;

	uri = uris;
	while (*uri) {
		bonfire_disc_add_uri (project->priv->current, *uri);
		uri ++;
	}
}

static void
bonfire_project_source_uri_activated_cb (BonfireURIContainer *container,
				     BonfireProject *project)
{
	project->priv->current_source = container;
	bonfire_project_transfer_uris_from_src (project);
}

static void
bonfire_project_source_uri_selected_cb (BonfireURIContainer *container,
					     BonfireProject *project)
{
	project->priv->current_source = container;
}

void
bonfire_project_add_source (BonfireProject *project,
			    BonfireURIContainer *source)
{
	g_signal_connect (source,
			  "uri-activated",
			  G_CALLBACK (bonfire_project_source_uri_activated_cb),
			  project);
	g_signal_connect (source,
			  "uri-selected",
			  G_CALLBACK (bonfire_project_source_uri_selected_cb),
			  project);
}

/******************************* menus/buttons *********************************/
static void
bonfire_project_save_cb (GtkAction *action, BonfireProject *project)
{
	bonfire_project_save_project (project);
}

static void
bonfire_project_save_as_cb (GtkAction *action, BonfireProject *project)
{
	bonfire_project_save_project_as (project);
}

static void
bonfire_project_add_uris_cb (GtkAction *action, BonfireProject *project)
{
	bonfire_project_transfer_uris_from_src (project);
}

static void
bonfire_project_remove_selected_uris_cb (GtkAction *action, BonfireProject *project)
{
	bonfire_disc_delete_selected (BONFIRE_DISC (project->priv->current));
}

static void
bonfire_project_empty_cb (GtkAction *action, BonfireProject *project)
{
	bonfire_disc_clear (BONFIRE_DISC (project->priv->current));
}

static void
bonfire_project_burn_cb (GtkAction *action, BonfireProject *project)
{
	bonfire_project_burn (project);
}

static void
bonfire_project_add_clicked_cb (GtkButton *button, BonfireProject *project)
{
	bonfire_project_transfer_uris_from_src (project);
}

static void
bonfire_project_remove_clicked_cb (GtkButton *button, BonfireProject *project)
{
	bonfire_disc_delete_selected (BONFIRE_DISC (project->priv->current));
}

static void
bonfire_project_burn_clicked_cb (GtkButton *button, BonfireProject *project)
{
	bonfire_project_burn (project);
}

void
bonfire_project_register_menu (BonfireProject *project, GtkUIManager *manager)
{
	GError *error = NULL;
	GtkAction *action;

	/* menus */
	project->priv->project_group = gtk_action_group_new ("ProjectActions1");
	gtk_action_group_set_translation_domain (project->priv->project_group, GETTEXT_PACKAGE);
	gtk_action_group_add_actions (project->priv->project_group,
				      entries_project,
				      G_N_ELEMENTS (entries_project),
				      project);

	gtk_ui_manager_insert_action_group (manager, project->priv->project_group, 0);
	if (!gtk_ui_manager_add_ui_from_string (manager,
						description_project,
						-1,
						&error)) {
		g_message ("building menus failed: %s", error->message);
		g_error_free (error);
	}

	action = gtk_action_group_get_action (project->priv->project_group, "Save");
	gtk_action_set_sensitive (action, FALSE);
	action = gtk_action_group_get_action (project->priv->project_group, "SaveAs");
	gtk_action_set_sensitive (action, FALSE);

	project->priv->action_group = gtk_action_group_new ("ProjectActions2");
	gtk_action_group_set_translation_domain (project->priv->action_group, GETTEXT_PACKAGE);
	gtk_action_group_add_actions (project->priv->action_group,
				      entries_actions,
				      G_N_ELEMENTS (entries_actions),
				      project);

	gtk_ui_manager_insert_action_group (manager, project->priv->action_group, 0);
	if (!gtk_ui_manager_add_ui_from_string (manager,
						description_actions,
						-1,
						&error)) {
		g_message ("building menus failed: %s", error->message);
		g_error_free (error);
	}

	action = gtk_action_group_get_action (project->priv->action_group, "Burn");
	gtk_action_set_sensitive (action, FALSE);
	action = gtk_action_group_get_action (project->priv->action_group, "Delete");
	gtk_action_set_sensitive (action, FALSE);
	action = gtk_action_group_get_action (project->priv->action_group, "DeleteAll");
	gtk_action_set_sensitive (action, FALSE);
}

/******************************* common to save/open ***************************/
static void
bonfire_project_set_path (BonfireProject *project,
			  const char *path,
			  BonfireDiscTrackType type)
{
	char *title;
	GtkAction *action;
	GtkWidget *toplevel;

	/* possibly reset the name of the project */
	if (path) {
		if (project->priv->project)
			g_free (project->priv->project);

		project->priv->project = g_strdup (path);
	}

	/* update the name of the main window */
	if (type == BONFIRE_DISC_TRACK_DATA)
		title = g_strdup_printf (_("Bonfire - %s (data disc)"),
					 project->priv->project);
	else
		title = g_strdup_printf (_("Bonfire - %s (audio disc)"),
					 project->priv->project);

	toplevel = gtk_widget_get_toplevel (GTK_WIDGET (project));
	gtk_window_set_title (GTK_WINDOW (toplevel), title);
	g_free (title);

	/* update the menus */
	action = gtk_action_group_get_action (project->priv->project_group, "Save");
	gtk_action_set_sensitive (action, FALSE);
}

/******************************* Projects **************************************/
static void
bonfire_project_invalid_project_dialog (BonfireProject *project,
					const char *reason)
{
	GtkWidget *dialog;
	GtkWidget *toplevel;

	toplevel = gtk_widget_get_toplevel (GTK_WIDGET (project));
	dialog = gtk_message_dialog_new (GTK_WINDOW (toplevel),
					 GTK_DIALOG_DESTROY_WITH_PARENT |
					 GTK_DIALOG_MODAL,
					 GTK_MESSAGE_ERROR,
					 GTK_BUTTONS_CLOSE,
					 _("Error while loading the project:"));

	gtk_window_set_title (GTK_WINDOW (dialog), _("Project loading error"));

	gtk_message_dialog_format_secondary_text (GTK_MESSAGE_DIALOG (dialog),
						  reason);

	gtk_dialog_run (GTK_DIALOG (dialog));
	gtk_widget_destroy (dialog);
}

static BonfireGraftPt *
_read_graft_point (xmlDocPtr project,
		   xmlNodePtr graft)
{
	BonfireGraftPt *retval;

	retval = g_new0 (BonfireGraftPt, 1);
	while (graft) {
		if (!xmlStrcmp (graft->name, (const xmlChar *) "uri")) {
			if (retval->uri)
				goto error;

			retval->uri = (char*) xmlNodeListGetString (project,
							    graft->xmlChildrenNode,
							    1);
			if (!retval->uri)
				goto error;
		}
		else if (!xmlStrcmp (graft->name, (const xmlChar *) "path")) {
			if (retval->path)
				goto error;

			retval->path = (char *) xmlNodeListGetString (project,
								      graft->xmlChildrenNode,
								      1);
			if (!retval->path)
				goto error;
		}
		else if (!xmlStrcmp (graft->name, (const xmlChar *) "excluded")) {
			xmlChar *excluded;

			excluded = xmlNodeListGetString (project,
							 graft->xmlChildrenNode,
							 1);
			if (!excluded)
				goto error;

			retval->excluded = g_slist_prepend (retval->excluded,
							    excluded);
		}
		else if (graft->type == XML_ELEMENT_NODE)
			goto error;

		graft = graft->next;
	}

	retval->excluded = g_slist_reverse (retval->excluded);
	return retval;

error:
	bonfire_graft_point_free (retval);
	return NULL;
}

static BonfireDiscTrack *
_read_data_track (xmlDocPtr project,
		  xmlNodePtr item)
{
	BonfireDiscTrack *track;

	track = g_new0 (BonfireDiscTrack, 1);
	track->type = BONFIRE_DISC_TRACK_DATA;

	while (item) {
		if (!xmlStrcmp (item->name, (const xmlChar *) "graft")) {
			BonfireGraftPt *graft;

			graft = _read_graft_point (project, item->xmlChildrenNode);
			if (!graft)
				goto error;

			track->contents.data.grafts = g_slist_prepend (track->contents.data.grafts, graft);
		}
		else if (!xmlStrcmp (item->name, (const xmlChar *) "restored")) {
			xmlChar *restored;

			restored = xmlNodeListGetString (project,
							 item->xmlChildrenNode,
							 1);
			if (!restored)
				goto error;

			track->contents.data.restored = g_slist_prepend (track->contents.data.restored, restored);
		}
		else if (item->type == XML_ELEMENT_NODE)
			goto error;

		item = item->next;
	}

	track->contents.data.grafts = g_slist_reverse (track->contents.data.grafts);
	return (BonfireDiscTrack *) track;

error :
	bonfire_track_free ((BonfireDiscTrack*) track);
	return NULL;
}

static BonfireDiscTrack *
_read_audio_track (xmlDocPtr project,
		   xmlNodePtr uris)
{
	BonfireDiscTrack *track;

	track = g_new0 (BonfireDiscTrack, 1);
	track->type = BONFIRE_DISC_TRACK_AUDIO;

	while (uris) {
		if (!xmlStrcmp (uris->name, (const xmlChar *) "uri")) {
			char *uri;

			uri = (char *) xmlNodeListGetString (project,
						    uris->xmlChildrenNode,
						    1);
			if (!uri)
				goto error;

			track->contents.uris = g_slist_prepend (track->contents.uris, uri);
		}
		else if (uris->type == XML_ELEMENT_NODE)
			goto error;

		uris = uris->next;
	}

	track->contents.uris = g_slist_reverse (track->contents.uris);
	return (BonfireDiscTrack*) track;

error:
	bonfire_track_free ((BonfireDiscTrack *) track);
	return NULL;
}

static gboolean
_get_tracks (xmlDocPtr project,
	     xmlNodePtr track_node,
	     BonfireDiscTrack **track)
{
	BonfireDiscTrack *newtrack;

	track_node = track_node->xmlChildrenNode;

	newtrack = NULL;
	while (track_node) {
		if (!xmlStrcmp (track_node->name, (const xmlChar *) "audio")) {
			if (newtrack)
				goto error;

			newtrack = _read_audio_track (project,
						      track_node->xmlChildrenNode);
			if (!newtrack)
				goto error;
		}
		else if (!xmlStrcmp (track_node->name, (const xmlChar *) "data")) {
			if (newtrack)
				goto error;

			newtrack = _read_data_track (project,
						     track_node->xmlChildrenNode);

			if (!newtrack)
				goto error;
		}
		else if (track_node->type == XML_ELEMENT_NODE)
			goto error;

		track_node = track_node->next;
	}

	if (!newtrack)
		goto error;

	*track = newtrack;
	return TRUE;

error :
	if (newtrack)
		bonfire_track_free (newtrack);

	bonfire_track_free (newtrack);
	return FALSE;
}

static gboolean
bonfire_project_open_project_xml (BonfireProject *proj,
				  const char *uri,
				  BonfireDiscTrack **track)
{
	xmlNodePtr track_node = NULL;
	xmlDocPtr project;
	xmlChar *version;
	xmlNodePtr item;
	gboolean retval;

	/* start parsing xml doc */
	project = xmlParseFile (uri);
	if (!project) {
		bonfire_project_invalid_project_dialog (proj, _("the project could not be opened."));
		return FALSE;
	}

	/* parses the "header" */
	item = xmlDocGetRootElement (project);
	if (!item) {
		bonfire_project_invalid_project_dialog (proj, _("the file is empty."));
		xmlFreeDoc (project);
		return FALSE;
	}

	version = NULL;
	if (xmlStrcmp (item->name, (const xmlChar *) "bonfireproject")
	||  item->next)
		goto error;

	item = item->children;
	while (item) {
		if (!xmlStrcmp (item->name, (const xmlChar *) "version")) {
			if (version)
				goto error;

			version =  xmlNodeListGetString (project,
							 item->xmlChildrenNode,
							 1);
		}
		else if (!xmlStrcmp (item->name, (const xmlChar *) "track")) {
			if (track_node)
				goto error;

			track_node = item;
		}
		else if (item->type == XML_ELEMENT_NODE)
			goto error;


		item = item->next;
	}

	if (!version)
		goto error;

	if (strcmp ((char*) version, BONFIRE_PROJECT_VERSION))
		goto error;

	xmlFree (version);
	retval = _get_tracks (project, track_node, track);
	xmlFreeDoc (project);

	if (!retval)
		bonfire_project_invalid_project_dialog (proj,
							_("it doesn't seem to be a valid bonfire project."));

	return retval;

error:

	if (version)
		xmlFree (version);

	xmlFreeDoc (project);
	bonfire_project_invalid_project_dialog (proj, _("it doesn't seem to be a valid bonfire project."));
	return FALSE;
}

BonfireProjectType
bonfire_project_open_project (BonfireProject *project,
			      const char *name)
{
	BonfireDiscTrack *track = NULL;
	BonfireProjectType type;
	GtkWidget *toplevel;
	char *path;

	if (!name) {
		GtkWidget *chooser;
		int answer;

		toplevel = gtk_widget_get_toplevel (GTK_WIDGET (project));
		chooser = gtk_file_chooser_dialog_new (_("Open a project"),
						      GTK_WINDOW (toplevel),
						      GTK_FILE_CHOOSER_ACTION_OPEN,
						      GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL,
						      GTK_STOCK_OK, GTK_RESPONSE_OK,
						      NULL);
		gtk_file_chooser_set_local_only (GTK_FILE_CHOOSER (chooser), FALSE);
		gtk_file_chooser_set_current_folder (GTK_FILE_CHOOSER (chooser),
						     g_get_home_dir ());
	
		gtk_widget_show (chooser);
		answer = gtk_dialog_run (GTK_DIALOG (chooser));
	
		if (answer == GTK_RESPONSE_OK)
			path = gtk_file_chooser_get_filename (GTK_FILE_CHOOSER (chooser));
		else
			path = NULL;
	
		gtk_widget_destroy (chooser);
	}
	else
		path = g_strdup (name);

	if (!path || *path =='\0')
		return BONFIRE_PROJECT_TYPE_INVALID;

	if (!bonfire_project_open_project_xml (project, path, &track)) {
		g_free (path);
		return BONFIRE_PROJECT_TYPE_INVALID;
	}

	bonfire_project_size_set_size (BONFIRE_PROJECT_SIZE (project->priv->size_display),
				       0.0,
				       NULL);

	if (track->type == BONFIRE_DISC_TRACK_AUDIO) {
		bonfire_project_switch (project, TRUE);
		type = BONFIRE_PROJECT_TYPE_AUDIO;
	}
	else if (track->type == BONFIRE_DISC_TRACK_DATA) {
		bonfire_project_switch (project, FALSE);
		type = BONFIRE_PROJECT_TYPE_DATA;
	}
	else 
		return BONFIRE_PROJECT_TYPE_INVALID;

	bonfire_disc_load_track (project->priv->current, track);
	bonfire_track_free (track);

	bonfire_project_set_path (project, path, type);
	return type;
}

/******************************** save project *********************************/
static void
bonfire_project_not_saved_dialog (BonfireProject *project)
{
	GtkWidget *dialog;
	xmlErrorPtr error;
	GtkWidget *toplevel;

	toplevel = gtk_widget_get_toplevel (GTK_WIDGET (project));
	dialog = gtk_message_dialog_new (GTK_WINDOW (toplevel),
					 GTK_DIALOG_DESTROY_WITH_PARENT|
					 GTK_DIALOG_MODAL,
					 GTK_MESSAGE_WARNING,
					 GTK_BUTTONS_CLOSE,
					 _("Your project has not been saved:"));

	gtk_window_set_title (GTK_WINDOW (dialog), _("Unsaved project"));

	error = xmlGetLastError ();
	if (error)
		gtk_message_dialog_format_secondary_text (GTK_MESSAGE_DIALOG (dialog),
							  error->message);
	else
		gtk_message_dialog_format_secondary_text (GTK_MESSAGE_DIALOG (dialog),
							  _("Unknown error."));	
	xmlResetLastError ();

	gtk_dialog_run (GTK_DIALOG (dialog));
	gtk_widget_destroy (dialog);
}

static gboolean
_save_audio_track_xml (xmlTextWriter *project,
		       BonfireDiscTrack *track)
{
	GSList *iter;
	int success;
	char *uri;

	for (iter = track->contents.uris; iter; iter = iter->next) {
		uri = iter->data;
		success = xmlTextWriterWriteElement (project, (xmlChar *) "uri", (xmlChar *) uri);
		if (success == -1)
			return FALSE;
	}

	return TRUE;
}

static gboolean
_save_data_track_xml (xmlTextWriter *project,
		      BonfireDiscTrack *track)
{
	char *uri;
	int success;
	GSList *iter;
	GSList *grafts;
	BonfireGraftPt *graft;

	for (grafts = track->contents.data.grafts; grafts; grafts = grafts->next) {
		graft = grafts->data;

		success = xmlTextWriterStartElement (project, (xmlChar *) "graft");
		if (success < 0)
			return FALSE;

		success = xmlTextWriterWriteElement (project, (xmlChar *) "path", (xmlChar *) graft->path);
		if (success < 0)
			return FALSE;

		if (graft->uri) {
			success = xmlTextWriterWriteElement (project, (xmlChar *) "uri", (xmlChar *) graft->uri);
			if (success < 0)
				return FALSE;
		}

		for (iter = graft->excluded; iter; iter = iter->next) {
			uri = iter->data;
			success = xmlTextWriterWriteElement (project, (xmlChar *) "excluded", (xmlChar *) uri);
			if (success < 0)
				return FALSE;
		}

		success = xmlTextWriterEndElement (project); /* graft */
		if (success < 0)
			return FALSE;
	}

	/* save restored uris */
	for (iter = track->contents.data.restored; iter; iter = iter->next) {
		uri = iter->data;
		success = xmlTextWriterWriteElement (project, (xmlChar *) "restored", (xmlChar *) uri);
		if (success < 0)
			return FALSE;
	}

	/* NOTE: we don't write symlinks and unreadable they are useless */
	return TRUE;
}

static gboolean 
bonfire_project_save_project_xml (BonfireProject *proj,
				  const char *path,
				  BonfireDiscTrack *track)
{
	xmlTextWriter *project;
	gboolean retval;
	int success;

	project = xmlNewTextWriterFilename (path, 0);
	if (!project) {
		bonfire_project_not_saved_dialog (proj);
		return FALSE;
	}

	xmlTextWriterSetIndent (project, 1);
	xmlTextWriterSetIndentString (project, (xmlChar *) "\t");

	success = xmlTextWriterStartDocument (project,
					      NULL,
					      NULL,
					      NULL);
	if (success < 0)
		goto error;

	success = xmlTextWriterStartElement (project, (xmlChar *) "bonfireproject");
	if (success < 0)
		goto error;

	/* write the name of the version */
	success = xmlTextWriterWriteElement (project,
					     (xmlChar *) "version",
					     (xmlChar *) BONFIRE_PROJECT_VERSION);
	if (success < 0)
		goto error;

	success = xmlTextWriterStartElement (project, (xmlChar *) "track");
	if (success < 0)
		goto error;

	if (track->type == BONFIRE_DISC_TRACK_AUDIO) {
		success = xmlTextWriterStartElement (project, (xmlChar *) "audio");
		if (success < 0)
			goto error;

		retval = _save_audio_track_xml (project, track);
		if (!retval)
			goto error;

		success = xmlTextWriterEndElement (project); /* audio */
		if (success < 0)
			goto error;
	}
	else if (track->type == BONFIRE_DISC_TRACK_DATA) {
		success = xmlTextWriterStartElement (project, (xmlChar *) "data");
		if (success < 0)
			goto error;

		retval = _save_data_track_xml (project, track);
		if (!retval)
			goto error;

		success = xmlTextWriterEndElement (project); /* data */
		if (success < 0)
			goto error;
	}
	else 
		retval = FALSE;

	success = xmlTextWriterEndElement (project); /* track */
	if (success < 0)
		goto error;

	success = xmlTextWriterEndElement (project); /* bonfireproject */
	if (success < 0)
		goto error;

	xmlTextWriterEndDocument (project);
	xmlFreeTextWriter (project);
	return TRUE;

error:
	xmlTextWriterEndDocument (project);
	xmlFreeTextWriter (project);
	g_remove (path);
	
	bonfire_project_not_saved_dialog (proj);
	return FALSE;
}

static gboolean
bonfire_project_save_project_real (BonfireProject *project,
				   const char *path)
{
	BonfireDiscResult result;
	BonfireDiscTrack track;
	gboolean retval;

	g_return_val_if_fail (path != NULL || project->priv->project != NULL, FALSE);

	result = bonfire_project_check_status (project, project->priv->current);
	if (result != BONFIRE_DISC_OK)
		return FALSE;

	bzero (&track, sizeof (track));
	result = bonfire_disc_get_track (project->priv->current, &track, FALSE);
	if (result == BONFIRE_DISC_ERROR_EMPTY_SELECTION) {
		if (BONFIRE_IS_AUDIO_DISC (project->priv->current))
			bonfire_project_no_song_dialog (project);
		else if (BONFIRE_IS_DATA_DISC (project->priv->current))
			bonfire_project_no_file_dialog (project);

		return FALSE;
	}
	else if (result != BONFIRE_DISC_OK) {
		bonfire_project_not_saved_dialog (project);
		return FALSE;
	}

	retval = bonfire_project_save_project_xml (project,
						   path ? path : project->priv->project,
						   &track);

	if (retval)
		bonfire_project_set_path (project, path, track.type);

	bonfire_track_clear (&track);
	return retval;
}

static char *
bonfire_project_save_project_ask_for_path (BonfireProject *project)
{
	GtkWidget *toplevel;
	GtkWidget *chooser;
	char *path = NULL;
	int answer;

	toplevel = gtk_widget_get_toplevel (GTK_WIDGET (project));
	chooser = gtk_file_chooser_dialog_new (_("Save current project"),
					       GTK_WINDOW (toplevel),
					       GTK_FILE_CHOOSER_ACTION_SAVE,
					       GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL,
					       GTK_STOCK_OK, GTK_RESPONSE_OK,
					       NULL);
	gtk_file_chooser_set_local_only (GTK_FILE_CHOOSER (chooser), FALSE);
	gtk_file_chooser_set_current_folder (GTK_FILE_CHOOSER (chooser),
					     g_get_home_dir ());

	gtk_widget_show (chooser);
	answer = gtk_dialog_run (GTK_DIALOG (chooser));
	if (answer == GTK_RESPONSE_OK) {
		path = gtk_file_chooser_get_filename (GTK_FILE_CHOOSER (chooser));
		if (*path == '\0') {
			g_free (path);
			path = NULL;
		}
	}

	gtk_widget_destroy (chooser);
	return path;
}

gboolean
bonfire_project_save_project (BonfireProject *project)
{
	char *path = NULL;
	gboolean result;

	if (!project->priv->project
	&&  !(path = bonfire_project_save_project_ask_for_path (project)))
		return FALSE;

	result = bonfire_project_save_project_real (project, path);
	g_free (path);

	return result;
}

gboolean
bonfire_project_save_project_as (BonfireProject *project)
{
	gboolean result;
	char *path;

	path = bonfire_project_save_project_ask_for_path (project);
	if (!path)
		return FALSE;

	result = bonfire_project_save_project_real (project, path);
	g_free (path);

	return result;
}
