#include <stdio.h>
#include <string.h>
#include <signal.h>
#include <errno.h>
#ifndef _TESTING_
# include <sys/reboot.h>
#endif
#ifdef _TESTING_
# define RB_AUTOBOOT 1
void reboot(int blah);
#endif
#include <sys/types.h>
#include <fcntl.h>
#include <slang.h>
#include <newt.h>
#include <dirent.h>

#ifdef USE_LANGUAGE_CHOOSER
#include <wchar.h>
#endif

/* shrink_string is used for correct line break in euc-jp encoded strings */
extern char *
shrink_string(char *dest, char *source, int maxlinelen, int maxbuffersize);

/* This part is only defined for EUC-JP Japanese localized b-f.  -- dancerj */
#ifdef USE_SHRINK_STRING
char tmptext[BUFSIZ];
#define SHRINK_STRING(text,width)                      \
    (text) = (const char *)                            \
      shrink_string(tmptext, (char *)(text), (width) - 1, BUFSIZ)
#else
#define SHRINK_STRING(text,width);
#endif

#include "dbootstrap.h"
#include "lang.h"
#include "util.h"

#define LOCALBUFSIZE (256)

#ifndef NEWT_KEY_ESC
#define NEWT_KEY_ESC    0x1b
#endif

    static int choose_dir (const char *, const char *, const char *, char *, size_t);
#ifdef USE_LANGUAGE_CHOOSER
    static const struct language_item *boxChooseLanguageHelper (const struct language_list *);
#endif

int stderrToTTY(int tty) {
	static int fd=-1;
	char dev[10];

	/* stderr redirect only works on 2.0 kernels */
	/* FIXME: we need to figure out why it fails on 2.2 kernels.
	 *        In the meantime skip it. */
	if (strncmp(kver,"2.0",3))
		return 0;
	if (fd > 0) 
		close(fd);
	snprintf(dev,10,"/dev/tty%d",tty);
	if ((fd = open(dev, O_RDWR|O_NOCTTY)) < 0)
		return 1;
	if (-1 == dup2(fd,2))
		return 1;
	return 0;
}

void boxResume(void) {
	stderrToTTY(3);
	newtResume();
}

void boxSuspend(void) {
	newtSuspend();
	stderrToTTY(1);
}

void boxPopWindow(void) {
	newtPopWindow();
}

void boxFinished(void) {
	newtFinished();
	stderrToTTY(1);
}

static void suspend(void *data) {
	boxSuspend();
	raise(SIGTSTP);
	boxResume();
}

void boxInit (void) {
/* Print 24 blank lines right before erasing the screen. That way,
 * the user may see the boot messages using Shift-PgUp. */
	int w, h, j;
	char* msg;
	newtSetThreeD(1);
	newtInit();

	newtGetScreenSize(&w,&h);
	for(j=0;j<h;j++)
	    putchar('\n');
	newtCls();
	newtSetSuspendCallback(suspend, NULL); 
    /* MSS: I do think this should not be localized, as the boxInit
     * stuff is called BEFORE language chooser (or something else)
     * allowed the user to choose the language.
     */ 
	msg = "Debian GNU/Linux System Installation";
	newtDrawRootText((w-strlen(msg))/2, -5, msg);
	stderrToTTY(3);
}

void setMono(void) {
	if (!bootargs.ismono)
	{
		SLtt_Use_Ansi_Colors = 1;
		unsetenv("NEWT_MONO");
	}
	else
	{
		SLtt_Use_Ansi_Colors = 0;
		setenv("NEWT_MONO","1",1);
	}
}

/* A lot of this functionality is in newt-0.25 windows.c .
 * It would be nice to merge both works...
 */
#define PLEASEWAITBOX_LEN 59

int pleaseWaitBox(const char *text) {
    int height;
    newtComponent f1, t1;
    newtCenteredWindow (PLEASEWAITBOX_LEN + 1, 5, _("Please Wait"));
    f1 = newtForm(NULL, NULL, 0);
    t1 = newtTextbox (1, 1, PLEASEWAITBOX_LEN, 4, NEWT_FLAG_WRAP);
    SHRINK_STRING(text, PLEASEWAITBOX_LEN);
    newtTextboxSetText(t1,text);
    height=newtTextboxGetNumLines(t1);
    newtTextboxSetHeight(t1,height);
    newtFormAddComponent (f1 , t1);
    newtDrawForm(f1);
    newtRefresh();
    newtFormDestroy(f1);
    return 0;
}

int vaproblemBox(const char *title, const char *fmt, ...) {
  char *p;
  va_list ap;
  if ((p = malloc(256)) == NULL)
    return -1;
  
  va_start(ap, fmt);
  (void) vsnprintf(p, 256, fmt, ap);
  va_end(ap);

  problemBox(p, title);
  return 0;
}


#define PROBLEMBOX_LEN 60

static int _problemBox(const char *text, const char *title, int localize)
{
    newtComponent f1, t1, b1;
    int height;

    ERRMSG(text);
    t1 = newtTextbox(1, 1, PROBLEMBOX_LEN, 10, NEWT_FLAG_WRAP);
    SHRINK_STRING(text, PROBLEMBOX_LEN);
    newtTextboxSetText(t1, text);
    height = newtTextboxGetNumLines(t1);
    newtTextboxSetHeight(t1,height);
    newtCenteredWindow(PROBLEMBOX_LEN + 1, height + 3, title);
    if (localize)
	b1 = newtCompactButton(23, height + 2, _("Continue"));
    else
	b1 = newtCompactButton(23, height + 2, "Continue");
    f1 = newtForm(NULL, NULL, 0);
    newtFormAddComponents(f1, t1, b1, NULL);
    newtRunForm(f1);
    newtPopWindow();
    newtFormDestroy(f1);
    return 0;
}

int problemBoxEn(const char *text, const char *title) {
   return _problemBox(text,title,0);
}

int problemBox(const char *text, const char *title) {
   return _problemBox(text,title,1);
}

int wideMessageBox (const char *text, const char *title)
{
  newtComponent   f1, t1, b1;
  int             cols, rows, text_width, text_height;
  int             win_height, win_width, x_offset, y_offset;
  char           *cp;
#ifndef USE_LANGUAGE_CHOOSER
  char           *cpp;
#endif

  newtGetScreenSize (&cols, &rows);
#ifdef USE_LANGUAGE_CHOOSER
  {
    int new_width, k;
    wchar_t c;

    text_width = strwidth (title);

    mbtowc (0, 0, 0);

    for (text_height = 0, new_width = 0, cp = (char *)text; *cp != '\0' && (k = mbtowc (&c, cp, MB_LEN_MAX)) > 0; cp += k)
        if (*cp == '\n')
        {
            text_height++;

            if (new_width > text_width)
                text_width = new_width;

            new_width = 0;
        }
        else
            new_width += wcwidth (c);
  }
#else
  text_width = strlen (title);

  for (text_height = 0, cpp = cp = (char *)text; *cp != '\0'; cp++)
    if (*cp == '\n')
      {
        text_height++;
        if (cp - cpp > text_width)
          text_width = cp - cpp;
        cpp = cp;
      }
#endif
  text_width  += 5;
  text_height += 5;

  win_height = (text_height > rows - 2) ? rows - 2 : text_height;
  win_width  = (text_width  > cols - 4) ? cols - 4 : text_width;

  x_offset   = (win_width   < cols) ? (cols - win_width)  / 2 : 0;
  y_offset   = (win_height  < rows) ? (rows - win_height) / 2 : 0;

  SHRINK_STRING(text, text_width);
  t1 = newtTextbox (1, 0, win_width - 4, win_height - 4, NEWT_FLAG_WRAP | NEWT_FLAG_SCROLL);
  newtTextboxSetText (t1, text);
  newtOpenWindow (x_offset + 1, y_offset + 1, win_width - 2, win_height - 3, title);
#ifdef USE_LANGUAGE_CHOOSER
  b1 = newtCompactButton ((win_width - strwidth (_("Continue"))) / 2 - 1, win_height - 4, _("Continue"));
#else
  b1 = newtCompactButton ((win_width - strlen (_("Continue"))) / 2 - 1, win_height - 4, _("Continue"));
#endif
  f1 = newtForm (NULL, NULL, 0);
  newtFormAddComponents (f1, t1, b1, NULL);
  newtRunForm (f1);
  newtPopWindow ();
  newtFormDestroy (f1);

  return 0;
}

int perrorBox(const char *text) {
    char buf[LOCALBUFSIZE];
    snprintf(buf,LOCALBUFSIZE,"%s: %s",text,strerror(errno));
    problemBox(buf,_("Error"));
    return 0;
}

/* Return Value:
 *   DLG_YES (-1)   left button pressed
 *   DLG_NO (0)    right button pressed
 */
#define TWOBOTTONBOX_LEN 60

int twoButtonBox(const char *text, const char *title, const char *button1, const char* button2, int def) {
    newtComponent f1, t1, b1, b2, ans;
    int height;

    t1 = newtTextbox (1, 1, TWOBOTTONBOX_LEN, 10, NEWT_FLAG_WRAP);
    SHRINK_STRING(text, TWOBOTTONBOX_LEN);
    newtTextboxSetText(t1, text);
    height=newtTextboxGetNumLines(t1);
    newtTextboxSetHeight(t1,height);
    newtCenteredWindow (TWOBOTTONBOX_LEN + 1, height+3, title);
    b1 = newtCompactButton ( 20, height+2, button1 );
    b2 = newtCompactButton ( 32, height+2, button2 );
    f1 = newtForm(NULL, NULL, 0);
    newtFormAddComponents (f1, t1, b1, b2, NULL);
    if (def == 2)
      newtFormSetCurrent (f1, b2);
    ans = newtRunForm(f1);
    newtPopWindow();
    newtFormDestroy(f1);
    if (ans == b1) return DLG_YES;
    return DLG_NO;
}

/* Return Value:
 *   DLG_YES (-1)    YES button pressed
 *   DLG_NO (0)       NO button pressed
 */
int yesNoBox(const char *text, const char *title) {
    return twoButtonBox(text, title, _("Yes"), _("No"), 1);
}

#define INPUTBOX_LEN 60

char *inputBox(const char *text, const char *title, const char *proto) { 
    newtComponent f1, t1, i1, b1, b2, ans;
    char *val, *res;
    int height;

    t1 = newtTextbox (1, 1, INPUTBOX_LEN, 10, NEWT_FLAG_WRAP);
    SHRINK_STRING(text, INPUTBOX_LEN);
    newtTextboxSetText(t1, text);
    height=newtTextboxGetNumLines(t1);
    newtTextboxSetHeight(t1,height);
    newtCenteredWindow (INPUTBOX_LEN + 1, height+5, title);
    i1 = newtEntry(1, height+2, proto, INPUTBOX_LEN - 1, &val, 
                     NEWT_FLAG_SCROLL | NEWT_FLAG_RETURNEXIT);
    b1 = newtCompactButton ( 20, height+4, _("OK"));
    b2 = newtCompactButton ( 32, height+4, _("Cancel"));
    f1 = newtForm(NULL, NULL, 0);
    newtFormAddComponents (f1, t1, i1, b1, b2, NULL);
    ans = newtRunForm(f1);
    res = strdup(val);
    newtPopWindow();
    newtFormDestroy(f1);
    if (ans != b2) return res;
    free(res);
    return NULL;
}

/* Return Values:
 *   0..oo       Selected item
 *  -1           Cancel
 */
#define _MAX_LIST_HEIGHT  (15)
#define _MIN_WIDTH  (60)
#define _MAX_WIDTH  (78)
int
menuBox (const char* text, const char* title, struct d_choices* choices, int nchoices, int cancel)
{
    return menuBox2 (text, title, choices, nchoices, cancel, -1);
}

int
menuBox2 (const char* text, const char* title, struct d_choices* choices, int nchoices, int cancel, int def)
{
  char buf[LOCALBUFSIZE];
  char format1[40],format2[40];
  newtComponent form, textbox, listbox, button, ans;
  int txheight, height, ix, string, tag, width;
  int mstring1=0, mstring2=0, mtag=0, lwidth=0;
  int result, *rs, *entry=calloc(nchoices,sizeof(int));

  pushHelpLine(_("<Up>/<Down> between elements   |  <Enter> selects"), 0);

  for (ix=0;ix<nchoices;ix++) {
      if (choices[ix].tag)
#ifdef USE_LANGUAGE_CHOOSER
        tag=strwidth(choices[ix].tag);
#else
        tag=strlen(choices[ix].tag);
#endif
      else
	tag=0;
      if(tag>mtag)
        mtag=tag;
      if (choices[ix].string)
#ifdef USE_LANGUAGE_CHOOSER
        string=strwidth(choices[ix].string);
#else
        string=strlen(choices[ix].string);
#endif
      else
	string=0;
      if (tag) {
	if (string > mstring2)
          mstring2=string;
	if (lwidth < mstring2+mtag+3)
	  lwidth=mstring2+mtag+3;
      } else {
	if (string > mstring1)
          mstring1=string;
	if (lwidth < mstring1+1)
	  lwidth=mstring1+1;
      }
  }
  if (mtag > 0)
    snprintf(format2,40,"%%-%ds: %%-%ds ",mtag,mstring2);
  snprintf(format1,40,"%%-%ds ",mstring1);

  lwidth+=7;
  width=((lwidth>_MIN_WIDTH) ?  ((lwidth<_MAX_WIDTH) ? lwidth : _MAX_WIDTH ) : _MIN_WIDTH);

  SHRINK_STRING(text, width);
  textbox = newtTextbox (1, 1, width, 5, NEWT_FLAG_WRAP);
  newtTextboxSetText(textbox, text);
  txheight = newtTextboxGetNumLines(textbox);
  newtTextboxSetHeight(textbox,txheight);

  height = _MAX_LIST_HEIGHT - txheight;
  height = ((nchoices<height) ? nchoices : height) + 2;
  listbox = newtListbox((width-lwidth)/2+1, txheight+2, height,
            NEWT_FLAG_BORDER | NEWT_FLAG_RETURNEXIT);
  height += txheight+1;

  newtCenteredWindow(width+2, height+2+cancel, title);

  for (ix=0;ix<nchoices;ix++) {
    if (mtag > 0) {
      if (choices[ix].string) {
        if (choices[ix].tag) {
          snprintf(buf,LOCALBUFSIZE,format2,choices[ix].tag,choices[ix].string);
        } else {
          snprintf(buf,LOCALBUFSIZE,format1,choices[ix].string);
        }
      } else {
        if (choices[ix].tag) {
          snprintf(buf,LOCALBUFSIZE,format2,choices[ix].tag,"");
        } else {
          // snprintf(buf,LOCALBUFSIZE,"");
          buf[0] = '\0';
        }
      }
    } else {
      if (choices[ix].string) {
        snprintf(buf,LOCALBUFSIZE,format1,choices[ix].string);
      } else {
        // snprintf(buf,LOCALBUFSIZE,"");
        buf[0] = '\0';
      }
    }
    entry[ix]=ix;
    if (strlen(buf))
      newtListboxAddEntry(listbox,buf,&entry[ix]);
    else
      newtListboxAddEntry(listbox,buf,NULL);
  }

  form = newtForm(NULL, NULL, 0);
  newtFormAddComponents(form, textbox, listbox, NULL);

  if (def != -1)
    newtListboxSetCurrent (listbox, def);

  sprintf(buf,_("Cancel"));
#ifdef USE_LANGUAGE_CHOOSER
  button = newtCompactButton ( width/2-strwidth (buf)/2, height+2, buf );
#else
  button = newtCompactButton ( width/2-strlen(buf)/2, height+2, buf );
#endif
  if (cancel) {
    newtFormAddComponent(form, button);
  }

  do {
    ans = newtRunForm(form);
    rs = newtListboxGetCurrent(listbox);
  } while ((rs == NULL) && (ans != button));

  if (ans == button) 
    result=-1;
  else
    result=*rs;

  newtPopWindow();
  newtFormDestroy(form);

  popHelpLine ();
  free(entry);

  return result;
}

#define SCALEBOX_LEN 49

int
scaleBox(const char *text, const char *title, long long value, int action)
{
  static newtComponent tx=NULL, scale=NULL, f=NULL;

  switch (action) {
    case SCALE_CREATE:
      if (f) return -1;
      newtCenteredWindow(SCALEBOX_LEN + 1, 6, title);
      f = newtForm(NULL, NULL, 0);
      tx= newtTextbox(1, 1, SCALEBOX_LEN, 2, NEWT_FLAG_WRAP);
      SHRINK_STRING(text, SCALEBOX_LEN);
      newtTextboxSetText(tx, text);
      scale = newtScale(10, 4, 30, value);
      newtFormAddComponents(f, tx, scale, NULL);
      newtDrawForm(f);
      newtRefresh();
      break;;
    case SCALE_REFRESH:
      if (!f) return -1;
      newtScaleSet(scale, value);
      newtRefresh();
      break;;
    case SCALE_DELETE:
      if (!f) return -1;
      newtPopWindow();
      newtFormDestroy(f);
      f=NULL;
      break;;
  }
  return 0;
}
	  
#define _ButtonH 1
#ifdef PCMCIA

int checkBox(const char* text, const char* title, int height, int width,
             char** choices, char** values, int nchoices)
{
  newtComponent form,tbox,okay,cancel,sform;
  newtComponent answer;
  newtComponent *chk;
  newtComponent sb=NULL;
  int rc=DLG_OKAY;
  int top,ix;
  char *result;

  int lheight=(nchoices>height - _ButtonH - 6)
    ? height - _ButtonH - 6 : nchoices;

  chk=calloc(nchoices,sizeof(newtComponent));
  if(!chk)
    return DLG_ERROR;

  result=calloc(nchoices+1,sizeof(char));
  if(!result)
    return DLG_ERROR;
  else
    result[nchoices]='\0';

  newtCenteredWindow (width, height, title);

  form=newtForm(NULL, NULL, 0);
  tbox=newtTextbox(1, 0, width-2, 10, NEWT_FLAG_WRAP);
  SHRINK_STRING(text, width - 2);
  newtTextboxSetText(tbox, text);
  top=newtTextboxGetNumLines(tbox);
  newtTextboxSetHeight(tbox,top);
  
  if(lheight!=nchoices) /* We need a scrollbar. */
    {
      sb=newtVerticalScrollbar(width - 4, top + 1,
                   lheight,
                               NEWT_COLORSET_CHECKBOX,
                               NEWT_COLORSET_ACTCHECKBOX);
      newtFormAddComponent(form,sb);
    }

  sform=newtForm(sb,NULL,0);
  newtFormSetBackground(sform, NEWT_COLORSET_CHECKBOX);
  for(ix=0;ix<nchoices;ix++)
    {
      chk[ix]=newtCheckbox(4 , top + 1 + ix, choices[ix],
               (*values)[ix],NULL,&result[ix]);
      newtFormAddComponent(sform, chk[ix]);
    }

  newtFormSetHeight(sform, lheight);
  newtFormSetWidth(sform,width - 10);
  okay = newtCompactButton ( (width-18)/3, height-_ButtonH, _("OK"));
  cancel = newtCompactButton ( (width-18)/3*2+9, height-_ButtonH, _("Cancel"));
  newtFormAddComponents(form,tbox,sform,okay,cancel,NULL);
  
  answer=newtRunForm(form);

  if(answer==cancel)
    rc=DLG_CANCEL;
  else
    strcpy(*values,result);

  newtPopWindow();
  newtFormDestroy(form);

  free(result);
  free(chk);
  return rc;
}

#endif /* PCMCIA */

static int help_level = 0;

void
pushHelpLine (const char *what, int replace)
{
    int w, h, _len, len;
    char *buffer;

    newtGetScreenSize (&w, &h);

    if (replace && help_level)
    {
        newtPopHelpLine ();

        --help_level;
    }

    _len = strlen (what);
    len = (w > _len ? (w + _len) / 2 : _len) + 1;

    buffer = (char *)malloc (len);

    memset (buffer, ' ', len - 1);
    strcpy (buffer + len - _len - 1, what);

    newtPushHelpLine (buffer);

    free (buffer);

    ++help_level;
}

void
popHelpLine (void)
{
    if (help_level > 0)
    {
        newtPopHelpLine ();

        --help_level;
    }
}

#define ENTERDIRBOX_LEN 49

int
enterDirBox (const char *title, const char *prompt, const char *prefix, const char *dir, char *buf, size_t bufsize)
{
    int result = DLG_ERROR;
    newtComponent form, text, entry, choose, obutton, cbutton;
    struct newtExitStruct exit_code;
    char *etext;
    int height;
    int done;

    text = newtTextbox (1, 1, ENTERDIRBOX_LEN, 10, NEWT_FLAG_WRAP);

    SHRINK_STRING(prompt, ENTERDIRBOX_LEN);
    newtTextboxSetText (text, prompt);
    height = newtTextboxGetNumLines (text);
    newtTextboxSetHeight (text, height);

    newtCenteredWindow (ENTERDIRBOX_LEN + 1, height + 5, title);

    form = newtForm (NULL, NULL, 0);

    newtFormAddHotKey (form, NEWT_KEY_ESC);

    entry = newtEntry (1, height + 2, dir, 42, &etext, NEWT_FLAG_RETURNEXIT | NEWT_FLAG_SCROLL);
    choose = newtCompactButton (43, height + 2, "...");
    obutton = newtCompactButton (13, height + 4, _("OK"));
    cbutton = newtCompactButton (28, height + 4, _("Cancel"));

    newtFormAddComponents (form, text, entry, choose, obutton, cbutton, NULL);

    done = 0;

    do {
        newtFormRun (form, &exit_code);

        switch (exit_code.reason)
        {
            case NEWT_EXIT_HOTKEY:
                switch (exit_code.u.key)
                {
                    default:    /* Unknown key??? */
                    case NEWT_KEY_ESC:
                        /* User decided to cancel the action. */
                        done = 1;
                        result = DLG_CANCEL;
                        break;
                }
                break;

            case NEWT_EXIT_COMPONENT:
                if (exit_code.u.co == choose)
                {
                    int tempo = choose_dir (title, prefix, etext, buf, bufsize);

                    if (tempo == DLG_OKAY)
                    {
                        newtEntrySet (entry, buf, 1);

                        newtFormSetCurrent (form, entry);
                    }
                }
                else if ((exit_code.u.co == obutton) || (exit_code.u.co == entry))
                {
                    done = 1;
                    result = DLG_OKAY;
                }
                else
                {
                    done = 1;
                    result = DLG_CANCEL;
                }
                break;

            default:    /* Unknown reason. */
                done = 1;
                result = DLG_ERROR;
        }
    } while (!done);

    if (result == DLG_OKAY)
        strcpy (buf, etext);

    newtFormDestroy (form);

    newtPopWindow ();

    return result;
}

struct _s_node {
    char *value;
    struct _s_node *next;
};

static struct _s_node *
new_node (const char *value)
{
    struct _s_node *r = (struct _s_node *)malloc (sizeof (struct _s_node));

    r->value = strdup (value);
    r->next = NULL;

    return r;
}

static void
free_node (struct _s_node *what)
{
    free (what->value);
    free (what);
}

static int
no_curpar (const struct dirent *what)
{
    return strcmp (what->d_name, ".") != 0 && strcmp (what->d_name, "..") != 0;
}

static int
choose_dir (const char *title, const char *prefix, const char *dir, char *buf, size_t bufsize)
{
    char *tempo = malloc (PATH_MAX);
    char mydir[PATH_MAX];
    int result;
    char *cp;
    struct stat statbuf;

    result = DLG_ERROR;
    if (!strcmp (prefix, "") && !strcmp (dir, ""))
      snprintf (mydir, sizeof (mydir), "/");
    else
      snprintf (mydir, sizeof (mydir), "%s%s", prefix, dir);

    if (!NAME_ISDIR (mydir, &statbuf)) {
	snprintf (prtbuf, sizeof (prtbuf),
		  _("The supplied directory %s does not exist. Please enter a new one."), dir);
	problemBox(prtbuf, _("Directory Error"));
	return DLG_ERROR;
    }

    if (normalize_dir (mydir, tempo, PATH_MAX) != NULL && tempo[0] == '/')
    {
        int done = 0;

        do {
            newtComponent form, list, obutton, cbutton;
            struct newtExitStruct exit_code;
            struct _s_node *node, *root;

            newtCenteredWindow (50, 17, title);

            form = newtForm (NULL, NULL, 0);

            newtFormAddHotKey (form, NEWT_KEY_ESC);

            list = newtListbox (0, 0, 15, NEWT_FLAG_RETURNEXIT | NEWT_FLAG_SCROLL);

            {
                char *fname = malloc (PATH_MAX), *lname = malloc (PATH_MAX);
                int spaces, index;

                index = 0;

                {
                    char *p, *q;

                    root = node = new_node ("/");

                    newtListboxAddEntry (list, "/", root);

                    for (++index, spaces = 0, p = tempo + 1 + strlen (prefix) ;
			 (q = strchr (p, '/')) != NULL ;
			 ++index, ++spaces, p = q + 1)
                    {
                        size_t l = q - p;

                        memset (lname, ' ', spaces);
                        memcpy (lname + spaces, p, l);

                        l += spaces;

                        memset (lname + l, ' ', 47 - l);
                        lname[47] = '\0';

                        l = q - tempo;

                        memcpy (fname, tempo, l);
                        fname[l] = '\0';

                        node->next = new_node (fname);

                        node = node->next;

                        newtListboxAddEntry (list, lname, node);
                    }

                    if (p[0] != '\0')   /* Process last name */
                    {
                        size_t l;

                        memset (lname, ' ', spaces);
                        strcpy (lname + spaces, p);

                        l = strlen (lname);

                        memset (lname + l, ' ', 47 - l);
                        lname[47] = '\0';

                        strcpy (fname, tempo);

                        node->next = new_node (fname);

                        node = node->next;

                        newtListboxAddEntry (list, lname, node);

                        newtListboxSetCurrent (list, index);
                    }
                }

                {
                    struct dirent **recs;
                    struct stat fs;
                    int i, no;

                    ++spaces;

                    if ((no = scandir (tempo, &recs, no_curpar, alphasort)) > 0)
                    {
                        for (i = 0 ; i < no ; ++i)
                        {
                            struct dirent *rec = recs[i];

                            if (tempo[1] == '\0')
                            {
                                strcpy (fname, "/");
                                strcat (fname, rec->d_name);
                            }
                            else
                            {
                                strcpy (fname, tempo);
                                strcat (fname, "/");
                                strcat (fname, rec->d_name);
                            }

                            if (stat (fname, &fs) == 0 && S_ISDIR (fs.st_mode))
                            {
                                size_t l;

                                memset (lname, ' ', spaces);
                                strcpy (lname + spaces, rec->d_name);

                                l = strlen (lname);

                                memset (lname + l, ' ', 47 - l);
                                lname[47] = '\0';

                                node->next = new_node (fname);

                                node = node->next;

                                newtListboxAddEntry (list, lname, node);
                            }

                            free (rec);
                        }

                        free (recs);
                    }
                }

                free (lname);
                free (fname);
            }

            obutton = newtCompactButton (13, 16, _("OK"));
            cbutton = newtCompactButton (28, 16, _("Cancel"));

            newtFormAddComponents (form, list, obutton, cbutton, NULL);

            newtFormRun (form, &exit_code);

            switch (exit_code.reason)
            {
                case NEWT_EXIT_HOTKEY:
                    switch (exit_code.u.key)
                    {
                        default:    // Unknown key???
                            ;

                        case NEWT_KEY_ESC:
                            // User decided to cancel the action.
                            node = NULL;
                            done = 1;
                            result = DLG_CANCEL;
                            break;

                        case NEWT_KEY_RIGHT:
                        case NEWT_KEY_ENTER:
                            node = (struct _s_node *)newtListboxGetCurrent (list);
                    }
                    break;

                case NEWT_EXIT_COMPONENT:
                    if (exit_code.u.co == list)
                        node = (struct _s_node *)newtListboxGetCurrent (list);
                    else if (exit_code.u.co == obutton)
                    {
                        node = (struct _s_node *)newtListboxGetCurrent (list);
                        done = 1;
                        result = DLG_OKAY;
                    }
                    else
                    {
                        node = NULL;
                        done = 1;
                        result = DLG_CANCEL;
                    }
                    break;

                default:    /* Unknown reason. */
                    node = NULL;
                    done = 1;
                    result = DLG_ERROR;
            }

            if (node != NULL)
                strcpy (tempo, node->value);
	    if (!strcmp (tempo, "/") && strcmp (prefix, ""))
	      sprintf (tempo, "%s", prefix);

            while (root != NULL)
            {
                node = root->next;

                free_node (root);

                root = node;
            }

            newtFormDestroy (form);

            newtPopWindow ();

        } while (!done);

	cp = tempo;
	cp += strlen (prefix);

        strcpy (buf, cp);
    }

    free (tempo);

    return result;
}


#define PROGRESS_SCALE	62
#define PROGRESS_TLEN	255
/*
   boxProgress() - display a progress bar during a lengthy operation. The
   caller controls the percent of the operation that is complete as well as
   the message displayed explaining what is going on.
   
   action - do something to the progress screen (start, stop, update etc.)
   arg - argument for the action

   supported actions:
   
   PROGRESS_INIT - initialize progress display, set window name to text to 
   the argument passed in, and percent complete to zero.
   
   PROGRESS_UPDATE - update the progress bar, argument is fraction of
   operation complete as two ascii numbers separated by a space.  Example:
   "50 100" means 50% done, "3 13" 3 of thirteen something's are complete.
   
   PROGRESS_SETTEXT - set the text displayed inside the window to
   arg.
   
   PROGRESS_PAUSE - pause the progress window, probably because we want to
   interrupt it with another message.  The operation can be resumed later
   with PROGRESS_RESUME (text and percent will be what they were before it
   was paused).
   
   PROGRESS_RESUME - resume a paused progress display.

   PROGRESS_CLOSE - operation is complete, progress bar is removed, 
   memory is freed.

*/
void
boxProgress (int action, char *arg)
{
  static newtComponent tx = NULL, scale = NULL, f = NULL;
  static int cur, max;
  static char window_content[PROGRESS_TLEN + 1];
  static char window_header[64];

  switch (action)
    {
    case PROGRESS_INIT:
      {
	if (arg)
	    snprintf (window_header, sizeof window_header, "%s", arg);
	else
	    window_header[0] = '\0';
	cur = 0;
	max = PROGRESS_SCALE;
      }
      /* fall through */
    case PROGRESS_RESUME:
      {
	newtCenteredWindow (72, 8, window_header);
	f = newtForm (NULL, NULL, 0);
	tx = newtTextbox (1, 1, 71, 4, NEWT_FLAG_WRAP);
	newtTextboxSetText (tx, window_content);
	scale = newtScale (4, 6, 62, PROGRESS_SCALE);
	newtFormAddComponents (f, tx, scale, NULL);
	newtDrawForm (f);
	newtScaleSet (scale, cur * PROGRESS_SCALE / max);
	newtRefresh ();
        break;
      }
    case PROGRESS_CLOSE:
    case PROGRESS_PAUSE:
      {
	newtPopWindow ();
	newtFormDestroy (f);
        break;
      }
    case PROGRESS_UPDATE:
      {
	/* Progress, just update scale */
	sscanf (arg, "%d %d", &cur, &max);
	if (cur > max)
	  cur = max;
	newtScaleSet (scale, cur * PROGRESS_SCALE / max);

	newtDrawForm (f);
	newtRefresh ();
	break;
      }
    case PROGRESS_SETTEXT:
      {
	if (arg)
	    snprintf (window_content, sizeof window_content, "%s", arg);
	else
	    window_content[0] = '\0';
	newtTextboxSetText (tx, window_content);
	newtDrawForm (f);
	newtRefresh ();
	break;
      }
    }
}

#ifdef USE_LANGUAGE_CHOOSER

int setFont (const char *fontname, const char *acm)
{
#ifdef _TESTING_
  sprintf(prtbuf, "consolechars -f %s", fontname);

  if (acm != NULL && acm[0] != '\0')
  {
      strcat (prtbuf, " -m ");
      strcat (prtbuf, acm);
  }
  execlog(prtbuf, LOG_INFO);
#else  
  sprintf(prtbuf, "loadfont < /etc/i18n/%s", fontname);
  execlog(prtbuf, LOG_INFO);

  if (acm != NULL && acm[0] != '\0')
  {
    sprintf(prtbuf, "loadacm < /etc/i18n/%s", acm);
    execlog(prtbuf, LOG_INFO);
  }
#endif
  return 0;
}

void
doIt (newtComponent lb, void *data)
{
    struct language_definition *lang = (struct language_definition *)newtListboxGetCurrent (lb);

    setFont (lang->font, lang->acm);

    pushHelpLine (lang->hint, 1);
}

const struct language_item *
boxChooseLanguageVariant (const struct language_definition *list)
{
    struct language_definition *result;
    newtComponent form, lang_list;
    int i;
    struct newtExitStruct exit_code;
    size_t width;

    /* Let's calculate the maximum length */
    for (i = 0, width = 0 ; list[i].name != NULL ; ++i)
    {
        size_t l = strwidth (list[i].hint);

        if (l > width)
            width = l;
    }

    /* 2 -- main window borders
     * 2 -- listbox border
     * 4 -- spaces 
     */
    width = max (25, width + 4);

    newtCenteredWindow (width, 14, "Choose The Language");

//  pushHelpLine ("Help here!", 1);

    form = newtForm (NULL, NULL, 0);
#if 0
    newtFormAddHotKey (form, NEWT_KEY_ESC);
#endif
    newtFormAddHotKey (form, NEWT_KEY_LEFT);
    newtFormAddHotKey (form, NEWT_KEY_RIGHT);

    lang_list = newtListbox (0, 0, 14, NEWT_FLAG_RETURNEXIT | NEWT_FLAG_SCROLL);

    width -= 4;

    {
        char *tempo = (char *)malloc (width * MB_LEN_MAX + 1);
        for (i = 0 ; list[i].name != NULL ; ++i)
        {
	    struct language_item *li = list[i].list->items[0].p;
	    if (list[i].list->items[0].d != 1)
		continue;
	    sprintf(prtbuf, "/etc/messages.%s", li->msgcat);
	    if (access(prtbuf, R_OK))
		continue;

            wpadit (tempo, list[i].hint, width);
            newtListboxAddEntry (lang_list, tempo, list + i);
        }

        free (tempo);
    }

//  newtComponentAddCallback (lang_list, doIt, NULL);

    newtFormAddComponents (form, lang_list, NULL);

    newtListboxSetCurrent (lang_list, 0);

    newtFormRun (form, &exit_code);

    switch (exit_code.reason)
    {
        case NEWT_EXIT_HOTKEY:
            switch (exit_code.u.key)
            {
                default:    /* Unknown key??? */
                    ;
#if 0
                case NEWT_KEY_ESC:
#endif
                case NEWT_KEY_LEFT:
                    /* User decided to cancel the action. */
                    result = NULL;
                    break;

                case NEWT_KEY_RIGHT:
                case NEWT_KEY_ENTER:
                    result = (struct language_definition *)newtListboxGetCurrent (lang_list);
            }
            break;

        case NEWT_EXIT_COMPONENT:
            // Do something
            if (exit_code.u.co == lang_list)
                result = (struct language_definition *)newtListboxGetCurrent (lang_list);
            else
                result = NULL;
            break;

        default:    // Unknown reason.
            result = NULL;
    }

    newtFormDestroy (form);

    newtPopWindow ();

#ifdef USE_LANGUAGE_VARIANTS
    if (result == NULL)
	return NULL;

    if (result->list->items[1].d == 0)
	return (struct language_item *)result->list->items[0].p;

    return boxChooseLanguageHelper (result->list);
#else
    // just choose the first variant
    return result == NULL ? NULL : (struct language_item *)result->list->items[0].p;
#endif
}

static const struct language_item *
boxChooseLanguageHelper (const struct language_list *list)
{
    newtComponent form, variants;
    struct newtExitStruct exit_code;
    int i, width, done = 0;
    const struct language_item *result;
    const struct language_list_item *tempo;

    for (i = 0, width = 0 ; list->items[i].d != 0 ; ++i)
    {
        const char *name = list->items[i].d == 1 ? ((const struct language_item *)list->items[i].p)->name : ((const struct language_list *)list->items[i].p)->name;
        size_t l = strwidth (name);

        if (l > width)
            width = l;
    }

    width = max (width + 8, strwidth (list->name) + 6);

    newtCenteredWindow (width, 14, list->name);

    form = newtForm (NULL, NULL, 0);

//  newtFormAddHotKey (form, NEWT_KEY_ESC);
    newtFormAddHotKey (form, NEWT_KEY_LEFT);
    newtFormAddHotKey (form, NEWT_KEY_RIGHT);

    variants = newtListbox (2, 1, 11, NEWT_FLAG_BORDER | NEWT_FLAG_RETURNEXIT | NEWT_FLAG_SCROLL);

    for (i = 0 ; list->items[i].d != 0 ; ++i)
    {
        const char *name = list->items[i].d == 1 ? ((const struct language_item *)list->items[i].p)->name : ((const struct language_list *)list->items[i].p)->name;

        newtListboxAddEntry (variants, name, list->items + i);
    }

    newtFormAddComponents (form, variants, NULL);

    do {
        newtFormRun (form, &exit_code);

        switch (exit_code.reason)
        {
            case NEWT_EXIT_HOTKEY:
                switch (exit_code.u.key)
                {
                    default:    // Unknown key???
                        ;

#if 0
                    case NEWT_KEY_ESC:
#endif
                    case NEWT_KEY_LEFT: /* User decided to retyrn to previous list */
                        result = NULL;
                        done = 1;
                        break;

                    case NEWT_KEY_RIGHT:
                    case NEWT_KEY_ENTER:
                        tempo = (struct language_list_item *)newtListboxGetCurrent (variants);

                        if (tempo->d == 2)
                            result = boxChooseLanguageHelper ((const struct language_list *)tempo->p);
                        else
                            result = (const struct language_item *)tempo->p;
                }
                break;

            case NEWT_EXIT_COMPONENT:
                // Do something
                if (exit_code.u.co == variants)
                {
                    tempo = (struct language_list_item *)newtListboxGetCurrent (variants);

                    if (tempo->d == 2)
                        result = boxChooseLanguageHelper ((const struct language_list *)tempo->p);
                    else
                        result = (const struct language_item *)tempo->p;
                }
                else
                {
                    result = NULL;
                    done = 1;
                }
                break;

            default:    // Unknown reason.
                result = NULL;
                done = 1;
        }
    } while (result == NULL && !done);

    newtFormDestroy (form);

    newtPopWindow ();

    return result;
}
#endif

int edit_append_line ()
{
    char *def = NULL;
    char *tmp = NULL;

    snprintf (prtbuf, sizeof (prtbuf), _("Please add all parameters that the kernel needs at boot time. These parameters are required if you have to turn off builtin drivers that make the kernel fail or to configure others.\n\nExample: ibmmcascsi=1 aha152x=0x340,9\n"));
    if (append_opts)
      def = strdup (append_opts);
    else
      def = strdup("");
    tmp = inputBox(prtbuf, _("Edit Kernel Boot Parameters"), def);
    if (tmp == NULL)
	return DLG_CANCEL;
    else {
	free (append_opts);
	append_opts = strdup (tmp);
    }

    return DLG_OKAY;
}

#ifdef _TESTING_
/* To test, compile using: make boxes_test */
int main (void)
{
	struct d_choices opt[3];
	int rs;
	char buf[128];

	LOAD_TRMFILE("test.trm");

	get_kver();
	boxInit();

// Testing menuBox
	opt[0].tag="test one";
	opt[0].string="one string...";
	opt[1].tag="test two";
	opt[1].string=NULL;
	opt[2].tag=NULL;
	opt[2].string="This is a somewhat large string. Let's test resizable boxes";
	rs=menuBox("This is a little text","I'm the title",opt,3,1);

// Testing problemBox
	sprintf(buf,"You've chosen option %d",rs);
	problemBox(buf,"Answer");

// Testing yesNoBox
	if ( yesNoBox("Please chose yes or no", "yesNoBox test") == DLG_YES)
	    problemBox("You chose YES","Answer");
	else
	    problemBox("You chose No","Answer");


// Testing pleaseWaitBox
	pleaseWaitBox("testing the pleaseWaitBox");
	newtRefresh();
	sleep(3);

	boxFinished();

	sleep (2);
// Testing widemessageBox
	boxInit();
	/* a widemessageBox will scroll if the text it too large to fit the screen. */
	wideMessageBox("This document assumes that you know what CVS is, and that you at least know the fundamental concepts of CVS.  If that is not the case you should read the man page for CVS.\n\nPcl-cvs is only useful once you have checked out a module.  So before you invoke it you must have a copy of a module somewhere in the file system.\n\nYou can invoke pcl-cvs by typing `M-x cvs-examine RET'.  If your emacs responds with `[No match]' your system administrator has not installed pcl-cvs properly.  Try `M-x load-library RET pcl-cvs RET'. If that also fails, talk to your system administrator.  If it succeeds you might put this line in your `.emacs' file so that you don't have to type the `load-library' command every time you wish to use pcl-cvs:\n\nThis document assumes that you know what CVS is, and that you at least know the fundamental concepts of CVS.  If that is not the case you should read the man page for CVS.\n\nPcl-cvs is only useful once you have checked out a module.  So before you invoke it you must have a copy of a module somewhere in the file system.\n\nYou can invoke pcl-cvs by typing `M-x cvs-examine RET'.  If your emacs responds with `[No match]' your system administrator has not installed pcl-cvs properly.  Try `M-x load-library RET pcl-cvs RET'. If that also fails, talk to your system administrator.  If it succeeds you might put this line in your `.emacs' file so that you don't have to type the `load-library' command every time you wish to use pcl-cvs:", "This is a test of really long text.!");

// Testing boxProgress
	boxProgress(PROGRESS_INIT, "Here is an initial string");
	sleep(1);
	boxProgress(PROGRESS_SETTEXT, "Now another, 10\% done");
	boxProgress(PROGRESS_UPDATE, "10 100");
	sleep(1);

	boxProgress(PROGRESS_PAUSE, NULL);
        problemBox("Hi friend","Useless box");
	boxProgress(PROGRESS_RESUME, NULL);
	boxProgress(PROGRESS_SETTEXT, "Now another, 50\% done");
	boxProgress(PROGRESS_UPDATE, "1 2");
	sleep(1);
	boxProgress(PROGRESS_CLOSE, NULL);

	
	boxFinished();

	return 0;
}
#endif

/*
 * Local variables:
 *   c-basic-offset: 4
 *   indent-tabs-mode: t
 * End:
 */


