//  ************************************************************************************************
//
//  BornAgain: simulate and fit reflection and scattering
//
//! @file      GUI/View/Instrument/ScanEditor.cpp
//! @brief     Defines class ScanEditor
//!
//! @homepage  http://www.bornagainproject.org
//! @license   GNU General Public License v3 or higher (see COPYING)
//! @copyright Forschungszentrum Jülich GmbH 2018
//! @authors   Scientific Computing Group at MLZ (see CITATION, AUTHORS)
//
//  ************************************************************************************************

#include "GUI/View/Instrument/ScanEditor.h"
#include "Base/Util/Assert.h"
#include "GUI/Model/Beam/BeamWavelengthItem.h"
#include "GUI/Model/Beam/SourceItems.h"
#include "GUI/Model/Device/InstrumentNotifier.h"
#include "GUI/View/Device/FootprintForm.h"
#include "GUI/View/Instrument/AlphaScanEditor.h"
#include "GUI/View/Instrument/DistributionEditor.h"
#include "GUI/View/Numeric/FixupDoubleValidator.h"
#include "GUI/View/Tool/GroupBoxCollapser.h"
#include <QFormLayout>
#include <QLineEdit>

ScanEditor::ScanEditor(QWidget* parent, ScanItem* item, InstrumentNotifier* ec)
    : QGroupBox("Beam and scan parameters", parent)
{
    ASSERT(item);

    auto* vLayout = new QVBoxLayout(this);
    vLayout->setContentsMargins(30, 8, 0, 0);
    auto* form = new QFormLayout();
    form->setFieldGrowthPolicy(QFormLayout::FieldsStayAtSizeHint);
    vLayout->addLayout(form);
    auto* intensityLineEdit = new QLineEdit(this);
    intensityLineEdit->setToolTip("Correction factor for normalized intensity");
    auto* validator = new FixupDoubleValidator(intensityLineEdit);
    validator->setNotation(QDoubleValidator::ScientificNotation);
    validator->setRange(0.0, 1e32, 1000);
    intensityLineEdit->setValidator(validator);
    form->addRow("Intensity:", intensityLineEdit);

    auto* wavelengthEditor =
        new DistributionEditor("Wavelength", MeanConfig{true}, GUI::ID::Distributions::Symmetric,
                               this, item->wavelengthItem());
    vLayout->addWidget(wavelengthEditor);

    auto* inclinationEditor = new AlphaScanEditor(this, item->grazingScanItem());
    vLayout->addWidget(inclinationEditor);

    auto* footprintEditor = new FootprintForm(this, item);
    vLayout->addWidget(footprintEditor);

    intensityLineEdit->setText(QString::number(item->intensity()));

    auto* collapser = GroupBoxCollapser::installIntoGroupBox(this);
    collapser->setExpanded(item->isExpandBeamParameters());
    connect(collapser, &GroupBoxCollapser::toggled, this,
            [item](bool b) { item->setExpandBeamParameters(b); });

    // order of the next two connections is important! Indicators have to be recalculated first,
    // then updated (recalculation is done in EditController)
    connect(wavelengthEditor, &DistributionEditor::distributionChanged, ec,
            &InstrumentNotifier::notifyWavelengthDistributionChanged);
    connect(wavelengthEditor, &DistributionEditor::distributionChanged, inclinationEditor,
            &AlphaScanEditor::updateIndicators);

    // order of the next two connections is important! Wavelength has to be recalculated first,
    // then updated (recalculation is done in EditController)
    connect(inclinationEditor, &AlphaScanEditor::dataChanged, ec,
            &InstrumentNotifier::notifyInclinationAxisChanged);
    connect(inclinationEditor, &AlphaScanEditor::dataChanged, wavelengthEditor,
            &DistributionEditor::updateData);

    connect(footprintEditor, &FootprintForm::dataChanged, this, &ScanEditor::dataChanged);

    // validate value while typing
    connect(intensityLineEdit, &QLineEdit::textEdited, [this, intensityLineEdit, item]() {
        QString str = intensityLineEdit->text();
        int pos;
        if (intensityLineEdit->validator()->validate(str, pos) == QValidator::Acceptable) {
            item->setIntensity(intensityLineEdit->text().toDouble());
            emit dataChanged();
        }
    });
    // if validator does not assert the input value, it says about that,
    // and here we return to the last correct value
    connect(validator, &FixupDoubleValidator::fixupSignal, [=]() {
        auto* editor = qobject_cast<QLineEdit*>(validator->parent());
        editor->setText(QString::number(item->intensity(), 'g'));
    });
}
