package org.bouncycastle.crypto.test;

import java.security.SecureRandom;

import org.bouncycastle.crypto.macs.SipHash;
import org.bouncycastle.crypto.params.KeyParameter;
import org.bouncycastle.util.Arrays;
import org.bouncycastle.util.Pack;
import org.bouncycastle.util.encoders.Hex;
import org.bouncycastle.util.test.SimpleTest;

/*
 * SipHash test values are taken from the output of the C reference implementations:
 * section "const uint8_t vectors_sip64[64][8]" of the output of command ./vectors.
 *
 * For SipHash-2-4, test values from "SipHash: a fast short-input PRF", by Jean-Philippe
 * Aumasson and Daniel J. Bernstein (https://131002.net/siphash/siphash.pdf), Appendix A
 * will also be evaluated.
 */
public class SipHashTest
    extends SimpleTest
{
    private static final int UPDATE_BYTES = 0;
    private static final int UPDATE_FULL = 1;
    private static final int UPDATE_MIX = 2;

    public String getName()
    {
        return "SipHash";
    }

    public void performTest()
        throws Exception
    {
        performTest_2_4();
        performTest_4_8();
    }

    private void performTest_2_4()
        throws Exception
    {
        /* Explicit test of the
         * SipHash test values from "SipHash: a fast short-input PRF", by Jean-Philippe
         * Aumasson and Daniel J. Bernstein (https://131002.net/siphash/siphash.pdf), Appendix A.
         */
        byte[] key = Hex.decode("000102030405060708090a0b0c0d0e0f");
        byte[] input = Hex.decode("000102030405060708090a0b0c0d0e");
        byte[] expected = Hex.decode("e545be4961ca29a1");
        runMAC(2, 4, expected, key, input);

        /* SipHash test values are taken from the output of the C reference implementations:
         * section "const uint8_t vectors_sip64[64][8]" of the output of command ./vectors.
         */
        int[][] vectors_sip64 = new int[][]{
            {0x31, 0x0e, 0x0e, 0xdd, 0x47, 0xdb, 0x6f, 0x72,},
            {0xfd, 0x67, 0xdc, 0x93, 0xc5, 0x39, 0xf8, 0x74,},
            {0x5a, 0x4f, 0xa9, 0xd9, 0x09, 0x80, 0x6c, 0x0d,},
            {0x2d, 0x7e, 0xfb, 0xd7, 0x96, 0x66, 0x67, 0x85,},
            {0xb7, 0x87, 0x71, 0x27, 0xe0, 0x94, 0x27, 0xcf,},
            {0x8d, 0xa6, 0x99, 0xcd, 0x64, 0x55, 0x76, 0x18,},
            {0xce, 0xe3, 0xfe, 0x58, 0x6e, 0x46, 0xc9, 0xcb,},
            {0x37, 0xd1, 0x01, 0x8b, 0xf5, 0x00, 0x02, 0xab,},
            {0x62, 0x24, 0x93, 0x9a, 0x79, 0xf5, 0xf5, 0x93,},
            {0xb0, 0xe4, 0xa9, 0x0b, 0xdf, 0x82, 0x00, 0x9e,},
            {0xf3, 0xb9, 0xdd, 0x94, 0xc5, 0xbb, 0x5d, 0x7a,},
            {0xa7, 0xad, 0x6b, 0x22, 0x46, 0x2f, 0xb3, 0xf4,},
            {0xfb, 0xe5, 0x0e, 0x86, 0xbc, 0x8f, 0x1e, 0x75,},
            {0x90, 0x3d, 0x84, 0xc0, 0x27, 0x56, 0xea, 0x14,},
            {0xee, 0xf2, 0x7a, 0x8e, 0x90, 0xca, 0x23, 0xf7,},
            {0xe5, 0x45, 0xbe, 0x49, 0x61, 0xca, 0x29, 0xa1,},
            {0xdb, 0x9b, 0xc2, 0x57, 0x7f, 0xcc, 0x2a, 0x3f,},
            {0x94, 0x47, 0xbe, 0x2c, 0xf5, 0xe9, 0x9a, 0x69,},
            {0x9c, 0xd3, 0x8d, 0x96, 0xf0, 0xb3, 0xc1, 0x4b,},
            {0xbd, 0x61, 0x79, 0xa7, 0x1d, 0xc9, 0x6d, 0xbb,},
            {0x98, 0xee, 0xa2, 0x1a, 0xf2, 0x5c, 0xd6, 0xbe,},
            {0xc7, 0x67, 0x3b, 0x2e, 0xb0, 0xcb, 0xf2, 0xd0,},
            {0x88, 0x3e, 0xa3, 0xe3, 0x95, 0x67, 0x53, 0x93,},
            {0xc8, 0xce, 0x5c, 0xcd, 0x8c, 0x03, 0x0c, 0xa8,},
            {0x94, 0xaf, 0x49, 0xf6, 0xc6, 0x50, 0xad, 0xb8,},
            {0xea, 0xb8, 0x85, 0x8a, 0xde, 0x92, 0xe1, 0xbc,},
            {0xf3, 0x15, 0xbb, 0x5b, 0xb8, 0x35, 0xd8, 0x17,},
            {0xad, 0xcf, 0x6b, 0x07, 0x63, 0x61, 0x2e, 0x2f,},
            {0xa5, 0xc9, 0x1d, 0xa7, 0xac, 0xaa, 0x4d, 0xde,},
            {0x71, 0x65, 0x95, 0x87, 0x66, 0x50, 0xa2, 0xa6,},
            {0x28, 0xef, 0x49, 0x5c, 0x53, 0xa3, 0x87, 0xad,},
            {0x42, 0xc3, 0x41, 0xd8, 0xfa, 0x92, 0xd8, 0x32,},
            {0xce, 0x7c, 0xf2, 0x72, 0x2f, 0x51, 0x27, 0x71,},
            {0xe3, 0x78, 0x59, 0xf9, 0x46, 0x23, 0xf3, 0xa7,},
            {0x38, 0x12, 0x05, 0xbb, 0x1a, 0xb0, 0xe0, 0x12,},
            {0xae, 0x97, 0xa1, 0x0f, 0xd4, 0x34, 0xe0, 0x15,},
            {0xb4, 0xa3, 0x15, 0x08, 0xbe, 0xff, 0x4d, 0x31,},
            {0x81, 0x39, 0x62, 0x29, 0xf0, 0x90, 0x79, 0x02,},
            {0x4d, 0x0c, 0xf4, 0x9e, 0xe5, 0xd4, 0xdc, 0xca,},
            {0x5c, 0x73, 0x33, 0x6a, 0x76, 0xd8, 0xbf, 0x9a,},
            {0xd0, 0xa7, 0x04, 0x53, 0x6b, 0xa9, 0x3e, 0x0e,},
            {0x92, 0x59, 0x58, 0xfc, 0xd6, 0x42, 0x0c, 0xad,},
            {0xa9, 0x15, 0xc2, 0x9b, 0xc8, 0x06, 0x73, 0x18,},
            {0x95, 0x2b, 0x79, 0xf3, 0xbc, 0x0a, 0xa6, 0xd4,},
            {0xf2, 0x1d, 0xf2, 0xe4, 0x1d, 0x45, 0x35, 0xf9,},
            {0x87, 0x57, 0x75, 0x19, 0x04, 0x8f, 0x53, 0xa9,},
            {0x10, 0xa5, 0x6c, 0xf5, 0xdf, 0xcd, 0x9a, 0xdb,},
            {0xeb, 0x75, 0x09, 0x5c, 0xcd, 0x98, 0x6c, 0xd0,},
            {0x51, 0xa9, 0xcb, 0x9e, 0xcb, 0xa3, 0x12, 0xe6,},
            {0x96, 0xaf, 0xad, 0xfc, 0x2c, 0xe6, 0x66, 0xc7,},
            {0x72, 0xfe, 0x52, 0x97, 0x5a, 0x43, 0x64, 0xee,},
            {0x5a, 0x16, 0x45, 0xb2, 0x76, 0xd5, 0x92, 0xa1,},
            {0xb2, 0x74, 0xcb, 0x8e, 0xbf, 0x87, 0x87, 0x0a,},
            {0x6f, 0x9b, 0xb4, 0x20, 0x3d, 0xe7, 0xb3, 0x81,},
            {0xea, 0xec, 0xb2, 0xa3, 0x0b, 0x22, 0xa8, 0x7f,},
            {0x99, 0x24, 0xa4, 0x3c, 0xc1, 0x31, 0x57, 0x24,},
            {0xbd, 0x83, 0x8d, 0x3a, 0xaf, 0xbf, 0x8d, 0xb7,},
            {0x0b, 0x1a, 0x2a, 0x32, 0x65, 0xd5, 0x1a, 0xea,},
            {0x13, 0x50, 0x79, 0xa3, 0x23, 0x1c, 0xe6, 0x60,},
            {0x93, 0x2b, 0x28, 0x46, 0xe4, 0xd7, 0x06, 0x66,},
            {0xe1, 0x91, 0x5f, 0x5c, 0xb1, 0xec, 0xa4, 0x6c,},
            {0xf3, 0x25, 0x96, 0x5c, 0xa1, 0x6d, 0x62, 0x9f,},
            {0x57, 0x5f, 0xf2, 0x8e, 0x60, 0x38, 0x1b, 0xe5,},
            {0x72, 0x45, 0x06, 0xeb, 0x4c, 0x32, 0x8a, 0x95,},
        };

        performTest(2, 4, vectors_sip64);
    }

    private void performTest_4_8()
        throws Exception
    {
        /* SipHash test values are taken from the output of the C reference implementations:
         * section "const uint8_t vectors_sip64[64][8]" of the output of command ./vectors.
         */
        int[][] vectors_sip64 = new int[][]{
            {0x41, 0xda, 0x38, 0x99, 0x2b, 0x05, 0x79, 0xc8,},
            {0x51, 0xb8, 0x95, 0x52, 0xf9, 0x14, 0x59, 0xc8,},
            {0x92, 0x37, 0x16, 0xf0, 0xbe, 0xdd, 0xc3, 0x33,},
            {0x6a, 0x46, 0xd4, 0x7d, 0x65, 0x47, 0xc1, 0x05,},
            {0xc2, 0x38, 0x59, 0x2b, 0x4a, 0xc1, 0xfa, 0x48,},
            {0xf6, 0xc2, 0xd7, 0xd9, 0xcf, 0x52, 0x47, 0xe1,},
            {0x6b, 0xb6, 0xbc, 0x34, 0xc8, 0x35, 0x55, 0x8e,},
            {0x47, 0xd7, 0x3f, 0x71, 0x5a, 0xbe, 0xfd, 0x4e,},
            {0x20, 0xb5, 0x8b, 0x9c, 0x07, 0x2f, 0xdb, 0x50,},
            {0x36, 0x31, 0x9a, 0xf3, 0x5e, 0xe1, 0x12, 0x53,},
            {0x48, 0xa9, 0xd0, 0xdb, 0x0a, 0x8d, 0x84, 0x8f,},
            {0xcc, 0x69, 0x39, 0x60, 0x36, 0x04, 0x0a, 0x81,},
            {0x4b, 0x6d, 0x68, 0x53, 0x7a, 0xa7, 0x97, 0x61,},
            {0x29, 0x37, 0x96, 0xe9, 0xf2, 0xc9, 0x50, 0x69,},
            {0x88, 0x43, 0x1b, 0xea, 0xa7, 0x62, 0x9a, 0x68,},
            {0xe0, 0xa6, 0xa9, 0x7d, 0xd5, 0x89, 0xd3, 0x83,},
            {0x55, 0x9c, 0xf5, 0x53, 0x80, 0xb2, 0xac, 0x70,},
            {0xd5, 0xb7, 0xc5, 0x11, 0x7a, 0xe3, 0x79, 0x4e,},
            {0x5a, 0x3c, 0x45, 0x46, 0x34, 0xad, 0x10, 0x2b,},
            {0xc0, 0xa4, 0x80, 0xaf, 0xa3, 0x5a, 0x3d, 0xbc,},
            {0x78, 0xc2, 0x27, 0x09, 0xe5, 0x28, 0x4b, 0xc8,},
            {0xef, 0x26, 0x70, 0x46, 0x0d, 0xeb, 0xd6, 0x9d,},
            {0xd9, 0x76, 0xef, 0x86, 0xa9, 0xd0, 0x84, 0xd8,},
            {0xe3, 0xd9, 0x81, 0x18, 0x19, 0xea, 0xd0, 0xe8,},
            {0x89, 0x33, 0x3c, 0xb5, 0x3e, 0xea, 0xec, 0x16,},
            {0x31, 0x15, 0x6c, 0x5f, 0x64, 0x73, 0x49, 0xc6,},
            {0xa5, 0x4c, 0xce, 0x35, 0x35, 0x76, 0x32, 0xa4,},
            {0x06, 0x5d, 0x89, 0x25, 0xc0, 0xa7, 0xd2, 0xfe,},
            {0x2b, 0xbb, 0xaa, 0x82, 0x22, 0x1a, 0x3a, 0x8b,},
            {0x87, 0x0b, 0xfb, 0xce, 0x64, 0x09, 0x7b, 0x70,},
            {0x40, 0xd8, 0xe0, 0xf9, 0x64, 0x95, 0xee, 0x8b,},
            {0x79, 0xfc, 0xa7, 0xf4, 0x0b, 0xfa, 0xdf, 0x12,},
            {0x00, 0x0b, 0xfb, 0xf2, 0x2f, 0x76, 0x9e, 0xd2,},
            {0x40, 0x68, 0x55, 0x91, 0xf8, 0xe5, 0x22, 0xfa,},
            {0x2b, 0xe6, 0xfe, 0x74, 0xd8, 0x14, 0x9d, 0x0d,},
            {0xba, 0x7e, 0x2f, 0x0e, 0x0b, 0x75, 0x60, 0xed,},
            {0x02, 0xe9, 0xe3, 0x84, 0xed, 0xa7, 0xe1, 0x97,},
            {0xc4, 0xe8, 0x0a, 0x62, 0x95, 0x27, 0x63, 0xb6,},
            {0x83, 0x27, 0xed, 0xc6, 0x5d, 0x5c, 0x6d, 0xd3,},
            {0x79, 0xfc, 0x64, 0xd1, 0x64, 0xa4, 0x2f, 0xc0,},
            {0x15, 0x4a, 0x75, 0x11, 0xcb, 0xfc, 0x61, 0x4e,},
            {0x8b, 0x14, 0x8d, 0x7c, 0xec, 0xa0, 0xe6, 0x6f,},
            {0xdf, 0xee, 0x69, 0xb6, 0x54, 0xc4, 0x03, 0xfa,},
            {0xc5, 0x8f, 0x36, 0xa6, 0x69, 0x7b, 0xb7, 0xc9,},
            {0xa6, 0xc5, 0xbe, 0x9c, 0x05, 0xc6, 0x31, 0x21,},
            {0xb5, 0x8a, 0x87, 0x59, 0xfb, 0xcd, 0x89, 0x31,},
            {0xd7, 0x68, 0x3a, 0x67, 0x04, 0xcc, 0xc4, 0x25,},
            {0xcb, 0x6a, 0xe6, 0xe1, 0xe5, 0xa2, 0x44, 0x8d,},
            {0x6e, 0x26, 0x69, 0x5b, 0x3a, 0x3a, 0x51, 0x73,},
            {0x78, 0x71, 0x07, 0xcf, 0x9f, 0x33, 0xac, 0x4a,},
            {0x16, 0x75, 0x90, 0xda, 0xd9, 0x7b, 0x74, 0x84,},
            {0x00, 0x6b, 0x68, 0x1e, 0xf0, 0x6b, 0xf3, 0x06,},
            {0x1c, 0x9b, 0x30, 0x02, 0x66, 0xef, 0xcf, 0xa6,},
            {0x28, 0x8d, 0x2f, 0x88, 0xd1, 0xb0, 0xb3, 0x4b,},
            {0xe0, 0x11, 0x06, 0xbd, 0xac, 0xf5, 0x6b, 0xfe,},
            {0xc0, 0x10, 0x1f, 0x0e, 0x5b, 0x6e, 0x03, 0x28,},
            {0xc3, 0xa7, 0x91, 0x45, 0x5b, 0x1b, 0x1c, 0x0a,},
            {0x57, 0x07, 0xaf, 0xe1, 0x9e, 0x0b, 0x3a, 0x0f,},
            {0xe6, 0x5a, 0x72, 0x29, 0xfe, 0x53, 0x59, 0x4f,},
            {0x00, 0x2f, 0x9d, 0xb9, 0xab, 0x1a, 0xaf, 0x4c,},
            {0x59, 0x28, 0xcb, 0x50, 0x44, 0xc1, 0x06, 0x06,},
            {0xd5, 0x38, 0x01, 0x96, 0x7b, 0x85, 0x73, 0x21,},
            {0x05, 0xdb, 0x36, 0x4f, 0x1a, 0x09, 0x99, 0xcc,},
            {0xe6, 0x77, 0x84, 0xbc, 0x55, 0x03, 0xde, 0x23,},
        };

        performTest(4, 8, vectors_sip64);
    }

    private void performTest(int cRounds, int dRounds, int[][] testvectorsInt)
        throws Exception
    {
        int n = testvectorsInt.length;
        int macSize = testvectorsInt[0].length;

        byte[][] testvectors = new byte[n][];
        for (int i = 0; i < n; i++)
        {
            testvectors[i] = new byte[macSize];
            for (int j = 0; j < macSize; j++)
            {
                testvectors[i][j] = (byte)testvectorsInt[i][j];
            }
        }

        byte[] key = Hex.decode("000102030405060708090a0b0c0d0e0f");

        for (int i = 0; i < n; i++)
        {
            byte[] input = new byte[i];
            for (int j = 0; j < input.length; j++)
            {
                input[j] = (byte)j;
            }

            runMAC(cRounds, dRounds, testvectors[i], key, input);
        }

        SecureRandom random = new SecureRandom();
        for (int i = 0; i < 100; ++i)
        {
            randomTest(cRounds, dRounds, random);
        }
    }

    private void runMAC(int cRounds, int dRounds, byte[] expected, byte[] key, byte[] input)
        throws Exception
    {
        runMAC(cRounds, dRounds, expected, key, input, UPDATE_BYTES);
        runMAC(cRounds, dRounds, expected, key, input, UPDATE_FULL);
        runMAC(cRounds, dRounds, expected, key, input, UPDATE_MIX);
    }

    private void runMAC(int cRounds, int dRounds, byte[] expected, byte[] key, byte[] input,
                        int updateType)
        throws Exception
    {
        long expectedLong = Pack.littleEndianToLong(expected, 0);

        SipHash mac = new SipHash(cRounds, dRounds);
        mac.init(new KeyParameter(key));

        updateMAC(mac, input, updateType);

        long result = mac.doFinal();

        if (expectedLong != result)
        {
            fail("Result does not match expected value for doFinal()");
        }

        updateMAC(mac, input, updateType);

        byte[] output = new byte[mac.getMacSize()];
        int len = mac.doFinal(output, 0);
        if (len != output.length)
        {
            fail("Result length does not equal getMacSize() for doFinal(byte[],int)");
        }
        if (!areEqual(expected, output))
        {
            fail("Result does not match expected value for doFinal(byte[],int)");
        }
    }

    private void randomTest(int cRounds, int dRounds, SecureRandom random)
    {
        byte[] key = new byte[16];
        random.nextBytes(key);

        int length = 1 + RNGUtils.nextInt(random, 1024);
        byte[] input = new byte[length];
        random.nextBytes(input);

        SipHash mac = new SipHash(cRounds, dRounds);
        mac.init(new KeyParameter(key));

        updateMAC(mac, input, UPDATE_BYTES);
        byte[] result1 = new byte[16];
        mac.doFinal(result1, 0);

        updateMAC(mac, input, UPDATE_FULL);
        byte[] result2 = new byte[16];
        mac.doFinal(result2, 0);

        updateMAC(mac, input, UPDATE_MIX);
        byte[] result3 = new byte[16];
        mac.doFinal(result3, 0);

        if (!Arrays.areEqual(result1, result2) ||
            !Arrays.areEqual(result1, result3))
        {
            fail("Inconsistent results in random test");
        }
    }

    private void updateMAC(SipHash mac, byte[] input, int updateType)
    {
        switch (updateType)
        {
        case UPDATE_BYTES:
        {
            for (int i = 0; i < input.length; ++i)
            {
                mac.update(input[i]);
            }
            break;
        }
        case UPDATE_FULL:
        {
            mac.update(input, 0, input.length);
            break;
        }
        case UPDATE_MIX:
        {
            int step = Math.max(1, input.length / 3);
            int pos = 0;
            while (pos < input.length)
            {
                mac.update(input[pos++]);
                int len = Math.min(input.length - pos, step);
                mac.update(input, pos, len);
                pos += len;
            }
            break;
        }
        default:
            throw new IllegalStateException();
        }
    }

    public static void main(String[] args)
    {
        runTest(new SipHashTest());
    }
}
