#ifndef _SONGITERATOR_CPP_
#define _SONGITERATOR_CPP_

#include <iostream.h>
#include <fstream.h>
#include <stdlib.h>
#include <string.h>
#include "songIterator.h"

#include "song.h"
#include "track.h"
#include "part.h"
#include "event.h"
#include "note.h"
#include "prPartEditor.h"
#include "scoreTrack.h"
#include "drumTrack.h"

SongIterator::SongIterator(const SongIterator& i) : _song(0), _event(i._event), _left(i._left), _right(i._right) { }

SongIterator::SongIterator(Song * song, Position left, Position right) : _song(song), _event(0), _left(left), _right(right) {
  _tracks   = _song->size();
  _track    = new Track*[_tracks];
  _part     = new Part*[_tracks];
  _new_part = new Part*[_tracks];
  _event    = new Event*[_tracks];
  init(_left,_right);
}

void SongIterator::init(Position left, Position right) {
  _left = left;
  _right = right;
  for (int i=0; i<_tracks; i++) {
    _track[i] = (Track*) _song->get(i);
    _part[i]  = (Part*)  _track[i]->first();
    if (_part[i]) {
      for (_event[i] = (Event*) _part[i]->first(); (_event[i]!=0) && (_part[i]->start(_event[i]) < _left); ) {
	_event[i] = (Event*) _part[i]->next(_event[i]);
	if (_event[i]==0 && _part[i]!=0) {
	  _part[i] = (Part*) _track[i]->next(_part[i]);
	  if (_part[i]!=0) _event[i] = (Event*) _part[i]->first();
	}

      }
    } else _event[i] = 0;
  }
  _memref = 0;
  _memtop = 0;
}

Track * SongIterator::track(int i) {
  if( i < _tracks )
    return _track[i];
  return 0;
}

bool SongIterator::operator!=(const SongIterator& i) const {
  bool ret = false;
  for (int k=0; k<_tracks; k++)
    if ((_part[k] != i._part[k]) || (_event[k] != i._event[k])) ret = true;
  return ret;
}

bool SongIterator::operator==(const SongIterator& i) const {
  bool ret = true;
  for (int k=0; k<_tracks; k++)
    if ((_part[k] != i._part[k]) || (_event[k] != i._event[k])) ret = false;
  return ret;
}

bool SongIterator::done(int i) {
  if (i>=_tracks) return true;
  bool ret = true;
  if (i==-1) {
    if (_memtop>0) ret = false;
    else {
      for (int k=0; k<_tracks; k++) {
	if (_event[k]!=0) ret = false;
      }
    }
  } else ret = (_event[i]==0);
  return ret;
}


bool SongIterator::startsAt(int i, long p) {
  if (i>=_tracks) return false;
  if (!_event[i] || !_part[i]) return false;
  return _part[i]->eventStartsAt(_event[i],p);
}


Element * SongIterator::operator[](int i) {
  if (i>=_tracks) return 0;
  else return _event[i];
}


void SongIterator::increment(int i) {
  if (i>=_tracks) return;
  if (_event[i]) {
    _event[i] = (Event*) _part[i]->next(_event[i]);
    if (_event[i]==0 && _part[i]!=0) {
      _part[i] = (Part*) _part[i]->next(_part[i]);
      if (_part[i]) {
	_event[i] = (Event*) _part[i]->first();
	_new_part[i] = _part[i];
      } else _new_part[i] = 0;
    } else _new_part[i] = 0;
  } else _new_part[i] = 0;
  if (_event[i]!=0 && _right!=0) {
    if (_part[i]->start(_event[i]) >= _right) _event[i] = 0; // done!
  }
}

Position SongIterator::start(int i) {
  if (i>=_tracks) return false;
  if(!_event[i] || !_part[i]) return Position(0);
  return _part[i]->start();
}

Part * SongIterator::change(int i) {
  if (i>=_tracks) return 0;
  else return _new_part[i];
}

int SongIterator::channel(int i) {
  int ret = -1;
  if (i<_tracks) {
    if (_track[i]->isA()==SCORETRACK) ret = ((ScoreTrack*)_track[i])->channel();
    if (_track[i]->isA()==DRUMTRACK)  ret = ((DrumTrack*)_track[i])->channel();
  }
  return ret;
}

int SongIterator::program(int i) {
  int ret = -1;
  if (i<_tracks) {
    if (_track[i]->isA()==SCORETRACK) ret = ((ScoreTrack*)_track[i])->program();
    if (_track[i]->isA()==DRUMTRACK)  ret = ((DrumTrack*)_track[i])->program();
  }
  if (ret==128) ret = _part[i]->program();
  return ret;
}

bool SongIterator::mute(int i) {
  int ret = true;
  if (i<_tracks) {
    ret = _track[i]->mute();
  }
  return ret;
}


void SongIterator::activate(int i) {
  if (i<_tracks && _event[i]->isA()==NOTE) {
    _mem[_memtop]    = (Note*) _event[i];
    _offset[_memtop] = _part[i]->start().ticks();
    _num[_memtop]    = i;
    if (_track[i]->isA()==SCORETRACK) _chan[_memtop] = ((ScoreTrack*)_track[i])->channel();
    if (_track[i]->isA()==DRUMTRACK)  _chan[_memtop] = ((DrumTrack*)_track[i])->channel();
    _memtop++;
  }
}

bool SongIterator::activeEndsAt(long p) {
  bool ret = false;
  for (int i=0; i<_memtop && !ret; i++) {
    if (_mem[i]->internalEnd() + _offset[i] == p) {
      ret = true;
      _memref = i;
    }
  }
  return ret;
}

Note * SongIterator::activeNote() { return _mem[_memref]; }

int SongIterator::activeChannel() { return _chan[_memref]; }

int SongIterator::activeTrack() { return _num[_memref]; }

void SongIterator::cutActive() {
  for (int i=_memref; i < _memtop; i++) {
    _mem[i] = _mem[i+1];
    _chan[i] = _chan[i+1];
    _offset[i] = _offset[i+1];
    _num[i] = _num[i+1];
  }
  _memtop--;
}



// vim:ts=8:sw=2:sts=2

#endif
