/***************************************************************************
 *            project-size.c
 *
 *  dim nov 27 14:33:46 2005
 *  Copyright  2005  Rouquier Philippe
 *  brasero-app@wanadoo.fr
 ***************************************************************************/

/*
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */


#include <string.h>

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <glib.h>
#include <glib/gi18n-lib.h>
#include <glib/gstdio.h>

#include <gtk/gtkhbox.h>
#include <gtk/gtkprogressbar.h>
#include <gtk/gtklabel.h>
#include <gtk/gtkliststore.h>
#include <gtk/gtkcellrenderer.h>
#include <gtk/gtkcellrenderertext.h>
#include <gtk/gtkcontainer.h>
#include <gtk/gtkcombobox.h>
#include <gtk/gtkalignment.h>
#include <gtk/gtkcelllayout.h>
#include <gtk/gtkdialog.h>
#include <gtk/gtkmessagedialog.h>
#include <gtk/gtkstock.h>

#include <nautilus-burn-drive.h>

#include "project-size.h"
#include "utils.h"
#include "brasero-ncb.h"

#ifdef NCB_2_15
#include <nautilus-burn-drive-monitor.h>
#endif

static void brasero_project_size_class_init (BraseroProjectSizeClass *klass);
static void brasero_project_size_init (BraseroProjectSize *sp);
static void brasero_project_size_finalize (GObject *object);

static void
brasero_project_size_combo_destroy_cb (GtkObject *object,
				       BraseroProjectSize *size);
static void
brasero_project_size_refresh (BraseroProjectSize *obj);

static void
brasero_project_size_check_active_drive (BraseroProjectSize *size,
					 gboolean rescan);

static void
brasero_project_size_combo_changed_cb (GtkComboBox *combo,
				       BraseroProjectSize *size);

static void
brasero_project_size_disc_added_cb (NautilusBurnDrive *drive,
				    BraseroProjectSize *size);
static void
brasero_project_size_disc_removed_cb (NautilusBurnDrive *drive,
				      BraseroProjectSize *size);

static gboolean
brasero_project_size_media_filter (GtkTreeModel *model,
				   GtkTreeIter *iter,
				   BraseroProjectSize *size);

#define AUDIO_SECTOR_SIZE 2352
#define DATA_SECTOR_SIZE 2048

struct BraseroProjectSizePrivate {
	GtkWidget *combo;

	gint64 disc_sectors;
	gint64 max_sectors;
	gint64 sectors;

	gchar *text;
	gint blink_id;
	GtkTreeRowReference *blink_row;

	gint refresh_id;

	gboolean is_audio_context:1;
	gboolean is_loaded:1;
};

enum {
	BRASERO_PROJECT_SIZE_COMBO_DISPLAY_COL,
	BRASERO_PROJECT_SIZE_COMBO_DRIVE_COL,
	BRASERO_PROJECT_SIZE_COMBO_SECTORS_COL,
	BRASERO_PROJECT_SIZE_COMBO_FRACTION_COL,
	BRASERO_PROJECT_SIZE_COMBO_MEDIA_COL,
	BRASERO_PROJECT_SIZE_COMBO_ICON_COL,
	BRASERO_PROJECT_SIZE_COMBO_BLANK_COL,
	BRASERO_PROJECT_SIZE_COMBO_NB_COL
};

static const gchar *type_icons [] = { 
	NULL,
	NULL,
	NULL,
	"gnome-dev-cdrom",
	"gnome-dev-disc-cdr",
	"gnome-dev-disc-cdrw",
	"gnome-dev-disc-dvdrom",
	"gnome-dev-disc-dvdr",
	"gnome-dev-disc-dvdrw",
	"gnome-dev-disc-dvdram",
	"gnome-dev-disc-dvdr-plus",
	"gnome-dev-disc-dvdrw", /* FIXME */
	"gnome-dev-disc-dvdr-plus" /* FIXME */,
	NULL
};

typedef enum {
	DISC_CHANGED_SIGNAL,
	LAST_SIGNAL
} BraseroProjectSizeSignalType;


static guint brasero_project_size_signals [LAST_SIGNAL] = { 0 };
static GObjectClass *parent_class = NULL;

GType
brasero_project_size_get_type()
{
	static GType type = 0;

	if(type == 0) {
		static const GTypeInfo our_info = {
			sizeof (BraseroProjectSizeClass),
			NULL,
			NULL,
			(GClassInitFunc)brasero_project_size_class_init,
			NULL,
			NULL,
			sizeof (BraseroProjectSize),
			0,
			(GInstanceInitFunc)brasero_project_size_init,
		};

		type = g_type_register_static (GTK_TYPE_HBOX, 
					       "BraseroProjectSize",
					       &our_info, 0);
	}

	return type;
}

static void
brasero_project_size_class_init (BraseroProjectSizeClass *klass)
{
	GObjectClass *object_class = G_OBJECT_CLASS (klass);
	
	parent_class = g_type_class_peek_parent (klass);
	object_class->finalize = brasero_project_size_finalize;

	brasero_project_size_signals [DISC_CHANGED_SIGNAL] =
	    g_signal_new ("disc_changed",
			  G_OBJECT_CLASS_TYPE (object_class),
			  G_SIGNAL_ACTION|G_SIGNAL_NO_RECURSE|G_SIGNAL_RUN_FIRST,
			  G_STRUCT_OFFSET (BraseroProjectSizeClass, disc_changed),
			  NULL, NULL,
			  g_cclosure_marshal_VOID__VOID,
			  G_TYPE_NONE,
			  0);

}

static gchar *
brasero_project_size_get_media_string (BraseroProjectSize *self,
				       NautilusBurnDrive *drive,
				       gint64 disc_sectors)
{
	gint64 size;
	gchar *text = NULL;
	gchar *drive_name = NULL;
	gchar *disc_sectors_str = NULL;
	gchar *selection_size_str = NULL;

	/* we should round the disc sizes / length */
	if (disc_sectors == -2) {
		/* this is an empty drive */
		return NULL;
	}
	else if (disc_sectors == -1) {
		char *name;

		/* this is a mounted drive */
		name = nautilus_burn_drive_get_name_for_display (drive);
		disc_sectors_str = g_strdup_printf (_("(mounted drive) %s "), name);
		g_free (name);
	}
	else {
		if (self->priv->is_audio_context) {
			size = disc_sectors * DATA_SECTOR_SIZE;
			disc_sectors_str = brasero_utils_get_time_string_from_size (size,
										    TRUE,
										    TRUE);
		}
		else {
			size = disc_sectors * DATA_SECTOR_SIZE;
			disc_sectors_str = brasero_utils_get_size_string (size,
									  TRUE,
									  TRUE);
		}

		if (drive)
			drive_name = nautilus_burn_drive_get_name_for_display (drive);
	}

	if (self->priv->is_audio_context) {
		size = self->priv->sectors / 75 * 1000000000;
		selection_size_str = brasero_utils_get_time_string (size,
								    TRUE);
	}
	else {
		size = self->priv->sectors * DATA_SECTOR_SIZE;
		selection_size_str = brasero_utils_get_size_string (size,
								    TRUE,
								    FALSE);
	}

	if (self->priv->sectors > disc_sectors) {
		gchar *tmp;

		/* FIXME: we don't want to break string for 0.4.X */
		if (drive_name)
			tmp = g_strdup_printf ("%s / %s %s",
					       selection_size_str,
					       disc_sectors_str,
					       drive_name);
		else
			tmp = g_strdup_printf ("%s / %s",
					       selection_size_str,
					       disc_sectors_str);

		text = g_strdup_printf (_("oversized (%s)"), tmp);
		g_free (tmp);
	}
	else if (self->priv->sectors == 0) {
		gchar *tmp;

		/* FIXME: we keep this for 0.5.X since we don't want to break 
		 * string */
		tmp = g_strdup (_("empty"));
		if (drive_name)
			text = g_strdup_printf ("%s (%s / %s, %s)",
						tmp,
						selection_size_str,
						disc_sectors_str,
						drive_name);
		else
			text = g_strdup_printf ("%s (%s / %s)",
						tmp,
						selection_size_str,
						disc_sectors_str);
		g_free (tmp);
	}
	else if (drive_name)
		text = g_strdup_printf ("%s / %s (%s)",
					selection_size_str,
					disc_sectors_str,
					drive_name);
	else
		text = g_strdup_printf ("%s / %s",
					selection_size_str,
					disc_sectors_str);

	g_free (selection_size_str);
	g_free (disc_sectors_str);
	g_free (drive_name);

	return text;
}

static void
brasero_project_size_update_fraction (BraseroProjectSize *self,
				      GtkTreeModel *model,
				      GtkTreeIter *row)
{
	gdouble fraction = 0.0;

	gint percent;
	gchar *string;
	gchar *prev_string;
	gint64 disc_sectors;
	NautilusBurnDrive *drive;

	gtk_tree_model_get (model, row,
			    BRASERO_PROJECT_SIZE_COMBO_DRIVE_COL, &drive,
			    BRASERO_PROJECT_SIZE_COMBO_SECTORS_COL, &disc_sectors,
			    -1);

	if (self->priv->sectors <= 0)
		fraction = 0.0;
	else if (self->priv->sectors > disc_sectors)
		fraction = 1.0;
	else
		fraction = (gdouble) ((gdouble) self->priv->sectors /
				       (gdouble) disc_sectors);

	percent = fraction * 100;
	string = brasero_project_size_get_media_string (self, drive, disc_sectors);
	gtk_tree_model_get (model, row,
			    BRASERO_PROJECT_SIZE_COMBO_DISPLAY_COL, &prev_string,
			    -1);

	if (prev_string && *prev_string == '\0') {
		/* the current text is blinking */
		if (self->priv->text)
			g_free (self->priv->text);

		self->priv->text = string;
		gtk_list_store_set (GTK_LIST_STORE (model), row,
				    BRASERO_PROJECT_SIZE_COMBO_FRACTION_COL, percent,
				    -1);
	}
	else {
		gtk_list_store_set (GTK_LIST_STORE (model), row,
				    BRASERO_PROJECT_SIZE_COMBO_FRACTION_COL, percent,
				    BRASERO_PROJECT_SIZE_COMBO_DISPLAY_COL, string,
				    -1);
		g_free (string);
	}

	g_free (prev_string);
}

static void
brasero_project_size_add_default_medias (BraseroProjectSize *self)
{
	gint i;
	gchar *text;
	GtkTreeIter row;
	GtkTreeModel *store;
	const gint64 len_array [] = { 333000,
					360000,
					405000,
					450000, 
					2295104 };

	store = gtk_combo_box_get_model (GTK_COMBO_BOX (self->priv->combo));
	store = gtk_tree_model_filter_get_model (GTK_TREE_MODEL_FILTER (store));

	/* we add the fixed sizes */
	for (i = 0; i < sizeof (len_array) / sizeof (len_array [0]); i++) {
		text = brasero_project_size_get_media_string (self,
							      NULL,
							      len_array [i]);

		gtk_list_store_prepend (GTK_LIST_STORE (store), &row);
		gtk_list_store_set (GTK_LIST_STORE (store), &row,
				    BRASERO_PROJECT_SIZE_COMBO_DISPLAY_COL, text,
				    BRASERO_PROJECT_SIZE_COMBO_SECTORS_COL, len_array [i],
				    BRASERO_PROJECT_SIZE_COMBO_DRIVE_COL, NULL,
				    BRASERO_PROJECT_SIZE_COMBO_MEDIA_COL, i > 3 ? NAUTILUS_BURN_MEDIA_TYPE_DVDR : NAUTILUS_BURN_MEDIA_TYPE_CDR,
				    BRASERO_PROJECT_SIZE_COMBO_BLANK_COL, TRUE,
				    BRASERO_PROJECT_SIZE_COMBO_ICON_COL, "gnome-dev-removable",
				    -1);
		g_free (text);
	}
}

static void
brasero_project_size_fill_model (BraseroProjectSize *self)
{
	GList *iter, *list;
	GtkTreeIter row;
	GtkTreeModel *store;
	NautilusBurnDrive *drive;
	NautilusBurnMediaType media;

	store = gtk_combo_box_get_model (GTK_COMBO_BOX (self->priv->combo));
	store = gtk_tree_model_filter_get_model (GTK_TREE_MODEL_FILTER (store));

	NCB_DRIVE_GET_LIST (list, TRUE, FALSE);
	for (iter = list; iter; iter = iter->next) {
		gboolean is_blank;
		gchar *text;
		gint64 size;
		gint64 sectors;

		drive = iter->data;

		/* add a callback if media changes */
		g_object_set (drive, "enable-monitor", TRUE, NULL);
		g_signal_connect (G_OBJECT (drive), "media-added",
				  G_CALLBACK (brasero_project_size_disc_added_cb),
				  self);
		g_signal_connect (G_OBJECT (drive), "media-removed",
				  G_CALLBACK (brasero_project_size_disc_removed_cb),
				  self);

		/* get all the information about the current media */
		media = nautilus_burn_drive_get_media_type_full (drive,
								 NULL,
								 &is_blank,
								 NULL,
								 NULL);

		size = nautilus_burn_drive_get_media_size (drive);
		size = size > 0 ? size : 0;
		sectors = size % 2048 ? size / 2048 + 1 : size / 2048;

		text = brasero_project_size_get_media_string (self,
							      drive,
							      size == -1 || size == -2 ? size : sectors);

		/* we add the drive to the combo */
		gtk_list_store_append (GTK_LIST_STORE (store), &row);
		gtk_list_store_set (GTK_LIST_STORE (store), &row,
				    BRASERO_PROJECT_SIZE_COMBO_DISPLAY_COL, text,
				    BRASERO_PROJECT_SIZE_COMBO_DRIVE_COL, drive,
				    BRASERO_PROJECT_SIZE_COMBO_MEDIA_COL, media,
				    BRASERO_PROJECT_SIZE_COMBO_BLANK_COL, is_blank,
				    BRASERO_PROJECT_SIZE_COMBO_SECTORS_COL, sectors,
				    BRASERO_PROJECT_SIZE_COMBO_ICON_COL, type_icons [media],
				    -1);

		g_free (text);
		/* FIXME: when we add an appendable disc maybe we could display 
		 * its label. Moreover we should get the name of the already 
		 * existing label in burn-dialog.c */
	}
	g_list_free (list);

	brasero_project_size_check_active_drive (self, FALSE);
}

static void
brasero_project_size_init (BraseroProjectSize *obj)
{
	GtkListStore *store;
	GtkTreeModel *filter;
	GtkCellRenderer *renderer;

	obj->priv = g_new0 (BraseroProjectSizePrivate, 1);
	gtk_box_set_spacing (GTK_BOX (obj), 6);

	/* combo box */
	obj->priv->combo = gtk_combo_box_new ();
	gtk_combo_box_set_focus_on_click (GTK_COMBO_BOX (obj->priv->combo), FALSE);
	g_object_set (obj->priv->combo, "has-frame", FALSE, NULL);
	g_signal_connect (G_OBJECT (obj->priv->combo),
			  "destroy",
			  G_CALLBACK (brasero_project_size_combo_destroy_cb),
			  obj);

	store = gtk_list_store_new (BRASERO_PROJECT_SIZE_COMBO_NB_COL,
				    G_TYPE_STRING,
				    G_TYPE_POINTER,
				    G_TYPE_INT64,
				    G_TYPE_INT,
				    G_TYPE_INT,
				    G_TYPE_STRING,
				    G_TYPE_BOOLEAN);

	filter = gtk_tree_model_filter_new (GTK_TREE_MODEL (store), NULL);
	gtk_tree_model_filter_set_visible_func (GTK_TREE_MODEL_FILTER (filter), 
						(GtkTreeModelFilterVisibleFunc) brasero_project_size_media_filter,
						obj,
						NULL);
	g_object_unref (store);

	gtk_combo_box_set_model (GTK_COMBO_BOX (obj->priv->combo), GTK_TREE_MODEL (filter));
	g_object_unref (filter);

	g_signal_connect_after (G_OBJECT (obj->priv->combo),
				"changed",
				G_CALLBACK (brasero_project_size_combo_changed_cb),
				obj);

	renderer = gtk_cell_renderer_pixbuf_new ();
	gtk_cell_layout_pack_start (GTK_CELL_LAYOUT (obj->priv->combo),
				    renderer,
				    FALSE);
	gtk_cell_layout_add_attribute (GTK_CELL_LAYOUT (obj->priv->combo),
				       renderer,
				       "icon-name", BRASERO_PROJECT_SIZE_COMBO_ICON_COL);
	g_object_set (renderer,
		      "follow-state", TRUE,
		      "xpad", 4,
		      "ypad", 3,
		      NULL);

	renderer = gtk_cell_renderer_progress_new ();
	gtk_cell_layout_pack_start (GTK_CELL_LAYOUT (obj->priv->combo),
				    renderer,
				    TRUE);
	gtk_cell_layout_set_attributes (GTK_CELL_LAYOUT (obj->priv->combo),
					renderer,
					"text", BRASERO_PROJECT_SIZE_COMBO_DISPLAY_COL,
					"value", BRASERO_PROJECT_SIZE_COMBO_FRACTION_COL,
					NULL);
	g_object_set (renderer,
		      "xpad", 6,
		      NULL);

	gtk_box_pack_end (GTK_BOX (obj), obj->priv->combo, TRUE, TRUE, 0);

	brasero_project_size_add_default_medias (obj);
}

static gboolean
brasero_project_size_media_filter (GtkTreeModel *model,
				   GtkTreeIter *iter,
				   BraseroProjectSize *size)
{
	NautilusBurnMediaType media;
	gboolean is_blank;

	gtk_tree_model_get (model, iter,
			    BRASERO_PROJECT_SIZE_COMBO_MEDIA_COL, &media,
			    BRASERO_PROJECT_SIZE_COMBO_BLANK_COL, &is_blank,
			    -1);

	if (size->priv->is_audio_context
	&&  media > NAUTILUS_BURN_MEDIA_TYPE_CDRW)
		return FALSE;

	if (!nautilus_burn_drive_media_type_is_writable (media, is_blank))
		return FALSE;

	return TRUE;
}

static void
brasero_project_size_free_store (BraseroProjectSize *size)
{
	NautilusBurnDrive *drive;
	GtkTreeModel *store;
	GtkTreeIter iter;

	store = gtk_combo_box_get_model (GTK_COMBO_BOX (size->priv->combo));
	store = gtk_tree_model_filter_get_model (GTK_TREE_MODEL_FILTER (store));

	if (!gtk_tree_model_get_iter_first (store, &iter))
		return;

	do {
		drive = NULL;
		gtk_tree_model_get (store, &iter,
				    BRASERO_PROJECT_SIZE_COMBO_DRIVE_COL, &drive,
				    -1);

		if (drive) {
			g_signal_handlers_disconnect_by_func (G_OBJECT (drive),
							      brasero_project_size_disc_added_cb,
							      size);
			g_signal_handlers_disconnect_by_func (G_OBJECT (drive),
							      brasero_project_size_disc_removed_cb,
							      size);
			nautilus_burn_drive_unref (drive);
		}
	} while (gtk_tree_model_iter_next (store, &iter));

	gtk_list_store_clear (GTK_LIST_STORE (store));
}

static void
brasero_project_size_blink_stop (BraseroProjectSize *self)
{
	if (self->priv->text && self->priv->blink_row) {
		GtkTreePath *treepath;
		GtkTreeModel *filter;
		GtkTreeModel *store;
		GtkTreeIter child;

		filter = gtk_combo_box_get_model (GTK_COMBO_BOX (self->priv->combo));
		store = gtk_tree_model_filter_get_model (GTK_TREE_MODEL_FILTER (filter));

		treepath = gtk_tree_row_reference_get_path (self->priv->blink_row);
		gtk_tree_row_reference_free (self->priv->blink_row);
		self->priv->blink_row = NULL;

		gtk_tree_model_get_iter (store, &child, treepath);
		gtk_tree_path_free (treepath);

		gtk_list_store_set (GTK_LIST_STORE (store), &child,
				    BRASERO_PROJECT_SIZE_COMBO_DISPLAY_COL, self->priv->text,
				    -1);

		g_free (self->priv->text);
		self->priv->text = NULL;
	}

	if (self->priv->text) {
		g_free (self->priv->text);
		self->priv->text = NULL;
	}

	if (self->priv->blink_id) {
		g_source_remove (self->priv->blink_id);
		self->priv->blink_id = 0;
	}
}

static gboolean
brasero_project_size_blink (BraseroProjectSize *self)
{
	GtkTreePath *treepath;
	GtkTreeModel *filter;
	GtkTreeModel *store;
	GtkTreeIter child;

	filter = gtk_combo_box_get_model (GTK_COMBO_BOX (self->priv->combo));
	store = gtk_tree_model_filter_get_model (GTK_TREE_MODEL_FILTER (filter));

	if (!gtk_combo_box_get_active_iter (GTK_COMBO_BOX (self->priv->combo), &child)) {
		/* the user is selecting a disc so we simply stop blinking */
		brasero_project_size_blink_stop (self);
		return FALSE;
	}
	    
	if (self->priv->text) {
		treepath = gtk_tree_row_reference_get_path (self->priv->blink_row);
		gtk_tree_row_reference_free (self->priv->blink_row);
		self->priv->blink_row = NULL;

		gtk_tree_model_get_iter (store, &child, treepath);
		gtk_tree_path_free (treepath);

		gtk_list_store_set (GTK_LIST_STORE (store), &child,
				    BRASERO_PROJECT_SIZE_COMBO_DISPLAY_COL, self->priv->text,
				    -1);

		g_free (self->priv->text);
		self->priv->text = NULL;
	}
	else {
		GtkTreeIter iter;

		if (!gtk_combo_box_get_active_iter (GTK_COMBO_BOX (self->priv->combo), &iter)) {
			/* we re-use or previous reference */
			treepath = gtk_tree_row_reference_get_path (self->priv->blink_row);
			gtk_tree_model_get_iter (store, &child, treepath);
			gtk_tree_path_free (treepath);
		}
		else
			gtk_tree_model_filter_convert_iter_to_child_iter (GTK_TREE_MODEL_FILTER (filter),
									  &child,
									  &iter);
			
		gtk_tree_model_get (store, &child,
				    BRASERO_PROJECT_SIZE_COMBO_DISPLAY_COL, &self->priv->text,
				    -1);
		gtk_list_store_set (GTK_LIST_STORE (store), &child,
				    BRASERO_PROJECT_SIZE_COMBO_DISPLAY_COL, "",
				    -1);

		if (!self->priv->blink_row) {
			treepath = gtk_tree_model_get_path (store, &child);
			self->priv->blink_row = gtk_tree_row_reference_new (store, treepath);
			gtk_tree_path_free (treepath);
		}
	}

	return TRUE;
}

static void
brasero_project_size_blink_start (BraseroProjectSize *self)
{
	if (!self->priv->blink_id)
		self->priv->blink_id = g_timeout_add (750,
						      (GSourceFunc) brasero_project_size_blink,
						      self);
}

static void
brasero_project_size_combo_destroy_cb (GtkObject *combo,
				       BraseroProjectSize *self)
{
	if (self->priv->combo) {
		brasero_project_size_free_store (self);
		self->priv->combo = NULL;
	}

	if (self->priv->text) {
		g_free (self->priv->text);
		self->priv->text = NULL;
	}

	if (self->priv->blink_id) {
		g_source_remove (self->priv->blink_id);
		self->priv->blink_id = 0;
	}
}

static void
brasero_project_size_finalize (GObject *object)
{
	BraseroProjectSize *cobj;

	cobj = BRASERO_PROJECT_SIZE (object);
	
	if (cobj->priv->refresh_id)
		g_source_remove (cobj->priv->refresh_id);

	g_free (cobj->priv);
	G_OBJECT_CLASS (parent_class)->finalize (object);
}

static gboolean
brasero_project_size_expose_event_cb (GtkWidget *widget,
				      gpointer event,
				      gpointer null_data)
{
	BraseroProjectSize *self = BRASERO_PROJECT_SIZE (widget);

	/* we delay the loading of real discs until project-size is actually
	 * showed not to block UI */
	if (!self->priv->is_loaded) {
		self->priv->is_loaded = 1;
		brasero_project_size_fill_model (self);
	}

	return FALSE;
}

GtkWidget *
brasero_project_size_new ()
{
	BraseroProjectSize *obj;
	
	obj = BRASERO_PROJECT_SIZE (g_object_new (BRASERO_TYPE_PROJECT_SIZE, NULL));
	g_signal_connect (obj,
			  "expose-event",
			  G_CALLBACK (brasero_project_size_expose_event_cb),
			  NULL);

	return GTK_WIDGET (obj);
}

/************************************* COMBO STUFF *****************************/
static void
brasero_project_size_disc_added_cb (NautilusBurnDrive *drive,
				    BraseroProjectSize *self)
{
	gint64 size;
	gint64 sectors;
	GtkTreeIter row;
	gboolean is_blank;
	GtkTreeModel *store;
	NautilusBurnMediaType media;


	/* search for the right line */
	store = gtk_combo_box_get_model (GTK_COMBO_BOX (self->priv->combo));
	store = gtk_tree_model_filter_get_model (GTK_TREE_MODEL_FILTER (store));

	if (!gtk_tree_model_get_iter_first (store, &row))
		return;

	media = nautilus_burn_drive_get_media_type_full (drive,
							 NULL,
							 &is_blank,
							 NULL,
							 NULL);

	size = nautilus_burn_drive_get_media_size (drive);
	size = size > 0 ? size : 0;
	sectors = size % 2048 ? size / 2048 + 1 : size / 2048;

	do {
		NautilusBurnDrive *list_drive;

		list_drive = NULL;
		gtk_tree_model_get (store, &row,
				    BRASERO_PROJECT_SIZE_COMBO_DRIVE_COL, &list_drive,
				    -1);

		if (list_drive && nautilus_burn_drive_equal (drive, list_drive)) {
			gtk_list_store_set (GTK_LIST_STORE (store), &row,
					    BRASERO_PROJECT_SIZE_COMBO_MEDIA_COL, media,
					    BRASERO_PROJECT_SIZE_COMBO_BLANK_COL, is_blank,
					    BRASERO_PROJECT_SIZE_COMBO_SECTORS_COL, sectors,
					    BRASERO_PROJECT_SIZE_COMBO_ICON_COL, type_icons [media],
					    -1);

			brasero_project_size_update_fraction (self, store, &row);
			brasero_project_size_check_active_drive (self, TRUE);
			break;
		}

	} while (gtk_tree_model_iter_next (store, &row));
}

static void
brasero_project_size_disc_removed_cb (NautilusBurnDrive *drive,
				      BraseroProjectSize *size)
{
	NautilusBurnDrive *list_drive;
	GtkTreeModel *store;
	GtkTreeIter row;

	store = gtk_combo_box_get_model (GTK_COMBO_BOX (size->priv->combo));
	store = gtk_tree_model_filter_get_model (GTK_TREE_MODEL_FILTER (store));

	if (gtk_tree_model_get_iter_first (store, &row) == FALSE)
		return;

	do {
		gtk_tree_model_get (store, &row,
				    BRASERO_PROJECT_SIZE_COMBO_DRIVE_COL, &list_drive,
				    -1);

		if (!list_drive)
			continue;

		if (nautilus_burn_drive_equal (drive, list_drive)) {
			gtk_list_store_set (GTK_LIST_STORE (store), &row,
					    BRASERO_PROJECT_SIZE_COMBO_DISPLAY_COL, NULL,
					    BRASERO_PROJECT_SIZE_COMBO_MEDIA_COL, NAUTILUS_BURN_MEDIA_TYPE_ERROR,
					    BRASERO_PROJECT_SIZE_COMBO_BLANK_COL, FALSE,
					    BRASERO_PROJECT_SIZE_COMBO_SECTORS_COL, (gint64) 0,
					    BRASERO_PROJECT_SIZE_COMBO_ICON_COL, NULL,
					    -1);

			brasero_project_size_check_active_drive (size, FALSE);
			break;
		}
	} while (gtk_tree_model_iter_next (store, &row) == TRUE);
}

static gboolean
brasero_project_size_find_active_disc (BraseroProjectSize *size,
				       GtkTreeIter *iter)
{
	NautilusBurnMediaType media;
	NautilusBurnDrive *drive;
	gboolean change_required;
	GtkTreePath *treepath;
	GtkTreeModel *filter;
	gint64 disc_sectors;

	filter = gtk_combo_box_get_model (GTK_COMBO_BOX (size->priv->combo));
	gtk_tree_model_get (filter, iter,
			    BRASERO_PROJECT_SIZE_COMBO_DRIVE_COL, &drive,
			    BRASERO_PROJECT_SIZE_COMBO_MEDIA_COL, &media,
			    BRASERO_PROJECT_SIZE_COMBO_SECTORS_COL, &disc_sectors,
			    -1);

	/* Check if this row is actually a drive containing a media that fits */
	if (drive
	&&  disc_sectors > 0
	&&  media > NAUTILUS_BURN_MEDIA_TYPE_UNKNOWN
        && (!size->priv->is_audio_context || media <= NAUTILUS_BURN_MEDIA_TYPE_CDRW))
		return FALSE;

	/* Check if this row is a virtual disc and could fit the current context
	 * that way if we can't find a suitable media, then we won't have to try
	 * to find another suitable virtual disc. */
	if (!drive
        && (!size->priv->is_audio_context || media <= NAUTILUS_BURN_MEDIA_TYPE_CDRW))
		change_required = FALSE;
	else
		change_required = TRUE;

	/* save the iter position in case we don't find a suitable real disc */
	treepath = gtk_tree_model_get_path (filter, iter);
	gtk_tree_model_get_iter_first (filter, iter);
	while (1) {
		gtk_tree_model_get (filter, iter,
				    BRASERO_PROJECT_SIZE_COMBO_DRIVE_COL, &drive,
				    BRASERO_PROJECT_SIZE_COMBO_MEDIA_COL, &media,
				    BRASERO_PROJECT_SIZE_COMBO_SECTORS_COL, &disc_sectors,
				    -1);

		/* we are going over the filter model so any drive not
		 * fitting is not seeable. Therefore as soon as we have
		 * a drive, it means that this line can be shown. */
		if (drive
		&&  disc_sectors > 0
		&&  media > NAUTILUS_BURN_MEDIA_TYPE_UNKNOWN
		&&  (!size->priv->is_audio_context || media <= NAUTILUS_BURN_MEDIA_TYPE_CDRW)) {
			gtk_tree_path_free (treepath);
			return TRUE;
		}
	
		if (!gtk_tree_model_iter_next (filter, iter))
			break;
	}

	if (!change_required) {
		gtk_tree_model_get_iter (filter, iter, treepath);
		gtk_tree_path_free (treepath);
		return FALSE;
	}

	/* We have to find another virtual disc fitting the context */
	gtk_tree_model_get_iter_first (filter, iter);
	while (1) {
		gtk_tree_model_get (filter, iter,
				    BRASERO_PROJECT_SIZE_COMBO_DRIVE_COL, &drive,
				    BRASERO_PROJECT_SIZE_COMBO_MEDIA_COL, &media,
				    -1);

		/* we are looking for a virtual disc that is not a DVD */
		if (!drive && media <= NAUTILUS_BURN_MEDIA_TYPE_CDRW) {
			gtk_tree_path_free (treepath);
			return TRUE;
		}
	
		if (!gtk_tree_model_iter_next (filter, iter))
			break;
	}

	gtk_tree_model_get_iter (filter, iter, treepath);
	gtk_tree_path_free (treepath);

	return FALSE;
}

static void
brasero_project_size_check_active_drive (BraseroProjectSize *size, gboolean rescan)
{
	GtkTreeModel *filter;
	GtkTreeIter iter;

	filter = gtk_combo_box_get_model (GTK_COMBO_BOX (size->priv->combo));
	if (!gtk_combo_box_get_active_iter (GTK_COMBO_BOX (size->priv->combo), &iter)) {
		gtk_tree_model_get_iter_first (filter, &iter);
		brasero_project_size_find_active_disc (size, &iter);
		gtk_combo_box_set_active_iter (GTK_COMBO_BOX (size->priv->combo), &iter);
	}
	else if (rescan && brasero_project_size_find_active_disc (size, &iter))
		gtk_combo_box_set_active_iter (GTK_COMBO_BOX (size->priv->combo), &iter);
}


static void
brasero_project_size_combo_changed_cb (GtkComboBox *combo,
				       BraseroProjectSize *self)
{
	GtkTreeModel *filter;
	GtkTreeIter iter;

	filter = gtk_combo_box_get_model (GTK_COMBO_BOX (self->priv->combo));
	if (!gtk_combo_box_get_active_iter (GTK_COMBO_BOX (self->priv->combo), &iter)) {
		/* we wait for the above function to set a valid drive */
		return;
	}
	
	gtk_tree_model_get (filter, &iter,
			    BRASERO_PROJECT_SIZE_COMBO_SECTORS_COL, &self->priv->disc_sectors,
			    -1);

	if (self->priv->disc_sectors < 0)
		self->priv->disc_sectors = 0;

	/* FIXME: This is not good since with a DVD 3% of 4.3G may be too much
	 * with 3% we are slightly over the limit of the most overburnable discs
	 * but at least users can try to overburn as much as they can. */

	/* The idea would be to test write the disc with cdrecord from /dev/null
	 * until there is an error and see how much we were able to write. So,
	 * when we propose overburning to the user, we could ask if he wants
	 * us to determine how much data can be written to a particular disc
	 * provided he has chosen a real disc. */
	self->priv->max_sectors = self->priv->disc_sectors * 103 / 100;
	//brasero_project_size_refresh (size);

	if (self->priv->disc_sectors < self->priv->sectors)
		brasero_project_size_blink_start (self);
	else
		brasero_project_size_blink_stop (self);

	g_signal_emit (self,
		       brasero_project_size_signals [DISC_CHANGED_SIGNAL],
		       0);
}

static void
brasero_project_size_update_strings (BraseroProjectSize *self)
{
	GtkTreeModel *store;
	GtkTreeIter iter;

	/* when the context changes we need to update the sizes */
	store = gtk_combo_box_get_model (GTK_COMBO_BOX (self->priv->combo));
	store = gtk_tree_model_filter_get_model (GTK_TREE_MODEL_FILTER (store));

	if (!gtk_tree_model_get_iter_first (store, &iter))
		return;

	do {
		NautilusBurnDrive *drive;
		gint64 size;
		gchar *text;

		gtk_tree_model_get (store, &iter,
				    BRASERO_PROJECT_SIZE_COMBO_DRIVE_COL, &drive,
				    BRASERO_PROJECT_SIZE_COMBO_SECTORS_COL, &size,
				    -1);

		text = brasero_project_size_get_media_string (self,
							      drive,
							      size);

		gtk_list_store_set (GTK_LIST_STORE (store), &iter,
				    BRASERO_PROJECT_SIZE_COMBO_DISPLAY_COL, text,
				    -1);
		g_free (text);
	} while (gtk_tree_model_iter_next (store, &iter));

	brasero_project_size_check_active_drive (self, FALSE);
}

void
brasero_project_size_set_context (BraseroProjectSize *self,
				  gboolean is_audio)
{
	GtkTreeModel *filter;

	self->priv->sectors = 0;
	if (is_audio == self->priv->is_audio_context)
		return;

	self->priv->is_audio_context = is_audio;

	filter = gtk_combo_box_get_model (GTK_COMBO_BOX (self->priv->combo));
	gtk_tree_model_filter_refilter (GTK_TREE_MODEL_FILTER (filter));
	brasero_project_size_update_strings (self);
}

static gboolean
brasero_project_size_update_sectors (BraseroProjectSize *self)
{
	GtkTreeModel *model;
	GtkTreeIter row;

	model = gtk_combo_box_get_model (GTK_COMBO_BOX (self->priv->combo));
	model = gtk_tree_model_filter_get_model (GTK_TREE_MODEL_FILTER (model));

	if (!gtk_tree_model_get_iter_first (model, &row)) {
		self->priv->refresh_id = 0;
		return FALSE;
	}

	do {
		brasero_project_size_update_fraction (self, model, &row);
	} while (gtk_tree_model_iter_next (model, &row));

	self->priv->refresh_id = 0;

	if (self->priv->disc_sectors < self->priv->sectors)
		brasero_project_size_blink_start (self);
	else
		brasero_project_size_blink_stop (self);

	return FALSE;
}

static void
brasero_project_size_refresh (BraseroProjectSize *self)
{
	if (!self->priv->refresh_id)
		self->priv->refresh_id = g_timeout_add (500,
						       (GSourceFunc) brasero_project_size_update_sectors,
						       self);
}

void
brasero_project_size_set_sectors (BraseroProjectSize *self,
				  gint64 sectors)
{
	/* we don't update the size right now but in half a second.
	 * when exploring directories size can changed repeatedly
	 * and we don't want to lose too much time updating.
	 * if a size is already set, we know that we're waiting for
	 * a size update, so, just replace the old size. otherwise
	 * we add a g_timeout_add */

	/* we add 175 sectors for a single session track (that's the needed
	 * overhead.
	 * for multisessions (we'll need ncb 2.15), the overhead is much
	 * larger since we'll have to add 2 sec gap between tracks (300 sectors)
	 * - first track : 6750 sectors (1.5 min) and leadout 4500 sectors (1 mn)
	 *   and 2 sec gap (150 sectors)
	 * - next tracks : leadin 6750 sectors, leadout 2250 sectors (0.5 mn)
	 * Now, for the moment we don't know exactly how much we need ...
	 * so we add the maximum number of sectors and if the user wants he can
	 * still use overburn
	 */
	/* FIXME: for now just add 500 sectors = 1Mib */
	if (sectors)
		self->priv->sectors = sectors + 500;
	else
		self->priv->sectors = 0;

	brasero_project_size_refresh (self);
}

gboolean
brasero_project_size_check_status (BraseroProjectSize *self,
				   gboolean *overburn)
{
	if (self->priv->disc_sectors <= 0) {
		if (overburn)
			*overburn = FALSE;

		return TRUE;
	}

	if (self->priv->max_sectors < self->priv->sectors) {
		if (overburn)
			*overburn = FALSE;

		return FALSE;
	}

	if (self->priv->disc_sectors < self->priv->sectors) {
		if (overburn)
			*overburn = TRUE;

		return FALSE;
	}

	return TRUE;
}
