
/*
 *  Diverse Bristol audio routines.
 *  Copyright (c) by Nick Copeland <nick.copeland@ntlworld.com> 1996,2002
 *
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */
/*
 * This will be a linear potmeter. Takes a bitmap and locates it according to
 * input from the mouse/keyboard. Previous positions need to be unmapped, since
 * there is total repositioning of the image bitmap.
 */

#include "brightoninternals.h"

int
destroyScale(brightonDevice *dev)
{
	printf("destroyScale()\n");

	if (dev->image)
		brightonFreeBitmap(dev->bwin, dev->image);
	dev->image = NULL;
}

static int
displayscale(brightonDevice *dev)
{
	brightonIResource *panel;
	float displayvalue;

	if (dev->bwin->app->resources[dev->panel].flags & BRIGHTON_WITHDRAWN)
		return;

	panel = &dev->bwin->app->resources[dev->panel];
	/*
	 * Build up a smooth position for the pot. We may need to adjust this based
	 * on the to/from values.
	 */

	if (dev->value == dev->lastvalue)
		return;

/*
	if ((dev->bwin->app->resources[dev->panel].devlocn[dev->index].flags
		& BRIGHTON_REVERSE) == 0)
		displayvalue = 1.0 - dev->value;
	else
		displayvalue = dev->value;
*/
	displayvalue = dev->value;

	if (dev->lastposition >= 0)
	{
		brightonDevUndraw(dev->bwin, dev->bwin->dlayer,
			dev->x + dev->bwin->app->resources[dev->panel].sx,
			dev->y + ((int) dev->lastposition)
				+ dev->bwin->app->resources[dev->panel].sy,
			dev->width, dev->height / 4);
		brightonRenderShadow(dev, 1);
	}
/*
	brightonFinalRender(dev->bwin,
		dev->x + dev->bwin->app->resources[dev->panel].sx,
		dev->y + dev->lastposition + dev->bwin->app->resources[dev->panel].sy,
		dev->width, dev->height / 4);
*/

	dev->position = displayvalue * (dev->height - dev->height / 4);

	if (dev->bwin->app->resources[dev->panel].devlocn[dev->index].flags
		& BRIGHTON_HAMMOND)
	{
		brightonDevUndraw(dev->bwin, dev->bwin->dlayer,
			dev->x + dev->bwin->app->resources[dev->panel].sx,
			dev->y + dev->bwin->app->resources[dev->panel].sy,
			dev->width, dev->height);

		brightonStretch(dev->bwin, dev->image2,
			dev->bwin->dlayer,
			(int) (dev->x + dev->bwin->app->resources[dev->panel].sx),
			(int) (dev->y + dev->bwin->app->resources[dev->panel].sy),
			dev->width, (int) dev->position,
				0);

		brightonFinalRender(dev->bwin,
			dev->x + dev->bwin->app->resources[dev->panel].sx,
			dev->y + dev->bwin->app->resources[dev->panel].sy,
			dev->width, (int) dev->height);
	}

	/*
	 * Only draw fixed number of steps.
	 */
	brightonStretch(dev->bwin, dev->image,
		dev->bwin->dlayer,
		dev->x + dev->bwin->app->resources[dev->panel].sx,
		(int)(dev->y + dev->position
			+ dev->bwin->app->resources[dev->panel].sy),
		dev->width, dev->height / 4,
		dev->position);

	brightonRenderShadow(dev, 0);

	/*
	 * And request the panel to put this onto the respective image.
	 */
	if (dev->position > dev->lastposition)
	{
		brightonFinalRender(dev->bwin,
			dev->x + dev->bwin->app->resources[dev->panel].sx,
			dev->y + dev->lastposition
				+ dev->bwin->app->resources[dev->panel].sy,
			dev->width * 2,
			dev->position - dev->lastposition + (dev->height >> 2)
				+ (dev->width >> 1));
	} else {
		brightonFinalRender(dev->bwin,
			dev->x + dev->bwin->app->resources[dev->panel].sx,
			dev->y + dev->position
				+ dev->bwin->app->resources[dev->panel].sy,
			dev->width * 2,
			dev->lastposition - dev->position + (dev->height >> 2) + 1
				+ (dev->width >> 1));
	}

	dev->lastvalue = dev->value;
	dev->lastposition = dev->position;
}

static int considercallback(brightonDevice *dev)
{
	brightonIResource *panel = &dev->bwin->app->resources[dev->panel];
	float callvalue;

	if (dev->value > 1.0)
		dev->value = 1.0;
	else if (dev->value < 0)
		dev->value = 0.0;

	/*
	 * Due to the co-ordinate system, if we do NOT have the reverse flags
	 * then we need to reverse the value.
	 */
	if ((dev->bwin->app->resources[dev->panel].devlocn[dev->index].flags
		& BRIGHTON_REVERSE) == 0)
		callvalue = 1.0 - dev->value;
	else
		callvalue = dev->value;

	if (dev->bwin->app->resources[dev->panel].devlocn[dev->index].to != 1.0)
	{
		callvalue =
			(callvalue
			* dev->bwin->app->resources[dev->panel].devlocn[dev->index].to);

		if ((callvalue - ((int) callvalue)) > 0.5)
			callvalue = ((float) ((int) callvalue) + 1);
		else
			callvalue = ((float) ((int) callvalue));
	}

	if (dev->lastvalue != dev->value)
	{
		if (panel->devlocn[dev->index].callback)
		{
			panel->devlocn[dev->index].callback(dev->bwin, dev->panel,
				dev->index, callvalue);
		} else {
			if (panel->callback)
				panel->callback(dev->bwin, dev->panel, dev->index, callvalue);
		}
	}
}

static int
configure(brightonDevice *dev, brightonEvent *event)
{
//	printf("configureScale()\n");

	if (event->command == -1)
		return(-1);

	if (event->command == BRIGHTON_RESIZE)
	{
		dev->originx = event->x;
		dev->originy = event->y;

		dev->x = event->x;
		dev->y = event->y;
		dev->width = event->w;
		dev->height = event->h;

		/*
		 * We should now rework our parent understanding of our window, since
		 * it will have altered. NOT NECESSARY FOR SCALE.
		brightonPanelLocation(dev->bwin,
			dev->panel, dev->index, dev->x, dev->y, dev->width, dev->height);

		considercallback(dev);
		 */

		/*
		 * Highlights need to be rendered dynamically in displayscale().
		 */

		dev->lastvalue = -1;
		displayscale(dev);

		return(0);
	}

	if (event->command == BRIGHTON_KEYRELEASE)
	{
		switch(event->key) {
			default:
				break;
			case 50:
			case 62:
				dev->flags &= ~BRIGHTON_SHIFTKEY;
				break;
		}
	}

	if (event->command == BRIGHTON_BUTTONRELEASE)
	{
		if (dev->bwin->app->resources[dev->panel].devlocn[dev->index].flags
			& BRIGHTON_CENTER)
		{
			dev->value = 0.5;

			considercallback(dev);

			displayscale(dev);
		}

		return(0);
	}

	if (event->command == BRIGHTON_KEYPRESS)
	{
		switch(event->key) {
			default:
				break;
			case 50:
			case 62:
				dev->flags |= BRIGHTON_SHIFTKEY;
				break;
			case 44:
			case 104:
				if (dev->flags & BRIGHTON_SHIFTKEY)
					dev->value += ((float) 100) / 16384;
				else
					dev->value += ((float) 1) / 16384;
				break;
			case 45:
			case 98:
				if (dev->flags & BRIGHTON_SHIFTKEY)
					dev->value -= ((float) 100) / 16384;
				else
					dev->value -= ((float) 1) / 16384;
				break;
		}

		considercallback(dev);

		displayscale(dev);
	}

	if (event->command == BRIGHTON_MOTION)
	{
		dev->value = (((float) (event->y - dev->y - (dev->height / 8)))
			/ (dev->height - dev->height / 4));

//printf("scale motion %i %i, %i %i, %i %i: %f\n",
//event->x, event->y, dev->x, dev->y, dev->width, dev->height, dev->value);

		/*
		 * We now need to consider rounding this to the resolution of this
		 * device. If the max value is not 1.0 then we need to put fixed steps
		 * into our new device value.
		 */
		if (dev->bwin->app->resources[dev->panel].devlocn[dev->index].to != 1.0)
		{
			dev->value = (float) ((int)
				(dev->value
				* dev->bwin->app->resources[dev->panel].devlocn[dev->index].to))
				/ dev->bwin->app->resources[dev->panel].devlocn[dev->index].to;
		}

		considercallback(dev);

		displayscale(dev);

		return(0);
	}

	if (event->command == BRIGHTON_PARAMCHANGE)
	{
		if (dev->bwin->app->resources[dev->panel].devlocn[dev->index].flags
			& BRIGHTON_REVERSE)
			dev->value = event->value
				/ dev->bwin->app->resources[dev->panel].devlocn[dev->index].to;
		else
			dev->value = (1.0 - event->value)
				/ dev->bwin->app->resources[dev->panel].devlocn[dev->index].to;

		considercallback(dev);

		displayscale(dev);

		return(0);
	}
}

int *
createScale(brightonWindow *bwin, brightonDevice *dev, int index, char *bitmap)
{
//	printf("createScale(%s)\n", bitmap);

	dev->destroy = destroyScale;
	dev->configure = configure;
	dev->index = index;

	dev->bwin = bwin;

	if (bitmap == 0)
	{
		if (dev->image)
			brightonFreeBitmap(bwin, dev->image);
		/*
		 * Open the default bitmap
		 */
		if (bwin->app->resources[dev->panel].devlocn[dev->index].image != 0)
			dev->image =
				bwin->app->resources[dev->panel].devlocn[dev->index].image;
		else
			dev->image = brightonReadImage(bwin, "bitmaps/knobs/slider1.xpm");
	} else {
		if (dev->image)
			brightonFreeBitmap(bwin, dev->image);
		dev->image = brightonReadImage(bwin, bitmap);
	}

	if (bwin->app->resources[dev->panel].devlocn[dev->index].flags &
		BRIGHTON_HAMMOND)
	{
		if (dev->image2)
			brightonFreeBitmap(bwin, dev->image2);
		dev->image2 = brightonReadImage(bwin, "bitmaps/knobs/extend.xpm");
	}

	/*
	 * These will force an update when we first display ourselves.
	 */
	if (bwin->app->resources[dev->panel].devlocn[dev->index].flags
		& BRIGHTON_CENTER)
		dev->value = 0.5;
	else
		dev->value = 0;

	dev->lastvalue = -1;
	dev->lastposition = 0;

	return(0);
}

