
/*
 *  Diverse Bristol audio routines.
 *  Copyright (c) by Nick Copeland <nick.copeland@ntlworld.com> 1996,2002
 *
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

//#define DEBUG

#include <math.h>

#include "bristol.h"
#include "palette.h"

//int atStatus = 0;

extern int bristolGlobalController();
extern int operateOneMMVoiceFinal();
extern int operateOneHammondVoice();
extern int buildCurrentTable(Baudio *, float);

//Baudio *baudio;

void *
audioThread(audioMain *audiomain)
{
	int i, rr;
	float *outbuf, *startbuf;
	char *device;

	audiomain->atStatus = BRISTOL_WAIT;

	audiomain->debuglevel = 0;
#ifdef DEBUG
	audiomain->debuglevel = -1;
#endif
printf("init audio thread, %x\n", audiomain);

	if (audiomain->audiodev != (char *) NULL)
		device = audiomain->audiodev;
	else if ((audiomain->flags & BRISTOL_AUDIOMASK) == BRISTOL_OSS)
		device = bOAD;
	else
		device = bAAD;

	audiomain->opCount = PALETTE_SIZE;

	if ((audiomain->debuglevel & BRISTOL_DEBUG_MASK) > BRISTOL_DEBUG1)
		printf("starting audio thread\n");

printf("bufsize is now %i\n", audiomain->iosize);
	/*
	 * open the audio device with our desired buffer size, and take its
	 * returned fragment size.
	 */
	if ((audiomain->iosize = bristolAudioOpen(device, audiomain->samplerate,
		audiomain->iosize, audiomain->flags|audiomain->preload))
			< 0)
	{
		printf("Problem opening audio device %s, exiting audio thread\n",
			device);
		audiomain->atStatus = 0;
		pthread_exit(-1);
	}

	/*
	 * This fragment size is in bytes. The audio will be stereo shorts, so
	 * find out how many samples this is:
	 *	samplecount/2
	 * Our samples are going to be floats, and Mono until the later stages?
	 */
	audiomain->samplecount = (audiomain->iosize / sizeof(short)) / 2;

	audiomain->segmentsize = audiomain->samplecount * sizeof(float);

	if ((audiomain->debuglevel & BRISTOL_DEBUG_MASK) > BRISTOL_DEBUG1)
		printf("segmentsize is %i\n", audiomain->segmentsize);

	/*
	 * A segment is enough for the given number of mono samples in float res.
	 * Our IO buffers deal with stereo, so we need twice as many.
	 */
	outbuf = (float *) bristolmalloc(audiomain->segmentsize * 2);
	startbuf = (float *) bristolmalloc(audiomain->segmentsize * 2);

	if ((audiomain->debuglevel & BRISTOL_DEBUG_MASK) > BRISTOL_DEBUG1)
		printf("IO %i, samples %i, segsize %i\n",
			audiomain->iosize, audiomain->samplecount, audiomain->segmentsize);

	initAudioThread(audiomain);

	bristolbzero(outbuf, audiomain->segmentsize * 2);

	/*
	 * Preload the output
	 */
	for (i = 0; i < audiomain->preload; i++)
		bristolAudioWrite(outbuf, audiomain->samplecount);

	bristolAudioStart();

	audiomain->atStatus = BRISTOL_OK;

	while (audiomain->atReq != BRISTOL_REQSTOP)
	{
//printf("cycle\n");
		/*
		 * We should call here to operate the voices. Suggest we pass a single
		 * destination buffer, which we will write, on return, to the audio
		 * device.
		 */
		if (bristolAudioRead(startbuf, audiomain->samplecount * 2) < 0)
			printf("Audio device read issue\n");
//printf("doneread\n");
		/*
		 * doAudioOps should not use baudio, these will become a linked list
		 */
		doAudioOps(audiomain, outbuf, startbuf);

		if ((rr = bristolAudioWrite(outbuf, audiomain->samplecount * 2)) < 0)
		{
			if (rr != -4)
			{
				printf("Audio device write issue: restart pl %i\n",
					audiomain->preload);
				bristolAudioClose();
				if (bristolAudioOpen(device, audiomain->samplerate,
					audiomain->iosize,
					audiomain->flags|audiomain->preload) < 0)
					/*
					 * Then we have an issue.
					 */
					audiomain->atReq = BRISTOL_REQSTOP;
				else
					/*
					 * Preload the output
					 */
					for (i = 0; i < audiomain->preload; i++)
						bristolAudioWrite(outbuf, audiomain->samplecount * 2);
			}
		}
	}
	
	bristolfree(outbuf);
	bristolfree(startbuf);

	resetAudioThread(audiomain);

	bristolAudioClose();

	if ((audiomain->debuglevel & BRISTOL_DEBUG_MASK) > BRISTOL_DEBUG1)
		printf("audioThread exiting\n");

	audiomain->atStatus = BRISTOL_EXIT;
	pthread_exit(0);
}

initBristolAudio(audioMain *audiomain, Baudio *baudio)
{
	int i, j;
	bristolOP *op;

	if (baudio == NULL)
		return;

	if ((audiomain->debuglevel & BRISTOL_DEBUG_MASK) > BRISTOL_DEBUG1)
		printf("initBristolAudio()\n");

	if (audiomain->atStatus != BRISTOL_OK)
		return(-1);

	if ((audiomain->debuglevel & BRISTOL_DEBUG_MASK) > BRISTOL_DEBUG3)
		printf("using baudio: %x->%x\n", audiomain->audiolist,
			baudio->next);

	baudio->leftbuf = (float *) bristolmalloc0(audiomain->segmentsize);
	baudio->rightbuf = (float *) bristolmalloc0(audiomain->segmentsize);

	/*
	 * These will come from some configuration file, eventually.
	 */
	baudio->note_diff = pow((double) 2, (double) 1/12);
	baudio->gain_diff = pow((double) 13, ((double) 1)/DEF_TAB_SIZE);
	baudio->midi_pitch = 12;
	baudio->midi_pitchfreq = 0.0;
	baudio->gtune = 1.0;
	buildCurrentTable(baudio, 1.0);

	baudio->contcontroller[1] = 0.5;
	baudio->pitchwheel = 0.5;
	/*
	 * This should call an init routine for any given baudio.
	 */
	if (bristolAlgos[(audiomain->midiflags & BRISTOL_PARAMMASK)].initialise)
		bristolAlgos[(audiomain->midiflags & BRISTOL_PARAMMASK)].initialise
			(audiomain, baudio);

	/*
	 * Put in a locals array pointer. This is two dimensional, since the locals
	 * are in the audio structure, but we need a copy for each individual voice.
	 *
	 * The net result is that locals at the baudio level is a pointer to an 
	 * array of pointers to arrays pointing to chars.
	 */
	baudio->locals = (char ***)
		bristolmalloc0(sizeof(void *) * audiomain->voiceCount);
	baudio->FXlocals = (char ***)
		bristolmalloc0(sizeof(void *) * audiomain->voiceCount);

	for (i = 0; i < audiomain->voiceCount; i++)
	{
		baudio->locals[i] =
			(void *) bristolmalloc0(baudio->soundCount * sizeof(void *));

		baudio->FXlocals[i] =
			(void *) bristolmalloc0(baudio->soundCount * sizeof(void *));

		if ((baudio->effect != NULL) && (baudio->effect[0] != NULL))
		{
			op = (audiomain->effects)[(*baudio->effect[0]).index];
			baudio->FXlocals[i][0]
				= (char *) bristolmalloc0(op->specs->localsize);
		}

		/*
		 * Put in the locals.
		 */
		for (j = 0; j < baudio->soundCount; j++)
		{
			if ((audiomain->debuglevel & BRISTOL_DEBUG_MASK) > BRISTOL_DEBUG1)
				printf("Adding new MIDI operator: %i\n", j);

			op = (audiomain->palette)[(*baudio->sound[j]).index];
			baudio->locals[i][j]
				= (char *) bristolmalloc0(op->specs->localsize);

			if ((audiomain->debuglevel & BRISTOL_DEBUG_MASK) > BRISTOL_DEBUG4)
				printf("malloc locals %x %i\n", baudio->locals[i][j],
					op->specs->localsize);
		}
	}
	baudio->mixflags &= ~BRISTOL_HOLDDOWN;
	baudio->lowkey = 0;
	baudio->highkey = 127;
}

initAudioThread(audioMain *audiomain)
{
	if ((audiomain->debuglevel & BRISTOL_DEBUG_MASK) > BRISTOL_DEBUG1)
		printf("initAudioThread()\n");

	/*
	 * Assign an array of operator pointers.
	 */
	audiomain->palette = (bristolOP **)
		bristolmalloc0(sizeof(bristolOP *) * audiomain->opCount);

	/*
	 * Now that we have some basic stuctures we should start initiating the
	 * operators. This is done one by one, for the
	 *
	 *	DCO/DCF/DCA/ADSR/NOISE/LFO/FX/OTHERS
	 *
	 * For extensibility there should be some nice way to configure the list
	 * of operators that will be placed in our palette.
	 */
	initPalette(audiomain, audiomain->palette);

	/*
	 * The effect chain is identical to the palette - they can be used for the
	 * same purpose. The difference is that when they are used as effects, they
	 * will appear on a different list in a bristolAudio structure.
	 */
	audiomain->effects = audiomain->palette;

	/*
	 * Assign an array of voice pointers. Need to call "initMidiVoices()"
	 */
	initMidiVoices(audiomain);
}

freeAudioMain(audioMain *audiomain)
{
	printf("freeAudioMain()\n");

	freePalette(audiomain, audiomain->palette);
//	freePalette(audiomain, audiomain->effects);

	if (audiomain->palette != (bristolOP **) NULL)
		bristolfree(audiomain->palette);
//	if (audiomain->effects != (bristolOP **) NULL)
//		bristolfree(audiomain->effects);
}

freeBristolAudio(audioMain *audiomain, Baudio *baudio)
{
	bristolVoice *voice = audiomain->playlist;

	if (baudio == (Baudio *) NULL)
		return;

	baudio->mixflags = BRISTOL_HOLDDOWN;

	if ((audiomain->debuglevel & BRISTOL_DEBUG_MASK) > BRISTOL_DEBUG1)
		printf("freeBristolAudio(%x, %x)\n", audiomain, baudio);

	freeSoundAlgo(baudio, 0, baudio->sound);
	bristolfree(baudio->sound);

	if (baudio->effect != NULL)
	{
		freeSoundAlgo(baudio, 0, baudio->effect);
		bristolfree(baudio->effect);
	}

	if (baudio->locals != NULL)
	{
		int i, j = 0;

		for (i = 0; i < audiomain->voiceCount; i++)
		{
			/*
			 * Free the inividual locals.
			 */
			for (j = 0; j < baudio->soundCount; j++)
				bristolfree(baudio->locals[i][j]);

			bristolfree(baudio->FXlocals[i][0]);
			/*
			 * Free the pointer for this voice.
			 */
			bristolfree(baudio->locals[i]);
			bristolfree(baudio->FXlocals[i]);
		}
	}

	bristolfree(baudio->leftbuf);
	bristolfree(baudio->rightbuf);

	/*
	 * Free the locals pointer itself.
	 */
	bristolfree(baudio->locals);
	bristolfree(baudio->FXlocals);

	if (baudio->next != NULL)
		baudio->next->last = baudio->last;

	if (baudio->last != NULL)
		baudio->last->next = baudio->next;
	else
		audiomain->audiolist = baudio->next;

	while (voice != NULL)
	{
		if ((voice->baudio != NULL)
			&& (voice->baudio->controlid == baudio->controlid))
			voice->baudio = NULL;
		voice = voice->next;
	}

	if (baudio->destroy != NULL)
		baudio->destroy(audiomain, baudio);

	if (baudio->mixlocals != NULL)
		bristolfree(baudio->mixlocals);

	bristolfree(baudio);
}

resetAudioThread(audioMain *audiomain)
{
	if ((audiomain->debuglevel & BRISTOL_DEBUG_MASK) > BRISTOL_DEBUG1)
		printf("resetAudioThread()\n");

	while (audiomain->audiolist != NULL)
		freeBristolAudio(audiomain, audiomain->audiolist);

	/*
	 * Free the array of voice pointers.
	 */
	freeMidiVoices(&audiomain, audiomain->playlist);
}

initPalette(audioMain *audiomain, bristolOP *palette[])
{
	int i;

	if ((audiomain->debuglevel & BRISTOL_DEBUG_MASK) > BRISTOL_DEBUG1)
		printf("initPalette()\n");

	/* 
	 * Go through the palette, and initialise all our operators.
	 */
	for (i = 0; i < PALETTE_SIZE; i++)
	{
		if (bristolPalette[i].initialise == NULL)
			break;

		palette[i] = bristolPalette[i].initialise
			(&palette[i], i, audiomain->samplerate,
				audiomain->samplecount);

		if ((audiomain->debuglevel & BRISTOL_DEBUG_MASK) > BRISTOL_DEBUG3)
			printf("Assigned operator %s to %i at %x\n",
				palette[i]->specs->opname, i, palette[i]);
		if ((audiomain->debuglevel & BRISTOL_DEBUG_MASK) > BRISTOL_DEBUG4)
			bristolOPprint(palette[i]);
	}
	audiomain->opCount = i - 1;
}

freePalette(audioMain *audiomain, bristolOP *palette[])
{
	int i;

	if ((audiomain->debuglevel & BRISTOL_DEBUG_MASK) > BRISTOL_DEBUG1)
		printf("freePalette()\n");

	/* 
	 * This needs to be sorted out when we move on to operators other than the
	 * DCO!
	 */
	for (i = 0; i < audiomain->opCount; i++)
	{
		if (palette[i] != (bristolOP *) NULL)
		{
			if ((audiomain->debuglevel & BRISTOL_DEBUG_MASK) > BRISTOL_DEBUG3)
				printf("Removed operator %s from %i at %x\n",
					palette[i]->specs->opname, i, palette[i]);

			palette[i]->destroy(palette[i]);

			palette[i] = (bristolOP *) NULL;
		}
	}
}

initMidiVoices(audioMain *audiomain)
{
	int i;
	bristolVoice *voice;

	if ((audiomain->debuglevel & BRISTOL_DEBUG_MASK) > BRISTOL_DEBUG1)
		printf("initMidiVoices(%x, %i)\n", audiomain, audiomain->voiceCount);

	audiomain->playlist = NULL;
	audiomain->playlast = NULL;

	/*
	 * Create the voice structures, put them on the playlist with some default
	 * flags, etc.
	 */
	for (i = 0; i < audiomain->voiceCount; i++)
	{
		if ((audiomain->debuglevel & BRISTOL_DEBUG_MASK) > BRISTOL_DEBUG1)
			printf("Adding new MIDI Voice: %i\n", i);

		voice = (bristolVoice *) bristolmalloc0(sizeof(bristolVoice));

		voice->flags = BRISTOL_KEYDONE;
		voice->last = NULL;
		voice->next = audiomain->playlist;

		/*
		 * Chain it into the voice list.
		 */
		voice->next = audiomain->playlist;
		voice->last = NULL;
		if (audiomain->playlist == NULL)
			audiomain->playlast = voice;
		else
			audiomain->playlist->last = voice;
		audiomain->playlist = voice;

		voice->index = i;
	}
}

freeMidiVoices(audioMain *audiomain, bristolVoice *voice)
{
//	if ((audiomain->debuglevel & BRISTOL_DEBUG_MASK) > BRISTOL_DEBUG1)
//		printf("freeMidiVoices(%x, %x)\n", audiomain, voice);

	if (voice == NULL)
		return;

	freeMidiVoices(audiomain, voice->next);

	audiomain->playlist = NULL;
	audiomain->playlast = NULL;
}

initMidiVoice(baudio)
{
}

/*
 * Change the frequencies of all active notes. Called when global tuning is
 * changed.
 */
doNoteChanges(bristolVoice *voice)
{
	if ((voice->flags & BRISTOL_KEYDONE) == 0)
		voice->cFreq = voice->dFreq = voice->baudio->ctab[voice->key.key];

//	if (voice->next != NULL)
//		doNoteChanges(voice->next);
}

