/***************************************************************************
*	Brutal Chess Alpha Build
*	- brutalchess.cpp
*
*	Authors: Maxwell Lazaroff, Michael Cook, and Joe Flint
*	Date Created : May 26th, 2005
*	Last Modified: December 19th, 2005
*
*	- description - Handles the main function calls for brutal chess.  Loads
*	the SDL window and controls the drawing of OpenGL objects.  Also handles
*	the creation of players and game flow control.
***************************************************************************/

// Necessary if we're in Windows so we don't get a million linker errors.
#ifdef WIN32
	#include <windows.h>
#endif

#include <iostream>
#include <math.h>

#include "SDL.h"
#include "board.h"
#include "glhead.h"
#include "objfile.h"
#include "fontloader.h"
#include "console.h"
#include "timer.h"
#include "timerchain.h"
#include "movelog.h"
#include "textureloader.h"

// Including Players
#include "chessplayer.h"
#include "alphabetaplayer.h"
#include "humanplayer.h"
#include "randomplayer.h"

using namespace std;

// Key array
bool keys[SDLK_LAST] = { false };

// Mouse button array
// Currently there are only 5 SDL constants, numbered 1-5
bool mouseclick[6] = {false};
bool mouseOut = false;
int mouseX, mouseY;
int mouseDeltaX = 0, mouseDeltaY = 0;

// Material settings and light positions
GLfloat light_position[]       = { 3.0,   3.0,  5.0,  0.0 };
const GLfloat light_ambient[]  = { 0.1f, 0.1f, 0.1f, 1.0f };
const GLfloat light_diffuse[]  = { 1.0f, 1.0f, 1.0f, 1.0f };
const GLfloat light_specular[] = { 1.0f, 1.0f, 1.0f, 1.0f };
const GLfloat specularmat[]    = { 1.0f, 1.0f, 1.0f, 1.0f };

// Storage for loaded pieces
ObjFile pawn, rook, knight, bishop, queen, king;

bool reflect = true;

void
Quit(int returnCode)
/***************************************************************************
*	Quit()
*
*	- Releases the resources and returns control to the desktop
***************************************************************************/
{
	FontLoader::unload();
	SDL_Quit();
	exit(returnCode);
}

bool
resizeWindow(int width, int height)
/***************************************************************************
*	resizeWindow()
*
*	- Resizes the game window to 'width' and 'height'.
***************************************************************************/
{
	// Protect against devide by zero
	if (height == 0) { height = 1; }
	
	GLfloat aspectRatio;
	aspectRatio = (float)width/(float)height;
	
	// Set up our viewport.
	glViewport(0, 0, width, height);
	
	// Change to the projection matrix and set our viewing volume.
	glMatrixMode(GL_PROJECTION);
   glLoadIdentity();
	
	// Sets our persepective.
	gluPerspective(30.0f, aspectRatio, 0.1f, 150.0f);
	//glOrtho( -36, 36, -27, 27, 0.1, 100 );
	
	// Change to the modelview matrix and set out viewing volume.
	glMatrixMode(GL_MODELVIEW);
	glLoadIdentity();
	
	return true;
}

void
buildModels()
{
	// Build display lists for all pieces
	pawn.build();
	rook.build();
	knight.build();
	bishop.build();
	queen.build();
	king.build();
}

void
unbuildModels()
{
	// Delete display lists for all pieces
	pawn.unbuild();
	rook.unbuild();
	knight.unbuild();
	bishop.unbuild();
	queen.unbuild();
	king.unbuild();
}

bool
loadModels()
/***************************************************************************
*	loadModels()
*
*	- Loads the models needed by the game, must be called after initGL
***************************************************************************/
{
	// Recenter pieces so that the center is at the base of the piece
	pawn.setRecenter( true, false, true );
	knight.setRecenter( true, false, true );
	rook.setRecenter( true, false, true );
	bishop.setRecenter( true, false, true );
	queen.setRecenter( true, false, true );
	king.setRecenter( true, false, true );

	// Load all of the piece models
	if( !pawn.load( "models/pawn.obj" ) )
		return false;
	if( !knight.load( "models/knight.obj" ) )
		return false;
	if( !rook.load( "models/rook.obj" ) )
		return false;
	if( !bishop.load( "models/bishop.obj" ) )
		return false;
	if( !queen.load( "models/queen.obj" ) )
		return false;
	if( !king.load( "models/king.obj" ) )
		return false;

	// Generate smooth normals for the models
	pawn.findNorms();
	rook.findNorms();
	knight.findNorms();
	bishop.findNorms();
	queen.findNorms();
	king.findNorms();

	// Make all the pieces have the same base size
	pawn.setScale( pawn.scale()*7 );
	rook.setScale( pawn.scale() );
	knight.setScale( pawn.scale() );
	bishop.setScale( pawn.scale() );
	queen.setScale( pawn.scale() );
	king.setScale( pawn.scale() );

	buildModels();
	
	return true;
}


bool
initGL()
/***************************************************************************
*	initGL()
*
*	- Initializes GL settings.
***************************************************************************/
{
	// Enable smooth shading. 
    glShadeModel(GL_SMOOTH);

    // Set the background color to black. 
    glClearColor(0.0f, 0.0f, 0.0f, 0.0f);

    // Depth buffer setup. 
    glClearDepth(1.0f);

    // Enables Depth Testing 
    glEnable(GL_DEPTH_TEST);

    // The Type Of Depth Test To Do 
    glDepthFunc(GL_LEQUAL);

    // Really Nice Perspective Calculations 
    glHint(GL_PERSPECTIVE_CORRECTION_HINT, GL_NICEST);

	glLightfv(GL_LIGHT0, GL_POSITION, light_position);
	glLightfv(GL_LIGHT0, GL_AMBIENT, light_ambient);
	glLightfv(GL_LIGHT0, GL_DIFFUSE, light_diffuse);
	glLightfv(GL_LIGHT0, GL_SPECULAR, light_specular);
	glEnable( GL_LIGHTING );
	glEnable( GL_LIGHT0 );
	glColorMaterial(GL_FRONT, GL_AMBIENT_AND_DIFFUSE);
	glEnable( GL_COLOR_MATERIAL );
	glMaterialfv(GL_FRONT, GL_SPECULAR, specularmat);
	glMaterialf(GL_FRONT, GL_SHININESS, 128);

	glEnable( GL_CULL_FACE );

	//glEnable( GL_NORMALIZE );

    return true;
}

BoardMove 
getMousePos(double & mx, double & my, double & mz)
{
	GLdouble mvmatrix[16];
	GLdouble pjmatrix[16];
	GLint viewport[4];

	// Get the current setup.
	glGetDoublev( GL_MODELVIEW_MATRIX, mvmatrix );
	glGetDoublev( GL_PROJECTION_MATRIX, pjmatrix );
	glGetIntegerv( GL_VIEWPORT, viewport );

	// Y Coordinate must be measured from the bottom of the window.
	int mouseycorr = viewport[3] - mouseY;

	GLfloat depth;
	GLdouble x, y, z;

	glReadPixels( mouseX, mouseycorr, 1, 1, GL_DEPTH_COMPONENT, GL_FLOAT, 
			&depth );
	gluUnProject( mouseX, mouseycorr, depth, mvmatrix, pjmatrix, viewport, 
			&x, &y, &z);
	
	mx = x;
	my = y;
	mz = z + 8;

	x += 0.5;
	z += 0.5;

	if( y < -0.05 )
		return BoardMove( 9, 9, 9, 9 );


	int boardy = (int)floor( x );
	int boardx = 8 + (int)floor( z );
	
	if( boardx >= 0 && boardx < 8 && boardy >= 0 && boardy < 8 )
		return BoardMove( boardx, boardy, 9, 9 );
		
	return BoardMove( 9, 9, 9, 9 );
}

int sign( int i )
{
	return ( i >= 0 ) ? 1 : -1; 
}

void 
setupMoveAnims( const BoardMove & move, const Board & board, TimerChain & chainx, TimerChain & chainy )
{
	int deltax, deltay;

	deltax = int( move.getEndx() ) - int( move.getStartx() );
	deltay = int( move.getEndy() ) - int( move.getStarty() );

	Timer temp, temp2( Timer::LOGARITHMIC );
	temp2.setDuration( 0.5 );

	chainx.clear();
	chainy.clear();

	// chainx moves things in the x direction, deltay is being used due to the stupid way that moves are done (x is the row!?)
	if( board._board[ move.getStartx() ][ move.getStarty() ].getUnit() != KNIGHT ) {
		// Always take the same ammount of time to move one square
		temp.setDuration( 0.25*sqrt(double(deltax*deltax + deltay*deltay)) );
		
		if( deltay == 0 ) {
			temp.setRange( 0, 0 );
			chainx.add( temp );
			temp2.setRange( 0, 0 );
			chainx.add( temp2 );
		} else {
			temp.setRange( 0, deltay - sign(deltay)*0.5 );
			chainx.add( temp );
			temp2.setRange( deltay - sign(deltay)*0.5, deltay );
			chainx.add( temp2 );
		}

		if( deltax == 0 ) {
			temp.setRange( 0, 0 );
			chainy.add( temp );
			temp2.setRange( 0, 0 );
			chainy.add( temp2 );
		} else {
			temp.setRange( 0, deltax - sign(deltax)*0.5 );
			chainy.add( temp );
			temp2.setRange( deltax - sign(deltax)*0.5, deltax );
			chainy.add( temp2 );
		}
	} else {
		temp.setDuration( 0.25*1.0 );
 
		// Always move along the short side first
		if( abs( deltax ) < abs( deltay ) ) {
			// Move along short side
			temp.setRange( 0, 0 );
			chainx.add( temp );

			temp.setRange( 0, deltax );
			chainy.add( temp );

			// Move along long side
			temp.setDuration( 0.25*1.5 );

			temp.setRange( deltax, deltax );
			chainy.add( temp );

			temp.setRange( 0, deltay - sign(deltay)*0.5 );
			chainx.add( temp );

			temp2.setRange( deltax, deltax );
			chainy.add( temp2 );

			temp2.setRange( deltay - sign(deltay)*0.5, deltay );
			chainx.add( temp2 );

		} else {
			// Move along short side
			temp.setRange( 0, 0 );
			chainy.add( temp );

			temp.setRange( 0, deltay );
			chainx.add( temp );

			// Move along long side
			temp.setDuration( 0.25*1.5 );

			temp.setRange( deltay, deltay );
			chainx.add( temp );

			temp.setRange( 0, deltax - sign(deltax)*0.5 );
			chainy.add( temp );

			temp2.setRange( deltay, deltay );
			chainx.add( temp2 );

			temp2.setRange( deltax - sign(deltax)*0.5, deltax );
			chainy.add( temp2 );
		}
	}
}

void
drawBoardEdge()
{
	glEnable( GL_LIGHTING );

	// Move back to the center of the board.
	glTranslated( 3.5, 0, -4.5 );
	
	// Select the edgestyle to be used.
	int edgestyle = 2;

	if (edgestyle == 1)
	{
		glBegin( GL_QUADS );
		
		glColor3d( 0.3, 0.3, 0.3 );

		glNormal3d( 0, 1.87083, 1.87083);
		glVertex3d( -4.075, -0.01,  4.075 );
		glVertex3d(  4.075, -0.01,  4.075 );
		glVertex3d(  4.0, 0,  4.0 );
		glVertex3d( -4.0, 0,  4.0 );

		glNormal3d( 0, 0, 7);
		glVertex3d( -4.075, -0.15, 4.075 );
		glVertex3d(  4.075, -0.15, 4.075 );
		glVertex3d(  4.075, -0.01, 4.075 );
		glVertex3d( -4.075, -0.01, 4.075 );

		glNormal3d( 1.87083, 1.87083, 0);
		glVertex3d(  4.075, -0.01,  4.075 );
		glVertex3d(  4.075, -0.01, -4.075 );
		glVertex3d(  4.0, 0, -4.0 );
		glVertex3d(  4.0, 0,  4.0 );

		glNormal3d( 7, 0, 0 );
		glVertex3d(  4.075, -0.15,  4.075 );
		glVertex3d(  4.075, -0.15, -4.075 );
		glVertex3d(  4.075, -0.01, -4.075 );
		glVertex3d(  4.075, -0.01,  4.075 );

		glNormal3d( 0, 1.87083, -1.87083);
		glVertex3d(  4.075, -0.01, -4.075 );
		glVertex3d( -4.075, -0.01, -4.075 );
		glVertex3d( -4.0, 0, -4.0 );
		glVertex3d(  4.0, 0, -4.0 );

		glNormal3d( 0, 0, -7 );
		glVertex3d(  4.075, -0.15, -4.075 );
		glVertex3d( -4.075, -0.15, -4.075 );
		glVertex3d( -4.075, -0.01, -4.075 );
		glVertex3d(  4.075, -0.01, -4.075 );

		glNormal3d( -1.87083, 1.87083, 0);
		glVertex3d( -4.075, -0.01, -4.075 );
		glVertex3d( -4.075, -0.01,  4.075 );
		glVertex3d( -4.0, 0,  4.0 );
		glVertex3d( -4.0, 0, -4.0 );

		glNormal3d( -7, 0, 0 );
		glVertex3d( -4.075, -0.15, -4.075 );
		glVertex3d( -4.075, -0.15,  4.075 );
		glVertex3d( -4.075, -0.01,  4.075 );
		glVertex3d( -4.075, -0.01, -4.075 );

		glEnd();
	}
	else if (edgestyle == 2)
	{
		glBegin( GL_QUADS );
		
		// Darker of the edge colors.
		glColor4d( 0.2, 0.2, 0.2, 0.95 );

		glNormal3d( 0, 7, 0);
		
		// White side.
		glVertex3d( -4.1, 0,  4.1 );
		glVertex3d(  4.1, 0,  4.1 );
		glVertex3d(  4.0, 0,  4.0 );
		glVertex3d( -4.0, 0,  4.0 );

		// Right of white side.
		glVertex3d( -4.1, 0, -4.1 );
		glVertex3d( -4.1, 0,  4.1 );
		glVertex3d( -4.0, 0,  4.0 );
		glVertex3d( -4.0, 0, -4.0 );

		// Black side.
		glVertex3d(  4.1, 0, -4.1 );
		glVertex3d( -4.1, 0, -4.1 );
		glVertex3d( -4.0, 0, -4.0 );
		glVertex3d(  4.0, 0, -4.0 );

		// Left of white side.
		glVertex3d(  4.1, 0,  4.1 );
		glVertex3d(  4.1, 0, -4.1 );
		glVertex3d(  4.0, 0, -4.0 );
		glVertex3d(  4.0, 0,  4.0 );

		glColor3d( 0.2, 0.2, 0.2 );

		// Drawing sides of the darker portion.
		glNormal3d( 0, 0, 7);
		glVertex3d( -4.1, -0.075, 4.1 );
		glVertex3d(  4.1, -0.075, 4.1 );
		glVertex3d(  4.1, 0, 4.1 );
		glVertex3d( -4.1, 0, 4.1 );

		glNormal3d( 7, 0, 0 );
		glVertex3d(  4.1, -0.075,  4.1 );
		glVertex3d(  4.1, -0.075, -4.1 );
		glVertex3d(  4.1, 0, -4.1 );
		glVertex3d(  4.1, 0,  4.1 );
		
		glNormal3d( 0, 0, -7 );
		glVertex3d(  4.1, -0.075, -4.1 );
		glVertex3d( -4.1, -0.075, -4.1 );
		glVertex3d( -4.1, 0, -4.1 );
		glVertex3d(  4.1, 0, -4.1 );

		glNormal3d( -7, 0, 0 );
		glVertex3d( -4.1, -0.075, -4.1 );
		glVertex3d( -4.1, -0.075,  4.1 );
		glVertex3d( -4.1, 0,  4.1 );
		glVertex3d( -4.1, 0, -4.1 );

		// Now draw the outer border in light gray.
		glColor4d( 0.45, 0.45, 0.45, 0.95 );

		glNormal3d( 0, 7, 0 );

		// White side.
		glVertex3d( -4.2, -0.075,  4.2 );
		glVertex3d(  4.2, -0.075,  4.2 );
		glVertex3d(  4.1, -0.075,  4.1 );
		glVertex3d( -4.1, -0.075,  4.1 );

		// Right of white side.
		glVertex3d( -4.2, -0.075, -4.2 );
		glVertex3d( -4.2, -0.075,  4.2 );
		glVertex3d( -4.1, -0.075,  4.1 );
		glVertex3d( -4.1, -0.075, -4.1 );

		// Black side.
		glVertex3d(  4.2, -0.075, -4.2 );
		glVertex3d( -4.2, -0.075, -4.2 );
		glVertex3d( -4.1, -0.075, -4.1 );
		glVertex3d(  4.1, -0.075, -4.1 );

		// Left of white side.
		glVertex3d(  4.2, -0.075,  4.2 );
		glVertex3d(  4.2, -0.075, -4.2 );
		glVertex3d(  4.1, -0.075, -4.1 );
		glVertex3d(  4.1, -0.075,  4.1 );

		glColor3d( 0.45, 0.45, 0.45 );

        // Drawing sides of the darker portion.
		glNormal3d( 0, 0, 7);
		glVertex3d( -4.2, -0.15, 4.2 );
		glVertex3d(  4.2, -0.15, 4.2 );
		glVertex3d(  4.2, -0.075, 4.2 );
		glVertex3d( -4.2, -0.075, 4.2 );

		glNormal3d( 7, 0, 0 );
		glVertex3d(  4.2, -0.15,  4.2 );
		glVertex3d(  4.2, -0.15, -4.2 );
		glVertex3d(  4.2, -0.075, -4.2 );
		glVertex3d(  4.2, -0.075,  4.2 );
		
		glNormal3d( 0, 0, -7 );
		glVertex3d(  4.2, -0.15, -4.2 );
		glVertex3d( -4.2, -0.15, -4.2 );
		glVertex3d( -4.2, -0.075, -4.2 );
		glVertex3d(  4.2, -0.075, -4.2 );

		glNormal3d( -7, 0, 0 );
		glVertex3d( -4.2, -0.15, -4.2);
		glVertex3d( -4.2, -0.15,  4.2 );
		glVertex3d( -4.2, -0.075,  4.2 );
		glVertex3d( -4.2, -0.075, -4.2 );

		glEnd();
	}

	return;
}

void 
drawSquares( uint mousex, uint mousey, bool allowselect )
{
	static double alpha[8][8] = { 0 };
	static Timer alphatimers[8][8];
	
	
	//bool selected;
	bool alternate = true;
	
	glEnable( GL_TEXTURE_2D );

	/*if( !TextureLoader::setTexture( "marblewhite" ) )
		cerr << "Failed to find texture" << endl;*/

	glPushMatrix();

	for( int i = 0; i < 8; i++ ) {
		for( int j = 0; j < 8; j++ ) {
			
			bool selected = i == mousex && j == mousey;

			if( alphatimers[i][j].started() ) {
				alphatimers[i][j]++;
				alpha[i][j] += alphatimers[i][j].change();
			}

			// !noselection && activePlayer->requiresInput()
			if( selected && allowselect ) {
				alpha[i][j] = 0.7;
				alphatimers[i][j].stop();
			}
			if( alpha[i][j] > 0 ) {
				alphatimers[i][j].setDuration( 0.25 );
				alphatimers[i][j].setRange( alpha[i][j], 0 );
				alphatimers[i][j].start();
			} else {
				alpha[i][j] = 0;
			}

			if( alternate )
				TextureLoader::setTexture( "marblewhite" );
			else
				TextureLoader::setTexture( "marbleblack" );
			//glColor4d( color[i][j], color[i][j], color[i][j], 0.7 );
			glColor4d( 1.0, 1.0, 1.0, 0.7 );
			alternate = !alternate;

			glBegin( GL_QUADS );
			glNormal3d( 0, 7, 0 );
			//glTexCoord2d( 0, 1 );
			glTexCoord2d( j*1/8.0, (i+1)*1/8.0 );
			glVertex3d( -0.5, 0,  0.5 );
			//glTexCoord2d( 1, 1 );
			glTexCoord2d( (j+1)*1/8.0, (i+1)*1/8.0 );
			glVertex3d(  0.5, 0,  0.5 );
			//glTexCoord2d( 1, 0 );
			glTexCoord2d( (j+1)*1/8.0, i*1/8.0 );
			glVertex3d(  0.5, 0, -0.5 );
			//glTexCoord2d( 0, 0 );
			glTexCoord2d( j*1/8.0, i*1/8.0 );
			glVertex3d( -0.5, 0, -0.5 );
			glEnd();

			if( alpha[i][j] != 0 ) {
				glDisable( GL_TEXTURE_2D );
				glDisable( GL_DEPTH_TEST );
				glBegin( GL_QUADS );
				glColor4d( 0.5, 0.5, 0.5, alpha[i][j] );
				glNormal3d( 0, 7, 0 );
				glVertex3d( -0.5, 0,  0.5 );
				glVertex3d(  0.5, 0,  0.5 );
				glVertex3d(  0.5, 0, -0.5 );
				glVertex3d( -0.5, 0, -0.5 );
				glEnd();
				glEnable( GL_DEPTH_TEST );
				glEnable( GL_TEXTURE_2D );
			}



			glTranslated( 1, 0, 0 );
		}
		glTranslated( -8, 0, 1 );
		alternate = !alternate;
	}

	glDisable( GL_TEXTURE_2D );

	drawBoardEdge();

	glPopMatrix();	
}

void 
drawPiece( Unit whichpiece, bool reverse )
{
	if( reverse )
		glRotated( 180, 0, 1, 0 );
	glScaled( 1/7.0, 1/7.0, 1/7.0 );
	switch( whichpiece ) {
		case PAWN:
			pawn.draw();
			break;
		case KNIGHT:
			glRotated( 90, 0, 1, 0 );
			knight.draw();
			glRotated( -90, 0, 1, 0 );
			break;
		case BISHOP:
			glRotated( 180, 0, 1, 0 );
			bishop.draw();
			glRotated( 180, 0, -1, 0 );
			break;
		case ROOK:
			rook.draw();
			break;
		case QUEEN:
			queen.draw();
			break;
		case KING:
			/*if( board.inDanger( board._board[ i ][ j ].getColor() == WHITE, i, j ) )
				glColor3d( 0.5, 0, 0 );*/
			king.draw();
		default:
			break;
	}
	glScaled( 7, 7, 7 );
	if( reverse )
		glRotated( 180, 0, -1, 0 );	
}			
			
double 
dist( double x1, double y1, double x2, double y2 )
{
		return sqrt( (x1 - x2)*(x1 - x2) + (y1 - y2)*(y1 - y2) );
}

void 
drawBoard( const Board & board, bool white, uint mousei, uint mousej, 
		   bool allowselect, bool firstclick, uint movei, uint movej, 
		   const TimerChain & moveanimx, const TimerChain & moveanimy,
		   double mousex, double mousez )
{
	glPushMatrix();
	static double height[8][8] = { 0 };
	static Timer timers[8][8];
	double alpha = 1.0;
	
	if( !firstclick ) {
		if( board._board[ movei ][ movej ].getColor() == WHITE )
			glColor3d( 0.7, 0.7, 0.7 );
		else
			glColor3d( 0.2, 0.2, 0.2 );
		if( board._board[ movei ][ movej ].getUnit() == KING &&
			board.inDanger( board._board[ movei ][ movej ].getColor() == WHITE, movei, movej ) )
					glColor3d( 0.5, 0, 0 );
		glTranslated( mousex, height[movei][movej], mousez );
		drawPiece( board._board[ movei ][ movej ].getUnit(), board._board[ movei ][ movej ].getColor() == WHITE );
		glTranslated( -mousex, -height[movei][movej], -mousez);
	}
	
	for( int i = 0; i < 8; i++ ) {
		for( int j = 0; j < 8; j++ ) {


			glTranslated( 0, height[i][j], 0 );
				
			if( movei == i && movej == j 
				&& ( moveanimx.started() || moveanimy.started() ) ) {
					glTranslated( moveanimx.value(), 0, moveanimy.value() );
			}

			alpha = 1;
			
			if( !firstclick && mousei == i && mousej == j)
				alpha = 0.2 + dist( mousex, mousez, j, i )/0.5*0.8;

			// Sets the piece render color.
			if( board._board[ i ][ j ].getColor() == WHITE )
				glColor4d( 0.7, 0.7, 0.7, alpha );
			else
				glColor4d( 0.2, 0.2, 0.2, alpha );
			
			// If the king is in check, sets the piece render color to red. 
			if( board._board[ i ][ j ].getUnit() == KING &&
				board.inDanger( board._board[ i ][ j ].getColor() == WHITE, i, j ) )
						glColor4d( 0.5, 0, 0, alpha );
			// If a piece has already been selected, translate is drawn location to match the cursors
			// position.  
			if( !firstclick && i == movei && j == movej ) {
				glColor4d( 0.5, 0.5, 0.5, 0.5 );
				glEnable( GL_BLEND );
				drawPiece( board._board[ i ][ j ].getUnit(), board._board[ i ][ j ].getColor() == WHITE );
				glDisable( GL_BLEND );
			} else {			
				if( alpha != 1 )
					glEnable( GL_BLEND );
				drawPiece( board._board[ i ][ j ].getUnit(), board._board[ i ][ j ].getColor() == WHITE );
				if( alpha != 1 )
					glDisable( GL_BLEND );
			}
			

			if( movei == i && movej == j 
				&& ( moveanimx.started() || moveanimy.started() ) ) {
					glTranslated( -moveanimx.value(), 0, -moveanimy.value() );
			}

			glTranslated( 1, -height[i][j], 0 );
			
			if( timers[i][j].started() ) {
				timers[i][j]++;
				height[i][j] += timers[i][j].change();
			} 
			if ( timers[i][j].done() ) {
				height[i][j] = timers[i][j].end();
				timers[i][j].resetDone();
			}
			
			static bool bounceup = true;
			if( board._board[i][j].getUnit() != EMPTY ) {
				if( !firstclick && i == movei && j == movej ) {
					/*if( bounceup ) {
						if( height[i][j] < 0.4  && !timers[i][j].started() ) {
							timers[i][j].setRange( 0.2, 0.4 );
							timers[i][j].setDuration( 0.4 );
							timers[i][j].setProgress( height[i][j] );
							timers[i][j].start();
						}
						if( height[i][j] >= 0.4 ) {
							height[i][j] = 0.4;
							bounceup = false;
						}
					} else {
						if( height[i][j] > 0.2 && !timers[i][j].started() ) {
							timers[i][j].setRange( 0.4, 0.2 );
							timers[i][j].setDuration( 0.4 );
							timers[i][j].setProgress( height[i][j] );
							timers[i][j].start();
						}
						if( height[i][j] <= 0.2 ) {
							height[i][j] = 0.2;
							bounceup = true;
						}
					}*/
				} else if( firstclick && i == mousei && j == mousej &&
						(board._board[i][j].getColor() == WHITE) == white && 
						allowselect ) {
					if( height[i][j] > 0.2 && !timers[i][j].started() ) {
						timers[i][j].setRange( 0.4, 0.2 );
						timers[i][j].setDuration( 0.25 );
						timers[i][j].setProgress( height[i][j] );
						timers[i][j].start();
					}
					if( height[i][j] < 0.2 ) {
						timers[i][j].stop();
						height[i][j] = 0.2;
					}
				} else { 
					if( height[i][j] < 0.01 )
						height[i][j] = 0.01;
					if( height[i][j] > 0.01 && !timers[i][j].started()) {
						timers[i][j].setType( Timer::LOGARITHMIC );
						timers[i][j].setRange( 0.4, 0.01 );
						timers[i][j].setDuration( 1.0 );
						timers[i][j].setProgress( height[i][j] );
						timers[i][j].start();
					}
				}
			} else {
				timers[i][j].stop();
				height[i][j] = 0.01;
			}
		}
		glTranslated( -8, 0, 1 );
	}
	glPopMatrix();
}

bool
drawGLScene()
/***************************************************************************
*	drawGLScene()
*
*	- Handles the drawing of a frame.
***************************************************************************/
{
	static bool firstclick = true;
	static bool white = true;
	static bool showwhite = true;
	static BoardMove move;
	static BoardMove mouse( 9, 9, 9, 9 );

	// Creating our players.
	static ChessPlayer *playOne = new HumanPlayer;
	static ChessPlayer *playTwo = new AlphaBetaPlayer;

	// Setting player 1 to the active player.
	static ChessPlayer *activePlayer = playOne;
	static ChessPlayer *inactivePlayer = playTwo;

	static double tablerot = 0;
	static MoveLog movelog;
	static double mousex, mousey, mousez;
	string filename = "log.txt";
	static bool oneframe = false;
	
	// These are to calculate our fps. 
    static int T0     = 0;
    static int Frames = 0;
	
	// Clear The Screen And The Depth Buffer.
    glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT | GL_STENCIL_BUFFER_BIT);

	glLoadIdentity();

	glTranslated( 0, 0, -106 );

	glRotated( 42, 1, 0, 0 );

	static Board test;
	static BoardMove animmove;
	static TimerChain moveanimx, moveanimy;

	// Update the board when the move animations are finished
	// Don't reset done yet, do it at the end of the frame otherwise selections can get messed up
	if( moveanimx.done() && moveanimy.done() ) {
		inactivePlayer->opponent_move( animmove, !white );
		test.updateBoard( animmove );
		white = !white;
		movelog.write(filename);

		if (inactivePlayer->inCheckmate(test, white))
		{
			Console::out << inactivePlayer->getName() << " in Checkmate!\n";
			Console::out << activePlayer->getName() << " wins!\n";
			test.reset();
			if (white)
			{
				ChessPlayer *temp;
				temp = activePlayer;
				activePlayer = inactivePlayer;
				inactivePlayer = temp;
			}
			white = true;
			light_position[0] = -light_position[0];
			light_position[2] = -light_position[2];
			showwhite = !showwhite;
			movelog.write(filename);
		}
		else if(inactivePlayer->inStalemate(test, white))
		{
			Console::out << "The game ended in a stalemate caused";
			Console::out << " by " << inactivePlayer->getName() << "!\n";
			test.reset();
			if (white)
			{
				ChessPlayer *temp;
				temp = activePlayer;
				activePlayer = inactivePlayer;
				inactivePlayer = temp;
			}
			white = true;
			showwhite = !showwhite;
			light_position[0] = -light_position[0];
			light_position[2] = -light_position[2];
			movelog.write(filename);
		}

		// Swap the players
		ChessPlayer *temp;
		temp = activePlayer;
		activePlayer = inactivePlayer;
		inactivePlayer = temp;
		oneframe = false;
		moveanimx.resetDone();
		moveanimy.resetDone();		
		firstclick = true;
	}

	if( showwhite ) {
		if( tablerot != 0 && tablerot < 360 )
			tablerot += 10;
		if( tablerot > 360 )
			tablerot = 0;
		tablerot = 0;
	} else {
		if( tablerot < 180 )
			tablerot += 10;
		if( tablerot > 180 )
			tablerot = 0;
		tablerot = 180;
	}
	glRotated( tablerot, 0, 1, 0 );

	glScaled( 7, 7, 7 );

	glTranslated( -3.5, 0, -3.5 );

	bool alternate = true;


	static int movei, movej;
	
	// This plane is drawn and then erased. The purpose is to obtain the mouse
	// coordinates, and set the stencil buffer for the reflections
	// It is exactly the same size as the board, and is in exactly the same spot
	glDisable( GL_DEPTH_TEST );	
	if( reflect ) {
		glEnable( GL_STENCIL_TEST );
		glStencilOp( GL_REPLACE, GL_REPLACE, GL_REPLACE );
		glStencilFunc( GL_ALWAYS, 1, 0xffffffff );
	}
	
	glPushMatrix();
	glTranslated( 3.5, 0, 3.5 );
	
	glBegin( GL_QUADS );
	glNormal3d( 0, 7, 0 );
	glVertex3d( -(8*0.5+0.1), 0,  (8*0.5+0.1) );
	glVertex3d(  (8*0.5+0.1), 0,  (8*0.5+0.1) );
	glVertex3d(  (8*0.5+0.1), 0, -(8*0.5+0.1) );
	glVertex3d( -(8*0.5+0.1), 0, -(8*0.5+0.1) );

	// White side.
	glVertex3d( -4.2, -0.075,  4.2 );
	glVertex3d(  4.2, -0.075,  4.2 );
	glVertex3d(  4.1, -0.075,  4.1 );
	glVertex3d( -4.1, -0.075,  4.1 );

	// Right of white side.
	glVertex3d( -4.2, -0.075, -4.2 );
	glVertex3d( -4.2, -0.075,  4.2 );
	glVertex3d( -4.1, -0.075,  4.1 );
	glVertex3d( -4.1, -0.075, -4.1 );

	// Black side.
	glVertex3d(  4.2, -0.075, -4.2 );
	glVertex3d( -4.2, -0.075, -4.2 );
	glVertex3d( -4.1, -0.075, -4.1 );
	glVertex3d(  4.1, -0.075, -4.1 );

	// Left of white side.
	glVertex3d(  4.2, -0.075,  4.2 );
	glVertex3d(  4.2, -0.075, -4.2 );
	glVertex3d(  4.1, -0.075, -4.1 );
	glVertex3d(  4.1, -0.075,  4.1 );

	glEnd();
	
	// Draw the plane through the board so the mouse moves right off the board
	// Will need to be tweaked a bit if the viewport changes, turn color writing
	// on to see the plane, make sure it covers the whole viewport
	glEnable( GL_DEPTH_TEST );	
	if(reflect) {
		glDisable( GL_STENCIL_TEST );
	}
	glBegin( GL_QUADS );
	glNormal3d( 0, 7, 0 );
	glVertex3d( -30*0.5, 0,  9*0.5 );
	glVertex3d(  30*0.5, 0,  9*0.5 );
	glVertex3d(  30*0.5, 0, -34*0.5 );
	glVertex3d( -30*0.5, 0, -34*0.5 );
	glEnd();

	// Get the mouse position
	glTranslated( -3.5, 0, 4.5 );
	mouse = getMousePos( mousex, mousey, mousez );
	glPopMatrix();
	
	// OK to draw to the color buffer again
	glColorMask( GL_TRUE, GL_TRUE, GL_TRUE, GL_TRUE );
	
    glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);

	if( reflect ) {
		// Only draw if the stencil buffer has a 1 here
		glEnable( GL_STENCIL_TEST );
		glStencilFunc( GL_EQUAL, 1, 0xffffffff );
		glStencilOp( GL_KEEP, GL_KEEP, GL_KEEP );
		// Invert over the board plane
		glScalef( 1, -1, 1 );
		glCullFace(GL_FRONT);
	
		// The lighting is actually wrong here, need to fix it
		glLightfv(GL_LIGHT0, GL_POSITION, light_position);
	
		// Draw the reflected pieces
		drawBoard( test, white, mouse.getStartx(), mouse.getStarty(),
							activePlayer->requiresInput(), firstclick, movei,
							movej, moveanimx, moveanimy, mousex, mousez );	
	
		// Return to normal (not inverted)
		glCullFace(GL_BACK);
		glScalef( 1, -1, 1 );
		glLightfv(GL_LIGHT0, GL_POSITION, light_position);
	
		// Draw the board squares
		glDisable( GL_STENCIL_TEST );
	}
	glEnable( GL_BLEND );
	//glDisable( GL_LIGHTING );
	glBlendFunc( GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA );
		
	drawSquares( mouse.getStartx(), mouse.getStarty(), 
					activePlayer->requiresInput() );
	//glEnable( GL_LIGHTING );
	glDisable( GL_BLEND );
	
	// Draw the "real" pieces
	drawBoard( test, white, mouse.getStartx(), mouse.getStarty(),
						activePlayer->requiresInput(), firstclick, movei, movej,
						moveanimx, moveanimy, mousex, mousez );		


	// Prevent glitch
	if( moveanimx.started() )
		moveanimx++;
	if( moveanimy.started() )
		moveanimy++;

	if( mouseclick[SDL_BUTTON_RIGHT] ) {
		firstclick = true;
		SDL_ShowCursor(SDL_ENABLE);
		mouseclick[SDL_BUTTON_RIGHT] = false;
	}

	if( mouseclick[SDL_BUTTON_LEFT] ) 
	{
		if( activePlayer->requiresInput() ) {
			if( firstclick ) 
			{
				Piece * undermouse = &test._board[ mouse.getStartx() ]
					[ mouse.getStarty() ];
				if( undermouse->getUnit() != EMPTY 
					&& ( (undermouse->getColor() == WHITE) == white ) ) 
				{
					move.setStart( mouse.getStartx(), mouse.getStarty() );
					movei = move.getStartx();
					movej = move.getStarty();
					SDL_ShowCursor(SDL_DISABLE);
					firstclick = false;
				}
				mouseclick[SDL_BUTTON_LEFT] = false;
			} 
			else 
			{
				move.setEnd( mouse.getStartx(), mouse.getStarty() );
				if( move.getStartx() != move.getEndx() 
					|| move.getStarty() != move.getEndy() ) 
				{
					if( activePlayer->check_move(move, test, white)) 
					{
						Console::out << (white ? "White" : "Black");
						Console::out << " moves " << test._board[ move.getStartx() ][ move.getStarty() ].getName() << " from " << move << endl;

						animmove = move;
						//test.updateBoard(move);

						//setupMoveAnims( move, test, moveanimx, moveanimy );
						moveanimx.clear();
						moveanimy.clear();

						cout << (white ? "White" : "Black");
						cout << " moves " << test._board[ move.getStartx() ][ move.getStarty() ].getName() << " from " << move << endl;
						moveanimx.start();
						moveanimy.start();
						movelog.push(test, move);

						// Move valid, allow another
						firstclick = true;
						SDL_ShowCursor(SDL_ENABLE);
					}
				} else {
					// Clicked on same piece, deselect
					firstclick = true;
					SDL_ShowCursor(SDL_ENABLE);
				}
				mouseclick[SDL_BUTTON_LEFT] = false;
			}
		} else {
			mouseclick[SDL_BUTTON_LEFT] = false;
		}
	}

	if( !activePlayer->requiresInput() && oneframe && !moveanimx.started() && !moveanimy.started() && !moveanimx.done() && !moveanimy.done() ) {
		animmove = activePlayer->decide_move( test, white );

		Console::out << (white ? "White" : "Black");
		Console::out << " moves " << test._board[ animmove.getStartx() ][ animmove.getStarty() ].getName() << " from " << animmove << endl;


		setupMoveAnims( animmove, test, moveanimx, moveanimy );

		movei = animmove.getStartx();
		movej = animmove.getStarty();
		cout << (white ? "White" : "Black");
		cout << " moves " << test._board[ animmove.getStartx() ][ animmove.getStarty() ].getName() << " from " << animmove << endl;
		moveanimx.start();
		moveanimy.start();
		movelog.push(test, animmove);
	}
   
   if( keys[SDLK_BACKQUOTE] ) {
		Console::toggle();
		keys[SDLK_BACKQUOTE] = false;
	}

   // Handle console commands
   Console::getKeyboardInput(keys);
	string cmd = Console::fetchCommand();
   if (cmd != "") {
      if (cmd == "reflections 1") {
         Console::out << "console: reflections enabled" << endl;
         reflect = true;
      } else if (cmd == "reflections 0") {
         Console::out << "console: reflections disabled" << endl;
         reflect = false;
      } else {
         Console::out << "console: bad command: " << cmd;
      }
   }
   
	Console::draw();

	#ifdef WIN32
	Sleep(1);
	#else
	usleep(1);
	#endif
	
	// Draw it to the screen. 
    SDL_GL_SwapBuffers();

    // Gather our frames per second. 
    Frames++;
    int t = SDL_GetTicks();
	if (t - T0 >= 5000) 
	{
		float sec = (float)((t-T0)/1000.0);
	    float fps = Frames / sec;
		cout << Frames << " frames in "<< sec << " seconds = "<< fps << " FPS\n";
		//Console::out << Frames << " frames in "<< sec << " seconds = "<< fps << " FPS\n";
	    T0 = t;
	    Frames = 0;
	}
    
	oneframe = true;
	
	return true;
}

int 
main(int argc, char *argv[])
/***************************************************************************
*	MAIN
***************************************************************************/
{
   // Screen Settings
   const int   WINDOW_WIDTH  = 800;
   const int   WINDOW_HEIGHT = 600;
   const int   WINDOW_BPP    = 16;
   
   // Declares our SDL surface
   SDL_Surface *surface;
   
   bool mouseb[6] = {false};
   
	vector<string> args;
	for( int i = 0; i < argc; i++ )
		args.push_back( argv[i] );

	// TODO Better commandline argument system
	for( int i = 1; i < argc; i++ )
		if( args[i] == "-r" )
			reflect = false;
	
    bool done = false;
    
	// Flags to pass to SDL_SetVideoMode.
    int videoFlags;
	
	// Used to collect events. 
    SDL_Event event;
    
	// This holds some info about our display.
    const SDL_VideoInfo *videoInfo;
    
	// Whether or not the window is active.
    bool isActive = true;

    // Initialize SDL.
    if (SDL_Init(SDL_INIT_VIDEO) < 0)
	{
	    cerr << "Video initialization faild: " << SDL_GetError() << endl;
	    Quit(1);
	}

    // Fetch the video info.
    videoInfo = SDL_GetVideoInfo();

    if (!videoInfo)
	{
		cerr << "Video query faild: " << SDL_GetError() << endl;
		Quit(1);
	}

    // The flags to pass to SDL_SetVideoMode.
    videoFlags  = SDL_OPENGL;          // Enable OpenGL in SDL.
    videoFlags |= SDL_GL_DOUBLEBUFFER; // Enable double buffering.
    videoFlags |= SDL_HWPALETTE;       // Store the palette in hardware.
    videoFlags |= SDL_RESIZABLE;       // Enable window resizing.

    // This checks to see if surfaces can be stored in memory.
    if (videoInfo->hw_available)
		videoFlags |= SDL_HWSURFACE;
    else
		videoFlags |= SDL_SWSURFACE;

    // This checks if hardware blits can be done.
    if (videoInfo->blit_hw)
		videoFlags |= SDL_HWACCEL;

    // Sets up OpenGL double buffering.
    SDL_GL_SetAttribute(SDL_GL_DOUBLEBUFFER, 1);
	
	// Need some a bit for the stencil buffer
	SDL_GL_SetAttribute( SDL_GL_STENCIL_SIZE, 1 );

	// Sets the string displayed as the title of the window.
	SDL_WM_SetCaption("Brutal Chess Alpha 0.3", NULL);

    // Get a SDL surface.
    surface = SDL_SetVideoMode(WINDOW_WIDTH,WINDOW_HEIGHT,WINDOW_BPP,videoFlags);

    // Verify there is a surface.
    if (!surface)
	{
		cerr << "Video mode set failed: " << SDL_GetError() << endl;
		Quit( 1 );
	}

    // Initialize OpenGL.
    initGL();


	if( !FontLoader::loadFont("sans", "VeraMono.ttf", 32) )
	{
		cerr << "Failed to load fonts" << endl;
	}

	if( !loadModels() )
	{
		cerr << "Failed to load models" << endl;
		Quit( 1 );
	}

	if( !TextureLoader::loadTexture( "marblehugewhite.png", "marblewhite" ) ||
		!TextureLoader::loadTexture( "marblehugeblack.png", "marbleblack" ) )
	{
		cerr << "Failed to load texture" << endl;
	}

    // Resize the initial window.
    resizeWindow(WINDOW_WIDTH, WINDOW_HEIGHT);

    // Wait for events.
    while (!done)
	{
	    // Handle the events in the queue.

	    while (SDL_PollEvent(&event))
		{
		    switch(event.type)
			{
			case SDL_MOUSEBUTTONDOWN:
				// Mouse button pressed
				mouseX = event.button.x;
				mouseY = event.button.y;
				mouseb[event.button.button] = true;
				mouseclick[event.button.button] = true;
				break;
			case SDL_MOUSEBUTTONUP:
				// Mouse button released
				mouseX = event.button.x;
				mouseY = event.button.y;
				mouseb[event.button.button] = false;
				break;
			case SDL_MOUSEMOTION:
				mouseX = event.motion.x;
				mouseY = event.motion.y;
				if(event.motion.state)
				{
					mouseDeltaX += event.motion.xrel;
					mouseDeltaY += event.motion.yrel;
				}
				break;
			case SDL_ACTIVEEVENT:
			    // Something's happened with our focus. If we lost focus or we 
			    // are iconified, we shouldn't draw the screen... Only stop 
			    // drawing if the app is actually minimized or hidden
			    if(   event.active.gain == 0 
				   && event.active.state != SDL_APPMOUSEFOCUS)
					isActive = false;
			    else
					isActive = true;

				// Track when the mouse enters and leaves the window
				if(   event.active.gain == 0 
				   && event.active.state == SDL_APPMOUSEFOCUS)
					mouseOut = true;
				else if(   event.active.gain == 1
						&& event.active.state == SDL_APPMOUSEFOCUS)
					mouseOut = false;
			    break;			    
			case SDL_VIDEORESIZE:
				// Delete display lists first
				unbuildModels();
				FontLoader::unloadGL();
				TextureLoader::unloadGL();

			    // Handle resize event.
			    surface = SDL_SetVideoMode(event.resize.w,event.resize.h,
							               16, videoFlags);
			    if (!surface)
				{
					cerr << "Could not get a surface after resize: " << SDL_GetError() << endl;
				    Quit( 1 );
				}
			    resizeWindow(event.resize.w, event.resize.h);
				initGL();
				// Windows driver bug, pieces disappear on resize

				buildModels();
				FontLoader::reload();
				TextureLoader::reload();

			    break;
			case SDL_KEYDOWN:
				keys[ event.key.keysym.sym ] = true;
				break;
			case SDL_KEYUP:
				keys[ event.key.keysym.sym ] = false;
				break;
			case SDL_QUIT:
			    // Handle quit requests.
			    done = true;
			    break;
			default:
			    break;
			}
		}

		if( keys[SDLK_q] || keys[SDLK_ESCAPE] )
			done = true;

#ifndef WIN32
		if(keys[SDLK_F2]) {
			//This only works with X
			SDL_WM_ToggleFullScreen( surface );
			keys[SDLK_F2]=false;
		}
#endif

	    // Draw the scene.
		if (isActive)
			drawGLScene();
		else
			#ifdef WIN32
			Sleep(1);
			#else
			usleep(1);
			#endif
	}

    // Clean ourselves up and exit.
    Quit(0);

    // Should never get here.
	cerr << "Shouldn't have reached this... End of brutalchess.cpp.\n";
    return 0;
}

// End of file brutalchess.cpp
