/********************************************************************
 * Copyright (C) 2005, 2006 Piotr Pszczolkowski
 *-------------------------------------------------------------------
 * This file is part of BSCommander (Beesoft Commander).
 *
 * BsC is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * BsC is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with BsC; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *******************************************************************/

/*------- include files:
-------------------------------------------------------------------*/
#include "Copier.h"
#include "Shared.h"
#include "Config.h"
#include <qlayout.h>
#include <qpushbutton.h>
#include <qlabel.h>
#include <qprogressbar.h>
#include <qgroupbox.h>
#include <qdir.h>
#include <qmessagebox.h>
#include <qcheckbox.h>
#include <qinputdialog.h>
#include "InfoField.h"
#include "CanOverwrite.h"
using namespace std;

/*------- local constants:
-------------------------------------------------------------------*/
const int     Copier::PROGRESS_MIN_SIZE   = 300;
const QString Copier::GBOX_INFO_TITEL     = QT_TR_NOOP( "Source and destination" );
const QString Copier::SRC_LABEL           = QT_TR_NOOP( "Source:" );
const QString Copier::DST_LABEL           = QT_TR_NOOP( "Destination:" );
const QString Copier::GBOX_PROGRESS_LABEL = QT_TR_NOOP( "Progress" );
const QString Copier::BTN_CONTINUE_LABEL  = QT_TR_NOOP( "&Continue" );
const QString Copier::BTN_PAUSE_LABEL     = QT_TR_NOOP( "&Pause" );
const QString Copier::NOT_WRITABLE_DIR    = QT_TR_NOOP( "You can't wite to this directory: %1.\nCheck permission access." );
const QString Copier::NOT_READABLE_FILE   = QT_TR_NOOP( "You can't read from this file: %1.\nCheck permission access." );
const QString Copier::MKDIR_ERROR         = QT_TR_NOOP( "Can't create a directory named: %1.\nMaybe exists already another file with this name." );
const QString Copier::OPEN_FOR_READ_ERROR = QT_TR_NOOP( "Can't open this file for reading: %1." );
const QString Copier::OPEN_FOR_WRITE_ERROR= QT_TR_NOOP( "Can't open this file for writing; %1." );
const QString Copier::CB_REMOVE_LABEL     = QT_TR_NOOP( "remove source file(s)" );

//*******************************************************************
// OneFileCopier
//*******************************************************************
Copier::Copier( QWidget* const in_parent, const ViewTable::SelectedItems& in_items, const QString& in_dst_dir )
: QDialog       ( in_parent )
, d_items       ( in_items )
, d_dst_dir     ( in_dst_dir )
, d_pause_button( new QPushButton( tr(Shared::RunBtnLabel), this ))
, d_break_button( new QPushButton( tr(Shared::CloseBtnLabel), this ))
, d_remove_cb   ( new QCheckBox  ( tr( CB_REMOVE_LABEL ), this ))
, d_src_info    ( 0 )
, d_dst_info    ( 0 )
, d_progress_bar( 0 )
, d_paused      ( FALSE )
, d_started     ( FALSE )
, d_stopped     ( FALSE )
, d_block_size  ( BufferSize )
, d_ask_again   ( TRUE )
, d_fm          ( font() )
{
	setFont( Config::instance()->lfs_default_font() );
	
	QVBoxLayout* const main_layout = new QVBoxLayout( this );
	main_layout->setMargin( Shared::LayoutMargin );
	main_layout->setSpacing( Shared::LayoutSpacing );

	main_layout->addWidget( make_info_area() );
	main_layout->addWidget( make_progress_area() );
	main_layout->addLayout( make_button_area() );

	connect( this, SIGNAL( start() ), this, SLOT( process() ));
}
// end of Copier

//###################################################################
//#                                                                 #
//#                    C O N S T R U C T I O N                      #
//#                                                                 #
//###################################################################

//*******************************************************************
// make_info_area                                            PRIVATE
//*******************************************************************
QWidget* Copier::make_info_area()
{
	QGroupBox* const gbox = new QGroupBox( 2, Qt::Horizontal, tr(GBOX_INFO_TITEL), this );
	QFrame* const frame = new QFrame( gbox );
	QGridLayout* const grid_layout = new QGridLayout( frame, 2, 2, 4 );
	grid_layout->setColStretch( 1, Shared::OverStretch );

	grid_layout->addWidget( new QLabel( tr(SRC_LABEL), frame ), 0, 0, Qt::AlignRight );
	grid_layout->addWidget( d_src_info = new InfoField( frame ), 0, 1 );
	grid_layout->addWidget( new QLabel( tr(DST_LABEL), frame ), 1, 0, Qt::AlignRight );
	grid_layout->addWidget( d_dst_info = new InfoField( frame ), 1, 1 );

	return gbox;
}
// end of make_info_area

//*******************************************************************
// make_progress_area                                        PRIVATE
//*******************************************************************
QWidget* Copier::make_progress_area()
{
	QGroupBox* const gbox = new QGroupBox( 1, Qt::Vertical, tr(GBOX_PROGRESS_LABEL), this );
	d_progress_bar = new QProgressBar( gbox );
	d_progress_bar->setCenterIndicator( TRUE );

	return gbox;
}
// end of make_progress_area

//*******************************************************************
// make_button_area                                          PRIVATE
//*******************************************************************
QLayout* Copier::make_button_area()
{
	QHBoxLayout* const layout = new QHBoxLayout;
	layout->setMargin( Shared::LayoutMargin );
	layout->setSpacing( Shared::LayoutSpacing );

	layout->addWidget( d_remove_cb );
	layout->addStretch( Shared::OverStretch );
	layout->addWidget( d_pause_button );
	layout->addWidget( d_break_button );

	connect( d_pause_button, SIGNAL( clicked() ), this, SLOT( pause()  ));
	connect( d_break_button, SIGNAL( clicked() ), this, SLOT( finish() ));

	return layout;
}
// end of make_button_area

//*******************************************************************
// polish                                                    PRIVATE
//*******************************************************************
void Copier::polish()
{
	Shared::polish_width( this, 40 );
	
	if( FALSE == d_items.empty() ) {
		const QString src = d_items[0]->path();
		const QString dst = d_dst_dir + "/" + d_items[0]->name();
		display( src, dst );
	}
}
// end of polish

//*******************************************************************
// pause                                                PRIVATE SLOT
//*******************************************************************
void Copier::pause()
{
	if( TRUE == d_started ) {
		if( TRUE == d_paused ) {
			d_pause_button->setText( tr(BTN_PAUSE_LABEL) );
			d_paused = FALSE;
		}
		else {
			d_pause_button->setText( tr(BTN_CONTINUE_LABEL) );
			d_paused = TRUE;
		}
	}
	else {
		d_pause_button->setText( tr(BTN_PAUSE_LABEL) );
		d_break_button->setText( tr(Shared::BreakBtnLabel) );

		d_started = TRUE;
		emit start();
	}
}
// end of pause

//*******************************************************************
// finish                                               PRIVATE SLOT
//*******************************************************************
void Copier::finish()
{
	if( TRUE == d_started ) {
		d_paused = FALSE;
		d_stopped = TRUE;
	}
	else {
		reject();
	}
}
// end of finish

//###################################################################
//#                                                                 #
//#                     P R O C E S S I N G                         #
//#                                                                 #
//###################################################################

//*******************************************************************
// process                                              PRIVATE SLOT
//*******************************************************************
void Copier::process()
{
	// Spawdzan czy uzytkownik ma prawo zapisu do
	// katalogu docelowego.
	const QFileInfo fi( d_dst_dir );
	
	// Uzytkownik nie ma uprawnien do zapisu w katalu docelowym.
	if( FALSE == fi.isWritable() ) {
		QMessageBox::warning( this, Shared::ProgramName, tr(NOT_WRITABLE_DIR).arg( d_dst_dir ));
		reject();
	}
	// Uzytkownik ma prawo zapisu do katalogu docelowego.
	else {
		d_remove_cb->setEnabled( FALSE );
		ViewTable::SelectedItems::const_iterator it = d_items.begin();
		while(( it != d_items.end() ) && ( FALSE == d_stopped )) {
			copy_next( (*it)->fi(), d_dst_dir );
			++it;
		}
		accept();
	}
}
// end of process

//*******************************************************************
// copy_next                                                 PRIVATE
//*******************************************************************
void Copier::copy_next( const QFileInfo& in_fi, const QString& in_dst_dir )
{
	Shared::idle();
	
	if( FALSE == d_stopped ) {
		if( Shared::is_regular_file( in_fi.fileName() )) {
			// KATALOG ---------------------------------------------
			if( in_fi.isDir() ) {
				const QDir dir( in_fi.absFilePath() );
				if( dir.isReadable() ) {
					// OK. Wejdz do katalogu i odczytaj jego zawartosc.
					QDir dst( in_dst_dir );
					dst.mkdir( in_fi.fileName(), FALSE );
					if( dst.cd( in_fi.fileName(), FALSE )) {
						copy_subdir( in_fi.absFilePath(), dst.absPath() );
					}
					else {
						//  Nie mozna utworzyc podkatalogu.
						QMessageBox::warning( this, Shared::ProgramName, tr(MKDIR_ERROR).arg( in_fi.fileName() ));
					}
				}
				else {
					// Brak praw dostepu. Nic nierob. Wracamy.
					QMessageBox::warning( this, Shared::ProgramName, tr(Shared::NotReadableDir).arg( in_fi.absFilePath() ));
				}
			}
			// NORMALNY PLIK ---------------------------------------
			else {
				copy_file( in_fi.absFilePath(), in_dst_dir );
			}
		}
	}
}
// end of copy_next

//*******************************************************************
// copy_subdir                                               PRIVATE
//*******************************************************************
void Copier::copy_subdir( const QString& in_src_dir, const QString& in_dst_dir )
{
	if( FALSE == d_stopped ) {
		QDir dir( in_src_dir );
		QFileInfoListIterator it( *dir.entryInfoList( QDir::All | QDir::Hidden | QDir::System ));
	
		while( it.current() ) {
			copy_next( *it.current(), in_dst_dir );
			++it;
		}

		if( d_remove_cb->isChecked() ) {
			Shared::remove_path_and_content( in_src_dir );
		}
	}
}
// end of copy_subdir

//*******************************************************************
// can_copy                                                  PRIVATE
//*******************************************************************
bool Copier::can_copy( const QString& in_src, QString& in_dst )
{
	bool       retval = TRUE;
	static int answer = CanOverwrite::OVER;
	
	if( TRUE == d_ask_again ) {
		if( TRUE == QFile::exists( in_dst ) ) {
			CanOverwrite* const dlg = new CanOverwrite( this, in_dst );
			if( dlg ) {
				answer = dlg->exec();
				if( answer != CanOverwrite::RENAME ) {
					d_ask_again = dlg->ask_again();
				}
				delete dlg;
			}
		}
	}

	switch( answer )
	{
		case CanOverwrite::RENAME:
			rename( in_dst );
			break;
		case CanOverwrite::OVER:
			break;
		case CanOverwrite::CANCEL:
			d_stopped = TRUE;
		case CanOverwrite::SKIP:
			retval = FALSE;
			break;
		case CanOverwrite::UPDATE:
			retval = can_update( in_src, in_dst );
			break;
	}
	return retval;
}
// end of can_copy

//*******************************************************************
// rename                                                    PRIVATE
//*******************************************************************
void Copier::rename( QString& inout_fpath )
{
	QString fpath = inout_fpath;
	Shared::auto_rename( fpath );
	
	const QFileInfo fi( fpath );
	const QString fname = fi.fileName();
	QString dir = fi.dirPath( TRUE );
	if( dir != "/" ) dir += "/";
	
	bool ok;
	const QString new_fname = QInputDialog::getText(	tr("Rename"),
												tr("New file name:"),
												QLineEdit::Normal,
												fname, &ok,
												this );
	if( ok && ( FALSE == new_fname.isEmpty() )) {
		inout_fpath = dir + new_fname;
	}
}
// end of rename

//*******************************************************************
// can_update                                                PRIVATE
//*******************************************************************
bool Copier::can_update( const QString& in_src, const QString& in_dst )
{
	const QFileInfo src_fi( in_src );
	const QFileInfo dst_fi( in_dst );

	return ( src_fi.lastModified() > dst_fi.lastModified() );
}
// end of can_update

//*******************************************************************
// copy_file                                                 PRIVATE
//-------------------------------------------------------------------
// Kopiowanie pliku z jednego katalogu do innego.
//
// PARAMETRY:
// 'in_src_path' - Sciezka pliku zrodlowego (czyli katalog + plik )
//                 np. /home/piotr/gnu/nazwa_pliku
// 'in_dst_dir'  - Katalog, do ktorego ma byc przekopiowany plik
//                 np. /home/piotr/backup/
//*******************************************************************
void Copier::copy_file( const QString& in_src_path, const QString& in_dst_dir )
{
	const QFileInfo fi( in_src_path );
	
	if( FALSE == fi.isReadable() ) {
		QMessageBox::warning( this, Shared::ProgramName, tr(NOT_READABLE_FILE).arg( in_src_path ));
	}
	else {
		// Tworzymy sciezke pliku wynikowego.
		// Wyodrebniamy ze sciezki zrodlowej tylko nazwe pliku i
		// dodajemy ja do katalogu docelowego.
		QString dst_path = in_dst_dir + "/" + QFileInfo( in_src_path ).fileName();
	
		// Pokazujemy uzytkownikowi co bedziemy kopiowac.
		display( in_src_path, dst_path );
		Shared::idle();

		// Najpierw sprawdzamy czy w ogole mozemy kopiowac plik.
		// Jezeli sa jakiekolwiek watpliwosci (np. plik juz istnieje)
		// zgode na kopiowanie musi wydac uzytkownik.
		if( TRUE == can_copy( in_src_path, dst_path ) ) {
			// Po uzyskaniu zgody na kopiowanie bierzemy sie do roboty.
			QFile in( in_src_path );
			QFile out( dst_path );
			if( TRUE == in.open( IO_ReadOnly ) ) {
				const unsigned int nbytes = in.size();	// Liczba bajtow do przekopiowania.
				unsigned int copied = 0;               // Licznik aktualnie przekopiowanych bajtow.
			
				if( TRUE == out.open( IO_WriteOnly | IO_Truncate ) ) {
					d_progress_bar->reset();
					d_progress_bar->setTotalSteps( nbytes );
				
					// Petla kopiujaca.
					while( FALSE == in.atEnd() ) {
						const unsigned int n = in.readBlock( d_buffer, d_block_size );
						out.writeBlock( d_buffer, n );
						d_progress_bar->setProgress( copied += n );
						Shared::idle();
					}
					d_progress_bar->setProgress( nbytes );
					out.close();
					if( d_remove_cb->isChecked() ) {
						Shared::remove_file( in_src_path );
					}
				}
				else {
					QMessageBox::warning( this, Shared::ProgramName, tr(OPEN_FOR_WRITE_ERROR).arg( dst_path ) );
				}
				in.close();
			}
			else {
				QMessageBox::warning( this, Shared::ProgramName, tr(OPEN_FOR_READ_ERROR).arg( in_src_path ) );
			}
		}
	}
}
// end of copy_file

//*******************************************************************
// display                                                   PRIVATE
//*******************************************************************
void Copier::display( const QString& in_src_path, const QString& in_dst_path )
{
	const int MAXLEN = d_src_info->width();

	// Preparujemy sciezke zrodlowa i wyswietlamy ja.
	QString src_path = in_src_path;
	if( d_fm.width( src_path ) > MAXLEN ) {
		Shared::clip_path( d_fm, d_src_info->width(), src_path );
	}
	d_src_info->setText( src_path );
	
	// Preparujemy sciezke docelowa i wyswietlamy ja.
	QString dst_path = in_dst_path;
	if( d_fm.width( dst_path ) > MAXLEN ) {
		Shared::clip_path( d_fm, d_dst_info->width(), dst_path );
	}
	d_dst_info->setText( dst_path );
}
// end of display
