/*
 * Main application class
 *
 * Copyright (C) 2003,2004,2005  Enrico Zini
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include "BuffyWindow.h"

#include "Environment.h"
#include "buffy/MailProgram.h"

#include <gtkmm/stock.h>
#include <gtkmm/fileselection.h>
#include <gtkmm/messagedialog.h>
#include <gtkmm/toggleaction.h>
#include <gtkmm/spinbutton.h>

#include <map>

using namespace std;
using namespace stringf;
using namespace buffy;


BuffyWindow::BuffyWindow()
	: updateButton("Update"),
	  prefDialog(*this)
{
	Config& conf = Environment::get().config();

	set_title("Buffy Folder List");
	//set_border_width(5);
	ConfigItemDictionary prefs = conf.application("buffy");
	string saved_w = prefs.get("winw", "500").get();
	string saved_h = prefs.get("winh", "300").get();
	
	set_default_size(atoi(saved_w.c_str()), atoi(saved_h.c_str()));

	add(mainVBox);

	// Create the actions used by Buffy

	actionGroup = Gtk::ActionGroup::create("Actions");
	actionGroup->add(Gtk::Action::create("FileMenu", "_File"));
	actionGroup->add(Gtk::Action::create("ViewMenu", "_View"));
	actionGroup->add(Gtk::Action::create("Quit", Gtk::Stock::QUIT, "_Quit", "Quit"),
			sigc::mem_fun(*this, &BuffyWindow::on_quit));
	
	actionGroup->add(actionPreferences = Gtk::Action::create("Preferences", "_Preferences..."),
			sigc::bind<BuffyWindow*>(
				sigc::mem_fun(prefDialog, &PrefDialog::do_dialog), this));

	actionGroup->add(actionRescanFolders = Gtk::Action::create("RescanFolders", "_Update folder list", ""),
			sigc::mem_fun(*this, &BuffyWindow::rescanFolders));
	actionGroup->add(actionViewEmpty = Gtk::ToggleAction::create("ViewAll", "View _all folders", "", conf.view_empty().get()),
			sigc::mem_fun(*this, &BuffyWindow::on_filter_changed));
	actionGroup->add(actionViewRead = Gtk::ToggleAction::create("ViewNonEmpty", "View all _non empty folders", "", conf.view_read().get()),
			sigc::mem_fun(*this, &BuffyWindow::on_filter_changed));
	actionGroup->add(actionViewImportant = Gtk::ToggleAction::create("ViewImportant", "View all folders with _important messages", "", conf.view_important().get()),
			sigc::mem_fun(*this, &BuffyWindow::on_filter_changed));

	uimanager = Gtk::UIManager::create();
	uimanager->insert_action_group(actionGroup);
	add_accel_group(uimanager->get_accel_group());

	// Layout the actions in the main menubar
	try
	{
		Glib::ustring ui_info = 
			"<ui>"
			"  <menubar name='MenuBar'>"
			"    <menu action='FileMenu'>"
			"      <menuitem action='RescanFolders'/>"
			"      <separator/>"
			"      <menuitem action='Preferences'/>"
			"      <separator/>"
			"      <menuitem action='Quit'/>"
			"    </menu>"
			"    <menu action='ViewMenu'>"
			"      <menuitem action='ViewAll'/>"
			"      <menuitem action='ViewNonEmpty'/>"
			"      <menuitem action='ViewImportant'/>"
			"    </menu>"
			"  </menubar>"
			"</ui>";
		uimanager->add_ui_from_string(ui_info);
	}
	catch(const Glib::Error& ex)
	{
		warning("Building menus failed: %.*s\n", PFSTR(ex.what()));
	}

	Gtk::Widget* menuBar = uimanager->get_widget("/MenuBar") ;
	mainVBox.pack_start(*menuBar, Gtk::PACK_SHRINK);

	folderList.configureFilter(conf);
	folderList.signal_open_folder().connect(sigc::mem_fun(*this, &BuffyWindow::on_open_folder));
	mainVBox.pack_start(folderList, Gtk::PACK_EXPAND_WIDGET);

	mainVBox.pack_start(buttonBox, Gtk::PACK_SHRINK);


	/*
	menuBar.items().push_back(Gtk::Menu_Helpers::MenuElem("_File", fileMenu));
	fileMenu.items().push_back(Gtk::Menu_Helpers::StockMenuElem(Gtk::Stock::QUIT,
				sigc::mem_fun(*this, &BuffyWindow::on_quit)));
	*/

	/*
	menuBar.items().push_back(Gtk::Menu_Helpers::MenuElem("_View", viewMenu));
	viewMenu.items().push_back(Gtk::Menu_Helpers::StockMenuElem(Gtk::Stock::QUIT,
				sigc::mem_fun(*this, &BuffyWindow::on_quit)));
*/

	buttonBox.pack_start(updateButton, Gtk::PACK_SHRINK);
	updateButton.signal_clicked().connect(
			sigc::mem_fun(folderList, &FolderList::do_update));

	set_timer(conf.update_interval().get());

	Environment::get().signal_child_exit().connect(sigc::mem_fun(*this, &BuffyWindow::on_program_exited));

	show_all_children();
}

void BuffyWindow::rescanFolders()
{
	folderList.clear();
	Environment::get().enumerateFolders(folderList);
}

void BuffyWindow::on_program_exited(int pid, int status)
{
	debug("Notified of exit of pid %d, status %d\n", pid, status);
	folderList.do_update();
}

bool BuffyWindow::on_configure_event(GdkEventConfigure* c)
{
	bool res = Gtk::Window::on_configure_event(c);
	int x, y;
	Gtk::Window::get_position(x, y);

	Config& conf = Environment::get().config();
	ConfigItemDictionary prefs = conf.application("buffy");

	//conf.set_application_data("buffy", "winx", fmt(c->x));
	//conf.set_application_data("buffy", "winy", fmt(c->y));
	prefs.get("winx", fmt(x)).set(fmt(x));
	prefs.get("winy", fmt(y)).set(fmt(y));
	prefs.get("winw", fmt(c->width)).set(fmt(c->width));
	prefs.get("winh", fmt(c->height)).set(fmt(c->height));

	return res;
}

void BuffyWindow::on_show()
{
	Gtk::Window::on_show();

	Config& conf = Environment::get().config();
	ConfigItemDictionary prefs = conf.application("buffy");
	string saved_x = prefs.get("winx", "").get();
	string saved_y = prefs.get("winy", "").get();
	if (saved_x != "" && saved_y != "")
		move(atoi(saved_x.c_str()), atoi(saved_y.c_str()));

	if (folderList.size() == 0)
		rescanFolders();
}

void BuffyWindow::on_open_folder(MailFolder folder)
{
	debug("OOF activated %.*s\n", PFSTR(folder.path()));

	MailProgram m = Environment::get().config().mail_programs().selected();
	//MailProgram m("Mutt", "x-terminal-emulator -e mutt -f %p");
	if (m)
	{
		verbose("Running %.*s\n", PFSTR(m.command()));
		m.run(folder);
	}
	else
		warning("No mail programs are currently selected.  This is probably due to hand-editing the configuration file and forgetting to add selected=\"true\" to one mail program.\n");
}

void BuffyWindow::on_quit()
{
	hide();
}

void BuffyWindow::on_filter_changed()
{
	Config& conf = Environment::get().config();

	bool empty = actionViewEmpty->get_active();
	bool read = actionViewRead->get_active();
	bool important = actionViewImportant->get_active();
	verbose("Filter changed %s %s %s\n",
			empty ? "empty" : "",
			read ? "read" : "",
			important ? "important" : "");

	conf.view_empty().set(empty);
	conf.view_read().set(read);
	conf.view_important().set(important);

	folderList.configureFilter(conf);
}

void BuffyWindow::set_timer(int interval)
{
	if (timer_connection.connected())
		timer_connection.disconnect();

	// interval is given in milliseconds
	timer_connection = Glib::signal_timeout().connect(
			sigc::mem_fun(folderList, &FolderList::do_timer_tick),
			interval*1000);
}

// vim:set ts=4 sw=4:
