/*
 * Main application class
 *
 * Copyright (C) 2003  Enrico Zini
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#pragma implementation

#include "BuffyWindow.h"

#include "Environment.h"
#include "buffy/MailProgram.h"

#include <gtkmm/stock.h>
#include <gtkmm/fileselection.h>
#include <gtkmm/messagedialog.h>
#include <gtkmm/toggleaction.h>
#include <gtkmm/spinbutton.h>

#include <map>

using namespace std;
using namespace stringf;


BuffyWindow::BuffyWindow()
	: updateButton("Update"),
	  prefDialog(*this)
{
	Config& conf = Environment::get().config();

	set_title("Buffy Folder List");
	//set_border_width(5);
	string saved_w = conf.get_application_data("buffy", "winw", "500");
	string saved_h = conf.get_application_data("buffy", "winh", "300");
	
	set_default_size(atoi(saved_w.c_str()), atoi(saved_h.c_str()));

	add(mainVBox);

	// Create the actions used by Buffy

	actionGroup = Gtk::ActionGroup::create("Actions");
	actionGroup->add(Gtk::Action::create("FileMenu", "_File"));
	actionGroup->add(Gtk::Action::create("ViewMenu", "_View"));
	actionGroup->add(Gtk::Action::create("Quit", Gtk::Stock::QUIT, "_Quit", "Quit"),
			sigc::mem_fun(*this, &BuffyWindow::on_quit));
	
	actionGroup->add(actionPreferences = Gtk::Action::create("Preferences", "_Preferences..."),
			sigc::bind<BuffyWindow&>(
				sigc::mem_fun(prefDialog, &PrefDialog::do_dialog), *this));

	actionGroup->add(actionRescanFolders = Gtk::Action::create("RescanFolders", "_Update folder list", ""),
			sigc::mem_fun(*this, &BuffyWindow::rescanFolders));
	actionGroup->add(actionViewEmpty = Gtk::ToggleAction::create("ViewAll", "View _all folders", "", conf.get_view_empty()),
			sigc::mem_fun(*this, &BuffyWindow::on_filter_changed));
	actionGroup->add(actionViewRead = Gtk::ToggleAction::create("ViewNonEmpty", "View all _non empty folders", "", conf.get_view_read()),
			sigc::mem_fun(*this, &BuffyWindow::on_filter_changed));
	actionGroup->add(actionViewImportant = Gtk::ToggleAction::create("ViewImportant", "View all folders with _important messages", "", conf.get_view_important()),
			sigc::mem_fun(*this, &BuffyWindow::on_filter_changed));

	uimanager = Gtk::UIManager::create();
	uimanager->insert_action_group(actionGroup);
	add_accel_group(uimanager->get_accel_group());

	// Layout the actions in the main menubar
	try
	{
		Glib::ustring ui_info = 
			"<ui>"
			"  <menubar name='MenuBar'>"
			"    <menu action='FileMenu'>"
			"      <menuitem action='RescanFolders'/>"
			"      <separator/>"
			"      <menuitem action='Preferences'/>"
			"      <separator/>"
			"      <menuitem action='Quit'/>"
			"    </menu>"
			"    <menu action='ViewMenu'>"
			"      <menuitem action='ViewAll'/>"
			"      <menuitem action='ViewNonEmpty'/>"
			"      <menuitem action='ViewImportant'/>"
			"    </menu>"
			"  </menubar>"
			"</ui>";
		uimanager->add_ui_from_string(ui_info);
	}
	catch(const Glib::Error& ex)
	{
		warning("Building menus failed: %.*s\n", PFSTR(ex.what()));
	}

	Gtk::Widget* menuBar = uimanager->get_widget("/MenuBar") ;
	mainVBox.pack_start(*menuBar, Gtk::PACK_SHRINK);

	folderList.configureFilter(conf.get_view_empty(), conf.get_view_read(), conf.get_view_important());
	folderList.signal_open_folder().connect(sigc::mem_fun(*this, &BuffyWindow::on_open_folder));
	mainVBox.pack_start(folderList, Gtk::PACK_EXPAND_WIDGET);

	mainVBox.pack_start(buttonBox, Gtk::PACK_SHRINK);


	/*
	menuBar.items().push_back(Gtk::Menu_Helpers::MenuElem("_File", fileMenu));
	fileMenu.items().push_back(Gtk::Menu_Helpers::StockMenuElem(Gtk::Stock::QUIT,
				sigc::mem_fun(*this, &BuffyWindow::on_quit)));
	*/

	/*
	menuBar.items().push_back(Gtk::Menu_Helpers::MenuElem("_View", viewMenu));
	viewMenu.items().push_back(Gtk::Menu_Helpers::StockMenuElem(Gtk::Stock::QUIT,
				sigc::mem_fun(*this, &BuffyWindow::on_quit)));
*/

	buttonBox.pack_start(updateButton, Gtk::PACK_SHRINK);
	updateButton.signal_clicked().connect(
			sigc::mem_fun(folderList, &FolderList::do_update));

	set_timer(conf.get_update_interval());

	Environment::get().signal_child_exit().connect(sigc::mem_fun(*this, &BuffyWindow::on_program_exited));

	/*
	entryHBox.pack_start(addLabel, Gtk::PACK_SHRINK);
	entryHBox.pack_start(addCombo, Gtk::PACK_EXPAND_WIDGET);
	entryHBox.pack_start(addButton, Gtk::PACK_SHRINK);
	//currentAdd.set_text("");
	
	//currentAdd.get_entry()->signal_changed().connect(
	addButton.signal_clicked().connect(
			sigc::mem_fun(*this, &BuffyWindow::on_addButton_clicked));
	*/

	/*
	menuBar.items().push_back(Gtk::Menu_Helpers::MenuElem("_Edit", editMenu));

	fileMenu.items().push_back(Gtk::Menu_Helpers::StockMenuElem(Gtk::Stock::OPEN,
				sigc::mem_fun(*this, &BuffyWindow::on_open)));
	fileMenu.items().push_back(Gtk::Menu_Helpers::MenuElem("Load _Debtags",
				sigc::mem_fun(*this, &BuffyWindow::on_open_debtags)));
	fileMenu.items().push_back(Gtk::Menu_Helpers::SeparatorElem());

	fileMenu.items().push_back(Gtk::Menu_Helpers::StockMenuElem(Gtk::Stock::SAVE,
				sigc::mem_fun(*this, &BuffyWindow::on_save)));
	saveMenuItem = &(fileMenu.items().back());
	saveMenuItem->set_sensitive(false);
	fileMenu.items().push_back(Gtk::Menu_Helpers::StockMenuElem(Gtk::Stock::SAVE_AS,
				sigc::mem_fun(*this, &BuffyWindow::on_save_as)));
	saveasMenuItem = &(fileMenu.items().back());
	fileMenu.items().push_back(Gtk::Menu_Helpers::SeparatorElem());


	editMenu.items().push_back(Gtk::Menu_Helpers::StockMenuElem(Gtk::Stock::UNDO,
				sigc::mem_fun(*this, &BuffyWindow::on_undo)));
	undoMenuItem = &(editMenu.items().back());
	editMenu.items().push_back(Gtk::Menu_Helpers::StockMenuElem(Gtk::Stock::REDO,
				sigc::mem_fun(*this, &BuffyWindow::on_redo)));
	redoMenuItem = &(editMenu.items().back());
	editMenu.items().push_back(Gtk::Menu_Helpers::SeparatorElem());

	editMenu.items().push_back(Gtk::Menu_Helpers::MenuElem("_Merge",
				sigc::mem_fun(*this, &BuffyWindow::on_merge)));
	mergeMenuItem = &(editMenu.items().back());
	editMenu.items().push_back(Gtk::Menu_Helpers::MenuElem("_Intersect",
				sigc::mem_fun(*this, &BuffyWindow::on_intersect)));
	intersectMenuItem = &(editMenu.items().back());
	editMenu.items().push_back(Gtk::Menu_Helpers::MenuElem("Co_py to other panel",
				sigc::mem_fun(*this, &BuffyWindow::on_copy_to_other)));
	copyToOtherMenuItem = &(editMenu.items().back());
	editMenu.items().push_back(Gtk::Menu_Helpers::MenuElem("Mo_ve to other panel",
				sigc::mem_fun(*this, &BuffyWindow::on_move_to_other)));
	moveToOtherMenuItem = &(editMenu.items().back());
	editMenu.items().push_back(Gtk::Menu_Helpers::MenuElem("Delete _unselected tags",
				sigc::mem_fun(*this, &BuffyWindow::on_delete_unselected)));
	deleteUnselectedMenuItem = &(editMenu.items().back());

	//menuBar.items().push_back(Gtk::Menu_Helpers::StockMenuElem(Gtk::Stock::HELP, helpMenu));
	*/

	/*
	signal_configure_event().connect(
			sigc::mem_fun(*this, &BuffyWindow::on_window_configure));
	*/

	show_all_children();
}

void BuffyWindow::rescanFolders()
{
	folderList.clear();
	Environment::get().enumerateFolders(folderList);
}

void BuffyWindow::on_program_exited(int pid, int status)
{
	debug("Notified of exit of pid %d, status %d\n", pid, status);
	folderList.do_update();
}

bool BuffyWindow::on_configure_event(GdkEventConfigure* c)
{
	bool res = Gtk::Window::on_configure_event(c);

	Config& conf = Environment::get().config();

	conf.set_application_data("buffy", "winx", fmt(c->x));
	conf.set_application_data("buffy", "winy", fmt(c->y));
	conf.set_application_data("buffy", "winw", fmt(c->width));
	conf.set_application_data("buffy", "winh", fmt(c->height));

	return res;
}

void BuffyWindow::on_show()
{
	Gtk::Window::on_show();

	Config& conf = Environment::get().config();
	string saved_x = conf.get_application_data("buffy", "winx");
	string saved_y = conf.get_application_data("buffy", "winy");
	if (saved_x != "" && saved_y != "")
		move(atoi(saved_x.c_str()), atoi(saved_y.c_str()));
}


/*
void BuffyWindow::consume(const MailFolder& folder)
{
	folderList.consume(folder);
}
*/

void BuffyWindow::on_open_folder(MailFolder folder)
{
	debug("OOF activated %.*s\n", PFSTR(folder.path()));

	MailProgram m = Environment::get().config().get_selected_mail_program();
	//MailProgram m("Mutt", "x-terminal-emulator -e mutt -f %p");
	if (m)
	{
		verbose("Running %.*s\n", PFSTR(m.command()));
		m.run(folder);
	}
	else
		warning("No mail programs are currently selected.  This is probably due to hand-editing the configuration file and forgetting to add selected=\"true\" to one mail program.\n");
	/*
	string cmdline = "x-terminal-emulator -e mutt -f " + folder.path();
	try {
		warning("Running: %.*s\n", PFSTR(cmdline));
		Environment::get().spawnCommand(cmdline);
	} catch (SystemException& e) {
		warning("%s: %.*s\n", e.type(), PFSTR(e.desc()));
	}
	*/
}

void BuffyWindow::on_quit()
{
	hide();
}

void BuffyWindow::on_filter_changed()
{
	Config& conf = Environment::get().config();

	bool empty = actionViewEmpty->get_active();
	bool read = actionViewRead->get_active();
	bool important = actionViewImportant->get_active();
	verbose("Filter changed %s %s %s\n",
			empty ? "empty" : "",
			read ? "read" : "",
			important ? "important" : "");

	conf.set_view_empty(empty);
	conf.set_view_read(read);
	conf.set_view_important(important);

	folderList.configureFilter(empty, read, important);
}

void BuffyWindow::set_timer(int interval)
{
	if (timer_connection.connected())
		timer_connection.disconnect();

	// interval is given in milliseconds
	timer_connection = Glib::signal_timeout().connect(
			sigc::mem_fun(folderList, &FolderList::do_timer_tick),
			interval*1000);
}


#if 0
void BuffyWindow::on_doc_change()
{
	checkUndo();
	setChanged(true);

	TagSet tags = doc.vocabulary().getTags();
	vector<string> string_tags;
	for (TagSet::const_iterator i = tags.begin(); i != tags.end(); i++)
		string_tags.push_back(i->fullname());
	addCombo.set_popdown_strings(string_tags);

	/*
	allTags = doc.collection().getAllTags();
	addCombo.set_popdown_strings(allTags);
	*/
}

void BuffyWindow::on_filename_change()
{
	if (doc.fileName().empty())
	{
		set_title(string("Tagged Collection Editor - debtags Database") + (changed ? " [*]" : ""));
		saveMenuItem->set_sensitive(false);
	}
	else
	{
		set_title("Tagged Collection Editor - " + doc.fileName() + (changed ? " [*]" : ""));
		saveMenuItem->set_sensitive(changed);
	}
}

void BuffyWindow::on_selection_change()
{
	int size = activePanel().getSelectionSize();

	/*
	if (size == 1)
	{
		std::map< string, OpSet<string> > data = activePanel().getSelection();
		OpSet<string>& tags = data.begin()->second;
		std::string val;
		for (OpSet<string>::const_iterator i = tags.begin();
				i != tags.end(); i++)
			if (i == tags.begin())
				val += "Current tag set: " + *i;
			else
				val += ", " + *i;
		currentLabel.set_text(val + " ");
		currentAdd.set_sensitive(true);
	} else*/ if (size >= 1) {
		addLabel.hide();
		addCombo.show();
		addButton.set_sensitive(true);
	} else {
		addLabel.show();
		addCombo.hide();
		addButton.set_sensitive(false);
	}

	mergeMenuItem->set_sensitive(size >= 2);
	intersectMenuItem->set_sensitive(size >= 2);
	copyToOtherMenuItem->set_sensitive(size >= 1);
	moveToOtherMenuItem->set_sensitive(size >= 1);
	deleteUnselectedMenuItem->set_sensitive(size >= 1);
}

void BuffyWindow::checkUndo()
{
	undoMenuItem->set_sensitive(doc.canUndo());
	redoMenuItem->set_sensitive(doc.canRedo());
}

void BuffyWindow::on_open_debtags()
{
	try {
		doc.loadDebtags();
	} catch (Exception& e) {
		warning("%s: %.*s", e.type(), PFSTR(e.desc()));
	}
}

void BuffyWindow::on_open()
{
	Gtk::FileSelection dialog("Please select the name of the file to open");
	dialog.set_transient_for(*this);
	//dialog.get_file_list()->get_parent()->hide(); //Prevent the user from selecting a file.

	int result = dialog.run();

	//Handle the response:
	switch (result)
	{
		case Gtk::RESPONSE_OK:
		{
			try {
				doc.load(dialog.get_filename());
			} catch (Exception& e) {
				warning("%s: %.*s", e.type(), PFSTR(e.desc()));
			}
			break;
		}
		case Gtk::RESPONSE_CANCEL:
		{
			break;
		}
		default:
		{
			warning("Unexpected button clicked in load file selection dialog");
			break;
		}
	}
}
	
void BuffyWindow::on_save()
{
	if (!doc.fileName().empty())
	{
		doc.save(doc.fileName());
		setChanged(false);
	}
}

void BuffyWindow::on_save_as()
{
	Gtk::FileSelection dialog("Please select the file name to save as");
	dialog.set_transient_for(*this);
	//dialog.get_file_list()->get_parent()->hide(); //Prevent the user from selecting a file.

	int result = dialog.run();

	//Handle the response:
	switch (result)
	{
		case Gtk::RESPONSE_OK:
		{
			try {
				doc.save(dialog.get_filename());
			} catch (Exception& e) {
				warning("%s: %.*s", e.type(), PFSTR(e.desc()));
			}
			setChanged(false);
			break;
		}
		case Gtk::RESPONSE_CANCEL:
		{
			break;
		}
		default:
		{
			warning("Unexpected button clicked in save as... file selection dialog");
			break;
		}
	}
}

void BuffyWindow::on_undo()
{
	doc.undo();
}

void BuffyWindow::on_redo()
{
	doc.redo();
}

void BuffyWindow::on_merge()
{
	// Get the contents of the selection
	std::map< string, TagSet > data = activePanel().getSelection();

	/*
	warning("LTP: %s\n", leftTagPanel.isActive() ? "Y" : "N");
	warning("RTP: %s\n", rightTagPanel.isActive() ? "Y" : "N");
	*/	

	// Compute the union
	OpSet<string> items;
	TagSet merged;
	for (std::map< string, TagSet >::const_iterator i = data.begin();
			i != data.end(); i++)
	{
		items += i->first;
		merged += i->second;
	}

	// Make the changes
	TagcollChange<string, Tag> change;

	for (OpSet<string>::const_iterator i = items.begin();
			i != items.end(); i++)
		change.insert(make_pair(*i, merged));

	doc.applyChange(change);
}

void BuffyWindow::on_intersect()
{
	// Get the contents of the selection
	std::map< string, TagSet > data = activePanel().getSelection();

	if (data.empty())
		return;

	// Compute the intersection
	std::map< string, TagSet >::const_iterator i = data.begin();
	TagSet intersected = i->second;
	OpSet<string> items;
	for ( ; i != data.end(); i++)
	{
		items += i->first;
		intersected ^= i->second;
	}

	// Make the changes
	TagcollChange<string, Tag> change;

	for (OpSet<string>::const_iterator i = items.begin();
			i != items.end(); i++)
		change.insert(make_pair(*i, intersected));

	doc.applyChange(change);
}

void BuffyWindow::on_copy_to_other()
{
	std::map< string, TagSet > sourceItems = activePanel().getSelection();
	TagSet targetTags = otherPanel().selectedTags();

	TagcollChange<string, Tag> change;

	// Merge the source tagsets with the currently selected tagset of the target panel
	for (std::map< string, TagSet >::const_iterator i = sourceItems.begin();
			i != sourceItems.end(); i++)
		change.insert(make_pair(i->first, i->second + targetTags));

	doc.applyChange(change);
}

void BuffyWindow::on_move_to_other()
{
	std::map< string, TagSet > sourceItems = activePanel().getSelection();
	TagSet targetTags = otherPanel().selectedTags();

	TagcollChange<string, Tag> change;

	// Replace the source tagsets with the currently selected tagset of the target panel
	for (std::map< string, TagSet >::const_iterator i = sourceItems.begin();
			i != sourceItems.end(); i++)
		change.insert(make_pair(i->first, targetTags));

	doc.applyChange(change);
}

void BuffyWindow::on_delete_unselected()
{
	// Get the contents of the selection
	std::map< string, TagSet > data = activePanel().getSelection();

	if (data.empty())
		return;

	// Prepare the change
	TagSet selected = activePanel().selectedTags();
	TagcollChange<string, Tag> change;
	for (std::map< string, TagSet >::const_iterator i = data.begin();
			i != data.end(); i++)
		change.insert(make_pair(i->first, i->second ^ selected));

	// Make the change
	doc.applyChange(change);
}

void BuffyWindow::on_request_tagcoll_change(TagcollChange<string, Tag> change)
{
	doc.applyChange(change);
}

void BuffyWindow::on_select_tagset_panel1(TagSet tagset)
{
	panels[0]->selectedTags(tagset);
}

void BuffyWindow::on_select_tagset_panel2(TagSet tagset)
{
	panels[1]->selectedTags(tagset);
}

void BuffyWindow::on_addButton_clicked()
{
	string tag = addCombo.get_entry()->get_text();
	if (tag == "")
		return;

	if (!doc.vocabulary().hasTag(tag))
	{
		Gtk::MessageDialog dialog(*this, "Tag " + tag + " does not exist: do you want to create it?",
				false, Gtk::MESSAGE_QUESTION, (Gtk::ButtonsType)(Gtk::BUTTONS_YES_NO));
		int result = dialog.run();

		//Handle the response:
		switch (result)
		{
			case Gtk::RESPONSE_CANCEL: return;
			case Gtk::RESPONSE_OK:
			default: break;
		}
	}

	Tag newTag = doc.vocabulary().obtainTag(tag);

	std::map< string, TagSet > sourceItems = activePanel().getSelection();
	TagcollChange<string, Tag> change;

	// Add `tag' to all selected items
	for (std::map< string, TagSet >::const_iterator i = sourceItems.begin();
			i != sourceItems.end(); i++)
		change.insert(make_pair(i->first, i->second + newTag));

	doc.applyChange(change);
}
#endif

// vim:set ts=4 sw=4:
