# bzr-avahi - share and browse Bazaar branches with mDNS
# Copyright (C) 2007-2008 James Henstridge
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along
# with this program; if not, write to the Free Software Foundation, Inc.,
# 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.

"""Advertise and browse branches on the local network via mDNS.

Commands:
---------

 * advertise: Specify whether a branch should be advertised or not.
   These branches will be advertised by ``bzr serve``.
 * browse: List branches advertised on the local network.

Advertised branches can be accessed via the local: URI scheme using
the branch's advertised name.  For example:

    bzr branch local:somebranch
"""

from bzrlib.commands import Command, Option, display_command, register_command
from bzrlib.directory_service import directories
from bzrlib.lazy_import import lazy_import
from bzrlib.plugins.dbus.server_mainloop import MainloopThread
lazy_import(globals(), """
import os

from bzrlib import errors, trace, urlutils
from bzrlib.branch import Branch
import advertise
import browse
""")

version_info = (0, 3, 0, 'dev', 0)

if version_info[3] == 'final':
    version_string = '%d.%d.%d' % version_info[:3]
else:
    version_string = '%d.%d.%d%s%d' % version_info
__version__ = version_string


# Register hooks to advertise branches.

global_advertiser = None

def server_started(backing_urls, server_url):
    trace.mutter('Started: %r, %r' % (backing_urls, server_url))
    global global_advertiser
    if global_advertiser is None:
        global_advertiser = advertise.Advertiser()
        global_advertiser.start()
    global_advertiser.add_server(backing_urls, server_url)

def server_stopped(backing_urls, server_url):
    trace.mutter('Stopped: %r, %r' % (backing_urls, server_url))
    assert global_advertiser is not None
    global_advertiser.remove_server(server_url)
    pass

MainloopThread.hooks.install_named_hook(
    'server_started', server_started, 'Advertising branches via mDNS')
MainloopThread.hooks.install_named_hook(
    'server_stopped', server_stopped, 'Stop advertising branches via mDNS')

# Register local: transport.
directories.register_lazy(
    'local:', 'bzrlib.plugins.avahi.browse', 'AvahiDirectoryService',
    'Branches on the local network')

# Test suite.

def test_suite():
    from bzrlib.plugins.avahi import tests
    return tests.test_suite()

class cmd_mdns_advertise(Command):
    """Advertise a branch via mDNS."""
    aliases=['advertise']
    takes_options = [
        Option('name', type=unicode, short_name='n',
               help='The name to advertise the service under.'),
        Option('delete', short_name='d',
               help='Stop advertising the branch.'),
        ]
    takes_args = ['branch?']

    def run(self, branch=None, name=None, delete=False):
        if branch is None:
            branch_obj = Branch.open_containing('.')[0]
        else:
            branch_obj = Branch.open(branch)
        # Set the advertised name:
        if name is not None:
            advertise.set_mdns_name(branch_obj, name)
        advertise.set_mdns_advertise(branch_obj, not delete)
        advertise.send_change_notification(branch_obj)

register_command(cmd_mdns_advertise)


class cmd_mdns_browse(Command):
    """Browse branches that have been advertised via mDNS."""
    aliases = ['browse']
    takes_options = [Option('domain', help='DNS-SD domain.', type=unicode)]

    @display_command
    def run(sellf, domain='local'):
        browser = browse.BranchBrowser()
        browser.run(domain)

register_command(cmd_mdns_browse)

