# bzr-avahi - share and browse Bazaar branches with mDNS
# Copyright (C) 2007-2008 James Henstridge
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along
# with this program; if not, write to the Free Software Foundation, Inc.,
# 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.

__metaclass__ = type

import avahi
import dbus
import dbus.mainloop.glib
import gobject

from bzrlib import errors, trace


def service_to_uri((iface, proto, name, stype, domain,
                    host, aproto, addr, port, txt, flags), numeric=False):
    txt = dict(item.split('=', 1)
               for item in avahi.txt_array_to_string_array(txt)
               if '=' in item)
    url = '%s://' % txt.get('scheme', 'bzr')
    if numeric:
        url += addr
    else:
        url += host
    if port != 0:
        url += ':%d' % port
    url += txt.get('path', '/')

    return url


class AvahiDirectoryService(object):
    """A branch directory service for locating mDNS advertised branches."""

    def __init__(self):
        super(AvahiDirectoryService, self).__init__()
        self.bus = None
        self.server = None
        self.numeric = False

    def _connect(self):
        if self.server is not None:
            return
        self.bus = dbus.SystemBus()
        self.server = dbus.Interface(
            self.bus.get_object(avahi.DBUS_NAME, avahi.DBUS_PATH_SERVER),
            avahi.DBUS_INTERFACE_SERVER)
        # Only show host names if NSS support is available.
        self.numeric = not self.server.IsNSSSupportAvailable()

    def _resolve(self, service_name):
        """Resolve a service name to a branch URL."""
        self._connect()
        result = self.server.ResolveService(
            avahi.IF_UNSPEC, avahi.PROTO_UNSPEC, service_name, '_bzr._tcp',
            '', avahi.PROTO_UNSPEC, dbus.UInt32(0))
        return service_to_uri(result, self.numeric)

    def look_up(self, name, url):
        try:
            target = self._resolve(name)
        except dbus.DBusException, exc:
            raise errors.InvalidURL(path=url, extra=str(exc))
        return target


class BranchBrowser:

    def __init__(self):
        self.numeric = False
        self.browser = None
        self.all_for_now = False
        self.seen_names = set()
        self.pending_resolvers = {}

    def initialize(self):
        dbus.mainloop.glib.DBusGMainLoop(set_as_default=True)
        self.bus = dbus.SystemBus()
        self.server = dbus.Interface(
            self.bus.get_object(avahi.DBUS_NAME, avahi.DBUS_PATH_SERVER),
            avahi.DBUS_INTERFACE_SERVER)
        # Only show host names if NSS support is available.
        self.numeric = not self.server.IsNSSSupportAvailable()
        self.main = gobject.MainLoop()

    def maybe_quit(self):
        if self.all_for_now and len(self.pending_resolvers) == 0:
            self.main.quit()

    def run(self, domain=''):
        self.initialize()
        self.all_for_now = False
        self.seen_names.clear()
        self.pending_resolvers.clear()

        self.browser = dbus.Interface(
            self.bus.get_object(avahi.DBUS_NAME, self.server.ServiceBrowserNew(
                avahi.IF_UNSPEC, avahi.PROTO_UNSPEC,
                '_bzr._tcp', domain, dbus.UInt32(0))),
            avahi.DBUS_INTERFACE_SERVICE_BROWSER)
        self.browser.connect_to_signal('ItemNew', self.browser_item_new_cb)
        self.browser.connect_to_signal('Failure', self.browser_failure_cb)
        self.browser.connect_to_signal('AllForNow', self.browser_all_for_now_cb)

        self.main.run()

        self.browser.Free()
        self.browser = None

    def browser_item_new_cb(self, iface, proto, name, stype, domain, flags):
        """See ServiceBrowser.ItemNew"""
        # Make sure we only look up each name once
        if name in self.seen_names:
            return
        self.seen_names.add(name)

        # Resolve the name asynchronously
        resolver = dbus.Interface(
            self.bus.get_object(avahi.DBUS_NAME, self.server.ServiceResolverNew(
                iface, proto, name, stype, domain,
                avahi.PROTO_UNSPEC, dbus.UInt32(0))),
            avahi.DBUS_INTERFACE_SERVICE_RESOLVER)
        resolver.connect_to_signal('Found', self.resolver_found_cb,
                                   path_keyword='object_path')
        resolver.connect_to_signal('Failure', self.resolver_failure_cb,
                                   path_keyword='object_path')
        self.pending_resolvers[resolver.object_path] = resolver

    def browser_failure_cb(self, error):
        """See ServiceBrowser.Failure"""
        trace.error(error)
        # Treat this the same as AllForNow.
        self.browser_all_for_now_cb()

    def browser_all_for_now_cb(self):
        """See ServiceBrowser.AllForNow"""
        self.all_for_now = True
        self.maybe_quit()

    def resolver_found_cb(self, iface, proto, name, stype, domain,
                       host, aproto, addr, port, txt, flags,
                       object_path=None):
        """See ServiceResolver.Found"""
        assert object_path in self.pending_resolvers
        resolver = self.pending_resolvers[object_path]
        resolver.Free()
        del self.pending_resolvers[object_path]

        uri = service_to_uri((iface, proto, name, stype, domain,
                               host, aproto, addr, port, txt, flags),
                             self.numeric)
        self.found_branch(name, uri)
        self.maybe_quit()

    def resolver_failure_cb(self, error, object_path=None):
        """See ServiceResolver.Failure"""
        assert object_path in self.pending_resolvers
        resolver = self.pending_resolvers[object_path]
        resolver.Free()
        del self.pending_resolvers[object_path]

        trace.error(error)
        self.maybe_quit()

    def found_branch(self, nick, url):
        """Tell the user about a local branch."""
        print '%s: %s' % (nick, url)
