# Copyright (C) 2010 Canonical Limited
# vim: ts=4 sts=4 sw=4
#
# This file is part of bzr-builddeb.
#
# bzr-builddeb is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# bzr-builddeb is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with bzr-builddeb; if not, write to the Free Software
# Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

"""Tagging related functions for bzr-builddeb."""

__all__ = ['is_upstream_tag', 'upstream_tag_version']


try:
    from debian.changelog import Version
except ImportError:
    # Prior to 0.1.15 the debian module was called debian_bundle
    from debian_bundle.changelog import Version


def is_upstream_tag(tag):
    """Return true if tag is an upstream tag.
    
    :param tag: The string name of the tag.
    :return: True if the tag name is one generated by upstream tag operations.

    >>> is_upstream_tag('2.1')
    False
    >>> is_upstream_tag('upstream-2.1')
    True
    """
    return tag.startswith('upstream-') or tag.startswith('upstream/')


def upstream_tag_version(tag):
    """Return the upstream version portion of an upstream tag name.

    :param tag: The string name of the tag.
    :return: The version portion of the tag.

    >>> upstream_tag_version('upstream-2.1')
    '2.1'
    """
    assert is_upstream_tag(tag), "Not an upstream tag: %s" % tag
    if tag.startswith('upstream/'):
        tag = tag[len('upstream/'):]
    elif tag.startswith('upstream-'):
        tag = tag[len('upstream-'):]
        if tag.startswith('debian-'):
            tag = tag[len('debian-'):]
        elif tag.startswith('ubuntu-'):
            tag = tag[len('ubuntu-'):]
    return tag


def sort_debversion(branch, tags):
    """Sort tags using Debian version in-place.

    :param branch: Branch to use
    :param tags: List of tuples with name and version.
    """
    def debversion_key((version, revid)):
        return Version(version)
    tags.sort(key=debversion_key)
