/**
* This file is a part of the Cairo-Dock project
*
* Copyright : (C) see the 'copyright' file.
* E-mail    : see the 'copyright' file.
*
* This program is free software; you can redistribute it and/or
* modify it under the terms of the GNU General Public License
* as published by the Free Software Foundation; either version 3
* of the License, or (at your option) any later version.
*
* This program is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
* GNU General Public License for more details.
* You should have received a copy of the GNU General Public License
* along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

#include <string.h>
#include <math.h>

#include <cairo-dock.h>

#include "applet-struct.h"
#include "applet-bookmarks.h"
#include "applet-disk-usage.h"
#include "applet-drives.h"
#include "applet-load-icons.h"


void cd_shortcuts_set_icon_order_by_name (Icon *pNewIcon, GList *pIconsList)
{
	GList *ic;
	Icon *pIcon;
	for (ic = pIconsList; ic != NULL; ic = ic->next)
	{
		pIcon = ic->data;
		if (pIcon->iGroup == pNewIcon->iGroup)
			break;
	}
	GList *ic0 = ic;
	if (! ic0)
	{
		pNewIcon->fOrder = 0;
		return;
	}
	
	pIcon = ic0->data;
	if (cairo_dock_compare_icons_name (pNewIcon, pIcon) <= 0)
	{
		pNewIcon->fOrder = pIcon->fOrder - 1;
		//g_print ("name : %s <= %s -> %.2f\n", pNewIcon->cName, pIcon->cName, pNewIcon->fOrder);
		return;
	}
	
	pNewIcon->fOrder = 0;
	for (ic = ic0; ic != NULL; ic = ic->next)
	{
		pIcon = ic->data;
		if (pIcon->iGroup != pNewIcon->iGroup)
			break;
		if (cairo_dock_compare_icons_name (pNewIcon, pIcon) < 0)
		{
			if (ic->prev == NULL)
				pNewIcon->fOrder = pIcon->fOrder - 1;
			else
			{
				Icon *pPrevIcon = ic->prev->data;
				pNewIcon->fOrder = (pIcon->fOrder + pPrevIcon->fOrder) / 2;
			}
			//g_print ("  name : %s < %s -> %.2f\n", pNewIcon->cName, pIcon->cName, pNewIcon->fOrder);
			break;
		}
		pNewIcon->fOrder = pIcon->fOrder + 1;
	}
}


static void _cd_shortcuts_on_network_event (CairoDockFMEventType iEventType, const gchar *cURI, CairoDockModuleInstance *myApplet)
{
	CD_APPLET_ENTER;
	
	//g_print (" * event %d on network '%s'\n", iEventType, cURI);
	GList *pIconsList = CD_APPLET_MY_ICONS_LIST;
	CairoContainer *pContainer = CD_APPLET_MY_ICONS_LIST_CONTAINER;
	CD_APPLET_LEAVE_IF_FAIL (pContainer != NULL);
	
	switch (iEventType)
	{
		case CAIRO_DOCK_FILE_DELETED :  // un reseau a ete deconnecte.
		{
			Icon *pConcernedIcon = cairo_dock_get_icon_with_base_uri (pIconsList, cURI);
			if (pConcernedIcon == NULL)  // on cherche par nom.
			{
				pConcernedIcon = cairo_dock_get_icon_with_name (pIconsList, cURI);
			}
			if (pConcernedIcon == NULL)
			{
				cd_warning ("  an unknown network was removed");
				return ;
			}
			//g_print (" %s will be removed\n", pConcernedIcon->cName);
			
			CD_APPLET_REMOVE_ICON_FROM_MY_ICONS_LIST (pConcernedIcon);
		}
		break ;

		case CAIRO_DOCK_FILE_CREATED :  // un reseau a ete connecte.
		{
			//\_______________________ on verifie qu'elle n'existe pas deja.
			Icon *pSameIcon = cairo_dock_get_icon_with_base_uri (pIconsList, cURI);
			if (pSameIcon != NULL)
			{
				cd_warning ("this mount point (%s) already exists.", pSameIcon->cName);
				return;  // on decide de ne rien faire, c'est surement un signal inutile.
			}
			
			//\_______________________ on cree une icone pour cette nouvelle URI.
			Icon *pNewIcon = cairo_dock_fm_create_icon_from_URI (cURI, pContainer, CAIRO_DOCK_FM_SORT_BY_NAME);
			if (pNewIcon == NULL)
			{
				cd_warning ("couldn't create an icon for this network");
				return ;
			}
			pNewIcon->iGroup = CD_NETWORK_GROUP;
			
			//\_______________________ on la place au bon endroit suivant son nom.
			cd_shortcuts_set_icon_order_by_name (pNewIcon, pIconsList);
			//g_print (" new network : %s, order = %.2f\n", pNewIcon->cName, pNewIcon->fOrder);
			
			//\_______________________ on l'insere dans la liste.
			CD_APPLET_ADD_ICON_IN_MY_ICONS_LIST (pNewIcon);
			
			//\_______________________ on affiche un message.
			cairo_dock_show_temporary_dialog_with_icon_printf (
				D_("%s has been connected"),
				pNewIcon, pContainer,
				4000,
				NULL,  // son icone n'est pas encore chargee
				pNewIcon->cName);
		}
		break ;

		case CAIRO_DOCK_FILE_MODIFIED :  // un point de montage a ete (de)monte
		{
			//\_______________________ on cherche l'icone concernee.
			Icon *pConcernedIcon = cairo_dock_get_icon_with_base_uri (pIconsList, cURI);
			if (pConcernedIcon == NULL)  // on cherche par nom.
			{
				pConcernedIcon = cairo_dock_get_icon_with_name (pIconsList, cURI);
			}
			if (pConcernedIcon == NULL)
			{
				cd_warning ("  an unknown network was modified");
				return ;
			}
			//g_print (" %s is modified\n", pConcernedIcon->cName);
			
			//\_______________________ on recupere les infos actuelles.
			Icon *pNewIcon = cairo_dock_fm_create_icon_from_URI (cURI, pContainer, CAIRO_DOCK_FM_SORT_BY_NAME);
			if (pNewIcon == NULL)
			{
				cd_warning ("couldn't create an icon for this network");
				return ;
			}
			pNewIcon->iGroup = CD_NETWORK_GROUP;
			
			//\_______________________ on remplace l'icone si des choses ont change.
			if (cairo_dock_strings_differ (pConcernedIcon->cName, pNewIcon->cName) || cairo_dock_strings_differ (pConcernedIcon->cFileName, pNewIcon->cFileName))
			{
				//g_print (" '%s' -> '%s'\n'%s' -> '%s'\n", pConcernedIcon->cName, pNewIcon->cName, pConcernedIcon->cFileName, pNewIcon->cFileName);
				
				CD_APPLET_REMOVE_ICON_FROM_MY_ICONS_LIST (pConcernedIcon);
				pIconsList = CD_APPLET_MY_ICONS_LIST;
				
				cd_shortcuts_set_icon_order_by_name (pNewIcon, pIconsList);
				CD_APPLET_ADD_ICON_IN_MY_ICONS_LIST (pNewIcon);
			}
			else
			{
				cairo_dock_free_icon (pNewIcon);
			}
		}
		break ;

		case CAIRO_DOCK_NB_EVENT_ON_FILES :
		break ;
	}
	CD_APPLET_LEAVE();
}


static inline GList * _load_icons (CDSharedMemory *pSharedMemory)
{
	GList *pIconList = NULL;
	
	if (pSharedMemory->bListDrives)
	{
		pIconList = cd_shortcuts_list_drives (pSharedMemory);
	}
	
	if (pSharedMemory->bListNetwork)
	{
		gchar *cFullURI = NULL;
		GList *pIconList2 = cairo_dock_fm_list_directory (CAIRO_DOCK_FM_NETWORK, CAIRO_DOCK_FM_SORT_BY_NAME, CD_NETWORK_GROUP, FALSE, 100, &cFullURI);
		cd_message ("  cFullURI : %s", cFullURI);
		
		pIconList = g_list_concat (pIconList, pIconList2);
		
		pSharedMemory->cNetworkURI = cFullURI;
	}
		
	if (pSharedMemory->bListBookmarks)
	{
		gchar *cBookmarkFilePath = g_strdup_printf ("%s/.gtk-bookmarks", g_getenv ("HOME"));
		if (! g_file_test (cBookmarkFilePath, G_FILE_TEST_EXISTS))  // on le cree pour pouvoir ajouter des signets.
		{
			FILE *f = fopen (cBookmarkFilePath, "a");
			fclose (f);
		}
		
		GList *pIconList2 = cd_shortcuts_list_bookmarks (cBookmarkFilePath, pSharedMemory->pApplet);
		
		pIconList = g_list_concat (pIconList, pIconList2);
		
		pSharedMemory->cBookmarksURI = cBookmarkFilePath;
	}
	
	return pIconList;
}
static void cd_shortcuts_get_shortcuts_data (CDSharedMemory *pSharedMemory)
{
	pSharedMemory->pIconList = _load_icons (pSharedMemory);
}

static gboolean cd_shortcuts_build_shortcuts_from_data (CDSharedMemory *pSharedMemory)
{
	CairoDockModuleInstance *myApplet = pSharedMemory->pApplet;
	g_return_val_if_fail (myIcon != NULL, FALSE);  // paranoia
	CD_APPLET_ENTER;
	
	//\_______________________ get the result of the thread.
	GList *pIconList = pSharedMemory->pIconList;
	pSharedMemory->pIconList = NULL;
	myData.cDisksURI = pSharedMemory->cDisksURI;
	pSharedMemory->cDisksURI = NULL;
	myData.cNetworkURI = pSharedMemory->cNetworkURI;
	pSharedMemory->cNetworkURI = NULL;
	myData.cBookmarksURI = pSharedMemory->cBookmarksURI;
	pSharedMemory->cBookmarksURI = NULL;
	
	//\_______________________ monitor the sets.
	if (myData.cDisksURI)
	{
		if (! cairo_dock_fm_add_monitor_full (myData.cDisksURI, TRUE, NULL, (CairoDockFMMonitorCallback) cd_shortcuts_on_drive_event, myApplet))
			cd_warning ("Shortcuts : can't monitor drives");
	}
	if (myData.cNetworkURI)
	{
		if (! cairo_dock_fm_add_monitor_full (myData.cNetworkURI, TRUE, NULL, (CairoDockFMMonitorCallback) _cd_shortcuts_on_network_event, myApplet))
			cd_warning ("Shortcuts : can't monitor network");
	}
	if (myData.cBookmarksURI)
	{
		if (! cairo_dock_fm_add_monitor_full (myData.cBookmarksURI, FALSE, NULL, (CairoDockFMMonitorCallback) cd_shortcuts_on_bookmarks_event, myApplet))
			cd_warning ("Shortcuts : can't monitor bookmarks");
	}
	
	//\_______________________ On efface l'ancienne liste.
	CD_APPLET_DELETE_MY_ICONS_LIST;
	
	//\_______________________ On charge la nouvelle liste.
	const gchar *cDeskletRendererName = NULL;
	switch (myConfig.iDeskletRendererType)
	{
		case CD_DESKLET_SLIDE :
		default :
			cDeskletRendererName = "Viewport";
		break ;
		
		case CD_DESKLET_TREE :
			cDeskletRendererName = "Tree";
		break ;
	}
	CD_APPLET_LOAD_MY_ICONS_LIST (pIconList, myConfig.cRenderer, cDeskletRendererName, NULL);  // takes ownership of 'pIconList'
	
	//\_______________________ add a progress bar on disk volumes (must be done after inserting the icons into a container)
	pIconList = CD_APPLET_MY_ICONS_LIST;
	Icon *pIcon;
	GList *ic;
	for (ic = pIconList; ic != NULL; ic = ic->next)
	{
		pIcon = ic->data;
		if (CD_APPLET_GET_MY_ICON_DATA (pIcon) != NULL)  // drive
			cd_shortcuts_add_progress_bar (pIcon, myApplet);
	}
	
	//\_______________________ On lance la tache de mesure des disques.
	cd_shortcuts_launch_disk_periodic_task (myApplet);

	if (myData.bShowMenuPending)
	{
		cairo_dock_notify_on_object (myContainer, NOTIFICATION_CLICK_ICON, myIcon, myDock, GDK_BUTTON1_MASK);
		myData.bShowMenuPending = FALSE;
	}
	
	cairo_dock_discard_task (myData.pTask);
	myData.pTask = NULL;
	
	CD_APPLET_LEAVE (TRUE);
}

static void _free_shared_memory (CDSharedMemory *pSharedMemory)
{
	g_free (pSharedMemory->cDisksURI);
	g_free (pSharedMemory->cNetworkURI);
	g_free (pSharedMemory->cBookmarksURI);
	g_list_foreach (pSharedMemory->pIconList, (GFunc)g_free, NULL);
	g_list_free (pSharedMemory->pIconList);
	g_free (pSharedMemory);
}
void cd_shortcuts_start (CairoDockModuleInstance *myApplet)
{
	if (myData.pTask != NULL)
	{
		cairo_dock_discard_task (myData.pTask);
		myData.pTask = NULL;
	}
	
	CDSharedMemory *pSharedMemory = g_new0 (CDSharedMemory, 1);
	pSharedMemory->bListDrives = myConfig.bListDrives;
	pSharedMemory->bListNetwork = myConfig.bListNetwork;
	pSharedMemory->bListBookmarks = myConfig.bListBookmarks;
	pSharedMemory->pApplet = myApplet;
	
	myData.pTask = cairo_dock_new_task_full (0,
		(CairoDockGetDataAsyncFunc) cd_shortcuts_get_shortcuts_data,
		(CairoDockUpdateSyncFunc) cd_shortcuts_build_shortcuts_from_data,
		(GFreeFunc) _free_shared_memory,
		pSharedMemory);

	if (cairo_dock_is_loading ())
		cairo_dock_launch_task_delayed (myData.pTask, 0); // 0 <=> g_idle
	else
		cairo_dock_launch_task (myData.pTask);
}

