/*
 * Calcurse - text-based organizer
 * Copyright (c) 2004-2005 Frederic Culot
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 *
 * Send your feedback or comments to : calcurse@culot.org
 * Calcurse home page : http://culot.org/calcurse
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <time.h>

#include "event.h"


struct event_s *eventlist;

struct event_s *event_new(char *mesg, long start, long dur)
{
	struct event_s *o, **i;
	o = (struct event_s *) malloc(sizeof(struct event_s));
	o->mesg = (char *) malloc(strlen(mesg) + 1);
	strcpy(o->mesg, mesg);
	o->start = start;
	o->dur = dur;
	i = &eventlist;
	for (;;) {
		if (*i == 0 || (*i)->start > start) {
			o->next = *i;
			*i = o;
			break;
		}
		i = &(*i)->next;
	}
	return o;
}

unsigned event_inday(struct event_s *i, long start)
{
	if (i->start <= start + 3600 * 24 && i->start + i->dur > start) {
		return 1;
	}
	return 0;
}

void event_sec2str(struct event_s *o, long day, char *start, char *end)
{
	struct tm *lt;
	time_t t;

	if (o->start < day) {
		strcpy(start, "..:..");
	} else {
		t = o->start;
		lt = localtime(&t);
		snprintf(start, HRMIN_SIZE, "%02u:%02u", lt->tm_hour,
			 lt->tm_min);
	}
	if (o->start + o->dur > day + 24 * 3600) {
		strcpy(end, "..:..");
	} else {
		t = o->start + o->dur;
		lt = localtime(&t);
		snprintf(end, HRMIN_SIZE, "%02u:%02u", lt->tm_hour,
			 lt->tm_min);
	}
}


long date2sec(unsigned year, unsigned month, unsigned day, unsigned hour,
	      unsigned min)
{
	struct tm start, *lt;
	time_t tstart, t;

	t = time(NULL);
	lt = localtime(&t);
	start = *lt;

	start.tm_mon = month;
	start.tm_mday = day;
	start.tm_year = year;
	start.tm_hour = hour;
	start.tm_min = min;
	start.tm_sec = 0;
	start.tm_isdst = -1;
	start.tm_year -= 1900;
	start.tm_mon--;
	tstart = mktime(&start);
	if (tstart == -1) {
		fputs("date2sec problem\n", stderr);
		fprintf(stderr, "%u %u %u %u %u\n", year, month, day, hour,
			min);
		exit(1);
	}
	return tstart;
}

/* 
 * Returns the date in seconds from year 1900.
 * If no date is entered, current date is chosen.
 */
long
get_sec_date(int year, int month, int day)
{
	struct tm *ptrtime;
	time_t timer;
	long long_date;
	char current_day[3], current_month[3] ,current_year[5];

	if (year == 0 && month == 0 && day == 0) {
		timer = time(NULL);
		ptrtime = localtime(&timer);
		strftime(current_day, 3, "%d", ptrtime);
		strftime(current_month, 3, "%m", ptrtime);
		strftime(current_year, 5, "%Y", ptrtime);
		month = atoi(current_month);
		day = atoi(current_day);
		year = atoi(current_year);
		
	} 
	long_date = date2sec(year, month, day, 0, 0);
	return long_date;
}

long min2sec(unsigned minutes)
{
	return minutes * 60;
}


void event_write(struct event_s *o, FILE * f)
{
	struct tm *lt;
	time_t t;

	t = o->start;
	lt = localtime(&t);
	fprintf(f, "%02u/%02u/%04u @ %02u:%02u",
		lt->tm_mon + 1, lt->tm_mday, 1900 + lt->tm_year,
		lt->tm_hour, lt->tm_min);

	t = o->start + o->dur;
	lt = localtime(&t);
	fprintf(f, " -> %02u/%02u/%04u @ %02u:%02u |%s\n",
		lt->tm_mon + 1, lt->tm_mday, 1900 + lt->tm_year,
		lt->tm_hour, lt->tm_min, o->mesg);
}

struct event_s *event_scan(FILE * f)
{
	struct tm start, end, *lt;
	char buf[MESG_MAXSIZE], *nl;
	time_t tstart, tend, t;

	t = time(NULL);
	lt = localtime(&t);
	start = *lt;
	end = *lt;

	if (fscanf(f, "%u / %u / %u @ %u : %u -> %u / %u / %u @ %u : %u |",
		   &start.tm_mon, &start.tm_mday, &start.tm_year,
		   &start.tm_hour, &start.tm_min,
		   &end.tm_mon, &end.tm_mday, &end.tm_year,
		   &end.tm_hour, &end.tm_min) != 10) {
		fputs("syntax error in the event\n", stderr);
		exit(1);
	}

	fgets(buf, MESG_MAXSIZE, f);
	nl = strchr(buf, '\n');
	if (nl) {
		*nl = '\0';
	}
	start.tm_sec = end.tm_sec = 0;
	start.tm_isdst = end.tm_isdst = -1;
	start.tm_year -= 1900;
	start.tm_mon--;
	end.tm_year -= 1900;
	end.tm_mon--;

	tstart = mktime(&start);
	tend = mktime(&end);
	if (tstart == -1 || tend == -1 || tstart > tend) {
		fputs("date error in the event\n", stderr);
		exit(1);
	}
	return event_new(buf, tstart, tend - tstart);
}


void event_delete_bynum(long start, unsigned num)
{
	unsigned n;
	struct event_s *i, **iptr;

	n = 0;
	iptr = &eventlist;
	for (i = eventlist; i != 0; i = i->next) {
		if (event_inday(i, start)) {
			if (n == num) {
				*iptr = i->next;
				free(i->mesg);
				free(i);
				return;
			}
			n++;
		}
		iptr = &i->next;
	}
	/* not reached */
	fputs("event_delete_bynum: no such event_s\n", stderr);
	exit(1);
}

/* checks if a new appointment time format is ok, should be HH:MM */
int check_event_time(char *string)
{
	int ok = 0;
	char hour[] = "  ";
	char minutes[] = "  ";

	if (			//format test
		   (strlen(string) == 5) &
		   (isdigit(string[0]) != 0) &
		   (isdigit(string[1]) != 0) &
		   (isdigit(string[3]) != 0) &
		   (isdigit(string[4]) != 0) & (string[2] == ':')
	    ) {			// check if we have a valid time
		strncpy(hour, string, 2);
		strncpy(minutes, string + 3, 2);
		if ((atoi(hour) <= 24) & (atoi(hour) >=
					  0) & (atoi(minutes) <
						60) & (atoi(minutes) >= 0))
			ok = 1;
	}
	return ok;
}
