/*
 * Calcurse - text-based organizer
 * Copyright (c) 2004-2005 Frederic Culot
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 *
 * Send your feedback or comments to : calcurse@culot.org
 * Calcurse home page : http://culot.org/calcurse
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <time.h>

#include "todo.h"
#include "event.h"
#include "io.h"
#include "vars.h"

/* 
 * Initialization of data paths. The argument cfile is the variable
 * which contains the calendar file. If none is given, then the default
 * one (~/.calcurse/apts) is taken. If the one given does not exist, it
 * is created.
 */
void
io_init(char *cfile)
{
	FILE *data_file;
	char *home;
	char apts_file[MAX_LENGTH] = "";
	int ch;

	home = getenv("HOME");
	if (home == NULL) {
		home = ".";
	}
	snprintf(path_dir, MAX_LENGTH, "%s/" DIR_NAME, home);
	snprintf(path_todo, MAX_LENGTH, "%s/" TODO_PATH, home);
	snprintf(path_conf, MAX_LENGTH, "%s/" CONF_PATH, home);
	if (cfile == NULL) {
		snprintf(path_apts, MAX_LENGTH, "%s/" APTS_PATH, home);
	} else {
		snprintf(apts_file, MAX_LENGTH, "%s", cfile);
		strncpy(path_apts, apts_file, MAX_LENGTH);
		/* check if the file exists, otherwise create it */
		data_file = fopen(path_apts, "r");
		if (data_file == NULL) {
			printf("%s does not exist, create it now [y or n] ? ", path_apts);
			ch = getchar();
			switch (ch) {
				case 'N':
				case 'n':
					printf("aborting...\n");
					exit(1);
					break;

				case 'Y':
				case 'y':
					data_file = fopen(path_apts, "w");
					if (data_file == NULL) {
						perror(path_apts);
						exit(1);
					} else {
						printf("%s successfully created\n",path_apts);
						printf("starting interactive mode...\n");
					}
					break;

				default:
					printf("aborting...\n");
					exit(1);
					break;
			} 
		}
		fclose(data_file);
	}
}

/* Save the calendar data */
void
save_cal(char *auto_save, char *confirm_quit, char *confirm_delete,
	 int colr, int layout)
{
	FILE *data_file;
	struct event_s *j;
	struct todo_s *i;
	char config_txt[] =
	    "#\n# Calcurse configuration file\n#\n# This file sets the configuration options used by Calcurse. These\n# options are usually set from within Calcurse. A line beginning with \n# a space or tab is considered to be a continuation of the previous line.\n# For a variable to be unset its value must be blank.\n# To set a variable to the empty string its value should be \"\".\n# Lines beginning with \"#\" are comments, and ignored by Calcurse.\n";



	//save the todo data file
	data_file = fopen(path_todo, "w");
	if (data_file == (FILE *) 0)
		mvprintw(1, 1, "Problems accessing data file ...");
	else {
		for (i = todolist; i != 0; i = i->next)
			fprintf(data_file, "%s\n", i->mesg);
		fclose(data_file);
	}

	//save the apts data file
	data_file = fopen(path_apts, "w");
	if (data_file == (FILE *) 0)
		mvprintw(1, 1, "Problems accessing data file ...");
	else {
		for (j = eventlist; j != 0; j = j->next)
			event_write(j, data_file);
		fclose(data_file);
	}

	//save the user configuration
	data_file = fopen(path_conf, "w");
	if (data_file == (FILE *) 0)
		mvprintw(1, 1, "Problems accessing data file ...");
	else {
		fprintf(data_file, "%s\n", config_txt);

		fprintf(data_file,
			"# If this option is set to yes, automatic save is done when quitting\n");
		fprintf(data_file, "auto_save=\n");
		fprintf(data_file, "%s\n", auto_save);

		fprintf(data_file,
			"\n# If this option is set to yes, confirmation is required before quitting\n");
		fprintf(data_file, "confirm_quit=\n");
		fprintf(data_file, "%s\n", confirm_quit);

		fprintf(data_file,
			"\n# If this option is set to yes, confirmation is required before deleting an event\n");
		fprintf(data_file, "confirm_delete=\n");
		fprintf(data_file, "%s\n", confirm_delete);

		fprintf(data_file,
			"\n# This is the color theme used for menus (1 to 8) :\n");
		fprintf(data_file, "color-theme=\n");
		fprintf(data_file, "%d\n", colr);

		fprintf(data_file,
			"\n# This is the layout of the calendar (1 to 4) :\n");
		fprintf(data_file, "layout=\n");
		fprintf(data_file, "%d\n", layout);
		fclose(data_file);
	}


	//print message
	erase_status_bar();
	attron(COLOR_PAIR(colr));
	mvprintw(row - 2, 0, "The data files were successfully saved");
	mvprintw(row - 1, 0, "Press [Enter] to continue");
	attroff(COLOR_PAIR(colr));
	getch();
	erase_status_bar();
}

  /* Load the Appointment data */
void load_app()
{
	FILE *data_file;
	int c;

	data_file = fopen(path_apts, "r");
	for (;;) {
		c = getc(data_file);
		if (c == EOF)
			break;
		ungetc(c, data_file);
		event_scan(data_file);
	}
	fclose(data_file);
}

  /* Load the todo data */
int
load_todo()
{
	FILE *data_file;
	char mesg[] = "failed to open todofile";
	int i, nb_tod = 0;
	char buf[100], e_todo[100];

	data_file = fopen(path_todo, "r");
	if (data_file == NULL) {
		cal_error(mesg);
	}
	for (;;) {
		if (fgets(buf, 99, data_file) == NULL) {
			break;
		}
		extract_data(e_todo, buf, strlen(buf));
		todo_add(e_todo);
		++nb_tod;
	}
	fclose(data_file);
	return nb_tod;
}




/* Checks if data files exist. If not, create them */
void
check_data_files(int colr)
{
	FILE *data_file;
	int no_data_file;
	char bufstr[2];

	no_data_file = 0;
	erase_status_bar();
	/* creates the calcurse repertory if not present */
	mkdir(path_dir, 0700);

	data_file = fopen(path_todo, "r");
	if (data_file == NULL) {
		no_data_file++;
		data_file = fopen(path_todo, "w");
		if (data_file == NULL) {
			perror(path_todo);
			exit(1);
		}
	}
	fclose(data_file);

	data_file = fopen(path_apts, "r");
	if (data_file == NULL) {
		no_data_file++;
		data_file = fopen(path_apts, "w");
		if (data_file == NULL) {
			perror(path_apts);
			exit(1);
		}
	}
	fclose(data_file);

	data_file = fopen(path_conf, "r");
	if (data_file == NULL) {
		no_data_file++;
		data_file = fopen(path_conf, "w");
		if (data_file == NULL) {
			perror(path_conf);
			exit(1);
		}
	}
	fclose(data_file);

	attron(COLOR_PAIR(colr));
	if (no_data_file != 0) {
		mvprintw(row - 2, 0,
			 "Welcome to Calcurse. Missing data files were created.");
		mvprintw(row - 1, 0, "Press [Enter] to continue");
	} else {
		mvprintw(row - 2, 0,
			 "Data files found. Data will be loaded now.");
		mvprintw(row - 1, 0, "Press [Enter] to continue");
	}
	attroff(COLOR_PAIR(colr));
	getch();
}
