/*
 * Caudium - An extensible World Wide Web server
 * Copyright  2000-2004 The Caudium Group
 * Based on IMHO  Stefan Wallstrm and Bosse Lincoln.
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */
/*
 * $Id: camas.pike,v 1.428.2.16 2004/04/14 09:50:06 vida Exp $
 */

#include <module.h>
inherit "module";
inherit "caudiumlib";

#include <camas/globals.h>      // Global definitions
#include <camas/smtp.h>         // SMTP client
#include <camas/maindefvars.h>  // Defaults values for defvar()
#include <camas/utf.h>          // UTF8 conversion macros
#include <camas/msg.h>          // MSG() macros
#include <camas/pmods.h>        // Local or normal camas pike modules

// This is the real Camas version. It should changed before each
// release by make bump_version
constant __camas_version__ = "1.2";
constant __camas_build__ = "21";
constant __camas_state__ = "RC4";

constant camas_version = __camas_version__+"."+__camas_build__+"-"+__camas_state__;

//
//! module: CAMAS: Main module
//!  Caudium Mail Access System, the official Webmail for Caudium Webserver
//! inherits: module
//! inherits: caudiumlib
//! type: MODULE_LOCATION|MODULE_PROVIDER
//! cvs_version: $Id: camas.pike,v 1.428.2.16 2004/04/14 09:50:06 vida Exp $
//

constant cvs_version   = "$Id: camas.pike,v 1.428.2.16 2004/04/14 09:50:06 vida Exp $";
constant module_type   = MODULE_LOCATION|MODULE_PROVIDER;
constant module_name   = "CAMAS: Main module ("+camas_version+")";
constant module_doc    = "CAudium Mail Access System, the official Webmail "
                         "for Caudium Webserver <br />"
                         "<i>CAMAS Version : </i>"+camas_version+".<br />"
                         "For setting up your IMAP server hostname, please go in a "
                         "CAMAS auth module.<br />"
                         "To enable debug mode, start Caudium with -DCAMAS_DEBUG.<br/>"
                         "Please see Status for more informations.";
constant thread_safe   = 1;
constant module_unique = 1;

#ifdef CAMAS_DEBUG
// is debug is defined, then don't load IMAPclient as a Pike module but in the Camas Main module
// so that we don't need to restart Caudium each time we modify it
program imapclient;
#endif

int global_kludges = 0, global_ok = 0; //oliv3: until we are sure the XML Parser has no problems anymore
mixed gc_call_out = 0;

int starttime = 0;

// is it the first time someone hit my mountpoint ?
int firststart = 1;

// a list of mailboxes not to reload
array(string) notreloadmailboxes;

// the current session module we use, currently can be "123sessions" or "gsession"
string session_module;

// These properties are saved and loaded for each user. To add a new, change
// here and in the SETUP menu and in prefprop_att below.
mapping prefproperties=([
                          "name" : "",
                          "address" : "",
                          "mailpath" : "",
                          "replymsgprefix" : "> ",
                          "extraheader" : "",
                          "signature" : "",
                          "autologout" : "20",
                          "visiblemail" : "20",
                          "sortorder" : "forward",
                          "sortcolumn" : "num",
                          "trashfolder" : "Trash",
                          "sentfolder" : "sent-mail",
                          "draftsfolder" : "Drafts",
                          "answeredfolder" : "answered-mail",
                          "autobcc" : "",
                          "bcc2admin": "",
                          "filterbook" : "",
                          "language" : "",
                          "addressbook" : "",
                          "organization" : "",
                          "layout": "Default",
                          "delaytime" : "1440", /*
                              this is 24*60 (default from bertrand)
                              warning: at use time, remember to cast this to an int
                              because the prefs are stored as strings

                              --oliv3
                                  */

                          "delaycalc" : "linear",
                          // oliv3: these properties _MUST_ be queried with
                          // QUERY (x)=="1" or QUERY (x)=="0",
                          // *NOT* with QUERY (x) and !QUERY (x)
                          "replyincludemsg" : "1",
													"replyontop" : "0",
                          "saveattachments" : "0",
                          "showhiddenheaders": "1",
                          "showhtml": "0",
                          "autofilter": "1",
                          "showtext": "1",
                          "showlinkforalternatives": "1",
                          "showinlineimages": "1",
                          "blindsend": "1",
                          "defaultpostloginscreen": "mailindex",
                          "feedimapcache": "0",
                        ]);

mapping (string:array) prefprop_att = ([
                                         /* These are the attribute used to generate the customization interface
                                          * in Caudium. This mapping should have the same keys as prefproperties above
                                          * and those the values should be a 3 (at least) element array :
                                          * ({ title, type, help [, stringlist] })
                                          * where title is a string, indexing a message in the configuration menus,
                                          * type is the type of widget used (an int : TYPE_INT, TYPE_STRING...) and
                                          * help is the html text displayed as an help message in those menus.
                                          * stringlist is a list of choices (for TYPE_STRING_LIST).
                                          *
                                          * !! These are not defvar()
                                          * They are defvar()'ed later in the create() function and can be accessed
                                          * as it with QUERY(pref_title)
                                          */
                                         "name"             : ({ "Name", TYPE_STRING, "no help available" }),
                                         "address"          : ({ "Mail address", TYPE_STRING, "The mail address." }),
                                         "mailpath"         : ({ "Mail searchpath", TYPE_STRING, "no help available" }),
                                         "replymsgprefix"   : ({ "Prefix", TYPE_STRING,
                                                                 "The bit of text prepended to each line of an "
                                                                 "included message text into a reply." }),
                                         "replyincludemsg"  : ({ "Include mail when replying", TYPE_FLAG,
                                                                 "Toggles the auto inclusion of a message text into a reply." }),
																				 "replyontop"				: ({ "Reply on top", TYPE_FLAG,
																				 													"Reply on top of the replied message. Enabling this is not netiquette compliant." }),
                                         "extraheader"      : ({ "Extra headers", TYPE_STRING,
                                                                 "Site-wide extra headers to be placed in each outgoing mails<br />"
                                                                 "Non-standard headers might begin with  \"X-\"<br />"
                                                                 "You can use the following replacements:"
                                                                 "<ul>"
                                                                 "<li>$USER: login of the user</li>"
                                                                 "<li>$VERSION: CAMAS version</li>"
                                                                 "</ul>"}),
                                         "signature"        : ({ "Signature", TYPE_TEXT_FIELD,
                                                                 "Site-wide signature/disclaimer to be placed in bottom of each outgoing mail" }),
                                         "autologout"       : ({ "Inactive logout (minutes)", TYPE_INT,
                                                                 "no help available" }),
                                         "visiblemail"      : ({ "Messages shown in mailbox", TYPE_INT,
                                                                 "Number of messages displayed in each page" }),
                                         "sortorder"        : ({ "Mail sort order", TYPE_STRING_LIST,
                                                                 "no help available", ({ "forward", "backward" }) }),
                                         "trashfolder"      : ({ "Trash folder", TYPE_STRING, "no help available" }),
                                         "sentfolder"       : ({ "Sent mail folder", TYPE_STRING, "no help available" }),
                                         "draftsfolder"     : ({ "Drafts folder", TYPE_STRING, "no help available" }),
                                         "answeredfolder"   : ({ "Answered folder", TYPE_STRING, "no help available" }),
                                         "autobcc"          : ({ "Default Bcc address", TYPE_STRING, "This email address will receive silently all mail from users" }),
                                         "bcc2admin"        : ({ "BCC to admin", TYPE_STRING, "Users can have a checkbox letting them send a BCC to this email address in the compose screen" }),
                                         "saveattachments"  : ({ "Include attachments", TYPE_FLAG, "no help available" }),
                                         "language"         : ({ "Language", TYPE_STRING, "no help available" }),
                                         "layout"           : ({ "User interface", TYPE_STRING, "The layout to use for the user interface, see CAMAS: layout manager module for more information." }),
                                         "showhtml"         : ({ "Show HTML messages", TYPE_FLAG, "Show HTML messages." }),
                                         "showtext"         : ({ "Show text messages", TYPE_FLAG, "Show text messages." }),
                                         "showlinkforalternatives" : ({ "Show link for alternatives", TYPE_FLAG, "Show link for accessing other parts of multipart/alternative messages" }),
                                         "showinlineimages" : ({ "Show inline images", TYPE_FLAG, "Show images directly in the readmail pages. Otherwise, shows a link for it. Only images types specified by the features module can be displayed." }),
                                         "autofilter"       : ({ "Auto filter mail", TYPE_FLAG, "no help available" }),
                                         "organization"     : ({ "Organization", TYPE_STRING, "Organization" }),
                                         "delaytime"        : ({ "Delay time slices", TYPE_INT, "Delay time slices"}),
                                         "delaycalc"        : ({ "Delay calculation mode", TYPE_STRING_LIST, "Delay calculation mode", ({ "linear", "dichotomic" }) }),
                                         "blindsend"        : ({ "Blind sending", TYPE_FLAG, "For help, please see CAMAS: Features -> Features -> User can use 'blind sending'" }),
                                         "defaultpostloginscreen": ({ "Default post login screen", TYPE_MULTIPLE_STRING,
                                             "Default login screen after the user successfully authentificated. "
                                             "Usually it is mailindex but other values like compose or folderlist can be usefull.",
                                             ({ "mailindex", "compose", "folderlist", "setup", "files", "searchmail" }) }),
                                         "feedimapcache"    : ({ "Start the background IMAP client at startup", TYPE_FLAG, "For help, please see CAMAS: Features -> Features -> User can enable/disable background IMAP client." }) 
                                       ]);

// inline display, oliv3 FIXME
multiset (string) image_types = (< >);

void init_image_types () {
  object features = my_configuration ()->get_provider ("camas_features");
  if (objectp (features)) {
    image_types = (< >);
    string q = features->QUERY (displayinlineimages);
    if (sizeof (q))
      foreach (q/"\n", string i)
      if (sizeof (i)) {
        string imgtype = lower_case (i);
        if (!image_types[imgtype])
          image_types[imgtype] = 1;
      }
  }
}

#if constant(thread_create)
// Lock for sessions
object global_lock = Thread.Mutex();
#endif

int camas_feature (int f) {
  object features = my_configuration ()->get_provider ("camas_features");
  if (objectp (features))
    return features->feature (f);
}

string server_url (object id, void|int no_prestate) {
  string s1, s2, server_url = "";
  int port = 443;

  if (sizeof (QUERY (urloverride)) > 0)
    return QUERY (urloverride);

  if (QUERY (ssl3redir) && id->conf->server_ports)
  {
    if ((!QUERY (sslfaulty)) || (!id->supports->ssl_faulty)) 
    {
      CDEBUG("SSL enabled...\n");
      foreach (indices (id->conf->server_ports), string s) 
      {
        if (sscanf (s, "ssl3://%s/%s", s1, s2) == 2) 
	{
          sscanf (s1, "%s:%d", s1, port);
          if (s1 == "ANY") 
	  {
            // FIXME: does caudium work w.o. gethostname()?
            // Yeah, but I don't of of a case where it would happen,
            // at least not on UNIX. This test is safe in any case. -- david
#if constant(gethostname)
            s1 = gethostname ();
#else
            s1 = "localhost";
#endif
          }
          server_url = "https://" + s1 + (port == 443 ? "" : (":" + port));
        }
      }
    }
    else 
    {
      CDEBUG("SSL disabled...\n");
      server_url = id->conf->query ("MyWorldLocation");
      server_url = server_url[0..sizeof (server_url)-2];
    }
  }

  if ((!id->cookies["SessionID"] || !sizeof (id->cookies["SessionID"])) && 
      (!no_prestate) && id->misc->session_id) 
  {
    server_url += "/(SessionID=" + id->misc->session_id;
    if (sizeof (id->prestate) > 1) {
      foreach (indices (id->prestate), string p)
        if (p[0..8] != "SessionID")
          server_url += "," + p;
    }
    server_url += ")";
  }

  CDEBUG("server_url: '" + server_url + "'\n");

  return server_url;
}

mapping redirect (object id, string nextpage) {
  string redir = server_url (id, 1) + QUERY (location);
  redir += (nextpage == "" ? "" : ("/" + nextpage));
  return HTTP_REDIRECT (redir, id);
}

void get_files(object sessobj) {
  if(sizeof(QUERY (uploaddir))) 
  {
    array (string) files;
    array (int) filestat;
    files=get_dir(QUERY (uploaddir));
    if(files) {
      string fname;
      foreach(files,string file){
        if(sscanf(file,lower_case(sessobj["login"])+"_%s",fname)) {
          if (!has_value (fname, "_")) {
            fname=CAMAS.Tools.decode_fname(fname);
            filestat=file_stat(QUERY (uploaddir)+"/"+file);
sessobj["files"]+= ({ ([ "fname":fname, "size":filestat[1], "type": CAMAS.Tools.filetype(fname, my_configuration())  ]) });
          }
        }
      }
    }
  }
}

//
// The Session class
//

class _Session {
    
  int ok = 0; // is the user has to be here ?
  string session = ""; // id for this session
  string nexturl = "";
  string lastaction = ""; // the last user's action (like compose, mailindex, ...)
  
  string login = ""; // login name
  string passwd = ""; // the password

  int administrator = 0; // user is an administrator
  
  string saved_state = 0; // used for pass-through login

  object lock;

  string ip; // user logged in from this ip

  object imapclient; // the IMAP and SMTP client object
  object imapcachefeeder; // the IMAP cache feeder
  string imapserver;
  int imapport = 143;
  int reqtrycreate = 0; // require [TRYCREATE] hints to create folders?
  int mboxencode = 1; // mailbox names encoded in modified UTF-7 encoding?
  string hidemboxes; // prefix for mailboxes that should be hidden
  int showprefsbox = 0; // should the preferences mailbox be shown?
  int imapidlelogout; // disconnect after this many seconds if there is no IMAP traffic.
  int connected = 0; // are we connected to the IMAP server ?
  int nb_mailindex_visits = 0; // number of visits in the mailindex, since Camas 1.2.7 we allow 
  // you not to scan a folder every time the user goes in mailindex, this counter is used for this,
  // see also feature scan_folders_call_out

  int cachefeeded = 0; // is the cache of the background IMAP fetcher feeded ?
  mixed dummy = 0; // dummy temporary object for the IMAP cache feeder

  string maildomain = "";
  string domain = ""; // domain name of the user, used indirectly by Auth modules

  string sharedpath = "";
  int shareddepth = 2;

  string overridelayout;

  string mailboxstyle;

  string smtpserver = "localhost";
  int smtpport = 25;

  int logintime = 0;
  int lastusage = 0; // last access using this session

  // login error, some auth modules provides it, not all
  string loginerror;

  int loadfiles = 0; // flag that attachments should be listed

  int status = LOGINPROMPT; // active screen

  object cmail = 0; // current mail
  int cmailidx = 0;
  int cmailuid = 0;

  mapping quota = ([ ]); // the quota, the format is :
  // ([ 
  //    "rootname": 
  //   ([ 
  //      "STORAGE": ({ 0, 56 })
  //   ]),
  //   ([ 
  //      "MESSAGE": ({ 5, 100 })
  //   ])
  // ])
  mapping mailbox2rootname = ([ ]); // map root name to mailbox name for the quotas

  string messageid;
  array messageids;
  array uids;

  int mailheadersloaded = 0;

  object prefsmailobj;
  // the full path to the preferences folder 
  string prefsbox;
  // the preferences folder 
  string baseprefsbox;
  array prefsmails = ({ });

  mapping lcfgmap = ([ ]);
  int filters_have_changed = 1;
  int foldershavechanged = 1;
  int allfolderssize = 0; // size of all the folders
  mapping cache = ([ ]);
  array folderstoscan = ({ });
  mapping nodefoldersinfo = ([ ]);
  mapping(string:array(int)) foldersinfo = ([ ]); // the cache for folders
  int foldershavechange;
  // used when renaming a folder
/*  string newfoldername;
    string oldfoldername;*/
  int exists_old; // temporary variable that contains the number of message in a given mailbox, used by imapclient
  int uidnext_old; // same as exists_old except it is for next uid in the mailbox

  string mailssortcolumn = ""; // the current column on which the IMAP client is sorting the mails 
  string currentcolumn; // current column, used for sorting
  int sortordertoggled; // does the user changed the sort order ?
  int currentpage; // if there are several pages in MAILINDEX, this contains the page where we are.
  int firstpage; // the first page number to display in the navbar
  int lastpage; // the last page to display in the navbar
  
  int firstvisiblemail = -1; // Initialised after in mailindex
  int lastvisiblemail = 0;
  int hasnewmail = 0;

  int utf8 = 0; // client supports utf-8 encoding?
  int cookies = 1; // client accepts cookies?
  string charset = "iso-8859-1"; // charset currently in use
  
  mapping lang_progs; // language modules

  array mailbox = ({ "INBOX", "INBOX", MB_NOINFERIORS, 0, ({ "INBOX" }) }); // current mailbox
  array mails = ({ }); // mail in current mailbox
  array mailboxes = ({ }); // all mailboxes format: ({ foldername, displayname, flags, separator, foldername/separator })

  array activemailboxinfo = ({ });

  string usernamemethod;
  string namedb;

  object extendedabook; // the extended address book object 
  object ebookmailobj; // same as extendedabook but decoded
  array ebookuids; // IMAP uids for the address book on IMAP server
  array(mapping(int:array(string))) addrbook_contents = ({ }); // contents of the current address book
  string selectedaddrbook; // the current addressbook name
  array(array(string)) imported_addrbook; // the imported address book is loaded into memory 
  string typeofabook; // the type of the imported address book, example: "csv"
  // before going into one of the addressbooks (see AddressBook2.pmod)
  int administrator_addressbook; // if set to yes, the address book permissions from the CIF
  // will be overwritten and the user will have write access 

  array files = ({ }); // stored attachments
  
  int prefsloaded = 0; // user preferences has been loaded

  int debug = 0; // debug output on for this session?
  
  int showheaders = 0;

  string buttonmode = ""; // the button mode of the current layout

  // folders names
  string defaultsentfolder = "";
  string defaulttrashfolder = "";
  string draftsfolder = "";
  string answeredfolder = "";

  int ldap = 0;
  string ldapserver = "";
  string ldapsearchroot ="";
  string ldapadd;
  
  int wrapcolumn = 70;

  string displayerrors;

  string sessionsortorder;
  string sessionsortcolumn = "num";

  string searchstring = "";
  int nothingfound = 0; // does the previous search was sucessfull or not ?
  int stickyfilter; // is it a sticky search filter (search that we repeat when we changed mailboxes)
  int historysearch; // is it a history search ?

  array prefuids = ({ });

  array deletemboxes = ({ });
  array deletemails = ({ });

  string copytobox;

  // send mail
  string sendmail;
  string sentmaildata;
  array sendrecipients;
  int dsnsuccess;
  int dsndelay;

  // mailboxes
  string lastpath = "";

  // compose
  string from; // currently selected from address
  string to;
  mapping(string:string) tos;
  string cc;
  string bcc;
  string subject;
  string message;
  string originalmessage; // message of the mail without the '>' and the like
  array attachments = ({ });
  string replytocharset;
  int userblindsend = 1;
  int mdn; // do the user enable MDN ?
  string in_reply_to; // in-reply-to message-id

  // address book & filter screen
  string recipientfield;
  string editaddressmode;
  string filterbook = "";
  string editaddressfiltermode;
  string newaddressfilter;
  // the address to edit in the EDITADDRESS screen
  int|string editaddress;
  // the index in the list to edit in EDITADDRESS screen
  int editaddressfilter;
  // the name to add in the EDITADDRESS screen when we add an address from READMAIL
  int|string editname;

  // dialogs
  array dialogstrings;
  array dialogactions;
  string dialogtext;
  string dialogtarget;
  array(string) dialogoptions;
  int displayimaperrors = 0;
  int displaysmtperrors = 0;
  int displayaddrbookerrors = 0;

  // spell checker
  array spelling;
  array misspelled;
  int checkword = 0;
  string ispelldict;

  // Default prefs for users
  string name = "John Doe";
  string address = "";
  string mailpath = "";
  string replymsgprefix = "> ";
  string replyincludemsg = "yes";
	string replyontop = "0";
  string extraheader = "";
  string signature = "";
  string autologout = "20";
  string visiblemail = "20";
  string sortorder = "forward";
  string sortcolumn = "num";
  string trashfolder = "Trash";
  string sentfolder = "sent-mail";
  string autobcc = "";
  string saveattachments = "no";
  string language = 0; // selected language
  string addressbook = "";
  string layout = "Default";
  string showhiddenheaders = "yes";
  string activemailboxes = "INBOX";
  string draftmail = "";
  string showhtml = "1";
  string blindsend = "0";
  string organization = "";
  string autofilter = "1";
  string delaycalc = "";
  string delaytime = "0";
  string bcc2admin = "";
  string showtext = "1";
  string showinlineimages = "1";
  string showlinkforalternatives = "1";
  string defaultpostloginscreen = "mailindex"; 
  string feedimapcache = "0";

  // Default values for setup user _can_ change
  int usersetup = 0;
  int usersetupautobcc = 0;
  int usersetuptrashfolder = 0;
  int usersetupmailpath = 0;
  int usersetuplanguage = 0;
  int usersetupsignature = 0;
  int usersetupreplyincludemsg = 0;
  int usersetupsaveattachments = 0;
  int usersetupaddress = 0;
  int usersetupdraftmail = 0;
  int usersetupactivemailboxes = 0;
  int usersetupactiveinboxes = 0;
  int usersetupaddressbook = 0;
  int usersetupshowhiddenheaders = 0;
  int usersetuplayout = 0;
  int usersetupsortcolumn = 0;
  int usersetupsortorder = 0;
  int usersetupautologout = 0;
  int usersetupextraheader = 0;
  int usersetupsentfolder = 0;
  int usersetupname = 0;
  int usersetupshowhtml = 0;
  int usersetupvisiblemail = 0;
  int usersetupreplymsgprefix = 0;
	int usersetupreplyontop = 1;
  int usersetupdelaytime = 1;
  int usersetupdelaycalc = 1;
  int usersetupshowlinkforalternatives = 1;
  int usersetupshowinlineimages = 1;
  int usersetupblindsend = 0;
  int usersetuporganization = 0;
  int usersetupfilterbook = 1;
  int usersetupansweredfolder = 1;
  int usersetupdraftsfolder = 1;
  int usersetupautofilter = 1;
  int usersetupbcc2admin = 0;
  int usersetupshowtext = 1;
  int usersetupdefaultpostloginscreen = 1;
  int usersetupfeedimapcache = 0;
  
  // various temporary variables used in camas actions
  int replytouid = -1;
  int replytoidx = -1;
  int forwarduid = -1;
  int forwardidx = -1;
  int draftidx = -1;
  int draftuid = -1;
  // folder name for the send action (modified by eg forward, compose, reply,...)
  string sentfolderaction;

  array selected;
  array selectedfiles;
  array selectedattachments;
  int usermove;

  // used by logger module
  // the total size of all the email data the user sent 
  // in a given session
  int totalsentmaildata;
  // the number of messages the user sent in a given session
  int totalmsgsent;
  // the to, cc, subject, date and messageid
  // field the user fill in for the sent mail.
  // Each entry in these arrays equals to one mail sent
  array(string) userstat_to = ({ }), userstat_cc = ({ }), 
    userstat_subject = ({ }), userstat_replydate = ({ }), 
    userstat_replymessageid = ({ }), userstat_date = ({ });
  
  // runtime admin interface
  array users = ({ });
  int showfrom;
  int adminsortcolumn;
  int sortup;
  string screen; // current screen of each users
  // used in cont_imho_text
  string face;
  int size;
  // to be deprecated, only used in IMHO screens
  int notopbuttons;
  int nobottombuttons;
  int nobackbutton;
  int mixedtopbuttombuttons;
  string columns;
  string title; // title of the current screen
#ifdef CAMAS_DEBUG
  // debug variables to debug log
  array(string) debug_variables;
  int debug_variables_get;
  int debug_variables_set;
#endif

  mapping(string:string) _dump_me(object o, int loop) 
  {
    mapping dump = ([ ]);
    array(string) inds = indices(o);
    foreach(inds, string ind)
    {
      if(ind == "compat")
        continue;
      mixed value = o[ind];
      if(objectp(value) && value != o && 
          ind != "dump_me" && ind != "_dump_me"
          && ind != "id")
      {
        if(loop > 10)
          value = "infinite loop";
        else
          value = _dump_me(value, ++loop);
      }
      dump += ([ ind: value ]);
    }
    return dump;
  }

  string dump_me()
  {
    return sprintf("%O", _dump_me(this_object(), 0));
  }

  void destroy()
  {
#ifdef CAMAS_DEBUG
#if constant(Debug.locate_references)
     Debug.locate_references(this_object());
#endif
#endif
                  
  }
}

class Session
{
  inherit _Session;
  private string compat_error = "\nCompatibility problem !\n"
    "This won't preven you from using CAMAS "
    "but it is important for us to know your error. "
    "Please send a mail to camas-devel@oav.net "
    "with the following information: \n";

 
#ifdef CAMAS_DEBUG
  static void f_debug_variables(string ind, int set, void|mixed what)
  {
    if(debug_variables && sizeof(debug_variables) > 0)
      if(search(debug_variables, ind) >= 0)
      {
        if(!set)
        {
            if(QUERY(debugvariables_get))
            {
              write("CAMAS DEBUG get CSESSION->"+ind+"\n");
              write("Backtrace is: %s\n", describe_backtrace(backtrace()));
            }
        }
        else if(QUERY(debugvariables_set))
        {
          write("CAMAS DEBUG set CSESSION->"+ind+"="+sprintf("%O\n", what));
          write("Backtrace is: %s\n", describe_backtrace(backtrace()));
        }
      }
  }
#endif

  static mixed `[](string ind) {
#ifdef CAMAS_DEBUG
    f_debug_variables(ind, 0); 
#endif
    return _Session::`[](ind);
  }

#ifdef CAMAS_DEBUG
 static mixed `->(string ind) {
    f_debug_variables(ind, 0); 
    return _Session::`->(ind);
  }
#endif

  static mixed `[]=(string ind, mixed what) {
#ifdef CAMAS_DEBUG
    f_debug_variables(ind, 1, what); 
#endif
    return _Session::`[]=(ind, what);
  }

#ifdef CAMAS_DEBUG
  static mixed `->=(string ind, mixed what ) {
    f_debug_variables(ind, 1, what); 
    return _Session::`->=(ind, what);
  }
#endif
}

void default_session (object id) {
  array ind_prefproperties = indices (prefproperties);
  foreach (ind_prefproperties, string prop) {
    // first take default values
    CSESSION[prop] = prefproperties[prop];    
    // then overwrite with values from the CIF (some default values are not in the CIF)
    if(Array.search_array(indices(prefprop_att), lambda (string pro) { if(pro == prop) return 1; return 0; }) != -1)
    {
      string prop_value = (string)query("pref_"+prop);
      CSESSION[prop] = prop_value;
    }
    CSESSION["usersetup"+prop] = 1; // User is allowed to change all (so far)
  }

  CSESSION->displayimaperrors = QUERY(displayimaperrors);
  CSESSION->displaysmtperrors = QUERY(displaysmtperrors);
  CSESSION->displayaddrbookerrors = QUERY(displayaddrbookerrors);
  CSESSION->usersetupmailpath = camas_feature (FEAT_USERMAILPATH);
  CSESSION->hidemboxes = QUERY (hidemboxes);
  if (!sizeof (CSESSION->hidemboxes))
    CSESSION->hidemboxes = 0;
  CSESSION->usersetup = camas_feature (FEAT_USERSETUP);
  CSESSION->usersetupaddress = camas_feature (FEAT_USERADDRESS);
  CSESSION->usersetupextraheader = camas_feature (FEAT_USERHEADERS);
  CSESSION->usersetupreplymsgprefix = my_configuration ()->get_provider ("camas_features")->QUERY (chgreplymsgprefix) && camas_feature (FEAT_USEREDITSETUP);
	CSESSION->usersetupreplyontop = camas_feature(FEAT_USERCANREPLYONTOP); 
  CSESSION->usersetuptrashfolder = camas_feature (FEAT_USERTRASHFOLDER);
  CSESSION->usersetupsentfolder = camas_feature (FEAT_USERSENTFOLDER);
  CSESSION->usersetupansweredfolder = camas_feature (FEAT_USERANSWEREDFOLDER);
  CSESSION->usersetupsaveattachments = camas_feature (FEAT_USERSENTSAVEATTACHMENTS);
  CSESSION->usersetupautobcc = camas_feature (FEAT_USERBCCCOPY);
  CSESSION->usersetupname = camas_feature (FEAT_USERCANCHANGENAME);
  CSESSION->usersetuplanguage = camas_feature (FEAT_USERLANGUAGE);
  CSESSION->usersetupshowhiddenheaders = camas_feature (FEAT_USERSETUPSHOWHIDDENHEADERS);
  CSESSION->usersetupdefaultpostloginscreen = camas_feature(FEAT_USERDEFAULTPOSTLOGINSCREEN);
  CSESSION->usersetupfeedimapcache = camas_feature(FEAT_USERCANDISABLEIMAPCACHEFEEDER);
  CSESSION->showprefsbox = QUERY (showprefsbox);
  CSESSION->wrapcolumn = (int) QUERY (wrapcolumn);
  CSESSION->imapidlelogout = (int) QUERY (imapidlelogout);

  array(string) layouts = id->conf->get_provider ("camas_layout_manager")->list_layouts ();

  // don't set CSESSION->overridelayout to something _before_ the users really logs in
  // because if he hits the login page while allready logged in, his current layout will change
  if (sizeof (layouts) == 1)
    CSESSION->overridelayout = id->conf->get_provider ("camas_layout_manager")->get_login_layout();
  else
    CSESSION->layout = id->conf->get_provider ("camas_layout_manager")->get_login_layout();

  if(camas_feature(FEAT_EXTENDEDABOOK))
    CSESSION->extendedabook = CAMAS.ebook.ebook ();
  object lang_module = CAMAS_LANGUAGE;

  if (lang_module->QUERY(browserlang) && (id->request_headers["accept-language"])) {
    foreach (id->request_headers["accept-language"] / ",", string foo) {
      foo = (foo / ";")[0];
      if (lang_module->lang_progs_short[foo]) {
        CSESSION->language = lang_module->lang_progs_short[foo];
        break;
      }
    }
  }

  if (!CSESSION->language || (stringp(CSESSION->language) && !strlen(CSESSION->language)))
    CSESSION->language = lang_module->QUERY (defaultlang);

  //FIXME: better UTF-8 detection?
  CSESSION->utf8 = 0;
  if (QUERY (useutf-8)) {
    if (search (lower_case (id->request_headers["accept-charset"] || ""), "utf-8") != -1)
      CSESSION->utf8 = 1;
    else {
      int i;
      if ((i = search (id->client, "MSIE")) != -1)
        if (sizeof (id->client) > i)
          CSESSION->utf8 = ((int)(id->client[i + 1])) >= 4;
    }
  }

  CSESSION->charset = CSESSION->utf8 ? "utf-8" : (lang_module->lang_charsets[CSESSION->language] || "iso-8859-1");
#ifdef CAMAS_DEBUG
  CSESSION->imapclient = imapclient()->imapclient (camas_feature(FEAT_EXTENDEDABOOK), this_object());
#else
  CSESSION->imapclient = CAMASIMAPCLIENT.imapclient (camas_feature(FEAT_EXTENDEDABOOK), this_object());
#endif
  CSESSION->mboxencode = (QUERY (mboxencode) == "encode");
  // now comes the IMAP default settings
  CSESSION->sharedpath = QUERY(defaultsharedpath);
  CSESSION->shareddepth = QUERY (defaultshareddepth);
  CSESSION->mailpath = QUERY(defaultmailpath);
  CSESSION->maildomain = QUERY(smtpmaildomain);
#if constant(thread_create)
  CSESSION->lock = Thread.Mutex();
#endif
#ifdef CAMAS_DEBUG
  CSESSION->debug_variables = map(QUERY(debugvariables) / ",", 
      lambda(string s) { return String.trim_whites(s); });
  CSESSION->debug_variables_get = QUERY(debugvariables_get);
  CSESSION->debug_variables_set = QUERY(debugvariables_set);
  CSESSION->debug = QUERY(debug);
#else
  CSESSION->debug = 0;
#endif
}


string query_provides () {
  return "camas_main";
}

//! method: void create(mixed...foo)
//!  Constructor of the module
void create (mixed...foo) {
  if(sizeof(foo)==12){
    mapping g=foo[0];
    variables=foo[1];

    prefproperties=g["prefproperties"];
#if constant(thread_create)
    global_lock=g["global_lock"];
#endif
    return;
  }

  defvar ("location",
          "/mail/",
          "Mountpoint: Mountpoint",
          TYPE_LOCATION,
          "Mailreader server location.");
  defvar ("resetatlocationhit",
          0,
          "Mountpoint: Logout when hitting mountpoint",
          TYPE_FLAG,
          "<p>If the user hits the mountpoint (eg http://server.tld/mountpoint), he's logged out if already logged in "
          "and faces the login page.</p>"
          "<p>You may enable this feature if you have some problems that could get the session corrupted (eg frame recursion in a layout) "
          "and want to automatically reset the user's session.</p>"
          "<p>If you enable this feature, make sure you don't hit the mountpoint in normal CAMAS operation (standard CAMAS tags action don't)</p>");

  foreach ( indices (prefproperties), string prop ) {
    if (prefprop_att && prefprop_att[prop]) {

      string|int prop_value = prefproperties[prop];
      if(prefprop_att[prop][1] == TYPE_INT)
        prop_value = (int) prop_value;
      defvar("pref_" + prop, prop_value,
             "Default properties:" + prefprop_att[prop][0],
             prefprop_att[prop][1], prefprop_att[prop][2],
             (sizeof(prefprop_att[prop])>3)?prefprop_att[prop][3]:0,
             (sizeof(prefprop_att[prop])>4)?prefprop_att[prop][4]:0);
    }
  }

  defvar("indexdelayformat", DEFAULT_INDEXDELAY ,
         "Misc:User interface: Mailindex: Delay: Format",
         TYPE_TEXT_FIELD,
         "Set a format for the delay.<br />"
         "You can use the following replacements :<ul>"
         "<li><b>$DDAYS</b>: Day</li>"
         "<li><b>$DHOURS</b>: Hour</li>"
         "<li><b>$DMINUTES</b>: Minute</li>"
         "<li><b>$DSECONDS</b>: Second</li>"
         "</ul>"
        );

  defvar("displayimaperrors", 0, "Misc:User interface: Display IMAP errors to user", TYPE_FLAG,
         "If this is set to 'yes', the IMAP error string from the server "
         "will be shown to the user in the error dialog. The text comes from "
         "the server and cannot be altered nor translated.");

  defvar("displaysmtperrors", 0, "Misc: User interface: Display SMTP errors to user", TYPE_FLAG,
         "If this is set to 'yes', the SMTP error string from the server will be "
         "shown to the user in the error dialog. The text comes from the server "
         "and cannot be altered nor translated.");
  
  defvar("displayaddrbookerrors", 0, "Misc: User interface: Display address book errors to user", TYPE_FLAG,
         "If this is set to 'yes', the address book error string from the server will be "
         "shown to the user in the error dialog. The text comes from the Pike client "
         "and cannot be altered nor translated.");

  defvar("displaysmtpok", 1, "Misc: User interface: Display SMTP confirmations to user", TYPE_FLAG,
         "If this is set to 'yes', the user will see confirmation that his mail has been sent."
         "The text comes from CAMAS.");

  defvar("useutf-8", 1, "Misc: User interface: Output UTF-8 (if possible)", TYPE_FLAG,
         "UTF-8 is an encoding for 16-bit characters, which is used in CAMAS internally. "
         "If this option is enabled, CAMAS will output UTF-8 data to the browser if the "
         "browser supports it. If it is disabled, the character set will be taken from "
         "the current language file.");

  defvar("imapidlelogout", 300,  "Incoming mail:Idle IMAP connection timeout", TYPE_INT,
         "Close an open IMAP connection after this many seconds. The user session may still be "
         "active, and will not be disturbed. The IMAP connection will be reopened when the user "
         "issues an IMAP command.");

  defvar("mboxencode", "encode", "Incoming mail:Mailbox addressing", TYPE_STRING_LIST,
         "Choose whether CAMAS should use IMAPs modified UTF-7 coding when refering "
         "to mailboxes or if it should send mailbox names unmodified. If you're having trouble "
         "accessing mailboxes with non-USASCII characters, try changing this option.",
         ({ "encode", "8-bit" }));

  defvar("allowmailpathoverwrite", 1, "Incoming mail:Mail path: Allow IMAP server to overwrite namespace",
         TYPE_FLAG,
         "If set to yes, Camas will fetch the mail path informations from the IMAP server "
         "so that you don't need to fill in values for 'Mail path: Shared path' and "
         "'Mail path: Personal path'. You always need to fill in values for these if your IMAP server "
         "doesn't support <a href=\"http://www.ietf.org/rfc/rfc2342.txt\">RFC 2342</a>");

  defvar("defaultsharedpath", "",
      	 "Incoming mail:Mail path: Shared path", TYPE_STRING,
         "The default shared hierarchy path. Is usually \"shared.\""
      	 "(Courier-IMAP).<br>"
         "<b>Note that if you say yes to 'Allow IMAP server to overwrite namespace' and if the server "
         "provide informations for namespaces (most of them do) then this option will not be used.</b>");

  defvar("sharemailcache", 0, "Incoming mail: Share cache between users", TYPE_FLAG,
         "Camas uses a cache to store the mails of each user by default. If this option is set "
         "to yes then Camas will use a global cache for every users for special folders "
         "like shared ones. In order for this to work your users must have exactly the "
         "same data for the same shared folders name they use. If it is the case and if your users "
         "have lot of mails in their shared folders, the performances will be better both on "
         "Camas and on the IMAP server");
  
  defvar("defaultshareddepth",2, "Incoming mail:Default shared mail folder depth", TYPE_INT,
         "The default shared hierarchy depth. Might be usually 2, but might be more.");
 
  defvar("defaultmailpath", "", "Incoming mail:Mail path: Personal path", TYPE_STRING,
         "The default mail folder path. Is usually \"INBOX\" (WU-IMAP), \"INBOX.\" (Courier-IMAP) or "
	 "leave it empty when using the Dovecot IMAP server. "
         "Can be changed by the user, although the user's preferences is always saved here.<br/> "
         "<b>Note that if you say yes to 'Allow IMAP server to overwrite namespace' and if the server "
         "provide informations for namespaces (most of them do) then this option will not be used.</b>");
  
  defvar("hidemboxes", ".", "Incoming mail:Hide mailboxes starting with", TYPE_STRING,
         "Optionally hide mailboxes starting with the specified string. "
         "Mailboxes starting with '.' are often not really mailboxes.");

  defvar("notreloadmailboxes", "", "Incoming mail:Do not reload these mailboxes", TYPE_STRING|VAR_EXPERT,
         "If set, the mailboxes whose names match one of these will not be reloaded when "
         "the user check for new mail. This can be usefull for mailboxes like 'Trash' if it "
         "has lot of mails and if the IMAP server is slow so that check new mail is faster. "
         "<br>You must separate the mailboxes names with a comma.");

  defvar("sendmethod", "SMTP", "Outgoing mail:Method", TYPE_STRING_LIST,
         "Method to use when sending mail.",
         ({ "SMTP", "sendmail" }));

  defvar("sendmail", "/usr/lib/sendmail", "Outgoing mail:Sendmail",
         TYPE_STRING,
         "Location of sendmail.", 0, lambda () { return (QUERY (sendmethod) != "sendmail"); } );

  defvar("smtpserver", "localhost", "Outgoing mail:SMTP server address",
         TYPE_STRING,
         "The address of the SMTP-server.");

  defvar("smtpport", 25,  "Outgoing mail:SMTP server-port", TYPE_INT,
         "The portnumber of the SMTP-server.");

  defvar("smtpmaildomain", "your.domain.com", "Outgoing mail:SMTP mail domain", TYPE_STRING,
         "The outgoing mail domain.");

  defvar("addmaildomain", 1, "Outgoing mail:Complete unqualified addresses", TYPE_FLAG,
         "If this is enabled, the default mail domain will be added to addresses "
         "without a \"@\" before sending them to the SMTP server.");

  defvar("ssl3redir", 0, "URL:Redirect to SSL3", TYPE_FLAG,
         "Tries to find a SSL3 port to redirect the initial requests to. Useful if your virtual "
         "server has more than one listen port and you prefer SSL3 for CAMAS.");

  defvar("urloverride", "", "URL:Server URL override", TYPE_STRING,
         "Specify your server URL here if CAMAS fails to guess the correct URL."
         " Example: 'https://your.server.com'.");

  defvar("uploaddir", "", "Misc:Files: File upload directory", TYPE_STRING,
         "Directory to hold uploaded attachment files. Leave empty to disable persistent attachment support. ");

  defvar("uploadquota", 0, "Misc:Files: File upload user quota", TYPE_INT,
         "Amount of Kb each user is allowed to upload. 0 gives unlimited quota.");

  defvar("uploadsoftquota", 0, "Misc:Files: File upload soft user quota", TYPE_INT,
         "Amount of Kb by wich the user can overrun the quota for the last uploaded file.");

  defvar("ispell", file_stat("/usr/bin/ispell")?"/usr/bin/ispell":"",
         "Misc:Spelling : Speller location", TYPE_STRING,
         "Location of ispell binary. Empty string disables spell-checking.");

  defvar("ispelldict", "american", "Misc:Spelling : Speller dictionaries",TYPE_STRING,
         "Commaseparated list of speller dictionaries, default first."
         "(-d option). Display name can be specified after a colon (swedish:Svenska).");

  defvar("speller", "ispell", "Misc:Spelling : Spellchecker", TYPE_STRING_LIST,
         "CAMAS supports ispell and aspell. ",
         ({ "ispell", "aspell" }));

  defvar("showprefsbox", 0, "User preferences:Show user preferences folder", TYPE_FLAG,
         "If \"no\", the preferences folder is not displayed.");
#ifdef CAMAS_DEBUG
  defvar("debug", 0, "Debug: Main module debug", TYPE_FLAG,
         "When on, debug messages will be logged in Caudium's debug logfile. "
         "This information is very useful to the developers when fixing bugs.<br> "
         "<i>Note</i>: One of the usefull features is to put a prestate 'dumpcsession' "
         "to the url so that Camas dumps the current session at the end of the webpage");
  defvar("debugvariables", "", "Debug: Debug these session variables", TYPE_STRING,
         "A comma separated list of Camas session values to debug. Each call for reading or "
         "writing to this variable will be displayed in the debug log.<br>"
         "<i>Note</i>: You have to reload this module once you changed this.");
  defvar("debugvariables_get", 0, "Debug: Debug read only call", TYPE_FLAG,
         "If yes, read only calls from Camas to this variable will be output in "
         "the debug log", 0, hide_debugvariables);
  defvar("debugvariables_set", 1, "Debug: Debug write only call", TYPE_FLAG,
         "If yes, write only calls from Camas to this variable will be output in "
         "the debug log", 0, hide_debugvariables);
#endif
  defvar("wrapcolumn", 70, "Outgoing mail:Wrap line column", TYPE_INT,
         "Wrap outgoing message lines at this column. If set to zero, no wrapping will take place.");

  defvar("allowipchange", 1, "URL: Allow changing user IP-address", TYPE_FLAG,
         "Should CAMAS allow one session to use different IP addresses? "
         "If some users are using dynamic addresses which change even during a session "
         "(e.g. while composing mail) then this option should be \"yes\". A reason to set "
         "it to \"no\" is that a stolen session id (from cookie- or log file) cannot be "
         "used from a remote machine.");

  defvar("morecharsets", "", "Outgoing mail:Character sets",TYPE_STRING,
         "Commaseparated list of charctersets wich CAMAS tries to use "
         "for outgoing mail before the last resort, UTF-8. CAMAS first tries "
         "to fit the mail into the same characterset used by the mail the user "
         "replied to. Then it tries iso-8859-1, the charcterset of the "
         "active language file, this list of charactersets and UTF-8. "
         "Specify those charactersets your users usually use to avoid their "
         "mail being sent as UTF-8.");

  defvar("indexdatelongformat", DEFAULT_INDEXDATELONGFORMAT ,
         "Misc:User interface: Mailindex: Date: Long format",
         TYPE_TEXT_FIELD,
         "Set a format for the date if it is more than one week ago.<br />"
         "You can use the following replacements :<ul>"
         "<li><b>$DD</b>: Day</li>"
         "<li><b>$MM</b>: Month</li>"
         "<li><b>$YY</b>: Short year</li>"
         "<li><b>$YYYY</b>: Long year</li>"
         "<li><b>$HOURS</b>: Hours</li>"
         "<li><b>$MINUTES</b>: Minutes</li>"
         "<li><b>$SECONDS</b>: Seconds</li>"
         "<li><b>$TZ</b>: Timezone</li>"
         "</ul>"
        );

  defvar("indexdateshortformat", DEFAULT_INDEXDATESHORTFORMAT ,
         "Misc:User interface: Mailindex: Date: Short format",
         TYPE_TEXT_FIELD,
         "Set a format for the date if it is within the last week.<br />"
         "If nothing set, use the <i>&quot;normal&quot;</i> full output, "
         "specified in <i>&quot;Mailindex: Date: Long format&quot;</i><br />"
         "You can use the following replacements :<ul>"
         "<li><b>$WDAY</b>: Short format weekday-name (eg. Fri for friday)</li>"
         "<li><b>$WEEKDAY</b>: Full weekday-name (like Thursday)</li>"
         "<li><b>$HOURS</b>: Hours</li>"
         "<li><b>$MINUTES</b>: Minutes</li>"
         "<li><b>$SECONDS</b>: Seconds</li>"
         "<li><b>$TZ</b>: Timezone</li>"
         "</ul>"
        );

    defvar("delaystepnumber", 6, "Misc:User interface: Mailindex: Delay: Delay Step number", TYPE_INT,
         "Specify the number of delay step you want.<br />"
         "When set, you'll need as many CSS classes<br />"
         "<p><b>Example:</b><br />"
         "If set to 3, entries <i>delay0</i>, <i>delay1</i> and <i>delay2</i> might be in your CSS.<br />"
         "Moreover, if delay image path is set, you'll need <i>delay0.gif</i>, <i>delay1.gif</i> and <i>delay2.gif</i></p>");

  defvar("mdntype", "Both", "Outgoing mail:MDN support", TYPE_STRING_LIST,
         "MUA Mail Delivery Notification type. Set \"None\" if you want to disable MDN",
         ({ "None", "Both", "Netscape", "Microsoft" }));

  defvar("rxmlparsercalls", 3, "Debug:Maximum calls to parse_rxml", TYPE_INT|VAR_EXPERT,
         "Maximum calls to parse_rxml");

  defvar("buggyimap", 0, "Incoming mail:IMAP server is not RFC-compliant", TYPE_FLAG | VAR_EXPERT,
         "Some IMAP servers (e.g. Courier 1.3) are not RFC-compliant.<br />"
         "For example, they will not return \"[TRYCREATE]\" when appending a message "
         "in a non-existent folder.<br /> This will enable various workarounds to make "
         "CAMAS work as needed. <br />"
         "<hr />RFC2060, section 6.3.11 (APPEND Command)<br />"
         "<pre>If the destination mailbox does not exist, a server <b>MUST</b> return an\n"
         "error, and <b>MUST NOT</b> automatically create the mailbox.  Unless it is\n"
         "certain that the destination mailbox can not be created, the server\n"
         "<b>MUST</b> send the response code \"[TRYCREATE]\" as the prefix of the text\n"
         "of the tagged NO response.  This gives a hint to the client that it\n"
         "can attempt a CREATE command and retry the APPEND if the CREATE is\n"
         "successful.</pre>");

  defvar("upgrading", 1, "Compatibility:Upgrading from IMHO to CAMAS", TYPE_FLAG,
         "CAMAS will read 'IMHO Preferences' if 'CAMAS Preferences' are not found.");

  defvar("defaultlayout", "Default", "Compatibility:Layout to use", TYPE_STRING,
         "The layout to use if none found in the old preferences files.", 0, lambda () { return !QUERY (upgrading); });

  defvar("foldernamemaxlength", 0, "Limits:Max length allowed for a mailbox name",
         TYPE_INT | VAR_MORE,
         "Set the max length for a new mailbox name. If set to 0, there will be no check.");

  defvar("maxmailsize", 50000,"Limits:Max mail size to display", TYPE_INT,
         "If the mail data exeeds this size (in bytes), "
         "it will only be shown as a download link.");

  defvar ("gctimeout", 600, "Internals:Garbage collecting frequency",
          TYPE_INT,
          "Sets the delay (in seconds) between two garbage collections.");

  defvar ("sslfaulty", 0, "URL:No SSL if ssl_faulty", TYPE_FLAG,
          "If set, SSL will be disabled if the client has <b>id->supports->ssl_faulty</b> set.");

  // New options handled by optionnal external module_provider
  defvar("cadmif", 1, "Misc:Options: Enable Runtime Admin Interface", TYPE_FLAG,
         "Allow CAMAS Runtime Admin Interface to be accessed thru CAMAS "
         "login method. If you set this to \"No\", then you can remove the "
         "Runtime Admin Interface module to save memory...");
  defvar("allowfailauth", 1, "Misc:Options: Allow failed authentification from auth module",
         TYPE_FLAG, "If set to yes the user will be able to logged in if the IMAP server "
         "sucessfully authenticate him but the auth module failed to find him.");

  starttime = time ();
}

#ifdef CAMAS_DEBUG
int hide_debugvariables()
{
  return !(QUERY(debugvariables) && sizeof(QUERY(debugvariables)) > 0);
}
#endif

//! method: string status()
//!  Displays some information in the CIF in CAMAS: Main module: Status and debug info
//! returns:
//!  The string to display
string status ()
{
  string out = "";
  object sess123 = my_configuration ()->get_provider (session_module);
  if (objectp (sess123)) {
    if(session_module=="123sessions" && sess123->QUERY(storage) != "memory") {
      out += "<font color=red><font size=+3>WARNING !!!</font> "
                "123session module is not set to use <b>memory</b> storage "
                "method. CAMAS <b>CANNOT</b> work with other storage method "
                "than 'memory', this is not a bug from CAMAS or 123session "
                "</font><br />";
    }
    out += "Sesssion module used: " + session_module + ".<br />";
    out += "Active sessions:\n" + sess123->status () + "<br />";
  }
  out += "CAMAS Authentication :<br />";
  object auth_module = my_configuration ()->get_provider ("camas_auth");
  if (objectp (auth_module)) {
    out += auth_module->status ();
  }
  else {
    out += "<font color=\"red\">Warning:</font> no authentication module installed. ";
    out += "CAMAS will not work, you must install a suitable authentication module.";
  }
  out += "<br /><br />";

  out += "CAMAS user preferences :<br/>";
  object pref_module = my_configuration()->get_provider ("camas_preferences");
  if (objectp(pref_module))
    out += pref_module->status();
  else
  {
    out += "<font color=\"red\">Warning:</font> no preferences module installed. ";
    out += "Camas will not work properly, you should install a suitable preferences module.";
  }
  out += "<br/><br/>";

  out += my_configuration ()->get_provider ("camas_layout_manager")->status ();
  out += "<br />";

  if (my_configuration ()->get_provider ("rxml:core")->module_name[0..2] == "XML") {
    out += "XML Parser status: ";
    if (global_kludges)
      out += global_kludges + " kludges.";
    else {
      out += "fine, thanks.";
      if (global_ok)
        out += " (" + global_ok + " page" + ((global_ok == 1) ? "" : "s") + " correctly parsed).";
    }
  }

  return out;
}

//! method: void destroy_clients(object conf)
//!  Destroys all the sessions
void destroy_clients (object conf)
{
  CDEBUG("destroy_clients...\n");

  object sess123 = conf->get_provider (session_module);
  if (objectp (sess123)) {
    mapping sessions = sess123->sessions ();

    foreach(indices(sessions),string s)
    {
      CAMAS.Tools.low_destroy_session(sessions[s]->values->CAMAS);
      m_delete(sessions[s]->values, "CAMAS");
    }
  }
}

//! method: string|int check_variables(string var, mixed value)
//!  Checks the value of the vars set up via the CIF
//! arg: string var
//!  The name of the variable to check
//! arg: mixed value
//!  The value of the variable to check
//! returns:
//!  Some error message if needed, 0 if no error
string|int check_variable(string var, mixed value) {

  switch (var) {

  case "uploadquota":
  case "uploadsoftquota":
  case "frommaxlength":
  case "subjectmaxlength":
  case "foldernamemaxlength":
    if (value < 0)
      return "Must be positive integer or zero.";
    break;

  case "gctimeout":
    if (value <= 0)
      return "Must be positive integer.";
    break;

  case "uploaddir":
    if(sizeof(value)) {
      array stat=file_stat(value);
      if(!stat || stat[1]!=-2)
        return "Must be a directory or empty string.";
    }
    break;

  case "imapservers":
    if (sizeof(value/" ") < 2 && sizeof(value / "\t") < 2)
      return("Must contain at least one server in format 'domain imapserver'");
    break;

  case "ispell": {
      if(!sizeof(value))
        break;
      array stat=file_stat(value);
      if(!stat || !(stat[0] & 73))
        return "Must be empty or an executable file.";
    }
    break;

  case "ispelldict":
    if(!sizeof(value))
      return "Must not be empty.";
    break;

  case "sendmail":
    if (QUERY (sendmethod) == "SMTP")
      break;
    array stat = file_stat (value);
    if (!stat || !(stat[0] & 73))
      return "Must be an executable file.";
    break;

  }
  return 0;
}

//! method: void start(int num, object conf)
//!  Method called at CAMAS startup
void start (int num, object conf)
{
  array dependencies = ({ "graphic_text", "killframe", "navbar",
                          "tablesort", "scut", "contenttypes",
                          "camas_language",
                          "camas_layout_default", "camas_images",
                          "camas_html", "camas_formbuttons",
                          "camas_features", "camas_actions",
                          "camas_layout_manager",
                          "camas_features", "camas_actions", 
	                  		  "camas_tags", "camas_entities", 
                          });
  // Loads the Runtime Admin Interface only if needed...
  if (QUERY(cadmif)) dependencies += ({ "camas_runtime_admin" });

  // try to see if we can load a default camas_auth module if none exists
  array enabled_modules = indices(conf->retrieve("EnabledModules",conf));
  // a little function which returns true if the beginning of one element match the other
  function match = lambda(string haystack, string needle) 
  {
    return (search(haystack, needle) != -1);
  };
  if(Array.search_array(enabled_modules, match, "camas_auth") == -1)
  {
    CDEBUG("There is not auth_module, loading auth_basic...");
    dependencies += ({ "camas_auth_basic" });
  }
  if(Array.search_array(enabled_modules, match, "camas_preferences") == -1)
  {
    CDEBUG("There is no preferences module, loading imap preferences...");
    dependencies += ({ "camas_preferences_imap" });
  }
  if(Array.search_array(enabled_modules, match, "camas_addressbook") == -1)
  {
    CDEBUG("There is no addressbook module, loading addressbook preferences...");
    dependencies += ({ "camas_addressbook_preferences" });
  }
  module_dependencies (conf, dependencies);
  dependencies = 0;
#ifdef CAMAS_DEBUG
  array files = ({
                   "local_pmods/CAMAS.pmod/Local.pmod/_imapclient.pmod",
                   "../local/pmods/CAMAS.pmod/Local.pmod/_imapclient.pmod",
                   "etc/modules/CAMAS.pmod/imapclient.pmod"
                });
  foreach(files, string file)
  {
    string file_content;
    catch (file_content = Stdio.File(file, "r")->read());
    if(file_content)
    {
      write("\nDebug mode: Loading imapclient from file '" + file + "'\n");
      imapclient = compile_string(file_content);
      break;
    }
  }
#endif

  foreach (call_out_info (), array a)
  if ((sizeof (a) > 5) && (a[3] == "CAMAS_Survivor") && (sizeof (a) < 7 || a[6] == my_configuration ())) {
    remove_call_out (a[5]);
    break;
  }
  if (gc_call_out)
    remove_call_out (gc_call_out);
  gc_call_out = call_out (camas_gc, QUERY (gctimeout));

  array(string) layouts = conf->get_provider ("camas_layout_manager")->list_layouts ();

	
  // a list of mailboxes not to reload
  if(QUERY(notreloadmailboxes) && sizeof(QUERY(notreloadmailboxes)))
      notreloadmailboxes = map(QUERY(notreloadmailboxes) / ",",
            lambda(string mbox) { return String.trim_all_whites(mbox); });
  mixed err = catch {
    prefproperties->layout = layouts[0];
  };

  if(err)
    report_error("You propably don't have a Default layout or the layout files of your default "
        "layout are missing, please check that you have files in etc/include/camas/layouts/default");
  init_image_types ();
  global_kludges = global_ok = 0;
}

//! method: void stop()
//!  Method called at CAMAS shutdown
void stop ()
{
  mapping g = ([ ]);
  g["prefproperties"] = prefproperties;
#if constant(thread_create)
  g["global_lock"]=global_lock;
#endif

  if (gc_call_out)
    remove_call_out (gc_call_out);

  array a=({ 0 });
  // Caudium tries to kill us. Escape...
  // Destroy all sessions after 60 seconds if the new CAMAS hasn't been compiled by then.
  a[0]=call_out(lambda(mixed ... foo){foo[1]->destroy_clients(my_configuration());},60,"CAMAS_Survivor",object_program(this_object())(g,variables,0,0,0,0,0,0,0,0,0,0),a,my_configuration());
}

//! method: string query_location()
//!  MODULE_LOCATION method
//! returns:
//!  CAMAS mount point
string query_location()
{
  return QUERY (location);
}

// this class is used to fetch all headers and 
// feed the cache at the beginning
class FeedIMAPCache {
  private object id;
  private object cacheimapclient;
  int sharemailcache = QUERY(sharemailcache);
  int do_not_disconnect;
  int isimapcachefeeder = 1;
  object global_lock;
  
  string _sprintf()
  {
    return "FeedIMAPCache";
  }
  
  void create(object _id)
  {
    id = _id;
    CSESSION->cachefeeded = 0;
    global_lock = id->conf->get_provider("camas_main")->global_lock;
#ifdef CAMAS_DEBUG
    cacheimapclient = imapclient()->imapclient (0, this_object());
#else
    cacheimapclient = CAMASIMAPCLIENT.imapclient (0, this_object());
#endif
    CDEBUG("FeedIMAPCache created\n");
    array imap_commands = ({ });
    foreach(CSESSION->mailboxes, array mailbox)
    {
      if(mailbox[MB_FOLDERNAME_IDX] != CSESSION->mailbox[MB_FOLDERNAME_IDX]
	 && !(mailbox[MB_FLAGS_IDX] & MB_NOSELECT))
      {
          imap_commands += ({ CAMAS.IMAPTools.imap_cmd("status","mailbox",mailbox[MB_FOLDERNAME_IDX]), 
	  CAMAS.IMAPTools.imap_cmd("get_headers", "mailbox",mailbox[MB_FOLDERNAME_IDX], "output", CAMAS.IMAPTools.imap_cmd_var("dummy")) });
      }
    }
    CSESSION->dummy = 0;
    imap_commands += ({ CAMAS.IMAPTools.imap_cmd("low_logout", "noselect", 1) });
    cacheimapclient->imap_command(this_object(), CSESSION, imap_commands);
  }

  // dummy methods imapclient calls when finishing
  void create_output()
  {
  }
  
  void send_result()
  {
    CDEBUG("FeedIMAPCache finished\n");
    killall();
  }
    
  void killall()
  {
    CDEBUG("FeedIMAPCache destroyed\n");
    if(id && CSESSION)
    {
      CSESSION->cachefeeded = 1;
      CSESSION->dummy = 0;
    }
    destroy();
  }
  
  void destroy()
  {
    CDEBUG("FeedIMAPCache destroyed\n");
    if(cacheimapclient && objectp(cacheimapclient))
      destruct(cacheimapclient);
    destruct(this_object());
  }
};

//! method: array(string) get_cactions(object id)
//! arg: object id
//!  The Caudium id object
//!  Get, format and re-order the CAMAS actions.
//!  For example if we have
//!  a query like action3compose=1&action2readmail=1  then execute
//!  action readmail before compose. It also put actionlogin first if
//!  it exists.
//! returns:
//!  The actions as an array of string
array(string) get_cactions(object id)
{
  array ind_variables = indices (id->variables);
  array(string) actions = ({ });
  foreach (ind_variables, string var) {
    string foo;
    if (sscanf (var, "action%s", foo))
      if(id->variables[var] != "0")
        actions += ({ foo });
  }
  int n = sizeof(actions);
  array(string) ordered_actions = allocate(n);
  for(int i = 0; i < n; i++)
  {
    string action = actions[i];
    // first some checks if type=image in submit
    int foo = sizeof(action);
    if(foo > 2 && (action[foo-2..]==".x" || action[foo-2..]==".y"))
      action = action[..foo-3];
    if((int)action)
    {
      int order = (int)action;
      if(--order < n)
        ordered_actions[order] = action[1..];
    }
    else
      ordered_actions[0] = action;
    // login must always be the first action if it exists
    if(i > 0 && ordered_actions[i] == "login")
    {
      string tmp = ordered_actions[0];
      ordered_actions[0] = "login";
      ordered_actions[i] = tmp;
    }
  }
  return ordered_actions;
}

// This class takes care of launching Camas actions.
// With it Camas manage priority of Camas action modules
// as well as action overloading (a Camas action module can
// tell this class to execute other low priority Camas actions
// or not). Actions modules can also add an action for example
// if the user request the action forward, the action module can
// ask Camas to execute the move action on every action modules.
class ActionsHandler {
  
  // a FIFO queue of actions to execute
  // an int defines the behaviour of this class:
  //   -1: execute the same action for other low priority module 
  //   that can exists
  //   0: don't try to execute the same action for other low priority module
  //   that can exists
  private array(int) queue = ({ });
  // a queue of all IMAP commands to start
  private array imap_commands = ({ });
  private object camas_main;
  private object id;

  //! method: create(object _id, array(string) action)
  //!   The constructor of this class
  //! arg: object _id
  //!  The Caudium object id
  //! arg: array(string) action
  //!  An array of one or more actions to execute
  public void create(object _id, array(string) actions)
  {
    id = _id;
    camas_main = CAMAS_MODULE;
    if(actions && sizeof(actions) > 0 && actions[0])
    {
      // don't execute an action other than login if the user is not 
      // successfully authentificated
      if(!((CSESSION->status < 0 || id->misc->camas->status < 0) && actions[0] != "login"))
        foreach(actions, string action)
          if(action)
            add_action(action);
    }
  }

  //! method: public void add_action(string action)
  //!  Add an action to the list of pending action(s) to execute
  //!  arg: string action
  //!   The action to execute
  public void add_action(string action, mixed ...extra)
  {
    CDEBUG(sprintf("action %s added\n", action));
    CSESSION->lastaction = action;
    execute_action(action, @extra);
  }

  //! method: public void execute_action_for_module(object camas_actions, string action)
  //!  Try to execute an action for the given camas actions module
  //! arg: object camas_actions
  //!  The CAMAS action module to use
  //! arg: string action
  //!  The action to execute
  public void execute_action_for_module(object camas_actions, string action, mixed ...extra)
  {
    object actions = camas_actions->Actions(id, this_object());
    multiset availables = mkmultiset(indices(actions));

    // if the module handle this action
    if(availables[action])
      actions[action](@extra);
    destruct(actions);
  }
    
  //! method: private void execute_action (string action)
  //!  Execute the current action on queue for all modules
  //!  arg: string action
  //!   The action to execute
  private void execute_action (string action, mixed ...extra)
  {
    mapping(int:array(object)) actions_modules =
       CAMAS.Tools.get_providers_bypriority("camas_actions", id);
    array rev_sort_ind_actions_modules = reverse(sort(indices(actions_modules)));
    int warning = 1;
    foreach(rev_sort_ind_actions_modules, int priority)
    {
      // if execute_action return 0 or an array then we found the action
      // else continue
      foreach(actions_modules[priority], object module)
      {
        CDEBUG(sprintf("executing %O for action %s\n", module, action));
        execute_action_for_module(module, action, @extra);
        warning = 0;
        if(sizeof(queue) > 0)
          queue = queue[1..];
        if(sizeof(queue) == 0)
          queue = ({ -1 });
        // if queue is 0, then don't execute next modules for this action
        if(queue[0] == 0)
          return;
      }
    }
    if(warning)
      report_warning("Can't find action '%O' in camas_actions module(s)\n", action);
  }
  
  //! method: public void execute_next_module()
  //!  Execute next module for the same action (default case)
  //!  This is mainly intend to be used by the Camas action module
  public void execute_next_module()
  {
    queue += ({ -1 });
  }

  //! method: public void donot_execute_next_module()
  //!  Do not execute next module for the same action
  //!  This is mainly intend to be used by the Camas action module
  public void donot_execute_next_module()
  {
    CDEBUG("donot_execute_next_module called\n");
    queue += ({ 0 });
  }

  //! method: public void add_imap_commands(array cmd)
  //!  Add an imap commands to the imap commands the imapclient have to
  //!  execute
  //!  This is mainly intend to be used by the Camas action module
  public void add_imap_commands(array cmd)
  {
    CDEBUG("imap_commands added\n");
    imap_commands += cmd;
  }

  //! method: public mapping retvalue()
  //!  Gives back the return value for Camas main module
  public mapping retvalue()
  {
    if(sizeof(imap_commands) > 0)
    {
      // launch the bloody IMAP client
      // it may even work if you are lucky ;)
      CSESSION->imapclient->imap_command(id, CSESSION, imap_commands);
      return HTTP_PIPE_IN_PROGRESS();
    }
    else
      return camas_main->create_output(id);
  }
}; 

//! method: mapping process_request(object id)
//!  Called by find_file(string file, object id)
//!  Process the request
//! arg: object id
//!  The request id object
static mapping process_request (object id)
{
  object features = my_configuration ()->get_provider ("camas_features");
  if (!objectp(features)) {
    report_fatal("Can't find CAMAS features module");
    return 0;
  }
  
  if (CSESSION->administrator)
    return create_output (id);

  // CAMAS HTML module take care of rendering mail parts
  if (id->variables->mailpart)
    return id->conf->get_provider ("camas_html")->mailpart (id);

#if constant(Image.XFace.decode)
  if (id->variables->xface) {
    object img;
    if (catch (img = Image.XFace.decode (CSESSION->cmail->headers["x-face"])))
      return 0;
#if constant(Image.PNG.encode)
    if (id->supports->png)
      return HTTP_STRING_ANSWER (Image.PNG.encode (img), "image/png");
    else
      return HTTP_STRING_ANSWER (Image.GIF.encode (img), "image/gif");
#else
    return HTTP_STRING_ANSWER (Image.GIF.encode (img), "image/gif");
#endif
  }
#endif

  if(id->variables->download){
    if((int)id->variables->download < sizeof(CSESSION->files)){
      mapping file=CSESSION->files[(int)id->variables->download];
      object downfile=open(QUERY (uploaddir)+"/"+(CSESSION->login)+"_"+CAMAS.Tools.encode_fname(file->fname),"r");
      if (downfile)
        return HTTP_FILE_ANSWER(downfile,file->type,file->size);
      else
        return 0;
    }
    else
      return 0;
  }

  if (id->variables->layout && (id->variables->layout != "__user"))
    CSESSION->overridelayout = id->variables->layout;

  if (CSESSION->loadfiles) {
    CSESSION->loadfiles = 0;
    get_files(CSESSION);
  }

  id->misc->camas->actions = get_cactions(id);

  return ActionsHandler(id, id->misc->camas->actions)->retvalue();
}

//! method: mixed create_output(object id)
//!  Called by non blocking clients (IMAP for now) when they have finished 
//!  their job and when it's time to render the user interface.
//!  It's also there that we fetch the good layout and that CAMAS Tags gets
//!  called to parse it.
//! arg: object id
//!  The Caudium request id object
//! returns:
//!  Invoques really_output()
mixed create_output (object id)
{
  object features = my_configuration ()->get_provider ("camas_features");
  if (!objectp (features))
    return 0;

  string ret = "";

  CSESSION->nexturl = server_url(id) + QUERY (location);

  string screen = "unknown";
  if (screennames[id->misc->camas->status])
    screen = screennames[id->misc->camas->status][0];
  
  else if (screennames[CSESSION->status])
    screen = screennames[CSESSION->status][0];

  string action = "";
  if(id->misc->camas->actions && 
      sizeof(id->misc->camas->actions) > 0 && stringp(id->misc->camas->actions[0]))
    action = id->misc->camas->actions[0];

  // FIXME: Remove this hugly bloc of code / vida
  //
  // sendmarkeddraftsmove is particularly weird
  // we catch this action here after being dropped in create_output() by http_pipe_in_progree() in process_request()
  // the session variables are checked for getting all the mails to be send
  // then if the mails needs to be moved into sent mail, process_request is invoqued again with HTTP_PIPE_IN_PROGRESS()
  // id->variables being changed for another action to be done
  // -- Bertrand
  if(!id->misc->camas->loop && (action == "sendmarkeddrafts" || action == "sendmarkeddraftsmove"))
  {
    // prevent loops
    id->misc->camas->loop = 1;
    array imap_commands = ({ });
    object cmail = CSESSION->cmail;

    // Putting the headers of the mail in the correct session variables
    CSESSION->to = cmail->headers->to?CAMAS.Tools.fix_coding(cmail->headers->to):"";
    CSESSION->cc = cmail->headers->cc?CAMAS.Tools.fix_coding(cmail->headers->cc):"";
    CSESSION->bcc = cmail->headers->bcc?CAMAS.Tools.fix_coding(cmail->headers->bcc):"";
    CSESSION->replytoidx = -1;
    CSESSION->attachments = ({ });
    CSESSION->subject = CAMAS.Tools.fix_coding(cmail->headers->subject);
    CSESSION->message = "";

    // Putting the body of the mail in the correct session variables
    if(cmail->body_parts){
      int messageadded=0;
      foreach(cmail->body_parts, object mess) {
        if(!messageadded &&
            mess->type == "text" &&
            mess->subtype != "html") {
          CSESSION->message += mess->getdata()||"";
          messageadded=1;
        } else {
          string name=mess->disp_params["filename"] || mess->params["name"] || "unknown";
          name = MIME.decode_words_text_remapped(name);
          mapping file=([ ]);
          file["fname"]=name;
          file["data"]=mess->getdata()||"";
          file["size"]=sizeof(file->data);
          file["type"]=CAMAS.Tools.filetype(string_to_utf8(name), my_configuration());
          CSESSION->attachments += ({ file });
        }
      }
    }
    else
      CSESSION->message += cmail->getdata()||"";

    // Creating the mail, for output only
    mapping args = ([ 
      "includeipnumber": features->QUERY(includeipnumber),
      "siteorg": features->QUERY(siteorg),
      "addmaildomain": QUERY(addmaildomain),
      "morecharsets": QUERY(morecharsets),
      "shareddepth": QUERY(defaultshareddepth),
      "pref_extraheader": QUERY(pref_extraheader),
      "mdntype": QUERY(mdntype),
      "uploaddir": QUERY(uploaddir),
      "lang_charsets": CAMAS_LANGUAGE->lang_charsets,
      "camas_version": camas_version
    ]);
    array sendmailres = sendmail (CSESSION, CSESSION->to, CSESSION->cc, 
      CSESSION->bcc, CSESSION->messageid, CSESSION->subject, CSESSION->message, 1, args);
    string maildata = sendmailres[0];
    array recipients = sendmailres[1];

    if (QUERY (sendmethod) == "SMTP")
    {
      void|string error = smtp_send (CSESSION, recipients, maildata, QUERY (smtpserver),
          (int)QUERY (smtpport), QUERY (smtpmaildomain));

      if (error && sizeof(error) > 0)
        report_error("** STMP error on message uid "+CSESSION->selected[0]+"\n");
      else
        CDEBUG("** no errors on message uid "+CSESSION->selected[0]+"\n");
    }
    else
      CSESSION->attachments = ({ }); // Remove all attachments from mem if already sent
    string currentbox =  CAMASFOLDERTOOLS.getmailboxroot(CSESSION->mailbox, CSESSION)
      + CSESSION->draftsfolder;
      id->variables  = ([ "actionmailindex":"1" ]);
      id->variables += ([ "mbox":currentbox ]);

    if(CSESSION->selected[0] && action == "sendmarkeddraftsmove")
    {
      CSESSION->copytobox = CAMASFOLDERTOOLS.getmailboxroot(CSESSION->mailbox, CSESSION)
      + CSESSION->sentfolder;
      imap_commands += ({ CAMAS.IMAPTools.imap_cmd("move", "uids", ({ CSESSION->selected[0] })
            ,"updatembox",CAMAS.IMAPTools.imap_cmd_var("mails") )  });
      CSESSION->imapclient->imap_command(id,CSESSION,imap_commands);
      return HTTP_PIPE_IN_PROGRESS();
    }
    CSESSION->status = MAILINDEX;
    id->misc->camas->status = MAILINDEX;
  }
  // end especially hugly code to remove

  switch (screen) {
  case "login":
    if (CSESSION->imapclient->loggedin) {
      if(session_module == "gsession")
      {
      	int tmax;
        if (!CSESSION->autologout
           || !sscanf(CSESSION->autologout, "%d", tmax))
           tmax = 20;
        if (CSESSION->status == COMPOSE || id->misc->camas->status == COMPOSE)
          tmax = 480; // Do not timeout when composing mail...
      	id->conf->get_provider(session_module)->set_expire_time(id->misc->session_id, tmax*60);
      }
      CAMAS.Log.log (my_configuration (), "login", ([ "login" : CSESSION->address ]));
      if(camas_feature(FEAT_USERCANDISABLEIMAPCACHEFEEDER))
      {
        if(CSESSION->feedimapcache == "1")
          CSESSION->imapcachefeeder = FeedIMAPCache(id);
      }
      else
        if(features->QUERY(feedimapcache))
          CSESSION->imapcachefeeder = FeedIMAPCache(id);
      if (CSESSION->prefsloaded || !CSESSION->usersetup)
      {
        array allowedpostloginscreens = ({ "compose", "folderlist", "setup", "files", "searchmail" });
        // execute actions needed by defaultpostloginscreen
        if(CSESSION->defaultpostloginscreen &&
            !id->misc->camas->alreadywenthere &&
            has_value(allowedpostloginscreens, CSESSION->defaultpostloginscreen))
        {
          id->misc->camas->alreadywenthere = 1;
          foreach(indices(screennames), int status)
          {
            array arr = screennames[status];
            if(arr[0] == CSESSION->defaultpostloginscreen)
            {
              CSESSION->status = status; 
              id->misc->camas->status = status;
              break;
            }
          }
          // execute the action before displaying the screen
          if(CSESSION->defaultpostloginscreen != "setup")
            return ActionsHandler(id, ({ CSESSION->defaultpostloginscreen }))->retvalue();
        }
        else
        {
          CSESSION->status = MAILINDEX;
          id->misc->camas->status = MAILINDEX;
        }
      }
      else
      {
        if (features->QUERY (newbiesetup))
        {
          CSESSION->status = SETUP;
          id->misc->camas->status = SETUP;
        }
        else
        {
          CSESSION->status = MAILINDEX;
          id->misc->camas->status = MAILINDEX;
        }
      }
      CSESSION->loadfiles = 1;
      // the user might have a layout in his preferences that is not 
      // available on this server thus we need to update his layout name
      if(!id->conf->get_provider("camas_layout_manager")->layouts[CSESSION->layout])
      {
        string new_layout = id->conf->get_provider("camas_layout_manager")->get_login_layout();
        report_notice("CAMAS: User with login '" + CSESSION->login + "' "
          "use layout '" + CSESSION->layout + "' which is not available on "
          "this server, setting '" + new_layout + "' instead\n");
           CSESSION->layout = id->conf->get_provider("camas_layout_manager")->get_login_layout();
      }
    }
    break;

  case "compose":
    if(session_module == "gsession")
      // Do not timeout when composing mail...
      id->conf->get_provider(session_module)->set_expire_time(id->misc->session_id, 28800);
    // Checking if we reply or forward a mail instead of just composing it
    switch (action)
    {
      case "reply":
      case "replymove":
      case "replytoallmove":
      case "replytoall":
        action = "reply";
        break;
      case "forward":
        action = "forward";
        break;
      default: action = "compose";
    }
    break;
  case "readmail":
      if(QUERY(mdntype) != "None")
      {
        object mail = CSESSION->cmail;
        int in_drafts = CAMASFOLDERTOOLS.is_draft(CSESSION->mailbox, CSESSION->draftsfolder);
        if (!in_drafts)
          CSESSION->draftuid = 0;
        if (!in_drafts && (mail->headers["disposition-notification-to"] || mail->headers["return-receipt-to"])
          && !(has_value (CSESSION->mails[CSESSION->cmailidx]->imap->FLAGS, IMAP_MDN_FLAG)
                || /* has_value (CSESSION->mails[CSESSION->cmailidx]->imap->FLAGS, "\\Seen") */ 0))
        {
           CSESSION->status = MDNDIALOG;
           id->misc->camas->status = MDNDIALOG;
        }
      }
      break;
      
  default: break;
  }

  return really_output(id, 0, action);
}

//! method: really_output(object id, void|string screen, void|string action)
//!  This part sends the data to the user.
//!  Please don't touch the CSESSION in this screen anymore to avoid side effects
//!  while displaying a screen that don't correspond to CSESSION->status
//! arg: object id
//!  The Caudium request id object
//! arg: void|string screen
//!  If set, displays this screen instead of the screen corresponding to CSESSION->status
//! arg: void|string action
//!  Action returned by create_output
//! returns:
//!  HTTP answer to client with screen HTML code 
mixed really_output(object id, void|string screen, void|string action)
{
  int status = 0;
  string ret;
  string state = "in";
  if (CSESSION->status < 0)
    state = "login";
  else
    if (id->misc->camas->status == LOGOUT || CSESSION->status == LOGOUT)
      state = "out";
  // imho_* stuff keeped for compatibility
  ret  = "<define name=\"imhostate\">"+state+"</define>";
  ret += "<define name=\"composescreen\">"+action+"</define>";
  ret += "<define name=\"imhouser\">"+CSESSION->login+"</define>";
  ret += "<define name=\"imhoframe\">"+id->misc->camas->frame+"</define>";
  ret += "<define name=\"imhomailnotify\">"+(camas_feature (FEAT_MAILNOTIFY)?"yes":"no")+"</define>";

  ret += CAMAS.Tools.make_container("define", ([ "name":"camasframe" ]), id->misc->camas->frame);

  if(!mappingp(id->misc->defines))
    id->misc->defines = ([ ]);
  id->misc->defines["camasframe"] = id->misc->camas->frame;
  
  
  array mboxes = ({ });
  foreach(CSESSION->mailboxes, array mailbox) 
    mboxes += ({ mailbox[MB_FOLDERNAME_IDX] });
  CSESSION->hasnewmail = 0;
  foreach (mboxes, string mbox)
  {
    object cache;
#ifdef CAMAS_DEBUG
    cache = imapclient()->Cache (mbox, CSESSION, this_object());
#else
    cache = CAMASIMAPCLIENT.Cache (mbox, CSESSION, this_object());
#endif
    if (cache->mcache && cache->mcache->new_mails &&
    	sizeof (cache->mcache->new_mails))
    {
        CSESSION->hasnewmail = 1;
    }
    destruct(cache);
  }

  ret += "<define name=\"imhonewmail\">" + CSESSION->hasnewmail + "</define>";

  ret+="<define name=\"imhoscreen\">"+screen+"</define>";

  if(!stringp(screen) || screen=="0")
  {
    screen = "unknown";
    if (screennames[id->misc->camas->status])
      screen = screennames[id->misc->camas->status][0];
    
    if (screennames[CSESSION->status])
      screen = screennames[CSESSION->status][0];
  }
  
  if(id->misc->camas->status != 0)
    status = id->misc->camas->status;
  else
    status = CSESSION->status;

  CDEBUG(sprintf ("Entering status %d\n",id->misc->camas->status || CSESSION->status));
 
  mapping answer;
  string charset = CSESSION->charset;
  string language = CSESSION->language;

  // Finish to handle load/unload of CAMAS Runtime admin if --Xavier
  if (CSESSION->administrator)
  {
    object camas_adminif = id->conf->get_provider("camas_adminif");
    if (objectp(camas_adminif))
      ret += camas_adminif->handle_admin_interface (id, CSESSION);
    else ret += "<b>Camas Runtime Admin Interface not installed</b><end_session />";
  }
  else
  {
    object layout_manager = id->conf->get_provider ("camas_layout_manager");

    string wanted_layout = CSESSION->overridelayout ? CSESSION->overridelayout : CSESSION->layout;
    
    CSESSION->buttonmode = layout_manager->get_button_mode(wanted_layout);

    ret += layout_manager->get_layout_data(wanted_layout, status, id->misc->camas->frame);
  }
  if(CSESSION)
  {
    if(CSESSION->debug && id->prestate->dumpcsession)
      ret += sprintf("<hr /><b>CSESSION:</b><br><pre>%s</pre>", HTML_ENCODE_STRING(CSESSION->dump_me()));
  }
  ret+="\n<!-- Page generated by CAMAS - http://caudium.net/camas/ -->";

  int kludges = 0;

	string val = parse_rxml (ret, id);
	
  val = Locale.Charset.encoder (charset,
                                       CAMAS_LANGUAGE->lang_replacement_strings[language] || "?"
                                      )->feed (val)->drain ();

#ifdef CAMAS_DEBUG
  array btags = ({ "<end_", "<imho_", "<camas_", "<define", "<comment", "<insert" });
  array bsearch = map (btags, lambda (string tag) { return search (val, tag); } );

  int broken = -1;
  foreach (bsearch, int pos)
  if (pos != -1) {
    if (broken == -1)
      broken = pos;
    else if (pos < broken)
      broken = pos;
  }

  if (broken == -1)
    ++global_ok;
  else
    while ((broken != -1) && (kludges < QUERY (rxmlparsercalls))) {
      ++kludges;
      ++global_kludges;
      CDEBUG ("found an unparsed tag. kludges= " + kludges + "\n");
      CDEBUG ("data= " + val[broken..broken+30] + " (...)\n");
      string ok = val[0..broken-1];
      string new_answer;

      new_answer = parse_rxml (val[broken..sizeof(val)-1], id);
      val = ok + new_answer;

      bsearch = map (btags, lambda (string tag) { return search (val, tag); } );
      broken = -1;
      foreach (bsearch, int pos)
      if (pos != -1) {
        if (broken == -1)
          broken = pos;
        else if (pos < broken)
          broken = pos;
      }
    }
#endif

  answer = HTTP_LOW_ANSWER (200, val);
  // Make sure all pages are reloaded and set Content-type
  // the user can set is own content type for example to 
  // output WML or raw XML
  if(id->misc->camas->content_type)
    answer["type"] = id->misc->camas->content_type;
  else
    answer["type"] = "text/html; charset=" + charset;
  id->misc->is_dynamic = 1;

  return answer;
}

//! method: string create_main(object id, int screen)
//!  Method that allows to call for some screen output even if this one isn't the current one. 
//!  Warning : This won't work with new template system, only for imho templates
//! returns:
//!  RXML code for the given screen
string create_main (object id, int screen)
{
  object camas_imhoscreens = id->conf->get_provider("camas_imhoscreens");
  object camas_features = id->conf->get_provider("camas_features");

  if(objectp(camas_imhoscreens) && objectp(camas_features))
  {
    switch (screen) {

    case LOGINFAILEDIMAP:
    case LOGINFAILED:
    case LOGINPROMPT:
      return camas_imhoscreens->login(id, screen, 0);
      break;

    case LOGOUT:
      return camas_imhoscreens->logout(id);
      break;

    case COMPOSE:
      return camas_imhoscreens->compose(id,
                                        camas_features->feature(FEAT_ADDRESSBOOK),
                                        camas_features->QUERY(ldap),
                                        sizeof(QUERY(ispell)),
                                        QUERY(ispelldict),
                                        camas_features->QUERY(siteorg),
                                        camas_features->feature(FEAT_MAILBOXES),
                                        camas_features->QUERY(composesavemail),
                                        camas_features->QUERY(dsn),
                                        camas_features->feature(FEAT_ATTACHMENTS));

      break;

    case ATTACHMENTS:
      return camas_imhoscreens->attachments(id,sizeof(QUERY(uploaddir)));
      break;

    case ADDRESSBOOK:
      return camas_imhoscreens->addressbook(id,camas_feature(FEAT_EDITADDRESSBOOK),1);
      break;

    case ADDRESSBOOK2:
      return camas_imhoscreens->addressbook2(id,1);
      break;

    case LDAPRESULT:
      return camas_imhoscreens->ldapresult(id,camas_features->QUERY(ldapshowou));
      break;

    case LDAPRESULT2:
      return camas_imhoscreens->ldapresult2(id,camas_features->QUERY(ldapshowou));
      break;

    case MAILFILTER:
      return camas_imhoscreens->mailfilter(id,camas_feature(FEAT_EDITFILTERBOOK));
      break;

    case LDAPSEARCH:
      return camas_imhoscreens->ldapsearch(id);
      break;

    case IMPORTADDRESS:
      return camas_imhoscreens->importaddress(id);
      break;

    case EDITADDRESS:
      return camas_imhoscreens->editaddress(id,camas_feature(FEAT_EXTENDEDABOOK));
      break;

    case EDITADDRESSFILTER:
      return camas_imhoscreens->editfilter(id);
      break;

    case MAILINDEX:
      return camas_imhoscreens->mailindex(id,CSESSION->mailpath,camas_features->feature(FEAT_MAILBOXES),camas_features->QUERY(deletemethod), QUERY(delaystepnumber), QUERY (indexdelayformat), QUERY (indexdateshortformat), QUERY (indexdatelongformat));
      break;

    case SEARCHMAIL:
      return camas_imhoscreens->searchmail(id);
      break;

    case FOLDERLIST:
      return camas_imhoscreens->folderlist(id,camas_features->QUERY(foldersinfo));
      break;

    case MDNDIALOG:
      return camas_imhoscreens->mdndialog(id);
      break;

    case READMAIL:
      return camas_imhoscreens->readmail(id,
                                         CSESSION->mailpath,
                                         camas_feature(FEAT_ADDRESSBOOK),
                                         camas_feature(FEAT_SHOWTEXT),
                                         camas_feature(FEAT_SHOWHTML),
                                         camas_features->QUERY(linkwindow),
                                         camas_features->QUERY(mime_remap),
                                         camas_features->QUERY(deletemethod),
                                         camas_feature(FEAT_MAILBOXES),
                                         camas_features->QUERY(showhiddenheaders),
                                         camas_features->QUERY(moveanswered),
					 QUERY (indexdateshortformat),
					 QUERY (indexdatelongformat));
      break;

    case FILES:
      return camas_imhoscreens->files(id,QUERY(uploadquota));
      break;

    case SETUP:
      return camas_imhoscreens->setup(id,
                                      camas_feature(FEAT_USERCANCHANGENAME),
                                      camas_feature(FEAT_USERADDRESS),
                                      camas_feature(FEAT_USERMAILPATH),
                                      camas_feature(FEAT_USERLANGUAGE),
                                      camas_features->QUERY(chgreplymsgprefix),
                                      camas_feature(FEAT_USERHEADERS),
                                      camas_feature(FEAT_SHOWHTML),
                                      camas_feature(FEAT_SHOWTEXT),
                                      camas_feature(FEAT_MAILBOXES),
                                      camas_features->QUERY(deletemethod),
                                      camas_feature(FEAT_USERDRAFTSFOLDER),
                                      camas_feature(FEAT_USERSENTFOLDER),
                                      camas_feature (FEAT_USERANSWEREDFOLDER),
                                      camas_feature(FEAT_USERSENTSAVEATTACHMENTS),
                                      camas_feature(FEAT_USERBCCCOPY),
                                      (camas_features->QUERY(filterbook) && camas_features->QUERY(userautofilter) && camas_features->QUERY(autofilter)),
                                      camas_feature(FEAT_USERSETUPSHOWHIDDENHEADERS),
                                      camas_feature(FEAT_ORGANIZATION),
                                      camas_feature(FEAT_USERTRASHFOLDER));
      break;

    case DIALOGBOX:
      return camas_imhoscreens->dialog(id);
      break;

    case SPELLCHECK:
      return camas_imhoscreens->spellcheck(id);
      break;

    default:
      return "<font color=\"red\" size=\"+2\">Error: no such screen= " + screen + "</font>";
    }
  }
  else
    return "<font color=\"red\" size=\"+2\">Error: Screen module provider was not ready for screen "
           +screen+"...Please contact system administrator with this information "
           "</font>";
}

//! method: void camas_gc()
//!  Used to clean up old sessions
void camas_gc () {
  CDEBUG ("garbage collecting: ");

  if (gc_call_out)
    remove_call_out (gc_call_out);

  int removed = 0;
  int bar = time ();
  // internal gc not needed if session module not load yet 
  // (it will be loaded when user hit find_file() or 
  // not using 123sessions but gsession
  if(!session_module ||
      (session_module && session_module != "123sessions"))
    return;
  
  object sess123 = my_configuration ()->get_provider (session_module);
  mapping sessions = sess123->sessions ();

  if (sessions)
    foreach (indices (sessions), string foo) 
    {
      object csession = sessions[foo]->values->CAMAS;
      int tmax;
      if(csession)
      {
        if (!csession->autologout
            || !sscanf(csession->autologout, "%d", tmax))
          tmax = 20;
        if (csession->status == COMPOSE)
          tmax = 480; // Do not timeout when composing mail...
        CDEBUG("session="+foo+" for "+csession->login+", tmax="+tmax+"\n");
        CDEBUG("lifetime="+(csession->lastusage + tmax * 60 - bar)); 
        if (csession->lastusage && (bar - (csession->lastusage) > (tmax * 60)))
        {
          CAMAS.Log.log (my_configuration (), "autologout", 
            ([ "login": csession->login ]));
          CAMAS.Tools.low_destroy_session(csession);
          m_delete(sessions[foo]->values, "CAMAS");
#if constant(thread_create)
          object lock = global_lock->lock();
#endif
          mixed err = catch {
            sess123->kill_session (foo);
          };
#if constant(thread_create)
          destruct(lock);
#endif
          if(err)
            report_error("error in camas.pike: %s\n", describe_backtrace(err));
          CDEBUG("session="+foo+" removed\n");
          removed++;
        }
      }
    }

    if (removed)
      CDEBUG("removed "+removed+" session="+((removed == 1) ? "" : "s")+"\n");
    CDEBUG("done.\n");
  gc_call_out = call_out (camas_gc, QUERY (gctimeout));
}

//! method: void load_other_modules(object conf)
//!  load some modules after startup because they depends on each other
//!  other modules are loaded in start()
void load_other_modules(object conf)
{
  array dependencies = ({ });
  object oldsess = conf->get_provider("123sessions");
  object gsess = conf->get_provider("gsession");
  // Loads the 123session module if gsession is not loaded
  if(!oldsess && !gsess)
    dependencies += ({ "123session" });
#if constant(thread_create)
  object lock = global_lock->lock();
#endif
  mixed err = catch {
    if(oldsess)
      session_module = "123sessions";
    if(gsess)
      session_module = "gsession";
    firststart = 0;
  };
#if constant(thread_create)
  destruct(lock);
#endif
  if(err)
    report_error("error in camas.pike: %s\n", describe_backtrace(err));
  module_dependencies (conf, dependencies);
}

//! method: mapping find_file(string file, object id);
//!  Method called when Caudium get a request under CAMAS mount point
//!  Some deals are made about sessions and some stuff, then the
//!  process_request(object id) method is called
//! arg: string file
//!  The path under the mount point of CAMAS
//! arg: object id
//!  The request object
mapping find_file (string file, object id)
{
  array (string) fileparts = file / "/";

  if(firststart)
    load_other_modules(id->conf);

  // If we want to logout the user when hitting the mountpoint,
  // destroy any existing camas session leaving
  // before displaying the login screen
  if(QUERY(resetatlocationhit) && sizeof(file) == 0 && CSESSION && 
      CSESSION->imapclient && CSESSION->imapclient->loggedin)
  {
    CDEBUG(sprintf("User %s hit my mountpoint while already logged: destroying session",
          CSESSION->login));
    CAMAS.Tools.destroy_csession(id);
  }

  id->misc->cacheable = 0;
  id->misc->is_dynamic = 1;

  // don't mess the root of id->misc for CAMAS-specific stuff
  id->misc->camas = ([ 
                      "target": "",
                      "nexturl": "",
                      "nexttarget": "",
                      "nextsidetarget": "",
                      "nextpage": "",
                      "sidetarget": "",
                      "uptarget": "",
                      "nextuptarget": "",
                    ]);
  
  if (id->supports->msie && /*id->prestate &&*/ !id->prestate->nocache)
    id->prestate += (< "nocache" >);

  if ((sizeof (fileparts) > 1) && ((fileparts[1] == "about") 
        || (fileparts[1] == "jump"))) 
  {
    object sess_module = id->conf->get_provider(session_module);
    CAMAS.Tools.destroy_csession(id);
    if (fileparts[1] == "about") 
      return HTTP_REDIRECT ("http://caudium.net/camas/", id);
    else
      if (fileparts[1] == "jump") 
      {
        int pos = search (file, "jump");
        string url = file[pos+5..sizeof (file)-1];
        if (has_value (url, "http:/") && (!has_value (url, "http://")))
          url = replace (url, "http:/", "http://");
        return HTTP_REDIRECT (url, id);
      }
  }

  if (!CSESSION)
    CSESSION = Session();	
  string frame  = "";
  string screen = "";

  if (sizeof (fileparts) >= 1)
  {
    array parts = fileparts[0]/"-";

    frame = parts[0];

    if(sizeof(parts)==3)
      screen = parts[1];
  }

  if (sizeof (frame) < 1)
    frame = "_top";

  // set status for fake screen output in really_output()
  if(sizeof(screen))
    id->misc->camas->status = CAMAS.ScreenTools->screen_name_to_id(screen);
	else
		id->misc->camas->status = CSESSION->status;
 
  // Template system files are mounted under <CAMAS mount point>_camastemplates
  // As a result, if a file is missing from the template, the request come to CAMAS main module
  // that believes that _camastemplates is the frame target and launches the whole system for nothing
  if(frame=="_camastemplates")
    return 0;

  // someone try to fool us
  if ((file != "" && fileparts[-1] != "sslredir")|| (file == "" && id->variables->actionchangesort))
  {
    if (!CSESSION->ok)
      return redirect (id, "");
  }

  // session has to be created or re created if the user change ip and don't have
  // the right to do so
  if (!CSESSION->ok || (!QUERY (allowipchange) && (CSESSION->ip != id->remoteaddr))) 
  {
    if(session_module == "gsession")
    {
      object sess_module = id->conf->get_provider(session_module);
      /* sess_module->set_expire_hook(id->misc->session_id,
        CAMAS.Tools.destroy_csession(id)); */
      sess_module->set_expire_time(id->misc->session_id, 120);
    }
    if ((QUERY (ssl3redir) || sizeof (QUERY (urloverride)) > 0)
     	&& QUERY(ssl3redir) && (!QUERY (sslfaulty) || !id->supports->ssl_faulty)
      && fileparts[-1] != "sslredir")
    {
      // To make sure the session has been
      // redirected (to a safe port) without sending the session in clear
      // during the redirect.
      CAMAS.Tools.destroy_csession(id);	
      return redirect (id, "sslredir");
    }
    
    default_session (id);

    id->misc->camas->status = LOGINPROMPT;
    CSESSION->status = LOGINPROMPT;
    // For example automatic reloads should not reset idle counter
    CSESSION->ip = id->remoteaddr;
    CSESSION->ok = 1;
    frame = "_top";
  }

  string foo;
  id->misc->camas->frame = frame;
  CSESSION->lastusage = time();

  return process_request (id);
}


/* START AUTOGENERATED DEFVAR DOCS */

//! defvar: location
//! Mailreader server location.
//!  type: TYPE_LOCATION
//!  name: Mountpoint: Mountpoint
//
//! defvar: resetatlocationhit
//! <p>If the user hits the mountpoint (eg http://server.tld/mountpoint), he's logged out if already logged in and faces the login page.</p><p>You may enable this feature if you have some problems that could get the session corrupted (eg frame recursion in a layout) and want to automatically reset the user's session.</p><p>If you enable this feature, make sure you don't hit the mountpoint in normal CAMAS operation (standard CAMAS tags action don't)</p>
//!  type: TYPE_FLAG
//!  name: Mountpoint: Logout when hitting mountpoint
//
//! defvar: indexdelayformat
//! Set a format for the delay.<br />You can use the following replacements :<ul><li><b>$DDAYS</b>: Day</li><li><b>$DHOURS</b>: Hour</li><li><b>$DMINUTES</b>: Minute</li><li><b>$DSECONDS</b>: Second</li></ul>
//!  type: TYPE_TEXT_FIELD
//!  name: Misc:User interface: Mailindex: Delay: Format
//
//! defvar: displayimaperrors
//! If this is set to 'yes', the IMAP error string from the server will be shown to the user in the error dialog. The text comes from the server and cannot be altered nor translated.
//!  type: TYPE_FLAG
//!  name: Misc:User interface: Display IMAP errors to user
//
//! defvar: displaysmtperrors
//! If this is set to 'yes', the SMTP error string from the server will be shown to the user in the error dialog. The text comes from the server and cannot be altered nor translated.
//!  type: TYPE_FLAG
//!  name: Misc: User interface: Display SMTP errors to user
//
//! defvar: displaysmtpok
//! If this is set to 'yes', the user will see confirmation that his mail has been sent.The text comes from CAMAS.
//!  type: TYPE_FLAG
//!  name: Misc: User interface: Display SMTP confirmations to user
//
//! defvar: useutf-8
//! UTF-8 is an encoding for 16-bit characters, which is used in CAMAS internally. If this option is enabled, CAMAS will output UTF-8 data to the browser if the browser supports it. If it is disabled, the character set will be taken from the current language file.
//!  type: TYPE_FLAG
//!  name: Misc: User interface: Output UTF-8 (if possible)
//
//! defvar: imapidlelogout
//! Close an open IMAP connection after this many seconds. The user session may still be active, and will not be disturbed. The IMAP connection will be reopened when the user issues an IMAP command.
//!  type: TYPE_INT
//!  name: Incoming mail:Idle IMAP connection timeout
//
//! defvar: mboxencode
//! Choose whether CAMAS should use IMAPs modified UTF-7 coding when refering to mailboxes or if it should send mailbox names unmodified. If you're having trouble accessing mailboxes with non-USASCII characters, try changing this option.
//!  type: TYPE_STRING_LIST
//!  name: Incoming mail:Mailbox addressing
//
//! defvar: allowmailpathoverwrite
//! If set to yes, Camas will fetch the mail path informations from the IMAP server so that you don't need to fill in values for 'Mail path: Shared path' and 'Mail path: Personal path'. You always need to fill in values for these if your IMAP server doesn't support <a href="http://www.ietf.org/rfc/rfc2342.txt">RFC 2342</a>
//!  type: TYPE_FLAG
//!  name: Incoming mail:Mail path: Allow IMAP server to overwrite namespace
//
//! defvar: defaultsharedpath
//! The default shared hierarchy path. Is usually "shared."(Courier-IMAP).<br />
//!  type: TYPE_STRING
//!  name: Incoming mail:Mail path: Shared path
//
//! defvar: sharemailcache
//! Camas uses a cache to store the mails of each user by default. If this option is set to yes then Camas will use a global cache for every users for special folders like shared ones. In order for this to work your users must have exactly the same data for the same shared folders name they use. If it is the case and if your users have lot of mails in their shared folders, the performances will be better both on Camas and on the IMAP server
//!  type: TYPE_FLAG
//!  name: Incoming mail: Share cache between users
//
//! defvar: defaultshareddepth
//! The default shared hierarchy depth. Might be usually 2, but might be more.
//!  type: TYPE_INT
//!  name: Incoming mail:Default shared mail folder depth
//
//! defvar: defaultmailpath
//! The default mail folder path. Is usually "INBOX" (WU-IMAP) or "INBOX." (Courier-IMAP). Can be changed by the user, although the user's preferences is always saved here.
//!  type: TYPE_STRING
//!  name: Incoming mail:Mail path: Personal path
//
//! defvar: hidemboxes
//! Optionally hide mailboxes starting with the specified string. Mailboxes starting with '.' are often not really mailboxes.
//!  type: TYPE_STRING
//!  name: Incoming mail:Hide mailboxes starting with
//
//! defvar: notreloadmailboxes
//! If set, the mailboxes whose names match one of these will not be reloaded when the user check for new mail. This can be usefull for mailboxes like 'Trash' if it has lot of mails and if the IMAP server is slow so that check new mail is faster. <br />You must separate the mailboxes names with a comma.
//!  type: TYPE_STRING|VAR_EXPERT
//!  name: Incoming mail:Do not reload these mailboxes
//
//! defvar: sendmethod
//! Method to use when sending mail.
//!  type: TYPE_STRING_LIST
//!  name: Outgoing mail:Method
//
//! defvar: sendmail
//! Location of sendmail.
//!  type: TYPE_STRING
//!  name: Outgoing mail:Sendmail
//
//! defvar: smtpserver
//! The address of the SMTP-server.
//!  type: TYPE_STRING
//!  name: Outgoing mail:SMTP server address
//
//! defvar: smtpport
//! The portnumber of the SMTP-server.
//!  type: TYPE_INT
//!  name: Outgoing mail:SMTP server-port
//
//! defvar: smtpmaildomain
//! The outgoing mail domain.
//!  type: TYPE_STRING
//!  name: Outgoing mail:SMTP mail domain
//
//! defvar: addmaildomain
//! If this is enabled, the default mail domain will be added to addresses without a "@" before sending them to the SMTP server.
//!  type: TYPE_FLAG
//!  name: Outgoing mail:Complete unqualified addresses
//
//! defvar: ssl3redir
//! Tries to find a SSL3 port to redirect the initial requests to. Useful if your virtual server has more than one listen port and you prefer SSL3 for CAMAS.
//!  type: TYPE_FLAG
//!  name: URL:Redirect to SSL3
//
//! defvar: urloverride
//! Specify your server URL here if CAMAS fails to guess the correct URL. Example: 'https://your.server.com'.
//!  type: TYPE_STRING
//!  name: URL:Server URL override
//
//! defvar: uploaddir
//! Directory to hold uploaded attachment files. Leave empty to disable persistent attachment support. 
//!  type: TYPE_STRING
//!  name: Misc:Files: File upload directory
//
//! defvar: uploadquota
//! Amount of Kb each user is allowed to upload. 0 gives unlimited quota.
//!  type: TYPE_INT
//!  name: Misc:Files: File upload user quota
//
//! defvar: uploadsoftquota
//! Amount of Kb by wich the user can overrun the quota for the last uploaded file.
//!  type: TYPE_INT
//!  name: Misc:Files: File upload soft user quota
//
//! defvar: ispell
//!  type: TYPE_STRING
//!  name: Misc:Spelling : Speller location
//
//! defvar: ispelldict
//! Commaseparated list of speller dictionaries, default first.(-d option). Display name can be specified after a colon (swedish:Svenska).
//!  type: TYPE_STRING
//!  name: Misc:Spelling : Speller dictionaries
//
//! defvar: speller
//! CAMAS supports ispell and aspell. 
//!  type: TYPE_STRING_LIST
//!  name: Misc:Spelling : Spellchecker
//
//! defvar: showprefsbox
//! If "no", the preferences folder is not displayed.
//!  type: TYPE_FLAG
//!  name: User preferences:Show user preferences folder
//
//! defvar: debug
//! When on, debug messages will be logged in Caudium's debug logfile. This information is very useful to the developers when fixing bugs.<br /> <i>Note</i>: One of the usefull features is to put a prestate 'dumpcsession' to the url so that Camas dumps the current session at the end of the webpage
//!  type: TYPE_FLAG
//!  name: Debug: Main module debug
//
//! defvar: debugvariables
//! A comma separated list of Camas session values to debug. Each call for reading or writing to this variable will be displayed in the debug log.<br /><i>Note</i>: You have to reload this module once you changed this.
//!  type: TYPE_STRING
//!  name: Debug: Debug these session variables
//
//! defvar: debugvariables_get
//! If yes, read only calls from Camas to this variable will be output in the debug log
//!  type: TYPE_FLAG
//!  name: Debug: Debug read only call
//
//! defvar: debugvariables_set
//! If yes, write only calls from Camas to this variable will be output in the debug log
//!  type: TYPE_FLAG
//!  name: Debug: Debug write only call
//
//! defvar: wrapcolumn
//! Wrap outgoing message lines at this column. If set to zero, no wrapping will take place.
//!  type: TYPE_INT
//!  name: Outgoing mail:Wrap line column
//
//! defvar: allowipchange
//! Should CAMAS allow one session to use different IP addresses? If some users are using dynamic addresses which change even during a session (e.g. while composing mail) then this option should be "yes". A reason to set it to "no" is that a stolen session id (from cookie- or log file) cannot be used from a remote machine.
//!  type: TYPE_FLAG
//!  name: URL: Allow changing user IP-address
//
//! defvar: morecharsets
//! Commaseparated list of charctersets wich CAMAS tries to use for outgoing mail before the last resort, UTF-8. CAMAS first tries to fit the mail into the same characterset used by the mail the user replied to. Then it tries iso-8859-1, the charcterset of the active language file, this list of charactersets and UTF-8. Specify those charactersets your users usually use to avoid their mail being sent as UTF-8.
//!  type: TYPE_STRING
//!  name: Outgoing mail:Character sets
//
//! defvar: indexdatelongformat
//! Set a format for the date if it is more than one week ago.<br />You can use the following replacements :<ul><li><b>$DD</b>: Day</li><li><b>$MM</b>: Month</li><li><b>$YY</b>: Short year</li><li><b>$YYYY</b>: Long year</li><li><b>$HOURS</b>: Hours</li><li><b>$MINUTES</b>: Minutes</li><li><b>$SECONDS</b>: Seconds</li><li><b>$TZ</b>: Timezone</li></ul>
//!  type: TYPE_TEXT_FIELD
//!  name: Misc:User interface: Mailindex: Date: Long format
//
//! defvar: indexdateshortformat
//! Set a format for the date if it is within the last week.<br />If nothing set, use the <i>&quot;normal&quot;</i> full output, specified in <i>&quot;Mailindex: Date: Long format&quot;</i><br />You can use the following replacements :<ul><li><b>$WDAY</b>: Short format weekday-name (eg. Fri for friday)</li><li><b>$WEEKDAY</b>: Full weekday-name (like Thursday)</li><li><b>$HOURS</b>: Hours</li><li><b>$MINUTES</b>: Minutes</li><li><b>$SECONDS</b>: Seconds</li><li><b>$TZ</b>: Timezone</li></ul>
//!  type: TYPE_TEXT_FIELD
//!  name: Misc:User interface: Mailindex: Date: Short format
//
//! defvar: delaystepnumber
//! Specify the number of delay step you want.<br />When set, you'll need as many CSS classes<br /><p><b>Example:</b><br />If set to 3, entries <i>delay0</i>, <i>delay1</i> and <i>delay2</i> might be in your CSS.<br />Moreover, if delay image path is set, you'll need <i>delay0.gif</i>, <i>delay1.gif</i> and <i>delay2.gif</i></p>
//!  type: TYPE_INT
//!  name: Misc:User interface: Mailindex: Delay: Delay Step number
//
//! defvar: notifysent
//! Display a "sent message" window to the user.
//!  type: TYPE_FLAG
//!  name: Outgoing mail:Sent message notification
//
//! defvar: mdntype
//! MUA Mail Delivery Notification type.
//!  type: TYPE_STRING_LIST
//!  name: Outgoing mail:MDN support
//
//! defvar: rxmlparsercalls
//! Maximum calls to parse_rxml
//!  type: TYPE_INT|VAR_EXPERT
//!  name: Debug:Maximum calls to parse_rxml
//
//! defvar: buggyimap
//! Some IMAP servers (e.g. Courier 1.3) are not RFC-compliant.<br />For example, they will not return "[TRYCREATE]" when appending a message in a non-existent folder.<br /> This will enable various workarounds to make CAMAS work as needed. <br /><hr />RFC2060, section 6.3.11 (APPEND Command)<br /><pre>If the destination mailbox does not exist, a server <b>MUST</b> return an
//!error, and <b>MUST NOT</b> automatically create the mailbox.  Unless it is
//!certain that the destination mailbox can not be created, the server
//!<b>MUST</b> send the response code "[TRYCREATE]" as the prefix of the text
//!of the tagged NO response.  This gives a hint to the client that it
//!can attempt a CREATE command and retry the APPEND if the CREATE is
//!successful.</pre>
//!  type: TYPE_FLAG|VAR_EXPERT
//!  name: Incoming mail:IMAP server is not RFC-compliant
//
//! defvar: feedimapcache
//! If set to yes, Camas will launch a background IMAP client each time a user logged in in order to feed the cache with data. From the user's point Camas looks faster but it will eat more memory or load on your IMAP server especially if your users usually saw a few mails while they have lot of other they don't read
//!  type: TYPE_FLAG
//!  name: Incoming mail:Feed the IMAP cache at startup
//
//! defvar: upgrading
//! CAMAS will read 'IMHO Preferences' if 'CAMAS Preferences' are not found.
//!  type: TYPE_FLAG
//!  name: Compatibility:Upgrading from IMHO to CAMAS
//
//! defvar: defaultlayout
//! The layout to use if none found in the old preferences files.
//!  type: TYPE_STRING
//!  name: Compatibility:Layout to use
//
//! defvar: foldernamemaxlength
//! Set the max length for a new mailbox name. If set to 0, there will be no check.
//!  type: TYPE_INT|VAR_MORE
//!  name: Limits:Max length allowed for a mailbox name
//
//! defvar: maxmailsize
//! If the mail data exeeds this size (in bytes), it will only be shown as a download link.
//!  type: TYPE_INT
//!  name: Limits:Max mail size to display
//
//! defvar: quota
//! Set an informational quota for the mailboxes (in kB). If set to 0, no quota will be displayed. Provided rxml tag: <b>&lt;imho_quota&gt;</b> (the quota itself), and <b>&lt;imho_use&gt;</b> for the percentage of the quota used (rounded to an integer).
//!  type: TYPE_INT
//!  name: Limits:Mailbox quota
//
//! defvar: gctimeout
//! Sets the delay (in seconds) between two garbage collections.
//!  type: TYPE_INT
//!  name: Internals:Garbage collecting frequency
//
//! defvar: sslfaulty
//! If set, SSL will be disabled if the client has <b>id->supports->ssl_faulty</b> set.
//!  type: TYPE_FLAG
//!  name: URL:No SSL if ssl_faulty
//
//! defvar: gabook
//! Adds the option Global AddressBook, an site-wide address book handled by Camas Global AddressBook module, which gives to all users of CAMAS a list of centralized emails. If you set this to "No", then you can remove the Global AddressBook module to save memory...
//!  type: TYPE_FLAG
//!  name: Misc:Options: Enable Global AddressBook
//
//! defvar: cadmif
//! Allow CAMAS Runtime Admin Interface to be accessed thru CAMAS login method. If you set this to "No", then you can remove the Runtime Admin Interface module to save memory...
//!  type: TYPE_FLAG
//!  name: Misc:Options: Enable Runtime Admin Interface
//
//! defvar: allowfailauth
//! If set to yes the user will be able to logged in if the IMAP server sucessfully authenticate him but the auth module failed to find him.
//!  type: TYPE_FLAG
//!  name: Misc:Options: Allow failed authentification from auth module
//

/*
 * If you visit a file that doesn't contain these lines at its end, please
 * cut and paste everything from here to that file.
 */

/*
 * Local Variables:
 * c-basic-offset: 2
 * c-set-style: k&r
 * End:
 *
 * vim: softtabstop=2 tabstop=2 expandtab autoindent formatoptions=croqlt smartindent cindent shiftwidth=2
 */

