/*
 * Caudium - An extensible World Wide Web server
 * Copyright  2000-2004 The Caudium Group
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */
/*
 * $Id: camas_tags.pike,v 1.311.2.7 2004/04/25 21:40:57 vida Exp $
 */

//
//! module: CAMAS: CAMAS Tags
//!  This module handle CAMAS tags and containers used in layouts.
//! inherits: module
//! inherits: caudiumlib
//! type: MODULE_PARSER
//! cvs_version: $Id: camas_tags.pike,v 1.311.2.7 2004/04/25 21:40:57 vida Exp $
//
#include <module.h>
inherit "module";
inherit "caudiumlib";
#include <camas/screens.h>	// For screennames
#include <camas/msg.h>		// MSG() Language macros
#include <camas/globals.h>	// Global definitions
#include <camas/pmods.h>        // Local or normal camas pike modules
#include <camas/addressbook.h> // Address book defines and translations

constant cvs_version   = "$Id: camas_tags.pike,v 1.311.2.7 2004/04/25 21:40:57 vida Exp $";
constant module_type   = MODULE_PARSER|MODULE_PROVIDER;
constant module_name   = "CAMAS: CAMAS Tags";
constant module_doc    = "This module handle CAMAS tags and containers used in layouts."
			"<br /><b>This module is automatically "
                        "selected if you select \"CAMAS: Main module\".</b>";
constant module_unique = 1;
constant thread_safe   = 1;

mapping(string:array) program_screens;
object parser = cache_parse_html();
#ifdef CAMAS_DEBUG
int rxml_debug;
#endif

void create()
{
#ifdef CAMAS_DEBUG
  defvar("debug",0,"Debug",TYPE_FLAG,"Debug the call / errors into Caudium "
         "error log ?");
  defvar("rxml_debug",0,"Debug RXML cache",TYPE_FLAG,"Debug the CAMAS RXML cache call into Caudium "
         "error log ?");
#endif
  defvar("ent_parse", 1,"Parse entities", TYPE_FLAG,
         "Parse &amp;camasmsg.*; entities even if RXML is not XML ?");
  defvar ("screenslocation",({ "modules/camas/camas_screens/" }), "Screens path", TYPE_DIR_LIST,
         "A list of path to Camas Tags screen directory separated by commas.");

}

void start(int num, object conf) 
{
  add_constant("camas_tags", this_object());
  parser->flush_cache();
  program_screens = ([ ]);
#ifdef CAMAS_DEBUG
  rxml_debug = QUERY(rxml_debug);
#endif
  foreach(QUERY(screenslocation), string path2file)
  {
    array(string) screens = get_dir(path2file) || ({ });
	
		if(!sizeof(screens))
			report_warning("No camas screen found in "+path2file);
	
    foreach(screens, string file)
    {
      if (file[0..5] == "camas_" && (lower_case (file[sizeof (file) - 5..]) == ".pike"))
      {
        string real_path = combine_path(path2file, file);
        string screen_name = upper_case(file[sizeof("camas_")..sizeof(file)-sizeof(".pike")-1]);
        if(program_screens[screen_name] && program_screens[screen_name][1] != real_path)
          report_warning("Warning: overriding screen name '" + screen_name + "', old file is "+
              program_screens[screen_name][1] + ", new file is "+ real_path + "\n");
        object ee = ErrorContainer();
        master()->set_inhibit_compile_errors (ee);
        mixed err = catch {
          program lscreen = compile_file(real_path);
          program_screens += ([
            screen_name: 
              ({ 
                 lscreen(),
                 real_path,
              })
          ]);
          if(program_screens[screen_name][0]->parser)
            program_screens[screen_name][0]->parser->flush_cache();
        };
        master()->set_inhibit_compile_errors (0);
        if(!err)
        {
           CDEBUG("Loading screen name "+ upper_case(screen_name) +" from file '"+real_path+"'");
        }
        else
        {
           report_error("CAMAS: could not load screen file: " + err[0]);
           if(strlen(ee->get()))
              report_error(sprintf("%s", ee->get()));
           report_error(describe_backtrace(err));
        }
      }
    }
  }
}

void stop()
{
  foreach(indices(program_screens), string screen_name)
  {
    if(program_screens[screen_name] && objectp(program_screens[screen_name][0]))
      destruct(program_screens[screen_name][0]);
  }
  program_screens = ([ ]);
}

string status()
{
  string out = "";
  out += "<center>The following screens are loaded from external files</center><br/>";
  out += "<table border=\"1\"><tr><th>Screen name</th><th>Source file</th></tr>";
  foreach(sort(indices(program_screens)), string screen_name)
  {
    out += "<tr><td>" + screen_name + "</td><td>" +
      program_screens[screen_name][1] + "</td></tr>\n";
  }
  out += "</table>";
  return out;
}

/* A wrapper around parse_html with camas specific cache

   The idea is quite simple, if we have nothing in the cache, we
   replace the normal callback functions with our ones for which
   we know the output, these functions also take care of putting
   the arguments given to them by the parser in cache.
 
   As we know the output from our functions and we get the result from
   the parser we know where the parser replaced the output from our 
   functions in the content. So we have all we need: the arguments given
   to the functions and where to put their result in the RXML output.

   If we have something in the cache (for the given layout and parser place 
   in the camas_tags pike code) then let's just start the callback functions
   with the arguments we have in cache and replace their output in the RXML 
   output.
*/
class cache_parse_html {
  // quite beautiful isn't it ? :)
  private mapping(string:array(string|mapping(string:array(string)))) rxmlcache;
  private object lock = Thread.Mutex();
  constant magic_string = "320948329009742";

  // lookup for an entry in the rxml cache
  // returns a string if it is here, 0 otherwise
  // note: It also launch the callback functions
  private string|int cache_lookup(string layout, string global_pos, mapping(string:function) tags,
    mapping(string:function) containers, mixed ...args)
  {
    string cache_key = layout + global_pos;
    RXML_CACHE_DEBUG("looking up cache for key " + cache_key);
    array cache = rxmlcache[cache_key];
    if(cache)
    {
      string contents = cache[0];
      if(contents)
      {
        RXML_CACHE_DEBUG("Hit!");
        mapping cached_tags = copy_value(cache[1]);
        array tag_names = indices(tags);
      	foreach(tag_names, string tag_name)
        {
          if(cached_tags[tag_name])
            foreach(cached_tags[tag_name], array tag_args)
            {
              int wide_string = 0;
      	      RXML_CACHE_DEBUG(sprintf("Calling %O with args %O\n", tags[tag_name], 
    	        ({ tag_name, tag_args[0], tag_args[1] }) + args));
              string fun_res;
      	      fun_res = tags[tag_name](copy_value(tag_name), tag_args[0], tag_args[1], @args);
              RXML_CACHE_DEBUG(sprintf("fun_res=%O", fun_res));
              // replace doesn't work well on wide strings
              if(String.width(contents) > 8)
              {
                contents = string_to_utf8(contents);
                fun_res = string_to_utf8(fun_res);
                wide_string = 1;
              }
#ifdef CAMAS_DEBUG
              string contents2 = replace(contents, magic_string + tag_name 
                  + tag_args[0], fun_res);
              if(contents == contents2)
                report_warning("replace failed for contents: %O\n", contents);
              else
                contents = contents2;
#else
              contents = replace(contents, magic_string + tag_name 
                  + tag_args[0], fun_res);
#endif
              if(wide_string)
                contents = utf8_to_string(contents);
            }
        }
        mapping cached_containers = copy_value(cache[2]);
        array container_names = indices(containers);
        foreach(container_names, string container_name)
        {
          if(cached_containers[container_name])
            foreach(cached_containers[container_name], array container_args)
            {
              int wide_string = 0;
              RXML_CACHE_DEBUG(sprintf("Calling %O with args %O\n", containers[container_name], 
                ({ container_name, container_args[0], container_args[1], container_args[2] })
                 + args));
              object id = args[0];
              string fun_res;
              fun_res = containers[container_name](copy_value(container_name),
                container_args[0], container_args[1], container_args[2], @args);
              RXML_CACHE_DEBUG(sprintf("fun_res=%O", fun_res));
              // replace doesn't work well on wide strings
              if(String.width(contents) > 8)
              {
                contents = string_to_utf8(contents);
                fun_res = string_to_utf8(fun_res);
                wide_string = 1;
              }
#ifdef CAMAS_DEBUG
              string contents2 = replace(contents, magic_string + container_name
                  + container_args[0], fun_res);
              if(contents == contents2)
                 report_warning("replace failed for contents: %O\n", contents);
              else
                contents = contents2;
#else
              contents = replace(contents, magic_string + container_name
                  + container_args[0], fun_res);
#endif
              if(wide_string)
                contents = utf8_to_string(contents);
            }
        }
      }
      return contents; 
    }
    RXML_CACHE_DEBUG("Miss!");
    return 0;
  }

  private void cache_set(string layout, string global_pos, string contents, 
      array(string) tag_names, array(string) container_names)
  {
    mapping(string:array(string)) tags2cache = ([ ]), containers2cache = ([ ]);
    mapping(string:function) tags_cb = ([ ]), containers_cb = ([ ]);
    tags_cb = mkmapping(tag_names, allocate(sizeof(tag_names),  
    lambda(string _tag_name, int pos, mapping _args)
				  {
		        if(!tags2cache[_tag_name])
					    tags2cache += ([ _tag_name:({ ({ global_pos + pos, _args }) })]);
					  else
					    tags2cache[_tag_name] += ({ ({ global_pos + pos, _args }) });
					  return magic_string + _tag_name + global_pos + pos;
		 }));
    containers_cb = mkmapping(container_names, allocate(sizeof(container_names),
              lambda(string _tag_name, int pos, mapping _args, string _contents)
						  {
				        if(!containers2cache[_tag_name])
						  	  containers2cache += ([ _tag_name: 
										({ ({ global_pos + pos, _args, _contents }) })
									    ]);
							  else
							    containers2cache[_tag_name] += ({ ({ global_pos + pos, _args, _contents }) });
							  return magic_string + _tag_name + global_pos + pos;
						  }));
    string output_parsed_contents = CAMAS.Parse.parse_html2(contents, tags_cb, containers_cb);
    RXML_CACHE_DEBUG(sprintf("filling cache with contents=%s, tags2cache=%O, containers2cache=%O\n",
    output_parsed_contents, tags2cache, containers2cache));
    string cache_key = layout + global_pos;
    object _lock = lock->lock();
    rxmlcache += ([ cache_key:
                    ({
              				  output_parsed_contents,
              				  tags2cache,
               				  containers2cache
        			       })
		              ]);
    destruct(_lock);
  }

  string run(string contents, mapping(string:function) tags,
    mapping(string:function) containers, string layout, int|string pos, mixed ...args)
  {
    string result;
    string global_pos = (string) pos;
    result = cache_lookup(layout, global_pos, tags, containers, @args);
    if(result)
      return result;
    RXML_CACHE_DEBUG("cache not found for key " + layout + global_pos); 
    cache_set(layout, global_pos, contents, indices(tags), indices(containers));
    return cache_lookup(layout, global_pos, tags, containers, @args);
  }

  void flush_cache()
  {
    RXML_CACHE_DEBUG("cache_parse_html: flushing cache");
    object _lock = lock->lock();
    rxmlcache = ([ ]);
    destruct(_lock);
  }
  
  void create()
  {
    rxmlcache = ([ ]);
  }

  void destroy()
  {
    // we don't use flush_cache here because CDEBUG can cause a backtrace
    // since QUERY(debug) will not work when this module is stopped 
    // (and this function should be call each time this module is stopped)
    object _lock = lock->lock();
    rxmlcache = ([ ]);
    destruct(_lock);
  }
};

// Callback for parsing scopes
mixed cb_scopes(object p, string scope, string name, object id, mixed ... extra)
{
  multiset available_entities = (< "camasmsg", "camas" >); 
  if (!available_entities[scope]) return 0;    // Do we need to support other kind of
  // Entities in camas ? Maybe...
  // Taken from xmlparse.pike, thanks David :))
  mixed ret;
  string encoding;
  array tmp = (p->tag_name()) / ":";
  encoding = tmp[1..] * ":";
  if(id->misc->scopes[scope]) {
    ret =  id->misc->scopes[scope]->get(name, id, @extra);
    if(!ret) return "";
    if(stringp(ret)) return roxen_encode(ret, encoding);
    if(arrayp(ret)) return Array.map(ret, roxen_encode, encoding);
  }
  return 0;
}

//! method: string parse_camas_entities(string contents, object id)
//!  Method to parse CAMAS entities in the string contents
//!  Mainly used if some external modules needs to get CAMAS entities
//!  parsed in their data
//! arg: string contents
//!  The string to parse
//! arg: object id
//!  The CAMAS request object id
//! returns:
//!  string with CAMAS entities parsed
string parse_camas_entities(string contents, object id)
{
  return parse_scopes(contents, cb_scopes, id);
}

//! method: string query_provides()
//!  Identifier for MODULE_PROVIDER
string query_provides()
{ 
  return "camas_tags";
} 


// Tags and containers now

//! method: query_tag_callers ()
//!  Returns the top level tags this module provides
mapping query_tag_callers()
{
  return 
    ([
      "meta"              : tag_meta,
      "camas_meta"        : tag_camas_meta,
      "camas_frame"       : tag_camas_frame,
      "camas_totalsize"   : tag_camas_totalsize,
      "camas_about"       : tag_camas_about,
      "camas_use"         : tag_camas_use,
      "camas_quota"       : tag_camas_quota,
      "camas_jslogout"    : tag_camas_jslogout,
    ]);
}

//! method: query_container_callers ()
//!  Returns the top level containers this module provides
mapping query_container_callers ()
{
  return 
    ([
      "camas_page"        : container_camas_page,
      "camas_mailboxlist" : container_camas_mailboxlist,
      "camas_form"        : container_camas_form,
      "camas_action"      : container_camas_action,
      "camas_is_sent"     : container_camas_is_sent,
      "camas_is_answered" : container_camas_is_answered,
      "camas_is_trash"    : container_camas_is_trash,
      "camas_is_draft"    : container_camas_is_draft,
      "camas_is_in_inbox" : container_camas_is_in_inbox, 
      "camas_is_in_shared": container_camas_is_in_shared
    ]);
}

//! container: camas_page
//!  Container delimiting code for CAMAS tags
//!  Usefull if you want to use tags and/or entities outside <camas_form></>
string container_camas_page(string tag_name, mapping args, string contents, object id)
{
  object camas_main = CAMAS_MODULE;
  if (CSESSION && objectp(camas_main))
  {
    CAMAS.Tools.setup_state(id,args);

    if (!id->misc->_xml_parser)
      if (QUERY(ent_parse))
        contents = parse_scopes(contents, cb_scopes, id);
  
    return contents;
  }
  return "";
}

//! tag: meta
//!  Parse the meta for use with the HTTP headers (content-type for now)
string tag_meta (string tag_name, mapping args, object id)
{
  if(id->misc && id->misc->camas && args["http-equiv"])
  {
    string http_equiv = lower_case(args["http-equiv"]);
    if(http_equiv == "content-type" && args->content)
    {
      if(sscanf(args->content, "%*s;%*s") == 2)
        id->misc->camas->content_type = args->content;
    }
  }
}

//! tag: camas_meta
//!  META tag generator for CAMAS pages 
//!  For content-type, it will add content="text/html; charset=&camas.charset;"
//!  For expires, it will add 0
string tag_camas_meta(string tag_name, mapping args, object id)
{
  if(args["http-equiv"])
  {
    string http_equiv = lower_case(args["http-equiv"]);
    switch(http_equiv)
    {
      case "content-type":
        args->content = "text/html; charset="+CSESSION->charset;
        return CAMAS.Tools.make_tag("meta", args);
      
      case "expires":
        args->content = "0";
        return CAMAS.Tools.make_tag("meta", args);  
    }
  }
}

//! tag: camas_frame
//!  <frame /> handler for CAMAS 
//! attribute: anchor
//!  Some anchor to append to the src arg (for going to <a name="#anchor"></a>)
//! attribute: name
//!  The name of the frame.
//!  This name is also used in the src attribute for feeding CAMAS internal frame system
//! attribute: screen
//!  The screen to display in the frame if not the current screen.
//! attribute: camas_action
//!  The action to execute in the given frame
//! attribute: all attributes available for <frame />
string tag_camas_frame(string tag_name, mapping args, object id)
{
  if (CSESSION)
  {
    // args->target isn't a valid attribute for <frame></>
    // but this is used by make_get_url for setting stuff
    args->target = args->name;
    mapping vars = ([ ]);

    if (args->camas_action)
      vars += ([ "action" + args->camas_action: "=1" ]);
    args->src = CAMAS.Tools.make_get_url(id, args, vars);

    m_delete(args, "target");
    m_delete(args, "screen");
    m_delete(args, "anchor");
    m_delete(args, "camas_action");

    return CAMAS.Tools.make_tag("frame", args);
  }
  return "";
}

// call the right pike file for the given screen
function callscreen(string screen)
{
  screen = upper_case(screen);
  CDEBUG("screen '"+ screen + "' called\n");
  function fun;
  if(!program_screens[screen])
  {
    fun = this_object()["screen_"+screen];
  }
  else
  {
    object external_screen = program_screens[screen][0];
    if(!external_screen["redirect"])
      fun = external_screen["screen"];
    else
      fun = callscreen(external_screen["redirect"]());
  }
  return fun;
}

//! container: camas_form
//!  CAMAS Form handler for all camas actions
//! attribute: all attributes available for <form></>
string container_camas_form(string tag_name, mapping args, string contents, object id)
{
  object camas_main = CAMAS_MODULE;
  CDEBUG(tag_name);
  // Take the name of the screen and go to handle it using magic function
  string screen="unknown";

  CSESSION->nexturl = camas_main->server_url(id) + camas_main->QUERY(location);

  int status;

  // fake screen support
  if(id->misc->camas->status != 0)
    status = id->misc->camas->status; 
  else
    status = CSESSION->status;

  if(screennames[status])
    screen = screennames[status][0];

  // We have screen name in the 'screen' variable let's add some magic auto
  // configuration here and fail back to awfull <imho_main> tag if we cannot handle
  // the screen yet :)
  // warning: fake screen won't work with failback
  // TODO: remove failback to <imho_main>
  screen = upper_case(screen);
  function fun = callscreen(screen);
  if (!fun) {
    CDEBUG("Screen "+screen+" is not handled yet. Failbacking to old system.");
    return("<!-- Not handled yet by CAMAS Tags modules !, failback to old system -->\n"
           "<imho_main>");
  }

  return fun(args, contents, id);
}

//! method: int camas_feature(int f)
//!  Conveniance method for invoquing feature() from camas_features module
int camas_feature(int f)
{
  object features = my_configuration()->get_provider("camas_features");
  if (objectp (features))
    return features->feature (f);
	else
		report_error("\nobject feature not there\n");
}

// ============================================================================
// Any screen tag/container
// ============================================================================

//
//! tag: camas_about
//!  Show version of CAMAS and eventualy a link to camas website.
//! attribute: nolink
//!  Do not add a link to camas website
//
string tag_camas_about(string tag_name, mapping arg, object id, object file)
{
  object camas_module = CAMAS_MODULE;
  if (objectp(camas_module))
  {
    CDEBUG(tag_name);
    return (arg->nolink ? "" : "<a target=\"_top\" href=\""
            + camas_module->server_url (id)
            + camas_module->QUERY(location) + "0/about\">" )
           + "CAMAS v" + camas_module->camas_version
           + (arg->nolink ? "" : "</a>");
  }
  return "";
}


int get_totalsize (object id) {
  return (CSESSION->allfolderssize >> SIZE_SHIFT);
}

//
//! tag: camas_totalsize
//!  Return the totalsize of all folders in kB
//
string tag_camas_totalsize(string tag_name, mapping arg, object id, object file)
{
  return CAMAS.Tools.display_size(CSESSION->allfolderssize);
}

//
//! tag: camas_use
//!  Return the current usage in percent of the diskquota the user has
//
string tag_camas_use (string tag_name, mapping arg, object id, object file) {
  return "0";
}

//
//! tag: camas_quota
//!  Return the diskquota the user can use.
//
string tag_camas_quota (string tag_name, mapping arg, object id, object file)
{
  write("rootname2mailbox=%O\nquota=%O\n", CSESSION->mailbox2rootname, CSESSION->quota);
}

//
//! tag: camas_jslogout
//!  Little javascript devil to autoclose the CAMAS session when the
//!  user close the brower window :))
//
string tag_camas_jslogout (string tag_name, mapping arg, object id, object file) {
  object camas_module = CAMAS_MODULE;
  object camas_features = CAMAS_FEATURES;

  if (CSESSION && objectp(camas_module) && objectp(camas_features))
    CDEBUG(tag_name);
  {
    string res = "";
    if (id->supports->javascript && camas_features->QUERY (jslogout)) {
      res = "<script language=\"JavaScript\" type=\"text/javascript\"><!--\n";
      res += "var CAMAS_Width;\nvar CAMAS_Height;\nvar CAMAS_Confirm=1;\n\n";
      res += "function popup () {\n";
      res += "  if (CAMAS_Confirm) {\n";
      res += "    if (confirm ('" + MSG (M_ENDSESSION) + "')) {\n";

      string logout_url = "http://" + id->host
                          + (sizeof (id->prestate) ? ("/(" + indices (id->prestate) * "," + ")") : "")
                          + camas_module->QUERY (location) + "_top?actionlogout=1";
      if (QUERY (debug))
        write("js_logout: onunload () => '" + logout_url + "');\n");
      res += "      window.open ('" + logout_url + "');\n";

      res += "    }\n";
      res += "    else {\n";
      res += "      day = new Date ();\n";
      res += "      id = day.getTime ();\n";
      res += "      window.open (this.location, 'CAMAS' + id, 'toolbar=1,scrollbars=1,location=1,";
      res += "status=1,menubar=1,resizable=1,personalbar=1,width=CAMAS_Width,height=CAMAS_Height');\n";
      res += "    }\n";
      res += "  }\n";
      res += " } \n\n";


      res += "function set_size () {\n";
      if (id->supports->msie) {
        res += "  CAMAS_Width  = document.body.offsetWidth;\n";
        res += "  CAMAS_Height = document.body.offsetHeight;\n";
      }
      else {
        res += "  CAMAS_Width  = window.outerWidth;\n";
        res += "  CAMAS_Height = window.outerHeight;\n";
      }
      res += " }\n\n";

      res += "window.onload = set_size;\n";
      res += "window.onunload = popup;\n\n";

      res += "//-->\n</script>\n";
    }
    return res;
  }
  return "";
}

//! tag: camas_select_selectaddressbook
//!  Select box for selecting addressbook
//!  Read only address books will not be shown when not using the addressbook from compose
//! note: screen: addressbook, addressbooktoccbcc
string tag_camas_select_addressbook(string tag_name, mapping args, object id, int hide_readonly_addressbook)
{
  array addressbooks = id->conf->get_providers("camas_addressbook");
  string options = "";
  string out = MSG(M_ADDRESSBOOKTITLE) + ": " ;
  
  foreach(Array.uniq(addressbooks), object addressbook)
  {
    mapping largs = ([ "value": addressbook->get_name() ]);
    if(CSESSION->selectedaddrbook == addressbook->get_name())
      largs->selected = "1";
    string option_content = addressbook->get_name();
    string container = CAMAS.Tools.make_container("option", largs, option_content);
    if(!hide_readonly_addressbook)
      options += container;
    else if(addressbook->writeable())
      options += container;
  }
  args->name = "selectaddressbook";
  out += CAMAS.Tools.make_container("select",args,options);
  return out;
}

/* code for most attachments screen tags */
string tag_camas_attachments(string tag_name, int pos, mapping args, object id)
{
  object camas_features = id->conf->get_provider("camas_features");
  if (!objectp(camas_features)){
    CDEBUG("module camas_features is not present");
    return "foobar";
  }

  string out = "";

  // TODO: should be taken from CIF. Set by hand for testing purposes
  int uploaddir=0;

  switch(tag_name)
  {
  case "camas_attachmentlist":
    //! tag: camas_attachmentlist
    //!  List of files selected
    //! note: screen: attachments
    if(camas_features->QUERY(attachments)) {
      if(sizeof (CSESSION->attachments)) {
	int no_att = sizeof (CSESSION->attachments);
	int s = (no_att > 4) ? 4 : ((no_att < 1) ? 1 : no_att);
	
	//out += "<select name=\"attachments\" multiple size=\"" + (string)s + "\">";
	int i = 0;
	
	string contents = "";
	foreach (CSESSION->attachments, mapping file) {
	  args -> value = (string)i;
	  contents += CAMAS.Tools.make_container("option", args, HTML_ENCODE_STRING (file->fname + " (" + file->size + " bytes)"));
	  i++;
	}
	args -> name = "attachments";
	args -> multiple = "";
	args -> size = (string)s;
	m_delete(args,"value");
	out += CAMAS.Tools.make_container("select", args, contents);
	//out += "</select>";
      }
    }
    // else attachments are not allowed
    else
    {
      out = "<!-- Attachments sending is not allowed -->";
      CDEBUG("Attachments sending is not allowed");
    }
    break;

    case "camas_removemarkedattachments":
    //! tag: camas_removemarkedattachments
    //!  Button for removing attachments selected in the camas_attachmentlist
    if(camas_features->QUERY(attachments) && sizeof (CSESSION->attachments))
    {
      out = CAMAS.Tools.formdrawbutton(id,"m_removemarkedattachments", "actionremoveattachment", MSG(M_REMOVEMARKEDATTACHMENTS), args);
    }
    break;
    

   case "camas_removemarkedattachments2":
    //! tag: camas_removemarkedattachments2
    //!  Button for removing attachments checked using checkbox in the camas_attachmentlist
    if(camas_features->QUERY(attachments) && sizeof (CSESSION->attachments))
    {
      out = CAMAS.Tools.formdrawbutton(id,"m_removemarkedattachments", "actionremoveattachment2", MSG(M_REMOVEMARKEDATTACHMENTS), args);
    }
    break;

    /*
    case "camas_addmarkedattachments":
      //! tag: camas_addmarkedattachments
      //!  Button for adding attachments and going back to compose screen
      //! note: screen: attachments
      if(sizeof (CSESSION->attachments))
      {
        out += CAMAS.Tools.formdrawbutton(id,"m_addmarkedtoattachments", "actionaddfileattachment", MSG(M_ADDMARKEDTOATTACHMENTS), args);
      }
      break;
    */

  case "camas_continuecompose":
    //! tag: camas_continuecompose
    //!  Button for returning to compose page
    //! note: screen: attachments
    out = CAMAS.Tools.formdrawbutton(id,"m_backtocompose", "actioncontinuecompose", MSG(M_BACKTOCOMPOSE), args);
    break;

  default:
    out += "<!-- " + tag_name + " is not supported yet -->";
    CDEBUG(tag_name + " isn't supported yet");
  }

  return out;
}

//! container: camas_attachments
//!  Zone to display when there are attachments
//! childcontainer: camas_contents
string container_camas_attachments(string tag_name, int pos, mapping args, string contents, object id, object f, mapping defines)
{
  string out = "";

	mapping tags = ([ "camas_input_removemarkedattachments" : tag_camas_attachments_removemarkedattachments ]);
	mapping containers = ([ "camas_contents" : container_camas_attachments_contents ]);

  if (sizeof (CSESSION->attachments))
    out += parser->run(contents, tags, containers, CSESSION->layout, __LINE__  + "/" + pos, id);

  return out;
}

//! tag: camas_input_removemarkedattachments
//!  Button for removing attachments selected in the camas_attachments
string tag_camas_attachments_removemarkedattachments(string tag_name, int pos, mapping args, object id)
{
	return CAMAS.Tools.formdrawbutton(id,"m_removemarkedattachments", "actionremoveattachment2", MSG(M_REMOVEMARKEDATTACHMENTS), args);
}

//! container: camas_contents
//!  Zone to loop for each attachment
//! childtag: camas_input_selectattachement
//! parentcontainer: camas_attachments
//! variable: name
//!  The name of the attachment
//! variable: size
//!  The size of the attachment
string container_camas_attachments_contents(string tag_name, int pos, mapping args, string contents, object id, object f, mapping defines)
{
  string out = "";

  for (int i = 0; i < sizeof (CSESSION->attachments); i++)
	{
    mapping file = CSESSION->attachments[i];
    
    string name_field = HTML_ENCODE_STRING (file->fname);
    string size_field = file->size;
   
		mapping tags = ([ "camas_input_selectattachment" : tag_camas_attachments_contents_selectattachment ]);
	
    array rxmlvariables =
			({
      	([
					"name"         : name_field,
					"size"         : size_field,
				])
			});
    
		string tempcontents = parser->run(contents, tags, ([ ]), CSESSION->layout, __LINE__  + "/" + pos, id, i);

    out += do_output_tag (args, rxmlvariables, tempcontents, id);
  }

  return out;
}

//! container: camas_input_selectattachment
//!  Generates the input for selecting attachments
//! parentcontainer: camas_contents
string tag_camas_attachments_contents_selectattachment(string tag_name, int pos, mapping args, object id, int attachmentid)
{
	args->type = "checkbox";
	args->name = "selectattachment_"+attachmentid;	

	return CAMAS.Tools.make_tag("input", args);

}

/* Code for most readmail screen tags */
string tag_camas_readmail(string tag_name, mapping args, object id, object largs, int nextuid, int prevuid)
{
  object camas_features = id->conf->get_provider("camas_features");
  if (!objectp(camas_features))
  {
    CDEBUG("module camas_features is not present");
    return "foobar";
  }

  int display = 1;

  string out="";

  switch(tag_name)
  {
  case "camas_reply":
    //! tag: camas_reply
    //!  Button for replying to current email
    //! note: screen: readmail
    out = CAMAS.Tools.formdrawbutton(id,"m_reply", "actionreply", MSG(M_REPLY), args);
    break;

  case "camas_replytoall":
    //! tag: camas_replytoall
    //!  Button for replying to all recipients of the current email
    //! note: screen: readmail
    out = CAMAS.Tools.formdrawbutton(id,"m_replytoall", "actionreplytoall", MSG(M_REPLYTOALL), args);
    break;

  case "camas_replymove":
    //! tag: camas_replymove
    //!  Button for replying to current email and move it in the answered folder
    //! note : The answered folder and sent-mail folder is setup in the CIF
    //! note: screen: readmail
    if(!camas_features->QUERY(moveanswered))
    {
      out = "<!-- You must enable Move answered in CAMAS: Features: Features to get this button -->\n";
      break;
    }
    out = CAMAS.Tools.formdrawbutton(id,"m_replymove", "actionreplymove", MSG(M_REPLYMOVE), args);
    break;

  case "camas_replytoallmove":
    //! tag: camas_replyatoallmove
    //!  Button for replying ro all recipient of the current mail and move it in the answered folder
    //! note : The answered folder can be configured in the CIF
    //! note: screen: readmail
    if(!camas_features->QUERY(moveanswered))
    {
      out = "<!-- You must enable Move answered in CAMAS: Features: Features to get this button -->\n";
      break;
    }
    out = CAMAS.Tools.formdrawbutton(id,"m_replytoallmove", "actionreplytoallmove", MSG(M_REPLYTOALLMOVE), args);
    break;

  case "camas_forward":
    //! tag: camas_forward
    //!  Button for forwarding current email
    //! note: screen: readmail
    out = CAMAS.Tools.formdrawbutton(id,"m_forward", "actionforward", MSG(M_FORWARD), args);
    break;

  case "camas_forwardmove":
    //! tag: camas_forwardmove
    //!  Button for forwarding current email and saving the email forwarded to sent-mail
    //! note: screen: readmail
    if(!camas_features->QUERY(moveanswered))
    {
      out = "<!-- You must enable Move answered in CAMAS: Features: Features to get this button -->\n";
      break;
    }
    out = CAMAS.Tools.formdrawbutton(id,"m_forwardmove", "actionforwardmove", MSG(M_FORWARDMOVE), args);
    break;

  case "camas_trashthis":
    //! tag: camas_trashthis
    //!  Button for trashing current email
    //! note: screen: readmail
    if (!CAMASFOLDERTOOLS.is_trash(CSESSION->mailbox, CSESSION->trashfolder))
    {
      if (camas_features->feature(FEAT_MAILBOXES) &&
          sizeof (CSESSION->trashfolder) &&
          (camas_features->QUERY(deletemethod)=="both" || 
	   camas_features->QUERY(deletemethod)=="move to trash"))
      {
          out = CAMAS.Tools.formdrawbutton(id,"m_movethistotrash", 
					   "actiontrashthis", 
					   MSG(M_MOVETHISTOTRASH), args);
      }
      if (!camas_features->feature(FEAT_MAILBOXES) || 
	  (camas_features->QUERY(deletemethod) != "move to trash") || 
	  !sizeof(CSESSION->trashfolder))
      {
        out = CAMAS.Tools.formdrawbutton(id,"m_delete", 
					 "actiondeletethis", 
					 MSG(M_DELETE), args);
      }
    }
    break;

  case "camas_showprev":
    //! tag: camas_showprev
    //!  Button for reading previous email
    //! note: screen: readmail
    if (prevuid != -1)
    {
      out = CAMAS.Tools.formdrawbutton(id,"m_readprev", "actionreadprev", MSG(M_READPREV), args);
    }
    else
    {
      CDEBUG("<camas_showprev> : there is no previous message");
    }
    break;

  case "camas_shownext":
    //! tag: camas_shownext
    //!  Button for reading next email
    //! note: screen: readmail
    if (nextuid != -1)
    {
      out = CAMAS.Tools.formdrawbutton(id,"m_readnext", 
				       "actionreadnext", 
				       MSG(M_READNEXT), args);
    }
    else
    {
      CDEBUG("<camas_shownext> : there is no next message");
    }
    break;

	case "camas_movemarkedlist":
		out += "<!-- the camas_movemarkedlist tag is now deprecated in READMAIL; use camas_option_movethislist instead -->";
	case "camas_option_movethislist":
    //! tag: camas_option_movethislist
    //!  List of directories for moving current mail into
    //! attribute: restrictmailboxes
    //!  Lists folders only in current mailbox
    //! attribute: showspecialfolders
    //!  Lists special folders like .sent-mail, .Drafts, .Trash et all
    //! attribute: onlymailboxes
    //!  Lists only mailboxes, not folders in it
    //! attribute: option_*
    //!  All attributes applicable to <option></>
    //! note: It works with camas_input_movethisbutton
    //! note: screen: readmail
    if(sizeof(CSESSION->mails))
    {
      array folders = CAMASFOLDERTOOLS.list_folders(id,!args->restrictmailboxes,args->showspecialfolders,args->onlymailboxes);
      array currentparts = ({ });

      mapping optionargs = CAMAS.Tools.extract_html_attributes(args, "option");

      optionargs->value = "imhonomailbox";
      string contents = CAMAS.Tools.make_container("option", optionargs, MSG(M_SELECT)+"...")+"\n";
      if(folders)
        foreach(folders, int folder)
        {
          currentparts = (CSESSION->mailboxes[folder][MB_HIERARCHY_IDX]);
    			optionargs->value = HTML_ENCODE_STRING (CSESSION->mailboxes[folder][MB_FOLDERNAME_IDX]);

          contents += CAMAS.Tools.make_container("option", optionargs, HTML_ENCODE_STRING(CAMAS.FolderTools.translate_frommboxindex(id, folder)));
        }

      m_delete(args, "restrictmailboxes");
      m_delete(args, "showspecialfolders");
      m_delete(args, "onlymailboxes");
      m_delete(args, "setshareduserhome");
      m_delete(args, "value");
      args->name = "mboxmove";

      out += CAMAS.Tools.make_container("select", args, contents) + "\n";
    }
		break;

	case "camas_movemarkedbutton":
		out += "<!-- the tag camas_movemarkedbutton is now deprecated in READMAIL; use camas_input_movethisbutton instead -->";
	case "camas_input_movethisbutton":
    //! tag: camas_input_movethisbutton
    //!  Button for moving current mail into a folder
    //! note: It works with camas_option_movethisselect
    //! note: screen: readmail
    if(sizeof(CSESSION->mails))
    {
      out = CAMAS.Tools.formdrawbutton(id,"m_movemarked", "actionmovethis", MSG(M_MOVEMARKED), args);
    }
		break;

  case "camas_continuecompose":
    //! tag: camas_continuecompose
    //!  Button for continuing compose of draft
    //! note: screen: readmail
    if(CAMASFOLDERTOOLS.is_draft(CSESSION->mailbox, CSESSION->draftsfolder))
    {
      out += CAMAS.Tools.formdrawbutton(id,"m_continuecompose", 
					"actioncomposedraft", 
					MSG(M_CONTINUECOMPOSE)+"...", 
					args);
    }
    {
      CDEBUG("<camas_continuecompose>: not in a draft folder");
    }
    break;

  case "camas_continuecomposemove":
    //! tag: camas_continuecomposemove
    //!  Button for continuing compose of draft
    //! note: screen: readmail
    if(CAMASFOLDERTOOLS.is_draft(CSESSION->mailbox, CSESSION->draftsfolder))
    {
      out += CAMAS.Tools.formdrawbutton(id,"m_continuecomposemove", 
					"actioncomposedraftmove", 
					MSG(M_CONTINUECOMPOSEMOVE)+"...", 
					args);
    }
    else
    {
      CDEBUG("<camas_continuecomposemove>: not in a draft folder");
    }
    break;

  case "camas_fullheaders":
    //! tag: camas_fullheaders
    //!  Button for hiding/showing all headers
    //! note: screen: readmail
    if (CSESSION->showheaders)
      out = CAMAS.Tools.formdrawbutton(id,"m_hidefullheaders", 
					"actionhideheaders", 
					MSG(M_HIDEFULLHEADERS), args);
    else
      out = CAMAS.Tools.formdrawbutton(id,"m_showfullheaders", 
					"actionshowheaders", 
					MSG(M_SHOWFULLHEADERS), args);
    break;

  case "camas_backtomailindex":
    //! tag: camas_backtomailindex
    //!  Button for going back to the mail list.
    //! note: screen: readmail
    if(camas_features->feature(FEAT_MAILBOXES))
    {
      out = CAMAS.Tools.formdrawbutton(id,"m_index", 
				       "actionindex", 
				       MSG(M_BACKTOMAILINDEX), 
				       args);
    }
    break;

  case "camas_face":
    //! tag: camas_face
    //!  The x-face image of the sender or x-image-url if one of them are
		//!  provided
    //! arg: priority
    //!  Either "x-face" or "x-image-url", tells Camas which type of face
    //!  to display in case both are provided.
    //!  x-image-url usually provide better quality since x-face is only 48*48
    //!  black and white while x-image-url could be any format of any quality,
		//!  while historically a 64*64 tiff image was adviced.
		//!  However using x-image-url means making a HTTP connexion
    //!  to an external site which is maybe not what you want if you are focused
		//!  on confidentiality.
    //!  By default priority is set to x-image-url.
		//! attribute: x-image-url_*
		//!  Attributes to pass to the HTML IMG tag when displaying X-Image-URL
		//! attribute: x-face_*
		//!  Attributes to pass to the HTML IMG tag when displaying X-Face
    //! note: screen: readmail
    if(CSESSION->cmail && CSESSION->cmail->headers)
    {
      int display_ximageurl = 1;
      if(CSESSION->cmail->headers["x-image-url"])
      {
        if(CSESSION->cmail->headers["x-face"] && args->priority == "x-face")
          display_ximageurl = 0;
        else
          display_ximageurl = 1; 
      }
      else if(CSESSION->cmail->headers["x-face"])
      {
        display_ximageurl = 0;
      }
      else
      {
        out = "";
        break;
      }
      if(display_ximageurl)
			{
				args += CAMAS.Tools.extract_html_attributes(args, "x-image-url");
        args->src = HTML_ENCODE_STRING (CSESSION->cmail->headers["x-image-url"]);
			}
      else
      {
#if constant(Image.XFace.decode)
				args += CAMAS.Tools.extract_html_attributes(args, "x-face");
        args->src = CAMAS.Tools.make_get_url(id, ([ ]), ([ "xface":"1" ]));
#else
        out = "<!-- Cannot find Xface support in your Pike, please install Pike with Xface support -->";
#endif
      }

      args->alt = ":-)";
      
      m_delete(args, "priority");
      
      out = CAMAS.Tools.make_tag("img", args);
    }
    else
      out = "";
    break;

  default:
    CDEBUG("tag_camas_readmail: "+tag_name+" is not supported yet");
    out = "<!-- "+tag_name+" is not supported yet -->";
  }
  return out;
}

// helper function for folder tools containers
inline int display_is_folder(int foldertoolsresult, mapping args)
{
  int display = 0;
  if(foldertoolsresult)
    display = 1;
  if(args->not)
    display = !display;
  return display;
}

// helper function for folder tools containers:
// get the internal view of a mailbox according to
// a given mailbox name
inline array getmailbox(string _mailbox, object id)
{
  if(!_mailbox)
    return CSESSION->mailbox;
  foreach(CSESSION->mailboxes, array mailbox)
  {
    if(mailbox[MB_FOLDERNAME_IDX] == _mailbox
        || (mailbox[MB_FOLDERNAME_IDX] + 
                 CAMASFOLDERTOOLS.getseparator(mailbox)) == _mailbox)
      return mailbox;
  }
  report_warning("Can't find internal mailbox view for mailbox " 
      + _mailbox + ", returning current mailbox\n");
  return CSESSION->mailbox;
}

//! container: camas_is_sent
//!  Return the contents inside this container only if we are inside a sent folder
//! arg: not
//!  If not is present, return the contents inside this container only if 
//!  we are not inside a sent folder
//! arg: mailbox
//!  The name of the mailbox to compare (optional)
string container_camas_is_sent(string tag_name, mapping args, string contents, object id)
{
  if(!CSESSION || display_is_folder(
      CAMASFOLDERTOOLS.is_sent(getmailbox(args->mailbox, id), CSESSION->sentfolder),
      args))
    return contents;
  return "";
}

//! container: camas_is_answered
//!  Return the contents inside this container only if we are inside an answered folder
//! arg: not
//!  If not is present, return the contents inside this container only if 
//!  we are not inside an answered folder
//! arg: mailbox
//!  The name of the mailbox to compare (optional)
string container_camas_is_answered(string tag_name, mapping args, string contents, object id)
{
  if(!CSESSION || display_is_folder(
      CAMASFOLDERTOOLS.is_answered(getmailbox(args->mailbox, id), CSESSION->answeredfolder),
      args))
    return contents;
  return "";
}

//! container: camas_is_trash
//!  Return the contents inside this container only if we are inside a trash folder
//! arg: not
//!  If not is present, return the contents inside this container only if 
//!  we are not inside a trash folder
//! arg: mailbox
//!  The name of the mailbox to compare (optional)
string container_camas_is_trash(string tag_name, mapping args, string contents, object id)
{
  if(!CSESSION || display_is_folder(
      CAMASFOLDERTOOLS.is_trash(getmailbox(args->mailbox, id), CSESSION->trashfolder),
      args))
    return contents;
  return "";
}

//! container: camas_is_draft
//!  Return the contents inside this container only if we are inside a draft folder
//! arg: not
//!  If not is present, return the contents inside this container only if 
//!  we are not inside a draft folder
//! arg: mailbox
//!  The name of the mailbox to compare (optional)
string container_camas_is_draft(string tag_name, mapping args, string contents, object id)
{
  if(!CSESSION || display_is_folder(
      CAMASFOLDERTOOLS.is_draft(getmailbox(args->mailbox, id), CSESSION->draftsfolder),
      args))
    return contents;
  return "";
}

//! container: camas_is_in_inbox
//!  Return the contents inside this container only if we are inside INBOX or a 
//!  sub directory of INBOX
//! arg: not
//!  If not is present, return the contents inside this container only if 
//!  we are not inside INBOX or a sub directory of INBOX
//! arg: mailbox
//!  The name of the mailbox to compare (optional)
string container_camas_is_in_inbox(string tag_name, mapping args, string contents, object id)
{
  if(!CSESSION || display_is_folder(
      CAMASFOLDERTOOLS.is_in_inbox(getmailbox(args->mailbox, id), CSESSION),
      args))
    return contents;
  return "";
}

//! container: camas_is_in_shared
//!  Return the contents inside this container only if we are inside shared or a 
//!  sub directory of shared
//! arg: not
//!  If not is present, return the contents inside this container only if 
//!  we are not inside shared or a sub directory of shared
//! arg: mailbox
//!  The name of the mailbox to compare (optional)
string container_camas_is_in_shared(string tag_name, mapping args, string contents, object id)
{
  if(!CSESSION || display_is_folder(
      CAMASFOLDERTOOLS.is_in_shared(getmailbox(args->mailbox, id), CSESSION),
      args))
    return contents;
  return "";
}

//! container: camas_mailboxlist
//!  Provides a list of mailboxes and nested folders and links to go to them
//! childcontainer : camas_mailbox
//! childcontainer : camas_folder
//! childcontainer : camas_link
//! variable : mailboxroot
//! note: screen: any
string container_camas_mailboxlist(string tag_name, mapping args, string contents, object id)
{
  string out = "";   // HTML to output
  int i =0;          // loop counter

  array sorted_mailboxes = CSESSION->mailboxes;

  for(i=0; i<sizeof(sorted_mailboxes); i++)
  {
    string tmp = parser->run(contents,
                      ([
                       ]),
                      ([
                     "camas_mailbox"       : container_camas_mailboxlist_mailbox,
                     "camas_folder"        : container_camas_mailboxlist_folder,
                     "camas_specialfolder" : container_camas_mailboxlist_specialfolder,
                     "camas_link"          : container_camas_mailboxlist_link,
                       ]),
                      CSESSION->layout, __LINE__ + hash(contents), id, i, sorted_mailboxes);
		
		string mailboxroot=CAMASFOLDERTOOLS.getmailboxroot(sorted_mailboxes[i], CSESSION);

		array rxmlvariables = ({ ([ "mailboxroot" : mailboxroot ]) });

  	out += do_output_tag(args, rxmlvariables, tmp, id);
  }

  return out;
}

//! subcontainer : camas_mailbox
//! attribute: quote
//!  Sets the quote character for variables.
//!  default : #
//! attribute: nameinbox
//!  Put the name of the user instead of "INBOX"
//! variable : name
//! variable : total
//! variable : recent
//! variable : unseen
//! parentcontainer : camas_mailboxlist
//! note: screen: where camas_mailboxlist is defined
string container_camas_mailboxlist_mailbox(string tag_name, int pos, mapping args, string contents, object id, int i, array sorted_mailboxes)
{
  string out = "";

  string name = "";
  // It it's top level mailbox (eg INBOX or shared.something)
  if(CAMASFOLDERTOOLS.is_a_top_level_mailbox(sorted_mailboxes[i], CSESSION)
      && !CAMASFOLDERTOOLS.is_a_special_folder(sorted_mailboxes[i], CSESSION))
  {
    if(CAMASFOLDERTOOLS.is_inbox(sorted_mailboxes[i], CSESSION))
    {
			if(args->nameinbox)
				name = CSESSION->name;
			else
      	name = CAMAS.FolderTools.translate_frommbox(id,sorted_mailboxes[i]);
    }
    else
    {      
      name = sorted_mailboxes[i][MB_HIERARCHY_IDX][-2];
    }

    string numbermessage;
    string numberrecent;
    string numberunseen;
    if(CSESSION->foldersinfo)
    {
      numbermessage = mailboxatt("total", id, i, sorted_mailboxes);
      numberrecent  = mailboxatt("recent", id, i, sorted_mailboxes);
      numberunseen  = mailboxatt("unseen", id, i, sorted_mailboxes);
    }
    string mark=displaymark(args->mark, id,  i, sorted_mailboxes);
    
    array outlet = ({
                      ([
                     "name"        : HTML_ENCODE_STRING(name),
                     "total"       : numbermessage,
                     "recent"      : numberrecent,
                     "unseen"      : numberunseen,
                     "mark"        : mark,
                       ])
                    });
    
    string s = parser->run(contents,
                      ([
                       ]),
                      ([
                     "camas_link"          : container_camas_mailboxlist_link,
                       ]),
                      CSESSION->layout, __LINE__ + "/" + pos, id, i, sorted_mailboxes);
    
    out += do_output_tag(args, outlet, s, id);
  }

  return out;
}

//! subcontainer : camas_folder
//! variable : name
//! variable : total
//! variable : recent
//! variable : unseen
//! parentcontainer : camas_mailboxlist
//! note: some folder computation for shared.* are still linkeo-specific
//! note: screen: where camas_mailboxlist is defined
string container_camas_mailboxlist_folder(string tag_name, int pos, mapping args, string contents, object id, int i, array sorted_mailboxes)
{
  string out = "";

  if(
     CAMASFOLDERTOOLS.is_a_folder(sorted_mailboxes[i], CSESSION)
     && !CAMASFOLDERTOOLS.is_a_special_folder(sorted_mailboxes[i], CSESSION)
     && !CAMASFOLDERTOOLS.is_a_top_level_mailbox(sorted_mailboxes[i], CSESSION)
    )
  {
    string numbermessage  = "";
    string numberrecent   = "";
    string numberunseen   = "";
    if(CSESSION->foldersinfo)
    {
      numbermessage = mailboxatt("total",id, i, sorted_mailboxes);
      numberrecent  = mailboxatt("recent",id, i, sorted_mailboxes);
      numberunseen  = mailboxatt("unseen",id, i, sorted_mailboxes);
    }
    string name = mailboxatt("name", id, i, sorted_mailboxes);
    string mark = displaymark(args->mark, id, i, sorted_mailboxes);

    array outlet = ({
                      ([
                     "name"        : name,
                     "total"       : numbermessage,
                     "recent"      : numberrecent,
                     "unseen"      : numberunseen,
                     "mark"        : mark,
                       ])
                    });

    string s = parser->run(contents,
                      ([
                       ]),
                      ([
                     "camas_link"          : container_camas_mailboxlist_link,
                       ]),
                      CSESSION->layout, __LINE__ + "/" + pos, id, i, sorted_mailboxes);
    
    out += do_output_tag(args, outlet, s, id);
  }
  return out;
}

//! subcontainer : camas_specialfolder
//! attribute: noshared
//!  Don't display specialfolders in shared hierarchy
//! attribute: nosharedsent
//!  Don't display sent folders in shared hierarchy
//! variable : name
//! variable : total
//! variable : recent
//! variable : unseen
//! parentcontainer : camas_mailboxlist
//! note: some folder computation for shared.* are still linkeo-specific
//! note: screen: where camas_mailboxlist is defined
string container_camas_mailboxlist_specialfolder(string tag_name, int pos, mapping args, string contents, object id, int i, array sorted_mailboxes)
{
  string out = "";

  int noshared = zero_type(args->noshared) ? 0 : args->noshared;
  int nosharedsent = zero_type(args->nosharedsent) ? 0 : args->nosharedsent;

  if(CAMASFOLDERTOOLS.is_a_special_folder(CSESSION->mailboxes[i], CSESSION)
     && !(noshared && CAMASFOLDERTOOLS.is_in_shared(CSESSION->mailboxes[i], CSESSION))
     && !(nosharedsent && CAMASFOLDERTOOLS.is_sent(CSESSION->mailboxes[i], CSESSION->sentfolder) && CAMASFOLDERTOOLS.is_in_shared(CSESSION->mailboxes[i], CSESSION)))
  {
    string numbermessage  = "";
    string numberrecent   = "";
    string numberunseen   = "";

    if(CSESSION->foldersinfo)
    {
      numbermessage = mailboxatt("total", id, i, sorted_mailboxes);
      numberrecent  = mailboxatt("recent", id, i, sorted_mailboxes);
      numberunseen  = mailboxatt("unseen", id, i, sorted_mailboxes);
    }
    string name = "&nbsp;&nbsp;" * sizeof(sorted_mailboxes[i][MB_HIERARCHY_IDX])
      + HTML_ENCODE_STRING(CAMASFOLDERTOOLS.translate_frommboxindex(id, i));
    string mark = displaymark(args->mark, id, i, sorted_mailboxes);

    array outlet = ({
                      ([
			"name"        : name,
			"total"       : numbermessage,
			"recent"      : numberrecent,
			"unseen"      : numberunseen,
			"mark"        : mark,
                       ])
                      });
    string s = parser->run(contents,
                      ([
                       ]),
                      ([
                       "camas_link"          : container_camas_mailboxlist_link,
                       ]),
                      CSESSION->layout, __LINE__ + "/" + pos, id, i, sorted_mailboxes);
                      
    out += do_output_tag(args, outlet, s, id);
  }

  return out;
}

/* Returns info about current mailbox */
string mailboxatt(string what, object id, int i, array sorted_mailboxes)
{
  string out="";
  string imappath=sorted_mailboxes[i][MB_FOLDERNAME_IDX];

  switch(what)
  {
  case "name":
    out = "&nbsp;&nbsp;" * sizeof(sorted_mailboxes[i][MB_HIERARCHY_IDX])
      + HTML_ENCODE_STRING(sorted_mailboxes[i][MB_HIERARCHY_IDX][-1]);
    break;
  case "total":
    if(!(sorted_mailboxes[i][MB_FLAGS_IDX] & (MB_NOSELECT)))
      out = (string)CSESSION->foldersinfo[imappath][MB_MESSAGES_IDX];
    else
      out = "";
    break;
  case "recent":
    if(!(sorted_mailboxes[i][MB_FLAGS_IDX] & (MB_NOSELECT)))
      out = (string)CSESSION->foldersinfo[imappath][MB_RECENT_IDX];
    else
      out = "";
    break;
  case "unseen":
    if(!(sorted_mailboxes[i][MB_FLAGS_IDX] & (MB_NOSELECT)))
      out = (string)CSESSION->foldersinfo[imappath][MB_UNSEEN_IDX];
    else
      out = "";
    break;
  default:
    CDEBUG("mailboxatt: " + what + " is not handled");
  }

  return out;
}

/* Returns the mark for showing the current mailbox */
string displaymark(string imagepath, object id, int i, array sorted_mailboxes)
{
  string out="";

  // If we display the line for the current mailbox
  if(CSESSION->mailbox[MB_FOLDERNAME_IDX] == sorted_mailboxes[i][MB_FOLDERNAME_IDX])
  {
    // If an image has been specified with <[mailbox|folder] mark="/path/to/image">
    if(!zero_type(imagepath))
    {
      out += CAMAS.Tools.make_tag("img", ([ "src" : imagepath ]));
    }
    else
    {
      out += "X";
    }
  }
  return out;
}

//! subcontainer : camas_link
//! parentcontainer : camas_mailboxlist
//!  Make a link to read the mailbox involved
//! note: screen: where camas_mailboxlist is defined
string container_camas_mailboxlist_link(string tag_name, int pos, mapping args, string contents, object id, int i, array sorted_mailboxes)
{
  string out = "";
  
  if(!(sorted_mailboxes[i][MB_FLAGS_IDX] & (MB_NOSELECT)) )
  {
    mapping vars = ([ "actionmailindex":"1", "mbox":sorted_mailboxes[i][MB_FOLDERNAME_IDX] ]);

    args->href = CAMAS.Tools->make_get_url(id, args, vars);

    out += CAMAS.Tools.make_container("a", args, contents);
  }
  else
    out += contents;
    
  return out;
}

//! container: camas_action
//!  To do some actions
//! attribute: action
//!  The action to perform, can be compose, index, folderlist, addressbook, filters, setup, files, 
//!  checknewmail or logout
//! attribute: button
//!  Set button to 1 if you want a button to be drawn instead of a <a></>
//! attribute: target
//!  Set the target where you want your action to take effect.
//!  This is incompatible with button="1". If you use buttons, you must use target from the parent <camas_form></>
string container_camas_action(string tag_name, mapping args, string contents, object id)
{
  object camas_features = id->conf->get_provider("camas_features");
  mapping vars = ([ ]);
  string notavailable = "<!-- You must enable this feature in the CAMAS: Features module to get it working -->\n";
  
  switch(args->action)
  {
  case "compose":
    if(args->button)
    {
      m_delete(args, "button");
      return CAMAS.Tools.formdrawbutton(id,"m_compose", "actioncompose", MSG(M_NEWMAIL), args);
    }
    else
      vars = ([ "actioncompose" : "1" ]);
    break;
    
  case "index":
    if(camas_features->feature(FEAT_MAILBOXES))
    {
      if(args->button)
      {
	m_delete(args, "button");
	return CAMAS.Tools.formdrawbutton(id,"m_index", "actionindex", MSG(M_CURRMAILBOX), args);
      }
      else
	vars = ([ "actionindex" : "1" ]);
    }
    else
      return notavailable;
    break;
    
  case "inbox":
    if (camas_features->feature(FEAT_MAILBOXES))
    {
      if (args->button)
      {
	m_delete(args, "button");
        return CAMAS.Tools.formdrawbutton(id,"m_inbox", "actionindex", MSG(M_INBOX), args);
      }
      else
        vars = ([ "actionindex" : "1", "mbox" : "INBOX" ]);
    }
    else
      return notavailable;
    break;

  case "folderlist":
    if(camas_features->feature(FEAT_MAILBOXES))
    {
      if(args->button)
      {
	m_delete(args, "button");
	return CAMAS.Tools.formdrawbutton(id,"m_folderlist", "actionfolderlist", MSG(M_MAILBOXES), args);
      }
      else
	vars = ([ "actionfolderlist" : "1" ]);
    }
    else
      return notavailable;
    break;
    
  case "addressbook":
    array addressbooks = id->conf->get_providers("camas_addressbook");
    if(sizeof(addressbooks) > 0)
    {
      if(args->button)
      {
        m_delete(args, "button");
        return CAMAS.Tools.formdrawbutton(id,"m_addressbook", "actionaddressbook", MSG(M_ADDRESSBOOKTITLE), args->onclick);
      }
      else
        vars = ([ "actionaddressbook" : "1" ]);
    }
    else
      return notavailable;
    break;
    
  case "filter":
  case "filters":
    if(camas_features->feature(FEAT_MAILFILTER))
    {
      if(args->button)
      {
	m_delete(args, "button");
	return CAMAS.Tools.formdrawbutton(id,"m_filterbook", "actionfilterbook", MSG(M_FILTERBOOKTITLE), args);
      }
      else
	vars = ([ "actionfilterbook" : "1" ]);
    }
    else
      return notavailable;
    break;
    
  case "setup":
    if(camas_features->feature(FEAT_USEREDITSETUP))
    {
      if(args->button)
      {
	m_delete(args, "button");
	return CAMAS.Tools.formdrawbutton(id,"m_setup", "actionsetup", MSG(M_PREFS), args);
      }
      else
	vars = ([ "actionsetup" : "1" ]);
    }
    else
      return notavailable;
    break;
    
  case "search":
  case "searchmail":
    if(args->button)
    {
      m_delete(args, "button");
      return CAMAS.Tools.formdrawbutton (id,
					 "m_searchmail", 
					 "actionsearchmail", 
					 MSG(M_SEARCHMAIL)+"...", args);
    }
    else
      vars = ([ "actionsearchmail" : "1" ]);
    break;
    
  case "files":
    object camas_main = CAMAS_MODULE;
    if(camas_features->feature(FEAT_ATTACHMENTS) && sizeof(camas_main->QUERY(uploaddir)))
    {
      if(args->button)
      {
	      m_delete(args, "button");
        return CAMAS.Tools.formdrawbutton (id, 
					   "m_files", 
					   "actionfiles", 
					   MSG(M_FILES), 
					   args);
      }
      else
	      vars = ([ "actionfiles" : "1" ]);
    }
    else
      return notavailable;
    break;
    
  case "checknewmail":
    if(args->button)
    {
      m_delete(args, "button");
      return CAMAS.Tools.formdrawbutton(id,"m_reload", "actionreload", MSG(M_CHECKNEWMAIL), args);
    }
    else
      vars = ([ "actionreload" : "1" ]);
    break;
    
  case "notify":
    //TODO: real need to clean up that

    /* script that allow the opening of a window */
    string out = "<script language=\"JavaScript\" type=\"text/javascript\">\n"
      "function openWin(url,name,opts)"
      "	{window.open(url, name, opts);} "
      "</script>\n";
    m_delete(args, "button");
    m_delete(args, "action");

    args -> href = "javascript:openWin('"+CAMAS.Tools.make_get_url(id, ([ "screen":"notify" ]), ([ "actioncheckactivemailboxes":"1" ])) + "','Notify','toolbar=no,directories=no,location=no,status=no,menubar=no,scrollbars=no,resizable=no,copyhistory=no,width=100,height=100');"; 

    return out + CAMAS.Tools.make_container("a", args, contents);
    break;

  case "logout":
    if(args->button)
    {
      m_delete(args, "button");
      return CAMAS.Tools.formdrawbutton(id,
					"m_logout", 
					"actionlogout", 
					MSG(M_LOGOUT), 
					args);
    }
    else
      vars = ([ "actionlogout" : "1" ]);
    break;

  default: 
    return "<!-- unknown action "+args->action+" -->\n";
  }

  m_delete(args,"action");
 
  args -> href = CAMAS.Tools.make_get_url(id, args, vars );

  // Build an <a></> if the output isn't a button
  return CAMAS.Tools.make_container("a", args, contents);
}

/* START AUTOGENERATED DEFVAR DOCS */

//! defvar: debug
//! Debug the call / errors into Caudium error log ?
//!  type: TYPE_FLAG
//!  name: Debug
//
//! defvar: ent_parse
//! Parse &amp;camasmsg.*; entities even if RXML is not XML ?
//!  type: TYPE_FLAG
//!  name: Parse entities
//

/*
 * If you visit a file that doesn't contain these lines at its end, please
 * cut and paste everything from here to that file.
 */

/*
 * Local Variables:
 * c-basic-offset: 2
 * c-set-style: k&r
 * End:
 *
 * vim: softtabstop=2 tabstop=2 expandtab autoindent formatoptions=croqlt smartindent cindent shiftwidth=2
 */

