(**************************************************************************)
(*                   Cameleon                                             *)
(*                                                                        *)
(*      Copyright (C) 2002 Institut National de Recherche en Informatique et   *)
(*      en Automatique. All rights reserved.                              *)
(*                                                                        *)
(*      This program is free software; you can redistribute it and/or modify  *)
(*      it under the terms of the GNU General Public License as published by  *)
(*      the Free Software Foundation; either version 2 of the License, or  *)
(*      any later version.                                                *)
(*                                                                        *)
(*      This program is distributed in the hope that it will be useful,   *)
(*      but WITHOUT ANY WARRANTY; without even the implied warranty of    *)
(*      MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the     *)
(*      GNU General Public License for more details.                      *)
(*                                                                        *)
(*      You should have received a copy of the GNU General Public License  *)
(*      along with this program; if not, write to the Free Software       *)
(*      Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA          *)
(*      02111-1307  USA                                                   *)
(*                                                                        *)
(*      Contact: Maxence.Guesdon@inria.fr                                *)
(**************************************************************************)

(** The application class. *)

open Chat_types
open Chat_proto

module M = Chat_messages
module C = Configwin
module G = Chat_global
module Com = Chat_com

let conf = Chat_args.parse () 

let safe_int_param h label f v =
  C.string ~help: h ~f: (fun s -> try f (int_of_string s) with _ -> ())
    label (string_of_int v)

let input_people () =
  let id = ref "" in
  let p_id = C.string ~help: M.h_id ~f: (fun s -> id := s) M.id !id in
  let host =  ref "" in
  let p_host = C.string ~f: (fun s -> host := s) M.host !host in
  let port = ref M.default_port in
  let p_port = safe_int_param M.h_port M.port 
      (fun n -> port := n)
      !port
  in
  match C.simple_get M.people [p_id ; p_host ; p_port] with
    C.Return_cancel -> None
  | C.Return_apply | C.Return_ok -> 
      Some  (!id, !host, !port)

(** Return true if conf was modified. In this case,
   the conf has already been saved.*)
let edit_conf conf =
  let p_id = C.string 
      ~help: M.h_id
      ~f: conf#set_id M.id conf#id 
  in
  let p_port = safe_int_param M.h_port M.port conf#set_port conf#port in
  let p_timeout = safe_int_param M.h_timeout M.h_timeout
      conf#set_timeout conf#timeout in
  let p_send_timeout = safe_int_param M.h_send_timeout M.h_send_timeout
      conf#set_send_timeout conf#send_timeout
  in
  let p_popup = C.bool ~help: M.h_popup_all
      ~f: conf#set_popup_all
      M.popup_all
      conf#popup_all
  in
  let p_dialog_buttons = C.bool ~help: M.h_dialog_buttons
      ~f: conf#set_dialog_buttons
      M.dialog_buttons
      conf#dialog_buttons
  in

  let p_col_connected = C.color
      ~help: M.h_color_connected 
      ~f: conf#set_color_connected
      M.h_color_connected conf#color_connected
  in
  let p_col_connected_temp = C.color
      ~help: M.h_color_connected_temp
      ~f: conf#set_color_connected_temp
      M.h_color_connected_temp conf#color_connected_temp
  in
  let p_col_not_connected = C.color
      ~help: M.h_color_not_connected
      ~f: conf#set_color_not_connected
      M.h_color_not_connected conf#color_not_connected
  in
  let p_col_myself = C.color
      ~help: M.h_color_myself
      ~f: conf#set_color_myself
      M.h_color_myself conf#color_myself
  in
  let add_people () =
    match input_people () with
      None -> []
    | Some p -> [p]
  in
  let p_people = C.list ~f: conf#set_people
      ~add: add_people
      ~titles: [M.id ; M.host ; M.port]
      M.people 
      (fun (i,h,p) -> [i ; h ; string_of_int p])
      conf#people
  in
  let structure = [
    C.Section (M.connection, 
	     [ p_id ; p_port ; p_timeout ; p_send_timeout ; 
	       p_popup ; p_dialog_buttons]) ;
    C.Section (M.colors,
	     [ p_col_connected ;
	       p_col_connected_temp ;
	       p_col_not_connected ;
	       p_col_myself ] ) ;
    C.Section (M.people,
	       [ p_people ]) ;
  ] 
  in
  match C.get M.options structure with
    C.Return_cancel -> false
  | C.Return_apply | C.Return_ok ->
      conf#save;
      true




(** {2 Application} *)

class app socket_server =
  object (self)
    val mutable working = false
	
    inherit Chat_gui.gui as gui
	
    method handle_info i =
      let ((version,id,(host,port)), iddest, proto) = i in
      (
       match proto with
       | Byebye -> ()
       | Hello -> Com.send id (host,port) HelloOk
       | HelloOk -> ()
       | Message mes ->
	   let show = 
	     conf#popup_all or 
	     (List.mem (id,host,port) conf#people)
	   in
	   let dial = Chat_gui.get_dialog ~show (Chat_gui.Single (id, host, port)) in
	   dial#handle_message id mes
	     
       | AddOpen (i, (h, p)) ->
	   Chat_messages.verbose (Printf.sprintf "received AddOpen i=%s h=%s p=%d" i h p);
	   self#handle_info ((version,i,(h,p)), iddest, Message "")
	     
       | RoomMessage (name, people, mes) ->
	   let dial = Chat_gui.get_dialog ~show: true (Chat_gui.Room (name, people)) in
	   dial#handle_message id mes
       | File _ -> 
	   ()
      );
      self#update_after_message id host port

    method update_after_message id host port =
      gui#update

    method accept =
      try
	match Com.receive socket_server with
	  None -> ()
	| Some (((v,id,(host,port)),iddest,proto), sock) -> 
	    Com.handle_info id host port proto sock;
	    self#handle_info ((v,id,(host,port)),iddest,proto)
      with
	Failure s ->
	  Chat_messages.verbose s;
	  ()


    method work () =
      if working then 
	false
      else
	(
	 working <- true;
	 self#accept;
	 working <- false;
	 true
	)

    method say_hello id host port =
      Com.send id (host, port) Hello

    method close =
      Hashtbl.iter 
	(fun (h,p) sock -> Com.send "" (h,p) Byebye)
	G.connections;

    method edit_conf =
      if edit_conf conf then
	(
	 (* A VOIR *)
	 gui#update
	)
      else
	()

    method toggle_temp_selected =
      (* A VOIR : conf#toggle_temp_selected i h p*)
      (*List.iter
	(fun (i,h,p,_,t) ->
	  if t then 
	  else data#remove_people i h p)
	selected_people;
      *)
      gui#update

    method kill_people_selected =
      (* A VOIR : conf#remove_people + deconnexion *)
(*    List.iter
	(fun (i,h,p,_,t) ->
	  data#remove_people ~kill: true i h p)
	selected_people;
*)
      gui#update

    method add_people =
      match input_people () with
	None -> ()
      | Some (id,host,port) -> 
	  (* A VOIR conf#add_people *)
(*
	  let l = data#people in
	  data#add_people id host port;
	  let l2 = data#people in
*)
	  self#say_hello id host port;
	  gui#update

    method connect_to_send_file iddest host port file =
      let size = (Unix.stat file).Unix.st_size in
      let sock = Chat_com.get_connection ~temp: true  iddest (host,port) in
      let cancel () = G.safe_close sock in
      let win = new Chat_gui.file_progress (M.to_id iddest) file size cancel in
      Chat_com.send ~sock 
	  iddest (host,port)
	  (File(size,Filename.basename file)) ;
      Chat_file.send_file sock win#progress win#message file

    method ask_send_file () =
      match  selected_people with
	[] -> ()
      |	l ->
	  match GToolbox.select_file ~title: M.m_send_file () with
	    None -> ()
	  | Some f ->
	      List.iter 
		(fun (i,h,p) -> 
		  ignore (Thread.create 
			    (self#connect_to_send_file i h p)
			    f
			 )
		)
		l

    method init_window (w : GWindow.window) =
      w#add_accel_group gui#accelgroup
	
     method ask_accept_file id (size,name,sock) =
       match GToolbox.question_box ~title: M.question
	   ~buttons: [M.yes ; M.no]
	   (M.accept_file_from name id)
       with
	 1 -> 
	   (
	    match GToolbox.select_file ~title: M.save ~filename: name () with
	      None -> Unix.close sock
	    | Some f -> 
		let cancel () = G.safe_close sock in
		let w = new Chat_gui.file_progress (M.from_id id) f size cancel in
		Chat_file.receive_file w#progress w#message (size,f,sock) 
	   )
       | _ -> Unix.close sock


    initializer
      ignore (itemOptions#connect#activate (fun () -> self#edit_conf));
      ignore (itemToggleTemp#connect#activate
		(fun () -> self#toggle_temp_selected));
      ignore (itemAddPeople#connect#activate (fun () -> self#add_people));
      ignore (itemRemovePeople#connect#activate (fun () -> self#kill_people_selected));
      ignore (itemSendFile#connect#activate self#ask_send_file);

      Chat_com.hook_file := self#ask_accept_file ;

      List.iter (fun (i,h,p) -> self#say_hello i h p) conf#people;
      ignore (GMain.Timeout.add ~ms: conf#timeout ~callback: self#work) ;

      gui#update
  end

