(**************************************************************************)
(*                   Cameleon                                             *)
(*                                                                        *)
(*      Copyright (C) 2002 Institut National de Recherche en Informatique et   *)
(*      en Automatique. All rights reserved.                              *)
(*                                                                        *)
(*      This program is free software; you can redistribute it and/or modify  *)
(*      it under the terms of the GNU General Public License as published by  *)
(*      the Free Software Foundation; either version 2 of the License, or  *)
(*      any later version.                                                *)
(*                                                                        *)
(*      This program is distributed in the hope that it will be useful,   *)
(*      but WITHOUT ANY WARRANTY; without even the implied warranty of    *)
(*      MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the     *)
(*      GNU General Public License for more details.                      *)
(*                                                                        *)
(*      You should have received a copy of the GNU General Public License  *)
(*      along with this program; if not, write to the Free Software       *)
(*      Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA          *)
(*      02111-1307  USA                                                   *)
(*                                                                        *)
(*      Contact: Maxence.Guesdon@inria.fr                                *)
(**************************************************************************)

module V = Omom_variables
module M = Omom_messages
module C = Configwin

class type box_variable =
    object
      method coerce : GObj.widget
      method variable : V.variable
      method name : string
      method action : string
      method value : V.v_value
      method set_name : string -> unit
    end

let combo_action () =
  let wc = GEdit.combo
      ~popdown_strings: Omom_actions.actions_names
      ~use_arrows:`DEFAULT
      ~case_sensitive:true
      ~value_in_list:true
      ~ok_if_empty:false
      ()
  in
  wc

class box_string (name, v, activated, action) =
  let hbox = GPack.hbox () in
  let wchk_acti = GButton.check_button 
      ~active: activated
      ~packing: (hbox#pack ~expand: false) ()
  in
  let wl_name = GMisc.label ~text: (name^" : ") 
      ~packing: (hbox#pack ~expand: false ~padding: 2) ()
  in
  let we_v = GEdit.entry
      ~editable: activated 
      ~packing: (hbox#pack ~expand: true ~padding: 2) ()
  in
  let _ = we_v#set_text v in
  let wl_action = GMisc.label ~text: (M.action^" : ") 
      ~packing: (hbox#pack ~expand: false ~padding: 2) ()
  in
  let wc_action = combo_action () in
  let _ = wc_action#entry#set_text action in
  let _ = hbox#pack ~expand: false ~padding: 2 wc_action#coerce in      
  object (self)
    val mutable name = name
    val mutable v = v
    val mutable activated = activated
			
    method variable = V.new_variable name self#value self#action
    method value = V.V_String (self#v, activated)
    method name = name
    method v = we_v#text
    method activated = activated
    method action = wc_action#entry#text

    method coerce = hbox#coerce

    method private toggle_activated () =
      activated <- not activated;
      wchk_acti#set_active activated;
      we_v#set_editable activated

    method set_name newname =
      name <- newname;
      wl_name#set_text (newname ^ " : ")

    initializer
      ignore (wchk_acti#connect#toggled self#toggle_activated)

  end

class slist (name, l, action) =
  let wf = GBin.frame ~label: name () in
  let vbox = GPack.vbox ~packing: wf#add () in
  let hbox = GPack.hbox ~packing: (vbox#pack ~expand: true) () in
  (* the scroll window and the clist *)
  let wscroll = GBin.scrolled_window
      ~vpolicy: `AUTOMATIC
      ~hpolicy: `AUTOMATIC
      ~packing: (hbox#pack ~expand: true)
      () 
  in
  let wlist = 
    GList.clist ~selection_mode: `EXTENDED 
      ~titles: [M.value ; M.activated]
      ~titles_show: true
      ~packing: wscroll#add ()
  in
  let hbox_action = GPack.hbox ~packing: (vbox#pack ~expand: false ~padding: 2) () in
  let wl_action = GMisc.label ~text: (M.action^" : ")
      ~packing: (hbox_action#pack ~expand: false ~padding: 2) ()
  in
  let wc_action = combo_action () in
  let _ = wc_action#entry#set_text action in
  let _ = hbox_action#pack ~expand: false wc_action#coerce in
  let vbox_buttons = GPack.vbox ~packing: (hbox#pack ~expand: false) () in
  let wb_add = GButton.button
      ~label: M.add
      ~packing: (vbox_buttons#pack ~expand:false ~padding:2)
      ()
  in
  let wb_edit = GButton.button ~label: M.edit 
      ~packing: (vbox_buttons#pack ~expand:false ~padding:2) () 
  in
  let wb_up = GButton.button
      ~label: M.up
      ~packing: (vbox_buttons#pack ~expand:false ~padding:2)
      ()
  in
  let wb_remove = GButton.button
      ~label: M.remove
      ~packing: (vbox_buttons#pack ~expand:false ~padding:2)
      ()
  in
  let f_strings (s, b) = [ s ; if b then M.yes else M.no ] in
  let eq (s1,_) (s2,_) = s1 = s2 in
  let compare (s1,_) (s2,_) = compare s1 s2 in
  object (self)
    val mutable name = name
    val mutable values = l

    method variable = V.new_variable name self#value self#action
    method value = V.V_List values
    val mutable values_selected = ([] : int list)
    method name = name
    method action = wc_action#entry#text

    method coerce = wf#coerce

    method set_name newname =
      name <- newname;
      wf#set_label newname

    method edit (value, bool) =
      let s1 = ref value 
      and s2 = ref bool in
      let p1 = C.string ~f:(fun s -> s1 := s) M.list_edit_name value
      and p2 = C.bool ~f:(fun s -> s2 := s) M.list_edit_activated bool in
	match C.simple_get M.list_edit [p1; p2] with
	  | Configwin.Return_ok ->
	      (!s1, !s2)
	  | _ -> raise Not_found

    method update l =
      values <- l;
      (* insert the elements in the clist *)
      wlist#freeze ();
      wlist#clear ();
      List.iter 
        (fun (s,b) -> 
          ignore (wlist#append (f_strings (s,b)));
	  if not b then
            try wlist#set_row ~foreground: (`NAME "Grey") (wlist#rows - 1)
            with _ -> ()
        ) 
	values;
      
      GToolbox.autosize_clist wlist;
      wlist#thaw ();
      (* the list of selectd elements is now empty *)
      values_selected <- []

    (** Move up the selected rows. *)
    method up_selected =
      let rec iter n selrows l =
        match selrows with
          [] -> (l, [])
        | m :: qrows ->
            match l with
              [] -> ([],[])
            | [_] -> (l,[])
            | e1 :: e2 :: q when m = n + 1 ->
                let newl, newrows = iter (n+1) qrows (e1 :: q) in
                (e2 :: newl, n :: newrows)
            | e1 :: q ->
                let newl, newrows = iter (n+1) selrows q in
                (e1 ::  newl, newrows)
      in
      let sorted_select = List.sort Pervasives.compare values_selected in
      let new_list, new_rows = iter 0 sorted_select values in
      self#update new_list;
      List.iter (fun n -> wlist#select n 0) new_rows

    (** Make the user edit the first selected row. *)
    method edit_selected =
      let sorted_select = List.sort Pervasives.compare values_selected  in
      match sorted_select with
        [] -> ()
      | n :: _ ->
          try
            let ele = List.nth values n in
            let ele2 = self#edit ele in
            let rec iter m = function
                [] -> []
              | e :: q ->
                  if n = m then
                    ele2 :: q
                  else
                    e :: (iter (m+1) q)
            in
            self#update (iter 0 values);
            wlist#select n 0
          with
            Not_found ->
              ()

    method private add () =
      let value =
	match GToolbox.input_string ~title:M.list_add M.list_add_name with
	  | None -> []
	  | Some s -> [(s, true)] in 
      let list = List.fold_left (fun acc ele ->
				   if List.exists (eq ele) acc then
				     acc
				   else acc @ [ele]) values value in
	self#update list

    method private remove () =
      let rec iter n = function
          [] -> []
        | h :: q ->
            if List.mem n values_selected then
              iter (n+1) q
            else
              h :: (iter (n+1) q) in
      let new_list = iter 0 values in
        self#update new_list

    method private select ~row ~column ~event =
      try
        values_selected <- row :: values_selected
      with
        | Failure _ -> ()
	    
    method private unselect ~row ~column ~event =
      try
        let new_values_selected = List.filter (fun n -> n <> row) values_selected in
          values_selected <- new_values_selected
      with
        | Failure _ -> ()


     initializer
      (* connect the functions to the buttons *)
      ignore (wb_add#connect#clicked self#add);
      ignore (wb_remove#connect#clicked self#remove);
      ignore (wb_up#connect#clicked (fun () -> self#up_selected));
      ignore (wb_edit#connect#clicked (fun () -> self#edit_selected));

      (* connect the select and deselect events *)
      ignore(wlist#connect#select_row self#select);
      ignore(wlist#connect#unselect_row self#unselect);

      self#update values
  end

let build_var (name, v, action) = 
  match v with
      V.V_String (value, acti) ->
	let box = new box_string (name, value, acti, action) in
	  ((name, (box :> box_variable)), false)
    | V.V_List l -> 
	let box = new slist (name, l, action) in
	  ((name, (box :> box_variable)), true)

let build_var_list l =
  let vbox = GPack.vbox () in
  let boxes = List.map build_var l in
  List.iter
    (fun ((_, box),expand) -> vbox#pack ~expand box#coerce)
    boxes;
    (vbox, List.map fst boxes)
