/*****************************************************************************
 * $CAMITK_LICENCE_BEGIN$
 *
 * CamiTK - Computer Assisted Medical Intervention ToolKit
 * (c) 2001-2013 UJF-Grenoble 1, CNRS, TIMC-IMAG UMR 5525 (GMCAO)
 *
 * Visit http://camitk.imag.fr for more information
 *
 * This file is part of CamiTK.
 *
 * CamiTK is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License version 3
 * only, as published by the Free Software Foundation.
 *
 * CamiTK is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License version 3 for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * version 3 along with CamiTK.  If not, see <http://www.gnu.org/licenses/>.
 *
 * $CAMITK_LICENCE_END$
 ****************************************************************************/
#include "FillWithPoints.h"

#include <Application.h>
#include <MeshComponent.h>
#include <Log.h>


#include <VtkMeshUtil.h>

#include <vtkPointLocator.h>
#include <vtkCallbackCommand.h>
#include <vtkPolyData.h>
#include <vtkSelectEnclosedPoints.h>
#include <vtkAppendPolyData.h>

#include <QDateTime>

// -------------------- FillWithPoints --------------------
FillWithPoints::FillWithPoints(ActionExtension* extension) : Action(extension) {
    setName("Fill With Points");
    setDescription("Fill a surfacic mesh with regularly spaced nodes (create new nodes inside the mesh).");

    setComponent("MeshComponent");
    setFamily("Mesh Processing");
    addTag("Add Nodes");

    setProperty("Bucket Size", QVariant(1));  // nv of points per buckets, higher = coarser filling, 1 = highest density possible
    setProperty("Randomize", QVariant(true)); // randomize the position of the point +/- 0.5
}

// --------------- apply -------------------
Action::ApplyStatus FillWithPoints::apply() {
    //CAMITK_INFO("FillWithPoints", "apply", "Adding inside nodes to " << getTargets().last()->getName().toStdString());

    // set waiting cursor and status bar
    QApplication::setOverrideCursor ( QCursor ( Qt::WaitCursor ) );
    Application::showStatusBarMessage("Filling With Nodes...");
    Application::resetProgressBar();

    // use the last target
    MeshComponent *targetMesh = dynamic_cast<MeshComponent*>(getTargets().last());

    //-- check if this is a polydata and if this is an unstructured grid, extract the surface
    vtkSmartPointer<vtkPolyData> targetPolyData = VtkMeshUtil::vtkPointSetToVtkPolyData(targetMesh->getPointSet());
    //CAMITK_INFO("FillWithPoints", "apply", "polydata: points:" << targetPolyData->GetNumberOfPoints() << " , cells:" << targetPolyData->GetNumberOfCells());
    
    //-- computes the subdivision depending on the mesh resolution
    vtkSmartPointer<vtkPointLocator> subdivisionLocator = vtkSmartPointer<vtkPointLocator>::New();
    subdivisionLocator->SetDataSet ( targetPolyData );
    subdivisionLocator->SetNumberOfPointsPerBucket(property("Bucket Size").toInt());
    subdivisionLocator->BuildLocator();
    // number of subdivisions for the x, y and z axis. If the mesh density is high the nbDiv will be high
    // nbDiv is computed so that there is bucketSize nodes per division
    int nbDiv[3];
    subdivisionLocator->GetDivisions(nbDiv);
    //CAMITK_INFO("FillWithPoints", "apply", "Nb divisions: " << nbDiv[0] << "," << nbDiv[1] << "," << nbDiv[2]);

    // length of the divisions in the x-axis, y-axis and z-axis
    double *bounds = targetPolyData->GetPoints()->GetBounds(); // [xmin,xmax, ymin,ymax, zmin,zmax]
    double xDiv = ( bounds[1] - bounds[0] ) / nbDiv[0]; 
    double yDiv = ( bounds[3] - bounds[2] ) / nbDiv[1];
    double zDiv = ( bounds[5] - bounds[4] ) / nbDiv[2];

    //-- generates a regular grid in the bounding box
    vtkSmartPointer<vtkPoints> points = vtkSmartPointer<vtkPoints>::New();

    // position for generated points
    double x, y, z;

    if (property("Randomize").toBool())
        qsrand ( QDateTime::currentDateTime().toTime_t() );
    for ( int i = 0; i <= nbDiv[0]; i++ ) {
        x = bounds[0] + i * xDiv;

        for ( int j = 0; j <= nbDiv[1]; j++ ) {
            y = bounds[2] + j * yDiv;

            for ( int k = 0; k <= nbDiv[2]; k++ ) {
                z = bounds[4] + k * zDiv;
                // randomize
		if (property("Randomize").toBool()) {
		    x += ( xDiv / 10.0 ) * ( 0.5 - double ( qrand() ) / ( double ( RAND_MAX ) + 1.0 ) );
		    y += ( yDiv / 10.0 ) * ( 0.5 - double ( qrand() ) / ( double ( RAND_MAX ) + 1.0 ) );
		    z += ( zDiv / 10.0 ) * ( 0.5 - double ( qrand() ) / ( double ( RAND_MAX ) + 1.0 ) );
		}
		//CAMITK_INFO("FillWithPoints", "apply", "insert point: (" << x << "," << y << "," << z << ")");
		points->InsertNextPoint ( x, y, z );
            }
        }
    }

    //-- select internal points only
    vtkSmartPointer<vtkPolyData> gridPoints = vtkSmartPointer<vtkPolyData>::New();
    gridPoints->SetPoints ( points );
    vtkSmartPointer<vtkSelectEnclosedPoints> select = vtkSmartPointer<vtkSelectEnclosedPoints>::New();
    select->SetInput ( gridPoints );
    select->SetSurface ( targetPolyData );
    select->Update();

    //-- create a new polydata that contains only the internal selected points
    vtkSmartPointer<vtkPoints> insidePointsPoints = vtkSmartPointer<vtkPoints>::New();
    for ( int i = 0; i < gridPoints->GetPoints()->GetNumberOfPoints(); i++ ) {
        if ( select->IsInside ( i ) ) {
            insidePointsPoints->InsertNextPoint ( gridPoints->GetPoints()->GetPoint ( i ) );
        }
    }
    
    vtkSmartPointer<vtkPolyData> insidePoints = vtkSmartPointer<vtkPolyData>::New();
    insidePoints->SetPoints ( insidePointsPoints );
    insidePoints->Allocate ( 2 );
    vtkIdType *vtkPointIndex = new vtkIdType [insidePointsPoints->GetNumberOfPoints()];
    for (vtkIdType i=0; i<insidePointsPoints->GetNumberOfPoints(); i++)
	vtkPointIndex[i] = i;
    insidePoints->InsertNextCell(VTK_POLY_VERTEX, insidePointsPoints->GetNumberOfPoints(),vtkPointIndex);
    insidePoints->Update();

    //-- Generates one structure with all the points (append targetPolyData and insidePoints)
    vtkSmartPointer<vtkAppendPolyData> appender = vtkSmartPointer<vtkAppendPolyData>::New();
    appender->AddInput ( targetPolyData );
    appender->AddInput ( insidePoints );
    appender->Update();
    //CAMITK_INFO("FillWithPoints", "apply", "insidePointsPoints: points:" << insidePointsPoints->GetNumberOfPoints());
    //CAMITK_INFO("FillWithPoints", "apply", "targetPolyData: points:" << targetPolyData->GetNumberOfPoints() << " , cells:" << targetPolyData->GetNumberOfCells());

    //-- Create a new mesh Component using an unstructured grid (to show the isolated nodes)
    vtkSmartPointer<vtkPointSet> resultPointSet = appender->GetOutput();
    //CAMITK_INFO("FillWithPoints", "apply", "resultPointSet: points:" << resultPointSet->GetNumberOfPoints() << " , cells:" << resultPointSet->GetNumberOfCells());
    new MeshComponent(resultPointSet, targetMesh->getName() + " filled");
    Application::refresh();    

    // restore the normal cursor and progress bar
    Application::resetProgressBar();
    QApplication::restoreOverrideCursor();
    return SUCCESS;
}
