/*****************************************************************************
 * $CAMITK_LICENCE_BEGIN$
 *
 * CamiTK - Computer Assisted Medical Intervention ToolKit
 * (c) 2001-2013 UJF-Grenoble 1, CNRS, TIMC-IMAG UMR 5525 (GMCAO)
 *
 * Visit http://camitk.imag.fr for more information
 *
 * This file is part of CamiTK.
 *
 * CamiTK is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License version 3
 * only, as published by the Free Software Foundation.
 *
 * CamiTK is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License version 3 for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * version 3 along with CamiTK.  If not, see <http://www.gnu.org/licenses/>.
 *
 * $CAMITK_LICENCE_END$
 ****************************************************************************/

#include "Component.h"
#include "VtkMeshComponentPopup.h"
#include "VtkMeshComponent.h"
#include "VtkMeshComponentRegistrationAddon.h"

#include <Application.h>

#include <InteractiveViewer.h>
#include <Geometry.h>

#include <QFileDialog>
#include <QFileInfo>
#include <QLibrary>
#include <QInputDialog>
#include <QCheckBox>
#include <QSlider>
#include <QPushButton>
#include <QDateTime>
#include <QMessageBox>

#include <vtkTransformFilter.h>
#include <vtkTransform.h>
#include <vtkBMPReader.h>
#include <vtkImageFlip.h>
#include <vtkTexture.h>
#include <vtkDelaunay3D.h>
#include <vtkCleanPolyData.h>
#include <vtkSelectEnclosedPoints.h>
#include <vtkPointLocator.h>
#include <vtkCell.h>
#include <vtkAppendPolyData.h>
#include <vtkPointData.h>
#include <vtkUnstructuredGrid.h>
#include <vtkPolyDataWriter.h>

#include <sstream>

// ---------------------- constructor ----------------------------
VtkMeshComponentPopup::VtkMeshComponentPopup ( Component *dc, QWidget *parent ) : ComponentPopupMenu ( dc, parent ) {

    myComponent = dynamic_cast<VtkMeshComponent * >( dc );

    addAction ( "Load Transformation", SLOT ( loadTransformationFile() ) );
    addSeparator();
    QMenu *pointDataMenu = new QMenu("Point Data");

    QAction *showNoPointDataAction = pointDataMenu->addAction("Hide Point Data");
    connect(showNoPointDataAction, SIGNAL(triggered()), this , SLOT( noPointData()));
    showNoPointDataAction->setCheckable(true);
    showNoPointDataAction->setChecked ( true );

    QAction * showVTKPointDataAction = pointDataMenu->addAction ( "Show VTK Point Data" );
    connect(showVTKPointDataAction, SIGNAL(triggered()), this , SLOT( vtkPointData()));
    showVTKPointDataAction->setCheckable(true);
    showVTKPointDataAction->setChecked ( false );

    QAction * showDemoPointDataAction = pointDataMenu->addAction ( "Show Demo Point Data" );
    connect(showDemoPointDataAction, SIGNAL(triggered()), this , SLOT( demoPointData()));
    showDemoPointDataAction->setCheckable(true);
    showDemoPointDataAction->setChecked ( false );

    addMenu(pointDataMenu);
    pointDataMenu->setEnabled(true);

    QActionGroup *pointDataGroup = new QActionGroup(this);
    pointDataGroup->addAction(showNoPointDataAction);
    pointDataGroup->addAction(showVTKPointDataAction);
    pointDataGroup->addAction(showDemoPointDataAction);

    addAction ( "Load Texture from BMP File", SLOT ( loadTextureFromBMPFile() ) );
    addAction ( "Mesh registration", SLOT( meshRegistration( ) ) );
    addAction ( "Save Displacement From Transformation", SLOT ( getDisplacementFromTransformation() ) );
    addAction ( "Export as MDL", SLOT ( exportMDL() ) );
    addAction ( "Clean Nodes...", SLOT ( cleanPolyData() ) );

}


// ---------------------- loadTransformationFile ----------------------------
void VtkMeshComponentPopup::loadTransformationFile() {
    // get the file name
    QString fn = QFileDialog::getOpenFileName ( this, "Load Transformation" );

    if ( !fn.isNull() ) {
        //-- load the transformation
        std::ifstream in ( fn.toStdString().c_str() );

        vtkSmartPointer<vtkTransform> transformation = vtkSmartPointer<vtkTransform>::New();
        double x, y, z, t;
        for ( unsigned int i = 0;i < 4; i++ ) {
            in >> x >> y >> z >> t;
            transformation->GetMatrix()->SetElement ( i, 0, x );
            transformation->GetMatrix()->SetElement ( i, 1, y );
            transformation->GetMatrix()->SetElement ( i, 2, z );
            transformation->GetMatrix()->SetElement ( i, 3, t );
        }

        //-- apply the transformation to myComponent data set
        vtkSmartPointer<vtkPointSet> newData;
        if ( vtkUnstructuredGrid::SafeDownCast ( myComponent->getPointSet() ) ) {
            newData = vtkSmartPointer<vtkUnstructuredGrid>::New();
        }
        else
            if ( vtkPolyData::SafeDownCast ( myComponent->getPointSet() ) ) {
                newData = vtkSmartPointer<vtkPolyData>::New();
            }
            else
                return;

        // apply a vtk transform filter and get the result
        vtkSmartPointer<vtkTransformFilter> vtkFilter = vtkSmartPointer<vtkTransformFilter>::New();
        vtkFilter->SetTransform ( transformation );
        vtkFilter->SetInputConnection ( myComponent->getDataPort() );
        myComponent->setDataConnection ( vtkFilter->GetOutputPort() );

        vtkSmartPointer<vtkPointSet> result = vtkPointSet::SafeDownCast ( vtkFilter->GetOutputDataObject ( 0 ) );
        if ( result )
            myComponent->setPointSet ( result );

        myComponent->setModified();
    }
}

// ---------------------- noPointData ----------------------------
void VtkMeshComponentPopup::noPointData() {
    dynamic_cast<VtkMeshComponent *> ( myComponent )->showPointData(VtkMeshComponent::NONE);
}


// ---------------------- vtkPointData ----------------------------
void VtkMeshComponentPopup::vtkPointData() {
    dynamic_cast<VtkMeshComponent *> ( myComponent )->showPointData(VtkMeshComponent::INITIAL);
}

// ---------------------- demoPointData ----------------------------
void VtkMeshComponentPopup::demoPointData() {
    dynamic_cast<VtkMeshComponent *> ( myComponent )->showPointData(VtkMeshComponent::DEMO);
}

// ---------------------- loadTextureFromBMPFile ----------------------------
void VtkMeshComponentPopup::loadTextureFromBMPFile() {
    QString trFile = QFileDialog::getOpenFileName ( this, tr ( "Open BMP File" ), QString(), "BMP File (*.bmp)" );

    vtkSmartPointer<vtkBMPReader> bmpReader = vtkSmartPointer<vtkBMPReader>::New();
    bmpReader->SetFileName ( trFile.toStdString().c_str() );

    // Create a brand new imageYFlip.
    vtkSmartPointer<vtkImageFlip> imageYFlip = vtkSmartPointer<vtkImageFlip>::New();
    imageYFlip->SetInputConnection ( bmpReader->GetOutputPort() );
    imageYFlip->SetFilteredAxis ( 1 );

    vtkSmartPointer<vtkTexture> texture = vtkSmartPointer<vtkTexture>::New();
    texture->SetInputConnection ( imageYFlip->GetOutputPort() );

    dynamic_cast<VtkMeshComponent *> ( myComponent )->setTexture ( texture );

}

// --------------------   meshRegistration   ------------------------------------
void VtkMeshComponentPopup::meshRegistration( ) {

    // Search for the calibration library using a dialog box
    QString libraryName = QFileDialog::getOpenFileName(this, "Open an mesh registration library", 0, "Add-on (*.so *.dll *.dylib)" );

    // Process library
    if ( libraryName != "" ) {
        processLibrary( libraryName );
    }
}

// --------------------   processLibrary   ------------------------------------
void VtkMeshComponentPopup::processLibrary( QString libraryName ) {
    PtrToVtkMeshComponentRegistrationAddon vtkManagerComponentRegistrationAddonFunction = NULL;

    // Getting the pointer of the addon constructor.
    vtkManagerComponentRegistrationAddonFunction = (PtrToVtkMeshComponentRegistrationAddon) QLibrary::resolve( libraryName, "getVtkMeshComponentRegistrationAddon");

    // Creating the calibration dialog by calling the addon constructor.
    if ( vtkManagerComponentRegistrationAddonFunction )
        vtkManagerComponentRegistrationAddonFunction( myComponent );
}


// ---------------------- getDisplacementFromTransformation ----------------------------
void VtkMeshComponentPopup::getDisplacementFromTransformation() {
    /* Compute the displacement of each points with a given transformation, and put the results in an
     *    ostream, formatted as Ansys batch file.
     *    Note that the Geometry is NOT transformed.
     */

    // get the transformation file name
    QString fn = QFileDialog::getOpenFileName ( this, "Load Transformation" );

    if ( !fn.isNull() ) {
        std::ifstream in ( fn.toStdString().c_str() );

        vtkSmartPointer<vtkTransform> transfo = vtkSmartPointer<vtkTransform>::New();
        double x, y, z, t;
        for ( unsigned int i = 0;i < 4; i++ ) {
            in >> x >> y >> z >> t;
            transfo->GetMatrix()->SetElement ( i, 0, x );
            transfo->GetMatrix()->SetElement ( i, 1, y );
            transfo->GetMatrix()->SetElement ( i, 2, z );
            transfo->GetMatrix()->SetElement ( i, 3, t );
        }

        // get the output file name
        QString dFile = QFileDialog::getSaveFileName ( this, tr ( "Save As Ansys..." ), QString(), "Displacement in Ansys batch format (*.txt)" );

        if ( !dFile.isNull() ) {
            std::ofstream os ( dFile.toStdString().c_str() );

            // 0. compute the tranformation (=the lazy way to compute it)
            vtkSmartPointer<vtkPointSet> ps = myComponent->getPointSet();

            // Actually transform the points of the VTK dataset
            vtkSmartPointer<vtkTransformFilter> transformFilter = vtkSmartPointer<vtkTransformFilter>::New();
            transformFilter->SetTransform ( transfo );
            transformFilter->SetInput ( ps );

            transformFilter->Update();

            vtkSmartPointer<vtkPoints> newPts = transformFilter->GetOutput()->GetPoints();//ps->GetPoints();
            vtkSmartPointer<vtkPoints> currentPts = ps->GetPoints();

            // 1. compute the displacement and save it
            double newPos[3];
            double currentPos[3];

            for ( int i = 0; i < newPts->GetNumberOfPoints(); i++ ) {
                // get the current position
                currentPts->GetPoint ( i, currentPos );
                // get the new position
                newPts->GetPoint ( i, newPos );

                // INDEX +1 because of Ansys!!!!
                // put the displacement in the os, in Ansys displacement format
                os << "D,  " << i + 1 << ", , " << newPos[0] - currentPos[0] << " \t, , , ,UX, , , , , " << std::endl;
                os << "D,  " << i + 1 << ", , " << newPos[1] - currentPos[1] << " \t, , , ,UY, , , , , " << std::endl;
                os << "D,  " << i + 1 << ", , " << newPos[2] - currentPos[2] << " \t, , , ,UZ, , , , , " << std::endl;
            }
        }

    }
}

// ---------------------- exportMDL ----------------------------
void VtkMeshComponentPopup::exportMDL() {
    // get the file name
    //    myComponent->getDataManager()->getFileName()
    QString fn = QFileDialog::getSaveFileName ( this, tr ( "Save As MDL..." ), QString(), "MDL format(*.mdl)" );

    if ( !fn.isEmpty() ) {
        // if the filename does not have the ".mdl" extension, add it
        QString extension = fn.right ( 4 );
        QString txt ( ".mdl" );
        if ( extension.compare ( txt ) != 0 ) {
            // add the extension to the filename
            fn += txt;
        }

        ( ( VtkMeshComponent * ) myComponent )->exportMDL ( fn.toStdString() );
    }
}



// ---------------------- cleanPolyData --------------------------
void VtkMeshComponentPopup::cleanPolyData() {

    // Get tolerance
    bool ok;
    int tolerance = QInputDialog::getInt ( NULL, "Tolerance", "Tolerance Value (in %)", 1, 0, 100, 1, &ok );
    if ( ok ) {
        // remove redundant points
        vtkSmartPointer<vtkCleanPolyData> cleaner = vtkSmartPointer<vtkCleanPolyData>::New();
        cleaner->SetInput ( myComponent->getPointSet() );
        cleaner->PointMergingOn();
        cleaner->SetTolerance ( double ( tolerance ) *0.01 );  // x% of the Bounding Box
        cleaner->ConvertLinesToPointsOff ();
        cleaner->ConvertPolysToLinesOff ();
        cleaner->ConvertStripsToPolysOff ();
        cleaner->Update();

        // generates output name
        QFileInfo inputFileInfo = QFileInfo(myComponent->getFileName());
        QString outputFilename = inputFileInfo.dir().canonicalPath() + "/" + inputFileInfo.baseName() + "_cleaned.vtk";

        // save it
        vtkSmartPointer<vtkPolyDataWriter> pdWriter = vtkSmartPointer<vtkPolyDataWriter>::New();
        pdWriter->SetInput ( cleaner->GetOutput() );
        pdWriter->SetFileName ( outputFilename.toStdString().c_str() );
        pdWriter->SetHeader ( QString("Cleaned from " + myComponent->getName()).toStdString().c_str() );
        pdWriter->SetFileType ( VTK_ASCII );
        pdWriter->Update();
    }
}


