/*****************************************************************************
 * $CAMITK_LICENCE_BEGIN$
 *
 * CamiTK - Computer Assisted Medical Intervention ToolKit
 * (c) 2001-2013 UJF-Grenoble 1, CNRS, TIMC-IMAG UMR 5525 (GMCAO)
 *
 * Visit http://camitk.imag.fr for more information
 *
 * This file is part of CamiTK.
 *
 * CamiTK is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License version 3
 * only, as published by the Free Software Foundation.
 *
 * CamiTK is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License version 3 for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * version 3 along with CamiTK.  If not, see <http://www.gnu.org/licenses/>.
 *
 * $CAMITK_LICENCE_END$
 ****************************************************************************/

// -- Core stuff
#include "Core.h"
#include "CamiTKVersion.h"
#include "Log.h"
#include "ExtensionManager.h"
#include "Action.h"

// -- QT stuff
#include <QDir>
#include <QApplication>


namespace camitk {
QString Core::binDirName;

// ------------- getConfig (static) -----------------
const QString Core::getConfig() {
    QStringList diagnosis;

    diagnosis << "CamiTK............................ " + QString(Core::version);
    diagnosis << "- CamiTK Short Version............ " + QString(Core::shortVersion);
    diagnosis << "- CamiTK SO NAME.................. " + QString(Core::soVersion);
    diagnosis << "- Installed....................... " + QString(isInstalled()?"Yes":"No");
    diagnosis << "- Root Dir........................ " + getRootDir();
    //diagnosis << "- Bin Dir......................... " + getBinDirName();
    diagnosis << "- Shared Lib Dir.................. " + sharedDirectory();

    if (!hasBuildTypeDirectory().isEmpty())
      diagnosis << "- Has Build Type Directory........ " + hasBuildTypeDirectory();

    if (hasMacOSDirectory())
        diagnosis << "- Has Mac OS Directory";

    if (hasBinaryVersionSubdir())
        diagnosis << "- Has Binary Version Sub Directory";

    diagnosis << "- Install Directory............... " + QString(Core::installDir);
    diagnosis << "- Core Library Directory.......... " + Core::getCoreLibDir();
    diagnosis << "- Test Data Directory............. " + Core::getTestDataDir();
    diagnosis << "- Component Extension Directory... " + Core::getComponentDir();
    diagnosis << "- Action Extension Directory...... " + Core::getActionDir();

    unsigned int componentCount = 0;
    QStringList components;
    const QMap<QString, ComponentExtension*> & allCE = ExtensionManager::getComponentExtensions();

    foreach(ComponentExtension *ce, allCE.values().toSet()) {
        components << "- " + ce->getName() + " (" + ce->getFileExtensions().join(" ") + ") : " + ce->getDescription();
        componentCount++;
    }

    diagnosis << "- Number of Component Extensions.. " + QString::number(componentCount);

    unsigned int actionCount = 0;
    QStringList actions;
    const QMap<QString, ActionExtension* >& allActions = ExtensionManager::getActionExtensions();

    foreach(ActionExtension *ae, allActions.values()) {
        actions << "- " + ae->getName() + ": " + ae->getDescription();

        foreach(Action *a, ae->getActions()) {
            actions << "\t- " + a->getName() + ": " + a->getDescription();
            actionCount++;
        }
    }

    diagnosis << "- Number of Action Extensions..... " + QString::number(actionCount);

    diagnosis << "- Registered components:";
    diagnosis += components;

    diagnosis << "- Registered actions:";
    diagnosis += actions;
    
    return diagnosis.join("\n");
}

// ------------- getComponentDir (static) -----------------
const QString Core::getComponentDir() {
    QDir rootDir(getRootDir()); // either install or build

    QString subDir = hasBuildTypeDirectory();
    QString componentSubDir = sharedDirectory();

    if (hasBinaryVersionSubdir())
        componentSubDir += "/" + QString(Core::shortVersion);

    componentSubDir += "/components/" + hasBuildTypeDirectory();

    // check components subdir
    if (!rootDir.cd(componentSubDir)) {
        CAMITK_ERROR("Core", "getComponentDir", "Component extension directory not found\n\tRoot dir: " << getRootDir().toStdString()<< "\n\tComponent subdir: " << componentSubDir.toStdString());
        return QString(); // null QString
    }

    return rootDir.canonicalPath().toUtf8();
}

// ------------- getActionDir (static) -----------------
const QString Core::getActionDir() {
    QDir rootDir(getRootDir()); // either install or build

    QString subDir = hasBuildTypeDirectory();
    QString actionSubDir = sharedDirectory();

    if (hasBinaryVersionSubdir())
        actionSubDir += "/" + QString(Core::shortVersion);

    actionSubDir += "/actions/" + hasBuildTypeDirectory();

    // check actions subdir
    if (!rootDir.cd(actionSubDir)) {
        CAMITK_ERROR("Core", "getActionDir", "Action extension directory not found\n\tRoot dir: " << getRootDir().toStdString()<< "\n\tAction subdir: " << actionSubDir.toStdString());
        return QString(); // null QString
    }

    return rootDir.canonicalPath().toUtf8();
}


// ------------- getModuleDir (static) -----------------
const QString Core::getModuleDir(const QString & subpath) {
    QDir rootDir(getRootDir()); // either install or build

    QString subDir = hasBuildTypeDirectory();
    QString moduleSubDir = sharedDirectory();

    if (hasBinaryVersionSubdir())
        moduleSubDir += "/" + QString(Core::shortVersion);

    moduleSubDir += "/modules/" + subpath + "/" + hasBuildTypeDirectory();

    // check components subdir
    if (!rootDir.cd(moduleSubDir)) {
        CAMITK_INFO("Core", "getModuleDir", "Module extension directory not found\nRoot dir: " << getRootDir().toStdString()<< "\n\tLooking for subdir: " << subpath.toStdString() << "\n\tModule subdir: " << moduleSubDir.toStdString());
        return QString(); // null QString
    }

    return rootDir.canonicalPath().toUtf8();
}

// ------------- getTestDataDir (static) -----------------
const QString Core::getTestDataDir() {
    QDir rootDir(getRootDir()); // either install or build

    QString exampleSubDir = "share";

    if (hasBinaryVersionSubdir())
        exampleSubDir += "/" + QString(Core::shortVersion);

    exampleSubDir += "/testdata";

    if (!rootDir.cd (exampleSubDir)) {
        CAMITK_ERROR("Core", "getTestDataDir", "Test data directory not found\n\tRoot dir: " << getRootDir().toStdString()<< "\n\tTest data subdirectory attempted: " << exampleSubDir.toStdString());
        return QString(); // null QString
    }

    return rootDir.canonicalPath().toUtf8();
}


// ------------- hasBuildTypeDirectory (static) -----------------
const QString Core::hasBuildTypeDirectory() {
    QString buildType;
    // take care of system specificities
#if defined(Q_CC_MSVC)
    // get the executable path
    QDir binDir ( qApp->applicationDirPath() );

    // check the name
    if ( binDir.dirName().toLower() == "debug" || binDir.dirName().toLower() == "release" ) {
        buildType = binDir.dirName();
    }

#endif
    return buildType;

}

// ------------- hasMacOSDirectory (static) -----------------
const bool Core::hasMacOSDirectory() {
    bool hasMacOS = false;

#if defined(Q_OS_MAC)
    // get the executable path
    QDir binDir ( qApp->applicationDirPath() );
    // get the executable path (bin dir)
    hasMacOS = (binDir.dirName() == "MacOS");
#endif

    return hasMacOS;
}

// ------------- hasBinaryVersionSubdir (static) -----------------
const bool Core::hasBinaryVersionSubdir() {
#if defined(Q_CC_GNU) && !defined(__MINGW32__) // linux but not mingw32

    if (isInstalled())
        return true; // only when installed

#endif
    return false;
}

// ------------- isInstalled (static) -----------------
const bool Core::isInstalled() {
    // compare the variable created at build time with the application directory
    return (getRootDir() == QDir(installDir).canonicalPath());
}

// ------------- getRootDir (static) -----------------
const QString Core::getRootDir() {
    // get the executable path (bin dir)
    QDir rootDir ( qApp->applicationDirPath() );
    binDirName = "bin";
    QString nullString;
    bool cdOk = true;

    QString subdir = hasBuildTypeDirectory();

    if (!subdir.isNull()) {
        cdOk = cdOk && rootDir.cdUp(); // bin dir
    }
    else
        if (hasMacOSDirectory()) {
            cdOk = cdOk && rootDir.cdUp();
            cdOk = cdOk && rootDir.cdUp();
            cdOk = cdOk && rootDir.cdUp();
        }

    // here rootDir should be "bin"
    if (rootDir.dirName() != binDirName) {
        // check for "binDebug" (install of Debug version)
        binDirName = "binDebug";

        if (rootDir.dirName() != binDirName) {
            CAMITK_ERROR("Core", "getRootDir", "Invalid binary directory.\tApplication " << qApp->applicationName().toStdString() << " in " << qApp->applicationDirPath().toStdString() << "\n\tbin directory:" << rootDir.canonicalPath().toStdString());
            return nullString;
        }
    }

    // So as we are in the bin directory, up we go!
    cdOk = cdOk && rootDir.cdUp();

    if ( !cdOk ) {
        CAMITK_ERROR("Core", "getRootDir", "Invalid root directory.\tApplication " << qApp->applicationName().toStdString() << " in " << qApp->applicationDirPath().toStdString() << "\n\tRoot directory:" << rootDir.canonicalPath().toStdString());
        return nullString;
    }
    else {
        // check now for sharedDir
        // starts from root dir
        QDir camitkcoreDir(rootDir);
        // check bin dir
        cdOk = camitkcoreDir.cd(binDirName);

        if (!cdOk)
            // try lib
            cdOk = camitkcoreDir.cd("lib");

        if (!subdir.isNull()) {
            cdOk = cdOk && camitkcoreDir.cd(subdir); // bin dir
        }

        // bin or bin/Debug or bin/Release (and if not there lib) were tested
        if (cdOk && checkCoreLibDir(camitkcoreDir.canonicalPath())) {
            // the corelib is where it should be: this is an internall call, or else a external call
            // but from the same build/install directory structure
            return rootDir.canonicalPath();
        }

        // either there is no shared directory or the shared directory does not contains
        // camitkcore, therefore getRootDir() must be called externally, root dir is installDir!
        return QDir(installDir).canonicalPath();
    }
}

// ------------- checkCoreLibDir (static) -----------------
bool Core::checkCoreLibDir(QString dir) {
    QStringList coreLibFilter;
    coreLibFilter << "*camitkcore.so." + QString(Core::soVersion) << "*camitkcore.dll" << "*camitkcore.dylib";
    QStringList coreLibs = QDir(dir).entryList(coreLibFilter, QDir::Files);
    return (coreLibs.size()==1);
}

// ------------- getLibDir (static) -----------------
const QString Core::getCoreLibDir() {
    QString rootDir = getRootDir();
    QString corelibSubDir = sharedDirectory();

    corelibSubDir += "/" + hasBuildTypeDirectory();

    // check: core lib should be here
    if (checkCoreLibDir(rootDir + "/" + corelibSubDir))
        return rootDir + "/" + corelibSubDir;
    else
        return QString();
}

// ------------- sharedDirectory (static) -----------------
const QString Core::sharedDirectory() {
    if (binDirName.isEmpty())
        binDirName="bin";

    QString sharedDir = binDirName;
#if defined(Q_CC_GNU) && !defined(__MINGW32__) // linux but not mingw32

    if (isInstalled())
        sharedDir = "lib";

#endif
    return sharedDir;
}

}





