/*****************************************************************************
 * $CAMITK_LICENCE_BEGIN$
 *
 * CamiTK - Computer Assisted Medical Intervention ToolKit
 * (c) 2001-2014 UJF-Grenoble 1, CNRS, TIMC-IMAG UMR 5525 (GMCAO)
 *
 * Visit http://camitk.imag.fr for more information
 *
 * This file is part of CamiTK.
 *
 * CamiTK is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License version 3
 * only, as published by the Free Software Foundation.
 *
 * CamiTK is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License version 3 for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * version 3 along with CamiTK.  If not, see <http://www.gnu.org/licenses/>.
 *
 * $CAMITK_LICENCE_END$
 ****************************************************************************/

#include <PMManagerDC.h>

#include <PhysicalModel.h>

#include <StructuralComponentDC.h>
#include <MultiComponentDC.h>

#include "CreateSC.h"
#include <QVector3D>
#include <Log.h>
#include <Application.h>
using namespace camitk;

// -------------------- CreateSC --------------------
CreateSC::CreateSC(ActionExtension* extension) : Action(extension) {
    this->setName("CreateSC");
    this->setDescription("Generates a Structural Component from a given box selection");
    this->setComponent("PMManagerDC");
    this->setFamily("Mesh Processing");
    this->addTag("CreateSC");
    this->addTag("Select");

    setProperty("Name Of The New Component", "selection");
    //TODO setProperty("Select Only On Surface", (bool) false);
    setProperty("Bottom Corner", QVector3D(0.0,0.0,0.0));
    setProperty("Top Corner", QVector3D(0.0,0.0,0.0));
}

// --------------- getWidget -------------------
QWidget * CreateSC::getWidget() {

    //-- init the property values using the selection state
    if (getTargets().size() == 1) {
        PMManagerDC *pmlDC = dynamic_cast<PMManagerDC*>(getTargets().last());

        //-- check if name exists, generate new if it does
        QString selectionName = property("Name Of The New Component").toString();

        while (pmlDC->getPhysicalModel()->getComponentByName(selectionName.toStdString())!=NULL) {
            // If the name contains a number then increment it
            QRegExp numSearch( "(^|[^\\d])(\\d+)" ); // we want to match as far left as possible, and when the number is at the start of the name

            // Does it have a number?
            int start = numSearch.lastIndexIn( selectionName );

            if (start != -1) {
                // It has a number, increment it
                start = numSearch.pos( 2 ); // we are only interested in the second group
                QString numAsStr = numSearch.capturedTexts()[ 2 ];
                QString number = QString::number( numAsStr.toInt() + 1 );
                number = number.rightJustified( numAsStr.length(), '0' );
                selectionName.replace( start, numAsStr.length(), number );
            } else {
                // no number
                start = selectionName.lastIndexOf('.');

                if (start != -1) {
                    // has a . somewhere, e.g. it has an extension
                    selectionName.insert(start, '1');
                } else {
                    // no extension, just tack it on to the end
                    selectionName += '1';
                }
            }
        }

        setProperty("Name Of The New Component",selectionName);
        double bounds[6]; // [xmin,xmax, ymin,ymax, zmin,zmax]
        pmlDC->getBounds(bounds);

        setProperty("Bottom Corner", QVector3D(bounds[0],bounds[2],bounds[4]));
        setProperty("Top Corner", QVector3D(bounds[1],bounds[3],bounds[5]));
    }

    return Action::getWidget();
}

// --------------- apply -------------------
Action::ApplyStatus CreateSC::apply() {
    CAMITK_INFO("CreateSC", "apply", "Create Structural Component in " << getTargets().last()->getName().toStdString());

    // set waiting cursor and status bar
    QApplication::setOverrideCursor ( QCursor ( Qt::WaitCursor ) );
    Application::showStatusBarMessage("Creating New Structural Component...");
    Application::resetProgressBar();

    // use the last target
    PMManagerDC *pmlDC = dynamic_cast<PMManagerDC*>(getTargets().last());
    PhysicalModel *pm=pmlDC->getPhysicalModel();

    // create the new component
    StructuralComponent * selection;
    QVector3D lower = property("Bottom Corner").value<QVector3D>();
    QVector3D top = property("Top Corner").value<QVector3D>();

    QString selectionName = property("Name Of The New Component").toString();
    selection = new StructuralComponent(pm, selectionName.toStdString());
    Cell *selectionC = new Cell(pm, StructureProperties::POLY_VERTEX);
    selection->addStructure(selectionC);

    unsigned int nbSelected = 0;
    Atom *a;
    double pos[3];

    for (unsigned int i = 0; i < pm->getAtoms()->getNumberOfStructures(); i++) {
        // get the src position
        a = (Atom *) pm->getAtoms()->getStructure(i);
        a->getPosition(pos);

        // should it be selected
        if (pos[0] >= lower.x() && pos[0] <= top.x() && pos[1] >= lower.y() && pos[1] <= top.y() && pos[2] >= lower.z() && pos[2] <= top.z()) {
            selectionC->addStructureIfNotIn(a);
            nbSelected++;
        }
    }

    Application::showStatusBarMessage("Done Creating New Structural Component (" + QString::number(nbSelected) + " atoms selected");

    // add the new SC to the informative component
    if (!pm->getInformativeComponents()) {
        MultiComponent *mc = new MultiComponent(pm, "Informative Components");
        pm->setInformativeComponents(mc);
    }

    pm->getInformativeComponents()->addSubComponent(selection);

    // Create the corresponding MCDC if needed
    MultiComponentDC *mcdc = pmlDC->getDC(pm->getInformativeComponents());
    StructuralComponentDC *scdc;

    if (!mcdc) {
        // no informative component => the creation of the informative component DC
        // is going to cascade down and create the new SCDC corresponding to selection
        new MultiComponentDC(pmlDC, pmlDC, pm->getInformativeComponents());
        scdc = pmlDC->getDC(selection);
    } else {
        // there is already the informative component, just add the new corresponding SCDC
        scdc = new StructuralComponentDC(mcdc,pmlDC,selection);
    }

    pmlDC->refreshInterfaceNode();

    Application::showStatusBarMessage("Created SC \"" + scdc->getName() + "\"");

    // -- refresh to show
    Application::refresh();

    // restore the normal cursor and progress bar
    Application::resetProgressBar();
    QApplication::restoreOverrideCursor();
    return SUCCESS;
}
