/*****************************************************************************
 * $CAMITK_LICENCE_BEGIN$
 *
 * CamiTK - Computer Assisted Medical Intervention ToolKit
 * (c) 2001-2014 UJF-Grenoble 1, CNRS, TIMC-IMAG UMR 5525 (GMCAO)
 *
 * Visit http://camitk.imag.fr for more information
 *
 * This file is part of CamiTK.
 *
 * CamiTK is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License version 3
 * only, as published by the Free Software Foundation.
 *
 * CamiTK is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License version 3 for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * version 3 along with CamiTK.  If not, see <http://www.gnu.org/licenses/>.
 *
 * $CAMITK_LICENCE_END$
 ****************************************************************************/

#ifndef LOADSANIMATIONMOTORADDON_H
#define LOADSANIMATIONMOTORADDON_H

#include "PMComponentAPI.h"

class Loads;
class LoadsManager;
class QWidget;
class QString;
class AnimationMotorAddon;

/// the ptr to the function that create ta new instance of the animation motor add-on
typedef AnimationMotorAddon * (*PtrToAnimationMotorAddon)(LoadsManager *);


/** The Animation Motor addon class.
  *
  * All animation motor addon have to inherit from this class (which is an interface).
  *
  * I <b>highly recommand</b> a look/study/thesis at/on the code of the
  * ShakerAnimationMotorAddon class.
  * It gives an good idea of all the different facets of a animation motor addon.
  *
  *

 */
class PHYSICALMODEL_COMPONENT_API AnimationMotorAddon {
public:

    /// the default constructor get the loads manager (and thus the Physical model, ...)
    AnimationMotorAddon(LoadsManager *lm);

    /// default virtual destructor
    virtual ~AnimationMotorAddon() {};

// virtual void clear()=0;

    /** get the optionnal operator widget
      * if there the return value is not null, then
      * the motor widget will be displayed as a new tab
      */
    virtual QWidget * getMotorWidget();

    /// start the simulation from the begining
    virtual void restart() = 0;

    /** This method is called everytime the motor has to do one step
      * of simulation.
      * I.e. this method is called everytime the application
      * needs to know the positions of the atoms at time t+dt.
      *
      * <b>Note:</b> There are two possibilities for motors to
      * produce deformations:
      * - Either by returning a Loads* filled up with the appropriate translations,
      * - or by modifying directly the position of the atoms. In this case
      *   doMove(..) <b>must return NULL</b>. It is a kind of "direct rendering".
      *
      * The latter is around 5% faster when a big percentage of the atoms are moving due to
      * the action of the motor; consequently for a very low 3D refresh rate, the
      * speed difference can become quite important. It is strongly advise to
      * decrease the 3D refresh rate and use direct rendering.
      *
      * The two methods are provided for convenience (sometimes it is easier to produce
      * loads, and sometimes easier to directly modify the atom positions).
      *
      * Again, see the artistic effect ShakerAnimationMotorAddon for more infos and examples
      * (you could also use the shaker to check performances for your objects).
      *
      * The Loads * boundaryConditions object can be used here to
      * set initial boundary conditions.
      *
      * The second paramater allows you to do more than one motor loop,
      * if non zero it means that the motor can compute all the iterations
      * up to time=forcedTime without producing any loads or updating any positions,
      * making the motor faster.
      *
      * @param dt the delta-t to use in the motor
      * @param forcedTime the time at which the user wants a result
      * @return the list of loads produce by the motor
      */
    virtual Loads * doMove(double dt, double forcedTime=0.0) = 0;

    /// get the current motor internal time
    virtual double getTime() const = 0;

    /// get a simple description (can be rich text)
    virtual QString getDescription() = 0;

    /// set the initial loads (for boundary conditions)
    virtual void setBoundaryConditions(Loads *);

protected:
    // the load manager
    LoadsManager * myLM;

    // the boundary conditions
    Loads * boundaryConditions;
};


#endif
