/*****************************************************************************
 * $CAMITK_LICENCE_BEGIN$
 *
 * CamiTK - Computer Assisted Medical Intervention ToolKit
 * (c) 2001-2014 UJF-Grenoble 1, CNRS, TIMC-IMAG UMR 5525 (GMCAO)
 *
 * Visit http://camitk.imag.fr for more information
 *
 * This file is part of CamiTK.
 *
 * CamiTK is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License version 3
 * only, as published by the Free Software Foundation.
 *
 * CamiTK is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License version 3 for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * version 3 along with CamiTK.  If not, see <http://www.gnu.org/licenses/>.
 *
 * $CAMITK_LICENCE_END$
 ****************************************************************************/

// Local
#include "MoveFrame.h"

// CamiTK
#include <InteractiveViewer.h>
#include <Log.h>
#include <Application.h>
#include <Component.h>
#include <SingleImageComponent.h>
using namespace camitk;

// Qt
#include <QFileDialog>
#include <QTextStream>

// QT DOM
#include <QDomDocument>
#include <QDomElement>
#include <QDomText>

// vtk
#include <vtkUnstructuredGrid.h>
#include <vtkPolyData.h>
#include <vtkPointData.h>
#include <vtkCellData.h>
#include <vtkAlgorithmOutput.h>


// --------------- constructor -------------------
MoveFrame::MoveFrame(ActionExtension* extension) : Action(extension) {
    setName("Move Frame");
    setDescription(tr("This action allow to move a frame from its parent frame (or the world frame if it has no parent) by setting translations and rotation parameters."));
    setComponent("Component");

    // Setting classification family and tags
    setFamily("Frame");
    addTag(tr("Test"));
    addTag(tr("Move"));
    addTag(tr("Visualization"));

    dialog = NULL;
    inputFrame = NULL;
}


// --------------- init -------------------
void MoveFrame::init() {
    dialog = new QDialog();

    transformation = vtkTransform::New();

    //-- init user interface
    ui.setupUi(dialog);

    // initialize slider names
    ui.tX->setName("X");
    ui.tY->setName("Y");
    ui.tZ->setName("Z");
    ui.rX->setName(tr("Around X"));
    ui.rY->setName(tr("Around Y"));
    ui.rZ->setName(tr("Around Z"));

    // connect everything
    connect(ui.translatePushButton, SIGNAL(clicked()), SLOT(translate()));
    connect(ui.setTranslationPushButton, SIGNAL(clicked()), SLOT(setTranslation()));
    connect(ui.rotatePushButton, SIGNAL(clicked()), SLOT(rotate()));
    connect(ui.setRotationPushButton, SIGNAL(clicked()), SLOT(setRotation()));
    connect(ui.resetButton, SIGNAL(clicked()), SLOT(reset()));
    connect(ui.setTransformButton, SIGNAL(clicked()), SLOT(apply()));
    connect(ui.savePushButton, SIGNAL(clicked()), SLOT(save()));
    connect(ui.parentFramePushButton, SIGNAL(clicked()), SLOT(changeParent()));

}


// --------------- destructor -------------------
MoveFrame::~MoveFrame() {
    if (dialog) {
        delete dialog;
    }
}

// --------------- getWidget -------------------
QWidget * MoveFrame::getWidget() {
    if (!dialog)
        init();

    inputFrame = dynamic_cast<Component *> (getTargets().last());
    vtkSmartPointer<vtkTransform> initialTransform = inputFrame->getTransform();

    initialPosition.setX(initialTransform->GetPosition()[0]);
    initialPosition.setY(initialTransform->GetPosition()[1]);
    initialPosition.setZ(initialTransform->GetPosition()[2]);

    initialOrientation.setX(initialTransform->GetOrientation()[0]);
    initialOrientation.setY(initialTransform->GetOrientation()[1]);
    initialOrientation.setZ(initialTransform->GetOrientation()[2]);

    // disconnect and reset previous state
    reset();

    return dialog;
}

void MoveFrame::changeParent() {
    if (inputFrame != NULL) {
        QString parentName = ui.parentFrameComboBox->currentText();
        // Find the corresponding component:
        const ComponentList existingComponents = Application::getAllComponents();
        Component * newParent = NULL;
        int i = 0;
        while ((i < existingComponents.size()) && (parentName != existingComponents.at(i)->getName())) {
            i++;
        }
        if (i < existingComponents.size()) {
            newParent = existingComponents.at(i);
        }

        inputFrame->setParentFrame(newParent);
    }
    InteractiveViewer::get3DViewer()->refresh();
}

//--------------- reset -------------
void MoveFrame::reset() {
    //-- initialize the transformation to Identity
    transformation->Identity();
    transformation->RotateX(initialOrientation.x());
    transformation->RotateY(initialOrientation.y());
    transformation->RotateZ(initialOrientation.z());
    transformation->Translate(initialPosition.x(), initialPosition.y(), initialPosition.z());

    //-- init values
    double bounds[6];
    InteractiveViewer::get3DViewer()->getBounds(bounds);
    double xLength = bounds[1] - bounds[0];
    double yLength = bounds[3] - bounds[2];
    double zLength = bounds[5] - bounds[4];
    ui.tX->init(-xLength, + xLength, initialPosition.x());
    ui.tY->init(-yLength, + yLength, initialPosition.y());
    ui.tZ->init(-zLength, + zLength, initialPosition.z());
    ui.rX->init(-180.0, 180.0, initialOrientation.x());
    ui.rY->init(-180.0, 180.0, initialOrientation.y());
    ui.rZ->init(-180.0, 180.0, initialOrientation.z());

    //-- Fill in the Possible Parent Frame Components ComboBox
    ui.parentFrameComboBox->clear();
    QStringList possibleParentFrames;
    possibleParentFrames << "World Frame";
    const ComponentList existingComponents = Application::getAllComponents();
    for (int i = 0; i < existingComponents.size(); i++) {
        Component * c =  existingComponents.at(i);
        if( c ) {
            SingleImageComponent * csi = dynamic_cast<SingleImageComponent *>( c );
            if( csi ) {
                //do nothing
                //A singleImageComponent cannot be a parent frame, it is always a frame child of another frame.
            } else {
                possibleParentFrames << c->getName();
            }
        }
    }
    ui.parentFrameComboBox->addItems(possibleParentFrames);

    if (inputFrame != NULL) {
        Component * currentParentComponent = dynamic_cast<Component *> (inputFrame->getParentFrame());

        if (currentParentComponent == NULL) {
            ui.parentFrameComboBox->setCurrentIndex(0);
        } else {
            ui.parentFrameComboBox->setCurrentIndex(possibleParentFrames.indexOf(currentParentComponent->getName()));
        }
    }
}


//--------------- apply ------------
Action::ApplyStatus MoveFrame::apply() {
    inputFrame = dynamic_cast<Component *> (getTargets().last());
    transformation->Identity();

    //-- Set the Rotation
    transformation->RotateX(double(ui.rX->getValue()));
    transformation->RotateY(double(ui.rY->getValue()));
    transformation->RotateZ(double(ui.rZ->getValue()));

    //-- Set the Translation
    transformation->Translate(double(ui.tX->getValue()), double(ui.tY->getValue()), double(ui.tZ->getValue()));

    inputFrame->setTransform(transformation);

    // Refresh all the used viewers
//    InteractiveViewer::get3DViewer()->refresh();
    Application::refresh();

    return SUCCESS;
}


void MoveFrame::translate() {
    QVector3D trans;
    trans.setX(double (ui.tX->getValue()));
    trans.setY(double (ui.tY->getValue()));
    trans.setZ(double (ui.tZ->getValue()));

    inputFrame->translate(trans.x(), trans.y(), trans.z());
}

void MoveFrame::setTranslation() {
    QVector3D trans;
    trans.setX(double (ui.tX->getValue()));
    trans.setY(double (ui.tY->getValue()));
    trans.setZ(double (ui.tZ->getValue()));

    inputFrame->setTransformTranslation(trans.x(), trans.y(), trans.z());
}

void MoveFrame::rotate() {
    QVector3D rot;
    rot.setX(double (ui.rX->getValue()));
    rot.setY(double (ui.rY->getValue()));
    rot.setZ(double (ui.rZ->getValue()));

    inputFrame->rotate(rot.x(), rot.y(), rot.z());

}
void MoveFrame::setRotation() {
    QVector3D rot;
    rot.setX(double (ui.rX->getValue()));
    rot.setY(double (ui.rY->getValue()));
    rot.setZ(double (ui.rZ->getValue()));

    inputFrame->setTransformRotation(rot.x(), rot.y(), rot.z());
}


void MoveFrame::save() {
    transformation->Identity();

    //-- Set the Rotation
    transformation->RotateX(double(ui.rX->getValue()));
    transformation->RotateY(double(ui.rY->getValue()));
    transformation->RotateZ(double(ui.rZ->getValue()));

    //-- Set the Translation
    transformation->Translate(double(ui.tX->getValue()), double(ui.tY->getValue()), double(ui.tZ->getValue()));


    QString filename = QFileDialog::getSaveFileName(NULL, "Save file");
    QFile file(filename);
    if (file.open(QIODevice::WriteOnly)) {
        QTextStream stream(&file);
        QDomDocument doc("frame");
        QDomElement root = doc.createElement("frame");
        root.setAttribute("name", "created frame");

        QDomElement transform = doc.createElement("transform");
        transform.setAttribute("type", "matrix");
        vtkSmartPointer<vtkMatrix4x4> matrix = transformation->GetMatrix();
        for (int i = 0; i < 4; i++) {
            QDomElement line = doc.createElement("line");
            // x
            QDomElement xElmt = doc.createElement("x");
            double xVal = matrix->GetElement(i, 0);
            QDomText x = doc.createTextNode(QString::number(xVal));
            xElmt.appendChild(x);
            line.appendChild(xElmt);

            // y
            QDomElement yElmt = doc.createElement("y");
            double yVal = matrix->GetElement(i, 1);
            QDomText y = doc.createTextNode(QString::number(yVal));
            yElmt.appendChild(y);
            line.appendChild(yElmt);

            // z
            QDomElement zElmt = doc.createElement("z");
            double zVal = matrix->GetElement(i, 2);
            QDomText z = doc.createTextNode(QString::number(zVal));
            zElmt.appendChild(z);
            line.appendChild(zElmt);

            // t
            QDomElement tElmt = doc.createElement("t");
            double tVal = matrix->GetElement(i, 3);
            QDomText t = doc.createTextNode(QString::number(tVal));
            tElmt.appendChild(t);
            line.appendChild(tElmt);
            transform.appendChild(line);
        }

        root.appendChild(transform);
        doc.appendChild(root);
        doc.save(stream, 0);

        file.close();
    }

}
