// CamiTK includes
#include "ComputeCurvatures.h"
#include <Application.h>
#include <MeshComponent.h>
#include <Log.h>
#include <Property.h>

// Vtk includes
#include <vtkDataSetAttributes.h>
#include <vtkPointData.h>
#include <vtkCallbackCommand.h>

using namespace camitk;

ComputeCurvatures::ComputeCurvatures ( camitk::ActionExtension* ext ) : Action ( ext ) {
    this->setName ( "Compute Curvatures" );
    this->setDescription (QString ( tr("Computes curvatures of a surface.<br/>" )));
    this->setComponent ( "MeshComponent" );
    this->setFamily ( "Mesh Processing" );
    this->addTag ( tr("curvatures" ));

    Property* curvatureTypeProperty = new Property(tr("Curvature type"), GAUSSIAN, tr("The type of curvature to compute on the surfacic mesh."), "");
    curvatureTypeProperty->setEnumTypeName("CurvatureType");
    QStringList curvatureTypeNames;
    curvatureTypeNames << tr("Gaussian") << tr("Mean") << tr("Maximum") << tr("Minimum");
    curvatureTypeProperty->setAttribute("enumNames", curvatureTypeNames);
    addParameter(curvatureTypeProperty);
}

ComputeCurvatures::~ComputeCurvatures() {

}

Action::ApplyStatus ComputeCurvatures::apply() {
    // set waiting cursor and status bar
    QApplication::setOverrideCursor ( QCursor ( Qt::WaitCursor ) );
    Application::showStatusBarMessage (tr( "Compute Curvatures...") );
    Application::resetProgressBar();
    vtkSmartPointer<vtkCallbackCommand> progressCallback = vtkSmartPointer<vtkCallbackCommand>::New();
    progressCallback->SetCallback(&Application::vtkProgressFunction);

    // target must be a polydata
    MeshComponent *mesh = dynamic_cast<MeshComponent*> ( getTargets().first() );

    if (!mesh) {
        CAMITK_DEBUG ( "ComputeCurvatures", "apply", "mesh " +
                       getTargets().first()->getName().toStdString() + " is not a MeshComponent" )
        return ABORTED;
    }

    vtkPolyData* pData = vtkPolyData::SafeDownCast(mesh->getPointSet());
    pData->AddObserver(vtkCommand::ProgressEvent, progressCallback);
    pData->Update();

    if (!pData) {
        CAMITK_DEBUG ( "ComputeCurvatures", "apply", "the mesh point set of " +
                       getTargets().first()->getName().toStdString() + " is not a polydata" )
        return ABORTED;
    }

    CAMITK_INFO ( "ComputeCurvatures", "apply", "Compute curvatures for mesh " +
                  getTargets().first()->getName().toStdString() )

    // set up curvatures
    vtkSmartPointer<vtkCurvatures> curv = vtkSmartPointer<vtkCurvatures>::New();
    curv->SetCurvatureType ( property("Curvature type").toInt() );
    curv->SetInput(pData);
    curv->AddObserver(vtkCommand::ProgressEvent, progressCallback);
    curv->Update();

    switch(property("Curvature type").toInt()) {
    case MINIMUM :
        mesh->addPointData("Minimum Curvature", vtkDataSetAttributes::SCALARS, curv->GetOutput()->GetPointData()->GetScalars("Minimum_Curvature"));
        break;
    case MAXIMUM :
        mesh->addPointData("Maximum Curvature", vtkDataSetAttributes::SCALARS, curv->GetOutput()->GetPointData()->GetScalars("Maximum_Curvature"));
        break;
    case GAUSSIAN :
        mesh->addPointData("Gauss Curvature", vtkDataSetAttributes::SCALARS, curv->GetOutput()->GetPointData()->GetScalars("Gauss_Curvature"));
        break;
    case MEAN:
        mesh->addPointData("Mean Curvature", vtkDataSetAttributes::SCALARS, curv->GetOutput()->GetPointData()->GetScalars("Mean_Curvature"));
        break;
    default :
        return ABORTED;
        break;
    }

    // refresh restore the normal cursor and progress bar
    Application::refresh();
    Application::resetProgressBar();
    Application::showStatusBarMessage ( "" );
    QApplication::restoreOverrideCursor();
    return SUCCESS;
}
