/*****************************************************************************
 * $CAMITK_LICENCE_BEGIN$
 *
 * CamiTK - Computer Assisted Medical Intervention ToolKit
 * (c) 2001-2014 UJF-Grenoble 1, CNRS, TIMC-IMAG UMR 5525 (GMCAO)
 *
 * Visit http://camitk.imag.fr for more information
 *
 * This file is part of CamiTK.
 *
 * CamiTK is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License version 3
 * only, as published by the Free Software Foundation.
 *
 * CamiTK is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License version 3 for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * version 3 along with CamiTK.  If not, see <http://www.gnu.org/licenses/>.
 *
 * $CAMITK_LICENCE_END$
 ****************************************************************************/


#include "ActionWidget.h"
#include "Action.h"
#include "ObjectController.h"
#include "Component.h"
#include <Application.h>
#include <MainWindow.h>

namespace camitk {

// -------------------- constructor --------------------
ActionWidget::ActionWidget(Action *action): QFrame() {
    myAction = action;

    // create the property editor tab itself
    setObjectName(myAction->getName());

    QVBoxLayout * widgetLayout = new QVBoxLayout;

    // Action's name and icon
    QHBoxLayout *nameLayout = new QHBoxLayout;
    QLabel *iconPicture = new QLabel;
    iconPicture->setPixmap(myAction->getIcon());
    iconPicture->setSizePolicy(QSizePolicy::Fixed, QSizePolicy::Fixed);
    nameLayout->addWidget(iconPicture);
    actionNameLabel= new QLabel("<h3><b>" + myAction->getName() + "</b></h3>");
    actionNameLabel->setSizePolicy(QSizePolicy::Fixed, QSizePolicy::Fixed);
    nameLayout->addWidget(actionNameLabel);
    widgetLayout->addLayout(nameLayout);

    // get the proper text height
    actionNameLabel->adjustSize();
    // fix the icon height
    iconPicture->setFixedSize(actionNameLabel->height(), actionNameLabel->height());
    iconPicture->setScaledContents(true);

    // Action description presentation
    descriptionLabelScrollArea = new QScrollArea();
    descriptionLabelScrollArea->setWidgetResizable(true);
    descriptionLabelScrollArea->setSizePolicy(QSizePolicy::MinimumExpanding, QSizePolicy::Expanding);
    descriptionLabelScrollArea->setHorizontalScrollBarPolicy(Qt::ScrollBarAlwaysOff);
    descriptionLabelScrollArea->setVerticalScrollBarPolicy(Qt::ScrollBarAsNeeded);
    descriptionLabel = new QLabel("<b>Description :</b> <br />" + myAction->getDescription());
    descriptionLabel->setFrameStyle(QFrame::StyledPanel | QFrame::Sunken);
    descriptionLabel->setTextFormat(Qt::RichText);
    descriptionLabel->setLineWidth(3);
    descriptionLabel->setWordWrap(true);
    descriptionLabelScrollArea->setWidget(descriptionLabel);
    widgetLayout->addWidget(descriptionLabelScrollArea);

    // Targets list display
    targetLabelScrollArea = new QScrollArea();
    targetLabelScrollArea->setWidgetResizable(true);
    targetLabelScrollArea->setSizePolicy(QSizePolicy::MinimumExpanding, QSizePolicy::Minimum);
    targetLabelScrollArea->setHorizontalScrollBarPolicy(Qt::ScrollBarAlwaysOff);
    targetLabelScrollArea->setVerticalScrollBarPolicy(Qt::ScrollBarAsNeeded);
    targetLabel = new QLabel(getTargetLabel());
    targetLabel->setFrameStyle(QFrame::StyledPanel | QFrame::Sunken);
    targetLabel->setLineWidth(3);
    targetLabel->setWordWrap(true);
    targetLabelScrollArea->setWidget(targetLabel);
    widgetLayout->addWidget(targetLabelScrollArea);

    // first the property editor
    objectController = new ObjectController(this, ObjectController::BUTTON);
    objectController->setSizePolicy(QSizePolicy::MinimumExpanding, QSizePolicy::MinimumExpanding);
    widgetLayout->addWidget(objectController);

    // then the buttons
    buttonFrame = new QFrame();
    QHBoxLayout *buttonLayout = new QHBoxLayout;
//    buttonLayout->setSizeConstraint(QHBoxLayout::SetMaximumSize);

    QPushButton *applyButton = new QPushButton("Apply");
    applyButton->setSizePolicy(QSizePolicy::MinimumExpanding, QSizePolicy::Fixed);
    buttonLayout->addWidget(applyButton);

    QPushButton *revertButton = new QPushButton("Revert");
    revertButton->setSizePolicy(QSizePolicy::MinimumExpanding, QSizePolicy::Fixed);
    buttonLayout->addWidget(revertButton);

    buttonFrame->setLayout(buttonLayout);
    widgetLayout->addWidget(buttonFrame);

    // connect the buttons
    QObject::connect(applyButton, SIGNAL(clicked()), objectController, SLOT(apply()));
    // from http://doc.trolltech.com/4.7/signalsandslots.html
    // "If several slots are connected to one signal, the slots will be executed one after the other,
    // in the order they have been connected, when the signal is emitted."
    // it means that here when the user click on apply, it will first update the properties
    // and then call the apply() method
    QObject::connect(applyButton, SIGNAL(clicked()), myAction, SLOT(applyAndRegister()));
    QObject::connect(revertButton, SIGNAL(clicked()), objectController, SLOT(revert()));

    // Now tell the ObjectController that this Action itself is the one to manage
    objectController->setObject(myAction);
    objectController->setMaximumHeight(objectController->sizeHint().height());

    setLayout(widgetLayout);
}

// -------------------- destructor --------------------
ActionWidget::~ActionWidget() {
    // TODO delete
}

// -------------------- setButtonVisibility --------------------
void ActionWidget::setButtonVisibility(bool visible) {
    buttonFrame->setVisible(visible);
}

// -------------------- setNameVisibility --------------------
void ActionWidget::setNameVisibility(bool visible) {
    this->actionNameLabel->setVisible(visible);
}

// -------------------- setDescriptionVisibility --------------------
void ActionWidget::setDescriptionVisibility(bool visible) {
    this->descriptionLabel->setVisible(visible);
}

// -------------------- update --------------------
void ActionWidget::updateTargets() {
    // force update
    objectController->setObject(NULL);
    objectController->setObject(myAction);
    targetLabel->setText(getTargetLabel());
}

// -------------------- getTargetLabel --------------------
QString ActionWidget::getTargetLabel() {
    QString targetNames = tr("<b> Target(s) to apply the action on: </b> <ul>");
    foreach (Component * comp, myAction->getTargets()) {
        targetNames += "<li> " + comp->getName() + "<i> (" + comp->metaObject()->className() + ") </i>" + "</li>";
    }
    return targetNames;
}

// -------------------- setAutoUpdateProperty --------------------
void ActionWidget::setAutoUpdateProperty(bool autoUpdate) {
    objectController->setAutoUpdateProperty(autoUpdate);
}

}
