# 2 "input.mll"
 
(*
  Copyright 2002-2004 Sebastien Ailleret
  Copyright 2004-2006 Martin Jambon

  This file is distributed under the terms of the GNU Public License
  http://www.gnu.org/licenses/gpl.txt
*)

open Printf
open Lexing

type token = [ `Comment of string
             | `Special_comment of string * string
	     | `Construct of string
	     | `Keyword of string
	     | `Newline
	     | `String of string
	     | `Quotation of string
	     | `Tab
	     | `Token of string
	     | `Start_annot of (Annot.layer_info * string)
	     | `Stop_annot of Annot.layer_info ]

type state = { mutable depth : int;
	       buf : Buffer.t;
	       lexbuf : lexbuf;
	       mutable tokens : token list;
	       mutable annot_tags : Annot.tag list;
	       mutable in_group : bool }

let init_state annot_tags lexbuf = { depth = 0;
				     buf = Buffer.create 1000;
				     lexbuf = lexbuf;
				     tokens = [];
				     annot_tags = annot_tags;
				     in_group = false }

let stringpair_of_token = function
    `Comment s -> "Comment", s
  | `Construct s -> "Construct", s
  | `Keyword s -> "Keyword", s
  | `Newline -> "Newline", ""
  | `String s -> "String", s
  | `Quotation s -> "Quotation", s
  | `Tab -> "Tab", ""
  | `Token s -> "Token", s
  | `Start_annot (_info, s) -> "Start_annot", s
  | `Stop_annot _info -> "Stop_annot", ""

let string_of_token x =
  match stringpair_of_token x with
      a, "" -> a
    | a, b -> sprintf "%s %S" a b

let print_tokens l =
  List.iter (fun s -> 
	       printf "%s\n" (string_of_token s))
    l

let keywords = [
  "and";
  "as";
  "asr";
  "assert";
  "begin";
  "class";
  "constraint";
  "do";
  "done";
  "downto";
  "else";
  "end";
  "exception";
  "external";
  "false";
  "for";
  "fun";
  "function";
  "functor";
  "if";
  "in";
  "include";
  "inherit";
  "initializer";
  "land";
  "lazy";
  "let";
  "lor";
  "lsl";
  "lsr";
  "lxor";
  "match";
  "method";
  "mod";
  "module";
  "mutable";
  "new";
  "object";
  "of";
  "open";
  "or";
  "private";
  "rec";
  "sig";
  "struct";
  "then";
  "to";
  "true";
  "try";
  "type";
  "val";
  "virtual";
  "when";
  "while";
  "with" ]

let is_keyword =
  let tbl = Hashtbl.create 100 in
  List.iter (fun key -> Hashtbl.add tbl key ()) keywords;
  Hashtbl.mem tbl

let tokenify s =
  if is_keyword s then `Keyword s
  else `Token s

let init_lexbuf lexbuf filename =
  let pos = lexbuf.lex_curr_p in
  lexbuf.lex_curr_p <- { pos with pos_fname = filename }


let compare_pos a b =
  let c = compare a.pos_lnum b.pos_lnum in
  if c <> 0 then c
  else compare (a.pos_cnum - a.pos_bol) (b.pos_cnum - b.pos_bol)

(* Consume the list of annotations *)
let get_annot state cur_pos =
  let rec loop () =
    match state.annot_tags with
	[] -> []
      | ((_, (tag_pos, _)) as tag) :: tl ->
	  if compare_pos tag_pos cur_pos <= 0 then
	    (state.annot_tags <- tl;
	     tag :: loop ())
	  else [] in
  loop ()

let simple_annot x =
  match x with
      (`Start typ, (_, info)) -> `Start_annot (info, typ)
    | (`Stop, (_, info)) -> `Stop_annot info

let simple_annots = List.map simple_annot

(* Add all unclosed tags that may remain *)
let finish_annot state =
  state.tokens <- 
    (List.rev_map simple_annot state.annot_tags) 
    @ state.tokens;
  state.annot_tags <- []

let newline state =
  let lexbuf = state.lexbuf in
  let pos = lexbuf.lex_curr_p in
  lexbuf.lex_curr_p <- { pos with
			   pos_lnum = pos.pos_lnum + 1;
			   pos_bol = pos.pos_cnum }

let shift x pos =
  { pos with pos_cnum = pos.pos_cnum + x }

let begin_group state =
  state.in_group <- true

let end_group state =
  state.in_group <- false

let add_token ?(offset = 0) state x =
  if x = `Newline then
    newline state;
  let annot1, annot2 =
    if not state.in_group then
      let annot1 = 
	get_annot state (shift offset (lexeme_start_p state.lexbuf)) in
      let annot2 = 
	get_annot state (shift offset (lexeme_end_p state.lexbuf)) in
      annot1, annot2
    else [], [] in
  state.tokens <- (List.rev_append (simple_annots annot2)
		     (x :: (List.rev_append 
			      (simple_annots annot1) state.tokens)))

let return_tokens state =
  let l = List.rev state.tokens in
  let tagged =
    List.map (function
		  `Start_annot _ as x -> (Tag.Start, x)
		| `Stop_annot _ as x -> (Tag.Stop, x)
		| x -> (Tag.Other, x))
      l in
  let annotate b x =
    if not b then x
    else
      match x with
	  `Start_annot (info, typ) -> 
	    `Start_annot ({ info with Annot.innermost = true }, typ)
	| `Stop_annot info ->
	    `Stop_annot { info with Annot.innermost = true }
	| _ -> assert false in
  let l = Tag.annotate_innermost annotate (Tag.remove_matches tagged) in
  let result = List.map snd l in
  result

# 217 "input.ml"
let __ocaml_lex_tables = {
  Lexing.lex_base = 
   "\000\000\241\255\081\000\211\000\060\000\243\255\244\255\245\255\
    \001\000\003\001\022\001\060\001\250\255\095\001\068\000\091\000\
    \132\001\167\001\204\001\080\000\082\000\095\000\007\002\042\002\
    \077\002\107\002\059\003\011\004\252\255\234\004\254\255\100\000\
    \204\004\040\005\248\005\215\006\051\007\051\007\003\008\253\255\
    \226\008\113\000\088\000\108\000\130\008\138\000\110\000\248\255\
    \246\255\113\000\191\002\115\000\145\000\117\000\247\255\118\000\
    \221\001\134\003\219\002\086\004\110\000\233\000\096\004\067\006\
    \234\001\222\000\234\004\042\002\039\001\249\255\250\255\251\255\
    \002\000\252\255\253\255\119\000\119\000\255\255\254\255\081\001\
    \250\255\251\255\252\255\004\000\253\255\155\000\255\255\254\255\
    \153\001\251\255\252\255\253\255\005\000\100\000\111\000\255\255\
    \112\000\254\255";
  Lexing.lex_backtrk = 
   "\255\255\255\255\014\000\014\000\013\000\255\255\255\255\255\255\
    \014\000\014\000\006\000\005\000\255\255\005\000\005\000\005\000\
    \005\000\005\000\005\000\005\000\014\000\005\000\005\000\005\000\
    \006\000\005\000\004\000\003\000\255\255\005\000\255\255\005\000\
    \000\000\255\255\000\000\006\000\002\000\255\255\255\255\255\255\
    \006\000\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\014\000\255\255\255\255\255\255\014\000\
    \014\000\014\000\014\000\255\255\255\255\255\255\255\255\255\255\
    \006\000\255\255\255\255\006\000\006\000\255\255\255\255\255\255\
    \255\255\255\255\255\255\005\000\255\255\005\000\255\255\255\255\
    \255\255\255\255\255\255\255\255\004\000\004\000\004\000\255\255\
    \255\255\255\255";
  Lexing.lex_default = 
   "\001\000\000\000\255\255\255\255\255\255\000\000\000\000\000\000\
    \255\255\043\000\255\255\255\255\000\000\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\000\000\255\255\000\000\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\000\000\
    \255\255\255\255\255\255\255\255\043\000\255\255\255\255\000\000\
    \000\000\255\255\255\255\255\255\255\255\255\255\000\000\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\069\000\000\000\000\000\000\000\
    \255\255\000\000\000\000\255\255\255\255\000\000\000\000\080\000\
    \000\000\000\000\000\000\255\255\000\000\255\255\000\000\000\000\
    \089\000\000\000\000\000\000\000\255\255\255\255\255\255\000\000\
    \255\255\000\000";
  Lexing.lex_trans = 
   "\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\006\000\007\000\007\000\071\000\008\000\082\000\091\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \004\000\024\000\030\000\012\000\010\000\010\000\023\000\009\000\
    \031\000\012\000\018\000\018\000\012\000\022\000\021\000\010\000\
    \003\000\002\000\002\000\002\000\002\000\002\000\002\000\002\000\
    \002\000\002\000\020\000\019\000\029\000\018\000\017\000\016\000\
    \010\000\027\000\027\000\027\000\027\000\027\000\027\000\027\000\
    \027\000\027\000\027\000\027\000\027\000\027\000\027\000\027\000\
    \027\000\027\000\027\000\027\000\027\000\027\000\027\000\027\000\
    \027\000\027\000\027\000\015\000\004\000\012\000\010\000\025\000\
    \028\000\026\000\026\000\026\000\026\000\026\000\026\000\026\000\
    \026\000\026\000\026\000\026\000\026\000\026\000\026\000\026\000\
    \026\000\026\000\026\000\026\000\026\000\026\000\026\000\026\000\
    \026\000\026\000\026\000\014\000\013\000\012\000\011\000\059\000\
    \012\000\002\000\002\000\002\000\002\000\002\000\002\000\002\000\
    \002\000\002\000\002\000\012\000\012\000\041\000\032\000\012\000\
    \012\000\042\000\012\000\048\000\049\000\047\000\058\000\012\000\
    \047\000\012\000\048\000\055\000\054\000\054\000\065\000\065\000\
    \078\000\077\000\096\000\056\000\056\000\056\000\056\000\056\000\
    \056\000\056\000\056\000\056\000\056\000\095\000\097\000\000\000\
    \002\000\048\000\000\000\000\000\000\000\000\000\058\000\000\000\
    \048\000\000\000\000\000\000\000\000\000\087\000\000\000\000\000\
    \027\000\027\000\027\000\027\000\027\000\027\000\027\000\027\000\
    \027\000\027\000\027\000\027\000\027\000\027\000\027\000\027\000\
    \027\000\027\000\027\000\027\000\027\000\027\000\027\000\012\000\
    \027\000\027\000\027\000\027\000\027\000\027\000\027\000\026\000\
    \026\000\026\000\026\000\026\000\026\000\026\000\026\000\026\000\
    \026\000\026\000\026\000\026\000\026\000\026\000\026\000\026\000\
    \026\000\026\000\026\000\026\000\026\000\026\000\026\000\087\000\
    \026\000\026\000\026\000\026\000\026\000\026\000\026\000\026\000\
    \005\000\059\000\000\000\002\000\002\000\002\000\002\000\002\000\
    \002\000\002\000\002\000\002\000\002\000\046\000\065\000\065\000\
    \045\000\000\000\000\000\000\000\000\000\060\000\000\000\000\000\
    \058\000\064\000\064\000\064\000\064\000\064\000\064\000\064\000\
    \064\000\000\000\061\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\255\255\062\000\000\000\000\000\000\000\000\000\
    \070\000\071\000\002\000\000\000\072\000\060\000\000\000\010\000\
    \058\000\000\000\010\000\010\000\010\000\065\000\000\000\000\000\
    \010\000\010\000\061\000\010\000\010\000\010\000\000\000\000\000\
    \000\000\074\000\000\000\062\000\000\000\000\000\000\000\076\000\
    \010\000\075\000\010\000\010\000\010\000\010\000\010\000\000\000\
    \000\000\000\000\081\000\082\000\000\000\040\000\083\000\044\000\
    \040\000\040\000\040\000\000\000\000\000\000\000\040\000\040\000\
    \000\000\040\000\040\000\040\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\086\000\010\000\000\000\040\000\000\000\
    \040\000\040\000\040\000\040\000\040\000\000\000\000\000\000\000\
    \010\000\000\000\000\000\010\000\010\000\010\000\000\000\000\000\
    \000\000\010\000\010\000\000\000\010\000\010\000\010\000\000\000\
    \000\000\000\000\010\000\000\000\010\000\000\000\000\000\000\000\
    \000\000\010\000\040\000\010\000\010\000\010\000\010\000\010\000\
    \000\000\000\000\090\000\091\000\000\000\040\000\092\000\000\000\
    \040\000\040\000\040\000\000\000\000\000\085\000\040\000\040\000\
    \000\000\040\000\040\000\040\000\000\000\000\000\000\000\000\000\
    \040\000\000\000\040\000\000\000\012\000\010\000\040\000\000\000\
    \040\000\040\000\040\000\011\000\040\000\000\000\000\000\000\000\
    \010\000\000\000\000\000\010\000\010\000\010\000\000\000\000\000\
    \000\000\010\000\010\000\000\000\010\000\010\000\010\000\094\000\
    \000\000\000\000\000\000\010\000\000\000\010\000\000\000\000\000\
    \000\000\010\000\040\000\010\000\010\000\010\000\010\000\010\000\
    \000\000\000\000\000\000\000\000\000\000\010\000\000\000\000\000\
    \010\000\010\000\010\000\000\000\000\000\093\000\010\000\010\000\
    \000\000\010\000\010\000\010\000\000\000\000\000\000\000\000\000\
    \040\000\000\000\040\000\255\255\012\000\010\000\010\000\000\000\
    \010\000\010\000\010\000\010\000\010\000\043\000\043\000\043\000\
    \043\000\043\000\043\000\043\000\043\000\043\000\043\000\000\000\
    \000\000\000\000\064\000\064\000\064\000\064\000\064\000\064\000\
    \064\000\064\000\000\000\010\000\012\000\010\000\000\000\073\000\
    \010\000\000\000\010\000\010\000\010\000\010\000\000\000\000\000\
    \000\000\010\000\010\000\000\000\010\000\018\000\010\000\003\000\
    \002\000\002\000\002\000\002\000\002\000\002\000\002\000\002\000\
    \002\000\010\000\000\000\010\000\010\000\018\000\010\000\010\000\
    \010\000\064\000\010\000\010\000\000\000\000\000\010\000\010\000\
    \018\000\084\000\000\000\000\000\010\000\010\000\000\000\010\000\
    \010\000\010\000\066\000\066\000\066\000\066\000\066\000\066\000\
    \066\000\066\000\066\000\066\000\010\000\010\000\010\000\010\000\
    \010\000\010\000\010\000\000\000\000\000\000\000\040\000\000\000\
    \000\000\040\000\040\000\040\000\000\000\000\000\000\000\040\000\
    \040\000\000\000\040\000\040\000\040\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\010\000\000\000\010\000\000\000\040\000\
    \010\000\040\000\011\000\040\000\040\000\040\000\000\000\000\000\
    \000\000\000\000\026\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\255\255\000\000\026\000\026\000\026\000\026\000\026\000\
    \026\000\026\000\026\000\026\000\026\000\000\000\010\000\000\000\
    \010\000\000\000\000\000\040\000\026\000\026\000\026\000\026\000\
    \026\000\026\000\026\000\026\000\026\000\026\000\026\000\026\000\
    \026\000\026\000\026\000\026\000\026\000\026\000\026\000\026\000\
    \026\000\026\000\026\000\026\000\026\000\026\000\000\000\000\000\
    \000\000\040\000\026\000\040\000\026\000\026\000\026\000\026\000\
    \026\000\026\000\026\000\026\000\026\000\026\000\026\000\026\000\
    \026\000\026\000\026\000\026\000\026\000\026\000\026\000\026\000\
    \026\000\026\000\026\000\026\000\026\000\026\000\048\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\057\000\
    \057\000\057\000\057\000\057\000\057\000\057\000\057\000\057\000\
    \057\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \057\000\057\000\057\000\057\000\057\000\057\000\067\000\000\000\
    \067\000\000\000\000\000\066\000\066\000\066\000\066\000\066\000\
    \066\000\066\000\066\000\066\000\066\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \057\000\057\000\057\000\057\000\057\000\057\000\000\000\000\000\
    \000\000\000\000\000\000\026\000\026\000\026\000\026\000\026\000\
    \026\000\026\000\026\000\026\000\026\000\026\000\026\000\026\000\
    \026\000\026\000\026\000\026\000\026\000\026\000\026\000\026\000\
    \026\000\026\000\000\000\026\000\026\000\026\000\026\000\026\000\
    \026\000\026\000\026\000\026\000\026\000\026\000\026\000\026\000\
    \026\000\026\000\026\000\026\000\026\000\026\000\026\000\026\000\
    \026\000\026\000\026\000\026\000\026\000\026\000\026\000\026\000\
    \026\000\026\000\026\000\026\000\026\000\026\000\026\000\026\000\
    \026\000\026\000\026\000\026\000\026\000\026\000\026\000\026\000\
    \026\000\026\000\026\000\026\000\026\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\026\000\026\000\026\000\026\000\
    \026\000\026\000\026\000\026\000\026\000\026\000\026\000\026\000\
    \026\000\026\000\026\000\026\000\026\000\026\000\026\000\026\000\
    \026\000\026\000\026\000\026\000\026\000\026\000\000\000\000\000\
    \000\000\000\000\026\000\000\000\026\000\026\000\026\000\026\000\
    \026\000\026\000\026\000\026\000\026\000\026\000\026\000\026\000\
    \026\000\026\000\026\000\026\000\026\000\026\000\026\000\026\000\
    \026\000\026\000\026\000\026\000\026\000\026\000\043\000\043\000\
    \043\000\043\000\043\000\043\000\043\000\043\000\043\000\043\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\043\000\
    \043\000\043\000\043\000\043\000\043\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\043\000\
    \043\000\043\000\043\000\043\000\043\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\026\000\026\000\026\000\026\000\026\000\
    \026\000\026\000\026\000\026\000\026\000\026\000\026\000\026\000\
    \026\000\026\000\026\000\026\000\026\000\026\000\026\000\026\000\
    \026\000\026\000\000\000\026\000\026\000\026\000\026\000\026\000\
    \026\000\026\000\026\000\026\000\026\000\026\000\026\000\026\000\
    \026\000\026\000\026\000\026\000\026\000\026\000\026\000\026\000\
    \026\000\026\000\026\000\026\000\026\000\026\000\026\000\026\000\
    \026\000\026\000\027\000\026\000\026\000\026\000\026\000\026\000\
    \026\000\026\000\026\000\027\000\027\000\027\000\027\000\027\000\
    \027\000\027\000\027\000\027\000\027\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\027\000\027\000\027\000\027\000\
    \027\000\027\000\027\000\027\000\027\000\027\000\027\000\027\000\
    \027\000\027\000\027\000\027\000\027\000\027\000\027\000\027\000\
    \027\000\027\000\027\000\027\000\027\000\027\000\000\000\000\000\
    \000\000\000\000\027\000\000\000\027\000\027\000\027\000\027\000\
    \027\000\027\000\027\000\027\000\027\000\027\000\027\000\027\000\
    \027\000\027\000\027\000\027\000\027\000\027\000\027\000\027\000\
    \027\000\027\000\027\000\027\000\027\000\027\000\059\000\059\000\
    \059\000\059\000\059\000\059\000\059\000\059\000\059\000\059\000\
    \063\000\063\000\063\000\063\000\063\000\063\000\063\000\063\000\
    \063\000\063\000\000\000\058\000\000\000\000\000\000\000\000\000\
    \000\000\063\000\063\000\063\000\063\000\063\000\063\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\059\000\000\000\000\000\
    \000\000\000\000\000\000\058\000\000\000\000\000\000\000\000\000\
    \000\000\063\000\063\000\063\000\063\000\063\000\063\000\000\000\
    \000\000\000\000\000\000\027\000\027\000\027\000\027\000\027\000\
    \027\000\027\000\027\000\027\000\027\000\027\000\027\000\027\000\
    \027\000\027\000\027\000\027\000\027\000\027\000\027\000\027\000\
    \027\000\027\000\000\000\027\000\027\000\027\000\027\000\027\000\
    \027\000\027\000\027\000\027\000\027\000\027\000\027\000\027\000\
    \027\000\027\000\027\000\027\000\027\000\027\000\027\000\027\000\
    \027\000\027\000\027\000\027\000\027\000\027\000\027\000\027\000\
    \027\000\027\000\000\000\027\000\027\000\027\000\027\000\027\000\
    \027\000\027\000\027\000\010\000\000\000\000\000\010\000\010\000\
    \010\000\000\000\000\000\000\000\010\000\010\000\000\000\018\000\
    \010\000\010\000\066\000\066\000\066\000\066\000\066\000\066\000\
    \066\000\066\000\066\000\066\000\035\000\000\000\036\000\010\000\
    \010\000\010\000\010\000\033\000\000\000\034\000\034\000\034\000\
    \034\000\034\000\034\000\034\000\034\000\034\000\034\000\034\000\
    \034\000\034\000\034\000\034\000\034\000\034\000\034\000\034\000\
    \034\000\034\000\034\000\034\000\034\000\034\000\034\000\000\000\
    \010\000\066\000\000\000\000\000\000\000\000\000\000\000\034\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \034\000\034\000\034\000\034\000\034\000\034\000\034\000\034\000\
    \034\000\034\000\000\000\000\000\000\000\000\000\010\000\000\000\
    \010\000\034\000\034\000\034\000\034\000\034\000\034\000\034\000\
    \034\000\034\000\034\000\034\000\034\000\034\000\034\000\034\000\
    \034\000\034\000\034\000\034\000\034\000\034\000\034\000\034\000\
    \034\000\034\000\034\000\000\000\000\000\000\000\000\000\034\000\
    \000\000\034\000\034\000\034\000\034\000\034\000\034\000\034\000\
    \034\000\034\000\034\000\034\000\034\000\034\000\034\000\034\000\
    \034\000\034\000\034\000\034\000\034\000\034\000\034\000\034\000\
    \034\000\034\000\034\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\034\000\034\000\034\000\034\000\034\000\
    \034\000\034\000\034\000\034\000\034\000\034\000\034\000\034\000\
    \034\000\034\000\034\000\034\000\034\000\034\000\034\000\034\000\
    \034\000\034\000\034\000\000\000\034\000\034\000\034\000\034\000\
    \034\000\034\000\034\000\034\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \034\000\034\000\034\000\034\000\034\000\034\000\034\000\034\000\
    \034\000\034\000\034\000\034\000\034\000\034\000\034\000\034\000\
    \034\000\034\000\034\000\034\000\034\000\034\000\034\000\000\000\
    \034\000\034\000\034\000\034\000\034\000\034\000\034\000\034\000\
    \034\000\034\000\034\000\034\000\034\000\034\000\034\000\034\000\
    \034\000\034\000\034\000\034\000\034\000\034\000\034\000\034\000\
    \034\000\034\000\034\000\034\000\034\000\034\000\034\000\034\000\
    \034\000\034\000\034\000\034\000\034\000\034\000\034\000\034\000\
    \034\000\034\000\034\000\034\000\034\000\034\000\034\000\034\000\
    \034\000\034\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\034\000\034\000\034\000\034\000\034\000\034\000\034\000\
    \034\000\034\000\034\000\034\000\034\000\034\000\034\000\034\000\
    \034\000\034\000\034\000\034\000\034\000\034\000\034\000\034\000\
    \034\000\034\000\034\000\000\000\000\000\000\000\000\000\034\000\
    \000\000\034\000\034\000\034\000\034\000\034\000\034\000\034\000\
    \034\000\034\000\034\000\034\000\034\000\034\000\034\000\034\000\
    \034\000\034\000\034\000\034\000\034\000\034\000\034\000\034\000\
    \034\000\034\000\034\000\063\000\063\000\063\000\063\000\063\000\
    \063\000\063\000\063\000\063\000\063\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\063\000\063\000\063\000\063\000\
    \063\000\063\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\063\000\000\000\063\000\063\000\063\000\063\000\
    \063\000\063\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \034\000\034\000\034\000\034\000\034\000\034\000\034\000\034\000\
    \034\000\034\000\034\000\034\000\034\000\034\000\034\000\034\000\
    \034\000\034\000\034\000\034\000\034\000\034\000\034\000\000\000\
    \034\000\034\000\034\000\034\000\034\000\034\000\034\000\034\000\
    \034\000\034\000\034\000\034\000\034\000\034\000\034\000\034\000\
    \034\000\034\000\034\000\034\000\034\000\034\000\034\000\034\000\
    \034\000\034\000\034\000\034\000\034\000\034\000\034\000\000\000\
    \034\000\034\000\034\000\034\000\034\000\034\000\034\000\034\000\
    \010\000\000\000\000\000\010\000\010\000\010\000\000\000\000\000\
    \000\000\010\000\010\000\000\000\010\000\010\000\010\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\010\000\000\000\010\000\010\000\010\000\010\000\010\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\010\000\037\000\000\000\
    \038\000\038\000\038\000\038\000\038\000\038\000\038\000\038\000\
    \038\000\038\000\038\000\038\000\038\000\038\000\038\000\038\000\
    \038\000\038\000\038\000\038\000\038\000\038\000\038\000\038\000\
    \038\000\038\000\000\000\010\000\010\000\010\000\000\000\010\000\
    \010\000\010\000\038\000\000\000\000\000\010\000\010\000\000\000\
    \010\000\010\000\010\000\038\000\038\000\038\000\038\000\038\000\
    \038\000\038\000\038\000\038\000\038\000\010\000\000\000\010\000\
    \010\000\010\000\010\000\010\000\038\000\038\000\038\000\038\000\
    \038\000\038\000\038\000\038\000\038\000\038\000\038\000\038\000\
    \038\000\038\000\038\000\038\000\038\000\038\000\038\000\038\000\
    \038\000\038\000\038\000\038\000\038\000\038\000\000\000\000\000\
    \000\000\010\000\038\000\000\000\038\000\038\000\038\000\038\000\
    \038\000\038\000\038\000\038\000\038\000\038\000\038\000\038\000\
    \038\000\038\000\038\000\038\000\038\000\038\000\038\000\038\000\
    \038\000\038\000\038\000\038\000\038\000\038\000\000\000\010\000\
    \000\000\010\000\000\000\000\000\000\000\000\000\038\000\038\000\
    \038\000\038\000\038\000\038\000\038\000\038\000\038\000\038\000\
    \038\000\038\000\038\000\038\000\038\000\038\000\038\000\038\000\
    \038\000\038\000\038\000\038\000\038\000\038\000\000\000\038\000\
    \038\000\038\000\038\000\038\000\038\000\038\000\038\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\038\000\038\000\038\000\038\000\038\000\
    \038\000\038\000\038\000\038\000\038\000\038\000\038\000\038\000\
    \038\000\038\000\038\000\038\000\038\000\038\000\038\000\038\000\
    \038\000\038\000\000\000\038\000\038\000\038\000\038\000\038\000\
    \038\000\038\000\038\000\038\000\038\000\038\000\038\000\038\000\
    \038\000\038\000\038\000\038\000\038\000\038\000\038\000\038\000\
    \038\000\038\000\038\000\038\000\038\000\038\000\038\000\038\000\
    \038\000\038\000\038\000\038\000\038\000\038\000\038\000\038\000\
    \038\000\038\000\038\000\038\000\038\000\038\000\038\000\038\000\
    \038\000\038\000\038\000\038\000\038\000\000\000\000\000\039\000\
    \000\000\000\000\000\000\000\000\038\000\038\000\038\000\038\000\
    \038\000\038\000\038\000\038\000\038\000\038\000\038\000\038\000\
    \038\000\038\000\038\000\038\000\038\000\038\000\038\000\038\000\
    \038\000\038\000\038\000\038\000\038\000\038\000\000\000\000\000\
    \000\000\000\000\038\000\000\000\038\000\038\000\038\000\038\000\
    \038\000\038\000\038\000\038\000\038\000\038\000\038\000\038\000\
    \038\000\038\000\038\000\038\000\038\000\038\000\038\000\038\000\
    \038\000\038\000\038\000\038\000\038\000\038\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\053\000\000\000\000\000\052\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\051\000\051\000\051\000\051\000\051\000\051\000\
    \051\000\051\000\051\000\051\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\038\000\038\000\038\000\038\000\038\000\
    \038\000\038\000\038\000\038\000\038\000\038\000\038\000\038\000\
    \038\000\038\000\038\000\038\000\038\000\038\000\038\000\038\000\
    \038\000\038\000\000\000\038\000\038\000\038\000\038\000\038\000\
    \038\000\038\000\038\000\038\000\038\000\038\000\038\000\038\000\
    \038\000\038\000\038\000\038\000\038\000\038\000\038\000\038\000\
    \038\000\038\000\038\000\038\000\038\000\038\000\038\000\038\000\
    \038\000\038\000\050\000\038\000\038\000\038\000\038\000\038\000\
    \038\000\038\000\038\000\040\000\000\000\000\000\040\000\040\000\
    \040\000\000\000\000\000\000\000\040\000\040\000\000\000\040\000\
    \040\000\040\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\040\000\000\000\040\000\040\000\
    \040\000\040\000\040\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \040\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\040\000\000\000\
    \040\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\255\255\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000";
  Lexing.lex_check = 
   "\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\000\000\000\000\008\000\072\000\000\000\083\000\092\000\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\004\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\002\000\
    \014\000\002\000\002\000\002\000\002\000\002\000\002\000\002\000\
    \002\000\002\000\002\000\019\000\020\000\021\000\031\000\020\000\
    \020\000\041\000\042\000\043\000\045\000\046\000\002\000\015\000\
    \049\000\015\000\051\000\052\000\053\000\055\000\060\000\060\000\
    \075\000\076\000\093\000\051\000\051\000\051\000\051\000\051\000\
    \051\000\051\000\051\000\051\000\051\000\094\000\096\000\255\255\
    \002\000\045\000\255\255\255\255\255\255\255\255\002\000\255\255\
    \052\000\255\255\255\255\255\255\255\255\085\000\255\255\255\255\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\015\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\085\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\003\000\255\255\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\009\000\065\000\065\000\
    \009\000\255\255\255\255\255\255\255\255\003\000\255\255\255\255\
    \003\000\061\000\061\000\061\000\061\000\061\000\061\000\061\000\
    \061\000\255\255\003\000\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\009\000\003\000\255\255\255\255\255\255\255\255\
    \068\000\068\000\003\000\255\255\068\000\003\000\255\255\010\000\
    \003\000\255\255\010\000\010\000\010\000\065\000\255\255\255\255\
    \010\000\010\000\003\000\010\000\010\000\010\000\255\255\255\255\
    \255\255\068\000\255\255\003\000\255\255\255\255\255\255\068\000\
    \010\000\068\000\010\000\010\000\010\000\010\000\010\000\255\255\
    \255\255\255\255\079\000\079\000\255\255\011\000\079\000\009\000\
    \011\000\011\000\011\000\255\255\255\255\255\255\011\000\011\000\
    \255\255\011\000\011\000\011\000\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\079\000\010\000\255\255\011\000\255\255\
    \011\000\011\000\011\000\011\000\011\000\255\255\255\255\255\255\
    \013\000\255\255\255\255\013\000\013\000\013\000\255\255\255\255\
    \255\255\013\000\013\000\255\255\013\000\013\000\013\000\255\255\
    \255\255\255\255\010\000\255\255\010\000\255\255\255\255\255\255\
    \255\255\013\000\011\000\013\000\013\000\013\000\013\000\013\000\
    \255\255\255\255\088\000\088\000\255\255\016\000\088\000\255\255\
    \016\000\016\000\016\000\255\255\255\255\079\000\016\000\016\000\
    \255\255\016\000\016\000\016\000\255\255\255\255\255\255\255\255\
    \011\000\255\255\011\000\255\255\013\000\013\000\016\000\255\255\
    \016\000\016\000\016\000\016\000\016\000\255\255\255\255\255\255\
    \017\000\255\255\255\255\017\000\017\000\017\000\255\255\255\255\
    \255\255\017\000\017\000\255\255\017\000\017\000\017\000\088\000\
    \255\255\255\255\255\255\013\000\255\255\013\000\255\255\255\255\
    \255\255\017\000\016\000\017\000\017\000\017\000\017\000\017\000\
    \255\255\255\255\255\255\255\255\255\255\018\000\255\255\255\255\
    \018\000\018\000\018\000\255\255\255\255\088\000\018\000\018\000\
    \255\255\018\000\018\000\018\000\255\255\255\255\255\255\255\255\
    \016\000\255\255\016\000\009\000\017\000\017\000\018\000\255\255\
    \018\000\018\000\018\000\018\000\018\000\056\000\056\000\056\000\
    \056\000\056\000\056\000\056\000\056\000\056\000\056\000\255\255\
    \255\255\255\255\064\000\064\000\064\000\064\000\064\000\064\000\
    \064\000\064\000\255\255\017\000\017\000\017\000\255\255\068\000\
    \022\000\255\255\018\000\022\000\022\000\022\000\255\255\255\255\
    \255\255\022\000\022\000\255\255\022\000\022\000\022\000\022\000\
    \022\000\022\000\022\000\022\000\022\000\022\000\022\000\022\000\
    \022\000\022\000\255\255\022\000\022\000\022\000\022\000\022\000\
    \018\000\064\000\018\000\023\000\255\255\255\255\023\000\023\000\
    \023\000\079\000\255\255\255\255\023\000\023\000\255\255\023\000\
    \023\000\023\000\067\000\067\000\067\000\067\000\067\000\067\000\
    \067\000\067\000\067\000\067\000\023\000\022\000\023\000\023\000\
    \023\000\023\000\023\000\255\255\255\255\255\255\024\000\255\255\
    \255\255\024\000\024\000\024\000\255\255\255\255\255\255\024\000\
    \024\000\255\255\024\000\024\000\024\000\255\255\255\255\255\255\
    \255\255\255\255\255\255\022\000\255\255\022\000\255\255\024\000\
    \023\000\024\000\024\000\024\000\024\000\024\000\255\255\255\255\
    \255\255\255\255\025\000\255\255\255\255\255\255\255\255\255\255\
    \255\255\088\000\255\255\025\000\025\000\025\000\025\000\025\000\
    \025\000\025\000\025\000\025\000\025\000\255\255\023\000\255\255\
    \023\000\255\255\255\255\024\000\025\000\025\000\025\000\025\000\
    \025\000\025\000\025\000\025\000\025\000\025\000\025\000\025\000\
    \025\000\025\000\025\000\025\000\025\000\025\000\025\000\025\000\
    \025\000\025\000\025\000\025\000\025\000\025\000\255\255\255\255\
    \255\255\024\000\025\000\024\000\025\000\025\000\025\000\025\000\
    \025\000\025\000\025\000\025\000\025\000\025\000\025\000\025\000\
    \025\000\025\000\025\000\025\000\025\000\025\000\025\000\025\000\
    \025\000\025\000\025\000\025\000\025\000\025\000\050\000\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\050\000\
    \050\000\050\000\050\000\050\000\050\000\050\000\050\000\050\000\
    \050\000\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \050\000\050\000\050\000\050\000\050\000\050\000\058\000\255\255\
    \058\000\255\255\255\255\058\000\058\000\058\000\058\000\058\000\
    \058\000\058\000\058\000\058\000\058\000\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \050\000\050\000\050\000\050\000\050\000\050\000\255\255\255\255\
    \255\255\255\255\255\255\025\000\025\000\025\000\025\000\025\000\
    \025\000\025\000\025\000\025\000\025\000\025\000\025\000\025\000\
    \025\000\025\000\025\000\025\000\025\000\025\000\025\000\025\000\
    \025\000\025\000\255\255\025\000\025\000\025\000\025\000\025\000\
    \025\000\025\000\025\000\025\000\025\000\025\000\025\000\025\000\
    \025\000\025\000\025\000\025\000\025\000\025\000\025\000\025\000\
    \025\000\025\000\025\000\025\000\025\000\025\000\025\000\025\000\
    \025\000\025\000\026\000\025\000\025\000\025\000\025\000\025\000\
    \025\000\025\000\025\000\026\000\026\000\026\000\026\000\026\000\
    \026\000\026\000\026\000\026\000\026\000\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\026\000\026\000\026\000\026\000\
    \026\000\026\000\026\000\026\000\026\000\026\000\026\000\026\000\
    \026\000\026\000\026\000\026\000\026\000\026\000\026\000\026\000\
    \026\000\026\000\026\000\026\000\026\000\026\000\255\255\255\255\
    \255\255\255\255\026\000\255\255\026\000\026\000\026\000\026\000\
    \026\000\026\000\026\000\026\000\026\000\026\000\026\000\026\000\
    \026\000\026\000\026\000\026\000\026\000\026\000\026\000\026\000\
    \026\000\026\000\026\000\026\000\026\000\026\000\057\000\057\000\
    \057\000\057\000\057\000\057\000\057\000\057\000\057\000\057\000\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\057\000\
    \057\000\057\000\057\000\057\000\057\000\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\057\000\
    \057\000\057\000\057\000\057\000\057\000\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\026\000\026\000\026\000\026\000\026\000\
    \026\000\026\000\026\000\026\000\026\000\026\000\026\000\026\000\
    \026\000\026\000\026\000\026\000\026\000\026\000\026\000\026\000\
    \026\000\026\000\255\255\026\000\026\000\026\000\026\000\026\000\
    \026\000\026\000\026\000\026\000\026\000\026\000\026\000\026\000\
    \026\000\026\000\026\000\026\000\026\000\026\000\026\000\026\000\
    \026\000\026\000\026\000\026\000\026\000\026\000\026\000\026\000\
    \026\000\026\000\027\000\026\000\026\000\026\000\026\000\026\000\
    \026\000\026\000\026\000\027\000\027\000\027\000\027\000\027\000\
    \027\000\027\000\027\000\027\000\027\000\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\027\000\027\000\027\000\027\000\
    \027\000\027\000\027\000\027\000\027\000\027\000\027\000\027\000\
    \027\000\027\000\027\000\027\000\027\000\027\000\027\000\027\000\
    \027\000\027\000\027\000\027\000\027\000\027\000\255\255\255\255\
    \255\255\255\255\027\000\255\255\027\000\027\000\027\000\027\000\
    \027\000\027\000\027\000\027\000\027\000\027\000\027\000\027\000\
    \027\000\027\000\027\000\027\000\027\000\027\000\027\000\027\000\
    \027\000\027\000\027\000\027\000\027\000\027\000\059\000\059\000\
    \059\000\059\000\059\000\059\000\059\000\059\000\059\000\059\000\
    \062\000\062\000\062\000\062\000\062\000\062\000\062\000\062\000\
    \062\000\062\000\255\255\059\000\255\255\255\255\255\255\255\255\
    \255\255\062\000\062\000\062\000\062\000\062\000\062\000\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\059\000\255\255\255\255\
    \255\255\255\255\255\255\059\000\255\255\255\255\255\255\255\255\
    \255\255\062\000\062\000\062\000\062\000\062\000\062\000\255\255\
    \255\255\255\255\255\255\027\000\027\000\027\000\027\000\027\000\
    \027\000\027\000\027\000\027\000\027\000\027\000\027\000\027\000\
    \027\000\027\000\027\000\027\000\027\000\027\000\027\000\027\000\
    \027\000\027\000\255\255\027\000\027\000\027\000\027\000\027\000\
    \027\000\027\000\027\000\027\000\027\000\027\000\027\000\027\000\
    \027\000\027\000\027\000\027\000\027\000\027\000\027\000\027\000\
    \027\000\027\000\027\000\027\000\027\000\027\000\027\000\027\000\
    \027\000\027\000\255\255\027\000\027\000\027\000\027\000\027\000\
    \027\000\027\000\027\000\029\000\255\255\255\255\029\000\029\000\
    \029\000\255\255\255\255\255\255\029\000\029\000\255\255\029\000\
    \029\000\029\000\066\000\066\000\066\000\066\000\066\000\066\000\
    \066\000\066\000\066\000\066\000\029\000\255\255\029\000\029\000\
    \029\000\029\000\029\000\032\000\255\255\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\255\255\
    \029\000\066\000\255\255\255\255\255\255\255\255\255\255\033\000\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \033\000\033\000\033\000\033\000\033\000\033\000\033\000\033\000\
    \033\000\033\000\255\255\255\255\255\255\255\255\029\000\255\255\
    \029\000\033\000\033\000\033\000\033\000\033\000\033\000\033\000\
    \033\000\033\000\033\000\033\000\033\000\033\000\033\000\033\000\
    \033\000\033\000\033\000\033\000\033\000\033\000\033\000\033\000\
    \033\000\033\000\033\000\255\255\255\255\255\255\255\255\033\000\
    \255\255\033\000\033\000\033\000\033\000\033\000\033\000\033\000\
    \033\000\033\000\033\000\033\000\033\000\033\000\033\000\033\000\
    \033\000\033\000\033\000\033\000\033\000\033\000\033\000\033\000\
    \033\000\033\000\033\000\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\255\255\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \033\000\033\000\033\000\033\000\033\000\033\000\033\000\033\000\
    \033\000\033\000\033\000\033\000\033\000\033\000\033\000\033\000\
    \033\000\033\000\033\000\033\000\033\000\033\000\033\000\255\255\
    \033\000\033\000\033\000\033\000\033\000\033\000\033\000\033\000\
    \033\000\033\000\033\000\033\000\033\000\033\000\033\000\033\000\
    \033\000\033\000\033\000\033\000\033\000\033\000\033\000\033\000\
    \033\000\033\000\033\000\033\000\033\000\033\000\033\000\034\000\
    \033\000\033\000\033\000\033\000\033\000\033\000\033\000\033\000\
    \034\000\034\000\034\000\034\000\034\000\034\000\034\000\034\000\
    \034\000\034\000\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\034\000\034\000\034\000\034\000\034\000\034\000\034\000\
    \034\000\034\000\034\000\034\000\034\000\034\000\034\000\034\000\
    \034\000\034\000\034\000\034\000\034\000\034\000\034\000\034\000\
    \034\000\034\000\034\000\255\255\255\255\255\255\255\255\034\000\
    \255\255\034\000\034\000\034\000\034\000\034\000\034\000\034\000\
    \034\000\034\000\034\000\034\000\034\000\034\000\034\000\034\000\
    \034\000\034\000\034\000\034\000\034\000\034\000\034\000\034\000\
    \034\000\034\000\034\000\063\000\063\000\063\000\063\000\063\000\
    \063\000\063\000\063\000\063\000\063\000\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\063\000\063\000\063\000\063\000\
    \063\000\063\000\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\063\000\255\255\063\000\063\000\063\000\063\000\
    \063\000\063\000\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \034\000\034\000\034\000\034\000\034\000\034\000\034\000\034\000\
    \034\000\034\000\034\000\034\000\034\000\034\000\034\000\034\000\
    \034\000\034\000\034\000\034\000\034\000\034\000\034\000\255\255\
    \034\000\034\000\034\000\034\000\034\000\034\000\034\000\034\000\
    \034\000\034\000\034\000\034\000\034\000\034\000\034\000\034\000\
    \034\000\034\000\034\000\034\000\034\000\034\000\034\000\034\000\
    \034\000\034\000\034\000\034\000\034\000\034\000\034\000\255\255\
    \034\000\034\000\034\000\034\000\034\000\034\000\034\000\034\000\
    \035\000\255\255\255\255\035\000\035\000\035\000\255\255\255\255\
    \255\255\035\000\035\000\255\255\035\000\035\000\035\000\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\035\000\255\255\035\000\035\000\035\000\035\000\035\000\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\035\000\035\000\255\255\
    \035\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\255\255\035\000\036\000\035\000\255\255\036\000\
    \036\000\036\000\037\000\255\255\255\255\036\000\036\000\255\255\
    \036\000\036\000\036\000\037\000\037\000\037\000\037\000\037\000\
    \037\000\037\000\037\000\037\000\037\000\036\000\255\255\036\000\
    \036\000\036\000\036\000\036\000\037\000\037\000\037\000\037\000\
    \037\000\037\000\037\000\037\000\037\000\037\000\037\000\037\000\
    \037\000\037\000\037\000\037\000\037\000\037\000\037\000\037\000\
    \037\000\037\000\037\000\037\000\037\000\037\000\255\255\255\255\
    \255\255\036\000\037\000\255\255\037\000\037\000\037\000\037\000\
    \037\000\037\000\037\000\037\000\037\000\037\000\037\000\037\000\
    \037\000\037\000\037\000\037\000\037\000\037\000\037\000\037\000\
    \037\000\037\000\037\000\037\000\037\000\037\000\255\255\036\000\
    \255\255\036\000\255\255\255\255\255\255\255\255\035\000\035\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\255\255\035\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\035\000\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\037\000\037\000\037\000\037\000\037\000\
    \037\000\037\000\037\000\037\000\037\000\037\000\037\000\037\000\
    \037\000\037\000\037\000\037\000\037\000\037\000\037\000\037\000\
    \037\000\037\000\255\255\037\000\037\000\037\000\037\000\037\000\
    \037\000\037\000\037\000\037\000\037\000\037\000\037\000\037\000\
    \037\000\037\000\037\000\037\000\037\000\037\000\037\000\037\000\
    \037\000\037\000\037\000\037\000\037\000\037\000\037\000\037\000\
    \037\000\037\000\038\000\037\000\037\000\037\000\037\000\037\000\
    \037\000\037\000\037\000\038\000\038\000\038\000\038\000\038\000\
    \038\000\038\000\038\000\038\000\038\000\255\255\255\255\038\000\
    \255\255\255\255\255\255\255\255\038\000\038\000\038\000\038\000\
    \038\000\038\000\038\000\038\000\038\000\038\000\038\000\038\000\
    \038\000\038\000\038\000\038\000\038\000\038\000\038\000\038\000\
    \038\000\038\000\038\000\038\000\038\000\038\000\255\255\255\255\
    \255\255\255\255\038\000\255\255\038\000\038\000\038\000\038\000\
    \038\000\038\000\038\000\038\000\038\000\038\000\038\000\038\000\
    \038\000\038\000\038\000\038\000\038\000\038\000\038\000\038\000\
    \038\000\038\000\038\000\038\000\038\000\038\000\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\044\000\255\255\255\255\044\000\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\044\000\044\000\044\000\044\000\044\000\044\000\
    \044\000\044\000\044\000\044\000\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\038\000\038\000\038\000\038\000\038\000\
    \038\000\038\000\038\000\038\000\038\000\038\000\038\000\038\000\
    \038\000\038\000\038\000\038\000\038\000\038\000\038\000\038\000\
    \038\000\038\000\255\255\038\000\038\000\038\000\038\000\038\000\
    \038\000\038\000\038\000\038\000\038\000\038\000\038\000\038\000\
    \038\000\038\000\038\000\038\000\038\000\038\000\038\000\038\000\
    \038\000\038\000\038\000\038\000\038\000\038\000\038\000\038\000\
    \038\000\038\000\044\000\038\000\038\000\038\000\038\000\038\000\
    \038\000\038\000\038\000\040\000\255\255\255\255\040\000\040\000\
    \040\000\255\255\255\255\255\255\040\000\040\000\255\255\040\000\
    \040\000\040\000\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\040\000\255\255\040\000\040\000\
    \040\000\040\000\040\000\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \040\000\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\040\000\255\255\
    \040\000\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\044\000\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255";
  Lexing.lex_base_code = 
   "\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\084\000\036\001\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000";
  Lexing.lex_backtrk_code = 
   "\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \004\000\000\000\010\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000";
  Lexing.lex_default_code = 
   "\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000";
  Lexing.lex_trans_code = 
   "\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\001\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\007\000\007\000\007\000\007\000\007\000\007\000\007\000\
    \007\000\007\000\007\000\007\000\007\000\007\000\007\000\007\000\
    \007\000\007\000\007\000\007\000\007\000\007\000\007\000\007\000\
    \007\000\007\000\007\000\007\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\007\000\007\000\007\000\007\000\
    \007\000\007\000\007\000\007\000\007\000\007\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\007\000\007\000\007\000\
    \007\000\007\000\007\000\007\000\007\000\007\000\007\000\007\000\
    \007\000\007\000\007\000\007\000\007\000\007\000\007\000\007\000\
    \007\000\007\000\007\000\007\000\007\000\007\000\007\000\000\000\
    \000\000\000\000\000\000\007\000\000\000\007\000\007\000\007\000\
    \007\000\007\000\007\000\007\000\007\000\007\000\007\000\007\000\
    \007\000\007\000\007\000\007\000\007\000\007\000\007\000\007\000\
    \007\000\007\000\007\000\007\000\007\000\007\000\007\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\007\000\
    \007\000\007\000\007\000\007\000\007\000\007\000\007\000\007\000\
    \007\000\007\000\007\000\007\000\007\000\007\000\007\000\007\000\
    \007\000\007\000\007\000\007\000\007\000\007\000\007\000\000\000\
    \007\000\007\000\007\000\007\000\007\000\007\000\007\000\007\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\007\000\007\000\007\000\007\000\
    \007\000\007\000\007\000\007\000\007\000\007\000\007\000\007\000\
    \007\000\007\000\007\000\007\000\007\000\007\000\007\000\007\000\
    \007\000\007\000\007\000\000\000\007\000\007\000\007\000\007\000\
    \007\000\007\000\007\000\007\000\007\000\007\000\007\000\007\000\
    \007\000\007\000\007\000\007\000\007\000\007\000\007\000\007\000\
    \007\000\007\000\007\000\007\000\007\000\007\000\007\000\007\000\
    \007\000\007\000\007\000\007\000\007\000\007\000\007\000\007\000\
    \007\000\007\000\007\000\007\000\007\000\007\000\007\000\007\000\
    \007\000\007\000\007\000\007\000\007\000\007\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\007\000\007\000\007\000\
    \007\000\007\000\007\000\007\000\007\000\007\000\007\000\007\000\
    \007\000\007\000\007\000\007\000\007\000\007\000\007\000\007\000\
    \007\000\007\000\007\000\007\000\007\000\007\000\007\000\000\000\
    \000\000\000\000\000\000\007\000\000\000\007\000\007\000\007\000\
    \007\000\007\000\007\000\007\000\007\000\007\000\007\000\007\000\
    \007\000\007\000\007\000\007\000\007\000\007\000\007\000\007\000\
    \007\000\007\000\007\000\007\000\007\000\007\000\007\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\007\000\007\000\007\000\007\000\
    \007\000\007\000\007\000\007\000\007\000\007\000\007\000\007\000\
    \007\000\007\000\007\000\007\000\007\000\007\000\007\000\007\000\
    \007\000\007\000\007\000\000\000\007\000\007\000\007\000\007\000\
    \007\000\007\000\007\000\007\000\007\000\007\000\007\000\007\000\
    \007\000\007\000\007\000\007\000\007\000\007\000\007\000\007\000\
    \007\000\007\000\007\000\007\000\007\000\007\000\007\000\007\000\
    \007\000\007\000\007\000\000\000\007\000\007\000\007\000\007\000\
    \007\000\007\000\007\000\007\000\000\000";
  Lexing.lex_check_code = 
   "\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\031\000\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\033\000\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\033\000\033\000\033\000\033\000\
    \033\000\033\000\033\000\033\000\033\000\033\000\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\033\000\033\000\033\000\
    \033\000\033\000\033\000\033\000\033\000\033\000\033\000\033\000\
    \033\000\033\000\033\000\033\000\033\000\033\000\033\000\033\000\
    \033\000\033\000\033\000\033\000\033\000\033\000\033\000\255\255\
    \255\255\255\255\255\255\033\000\255\255\033\000\033\000\033\000\
    \033\000\033\000\033\000\033\000\033\000\033\000\033\000\033\000\
    \033\000\033\000\033\000\033\000\033\000\033\000\033\000\033\000\
    \033\000\033\000\033\000\033\000\033\000\033\000\033\000\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\255\255\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\033\000\033\000\033\000\033\000\
    \033\000\033\000\033\000\033\000\033\000\033\000\033\000\033\000\
    \033\000\033\000\033\000\033\000\033\000\033\000\033\000\033\000\
    \033\000\033\000\033\000\255\255\033\000\033\000\033\000\033\000\
    \033\000\033\000\033\000\033\000\033\000\033\000\033\000\033\000\
    \033\000\033\000\033\000\033\000\033\000\033\000\033\000\033\000\
    \033\000\033\000\033\000\033\000\033\000\033\000\033\000\033\000\
    \033\000\033\000\033\000\034\000\033\000\033\000\033\000\033\000\
    \033\000\033\000\033\000\033\000\034\000\034\000\034\000\034\000\
    \034\000\034\000\034\000\034\000\034\000\034\000\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\034\000\034\000\034\000\
    \034\000\034\000\034\000\034\000\034\000\034\000\034\000\034\000\
    \034\000\034\000\034\000\034\000\034\000\034\000\034\000\034\000\
    \034\000\034\000\034\000\034\000\034\000\034\000\034\000\255\255\
    \255\255\255\255\255\255\034\000\255\255\034\000\034\000\034\000\
    \034\000\034\000\034\000\034\000\034\000\034\000\034\000\034\000\
    \034\000\034\000\034\000\034\000\034\000\034\000\034\000\034\000\
    \034\000\034\000\034\000\034\000\034\000\034\000\034\000\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\034\000\034\000\034\000\034\000\
    \034\000\034\000\034\000\034\000\034\000\034\000\034\000\034\000\
    \034\000\034\000\034\000\034\000\034\000\034\000\034\000\034\000\
    \034\000\034\000\034\000\255\255\034\000\034\000\034\000\034\000\
    \034\000\034\000\034\000\034\000\034\000\034\000\034\000\034\000\
    \034\000\034\000\034\000\034\000\034\000\034\000\034\000\034\000\
    \034\000\034\000\034\000\034\000\034\000\034\000\034\000\034\000\
    \034\000\034\000\034\000\255\255\034\000\034\000\034\000\034\000\
    \034\000\034\000\034\000\034\000\255\255";
  Lexing.lex_code = 
   "\255\002\255\255\001\255\255\003\255\255\001\002\000\003\255";
}

let rec token state lexbuf =
  lexbuf.Lexing.lex_mem <- Array.create 4 (-1) ;   __ocaml_lex_token_rec state lexbuf 0
and __ocaml_lex_token_rec state lexbuf __ocaml_lex_state =
  match Lexing.new_engine __ocaml_lex_tables __ocaml_lex_state lexbuf with
      | 0 ->
let
# 238 "input.mll"
                  handler_name
# 1091 "input.ml"
= Lexing.sub_lexeme_opt lexbuf lexbuf.Lexing.lex_mem.(1) lexbuf.Lexing.lex_mem.(0) in
# 239 "input.mll"
    ( 
      begin_group state;
      Buffer.clear state.buf;
      state.depth <- 1;
      (match handler_name with
	   Some name when Plugin.exists name ->
	     comment true state lexbuf;
	     let s = Buffer.contents state.buf in
	     let n = Plugin.count_newlines s in
	     (for i = 1 to n do newline state done);
	     add_token state (`Special_comment (name, s))
	 | None 
	 | Some _ ->
	     Buffer.add_string state.buf "(*";
	     (match handler_name with
		  Some name -> Buffer.add_string state.buf name
		| None -> ());
	     comment false state lexbuf;
	     Buffer.add_string state.buf "*)";
	     add_token state (`Comment (Buffer.contents state.buf));
      );
      end_group state;
      token state lexbuf 
    )
# 1118 "input.ml"

  | 1 ->
# 264 "input.mll"
    ( begin_group state;
      Buffer.clear state.buf;
      Buffer.add_char state.buf '"';
      string state false lexbuf;
      add_token state (`String (Buffer.contents state.buf));
      end_group state;
      token state lexbuf )
# 1129 "input.ml"

  | 2 ->
# 273 "input.mll"
    ( begin_group state;
      add_token state (`Construct (lexeme lexbuf));
      Buffer.clear state.buf;
      quotation state lexbuf;
      add_token ~offset:(-2) state (`Quotation (Buffer.contents state.buf));
      add_token state (`Construct ">>");
      end_group state;
      token state lexbuf )
# 1141 "input.ml"

  | 3 ->
# 283 "input.mll"
      ( add_token state (`Construct (lexeme lexbuf));
	token state lexbuf )
# 1147 "input.ml"

  | 4 ->
# 286 "input.mll"
      ( add_token state (tokenify (lexeme lexbuf));
	token state lexbuf )
# 1153 "input.ml"

  | 5 ->
# 294 "input.mll"
    ( add_token state (`Keyword (lexeme lexbuf));
      token state lexbuf )
# 1159 "input.ml"

  | 6 ->
# 298 "input.mll"
      ( add_token state (`Token (lexeme lexbuf));
	token state lexbuf )
# 1165 "input.ml"

  | 7 ->
# 302 "input.mll"
      ( List.iter (add_token state) [`String "'"; `Newline; `String "'"];
	token state lexbuf )
# 1171 "input.ml"

  | 8 ->
# 305 "input.mll"
      ( List.iter (add_token state) [`String "'\\"; `Newline; `String "'"];
	token state lexbuf )
# 1177 "input.ml"

  | 9 ->
# 308 "input.mll"
      ( add_token state (`String (lexeme lexbuf));
	token state lexbuf )
# 1183 "input.ml"

  | 10 ->
# 312 "input.mll"
    ( add_token state `Newline;
      token state lexbuf )
# 1189 "input.ml"

  | 11 ->
# 315 "input.mll"
    ( add_token state `Tab;
      token state lexbuf )
# 1195 "input.ml"

  | 12 ->
# 318 "input.mll"
    ( finish_annot state;
      return_tokens state )
# 1201 "input.ml"

  | 13 ->
# 321 "input.mll"
    ( add_token state (`Token (lexeme lexbuf));
      token state lexbuf )
# 1207 "input.ml"

  | 14 ->
# 332 "input.mll"
    ( add_token state (`Token (lexeme lexbuf));
      token state lexbuf )
# 1213 "input.ml"

  | __ocaml_lex_state -> lexbuf.Lexing.refill_buff lexbuf; __ocaml_lex_token_rec state lexbuf __ocaml_lex_state

and comment special state lexbuf =
    __ocaml_lex_comment_rec special state lexbuf 68
and __ocaml_lex_comment_rec special state lexbuf __ocaml_lex_state =
  match Lexing.engine __ocaml_lex_tables __ocaml_lex_state lexbuf with
      | 0 ->
# 337 "input.mll"
    ( state.depth <- state.depth + 1;
      Buffer.add_string state.buf "(*";
      comment special state lexbuf )
# 1226 "input.ml"

  | 1 ->
# 341 "input.mll"
    ( state.depth <- state.depth - 1; 
      if (state.depth > 0) then (
	Buffer.add_string state.buf "*)";
	comment special state lexbuf
      )
    )
# 1236 "input.ml"

  | 2 ->
# 348 "input.mll"
    ( Buffer.add_char state.buf '"';
      string state true lexbuf;
      comment special state lexbuf )
# 1243 "input.ml"

  | 3 ->
# 352 "input.mll"
    ( finish_annot state )
# 1248 "input.ml"

  | 4 ->
# 354 "input.mll"
    ( if special then (
	Buffer.add_char state.buf '\n';
	comment special state lexbuf
      )
      else (
	add_token state (`Comment (Buffer.contents state.buf));
	add_token state `Newline;
	Buffer.clear state.buf;
	comment special state lexbuf
      )
    )
# 1263 "input.ml"

  | 5 ->
# 366 "input.mll"
    ( add_token state (`Comment (Buffer.contents state.buf));
      add_token state `Tab;
      Buffer.clear state.buf;
      comment special state lexbuf )
# 1271 "input.ml"

  | 6 ->
# 371 "input.mll"
    ( Buffer.add_string state.buf (lexeme lexbuf);
      comment special state lexbuf )
# 1277 "input.ml"

  | __ocaml_lex_state -> lexbuf.Lexing.refill_buff lexbuf; __ocaml_lex_comment_rec special state lexbuf __ocaml_lex_state

and string state comment lexbuf =
    __ocaml_lex_string_rec state comment lexbuf 79
and __ocaml_lex_string_rec state comment lexbuf __ocaml_lex_state =
  match Lexing.engine __ocaml_lex_tables __ocaml_lex_state lexbuf with
      | 0 ->
# 377 "input.mll"
    ( Buffer.add_char state.buf '"' )
# 1288 "input.ml"

  | 1 ->
# 380 "input.mll"
    ( Buffer.add_string state.buf (lexeme lexbuf);
      string state comment lexbuf )
# 1294 "input.ml"

  | 2 ->
# 383 "input.mll"
    ( finish_annot state )
# 1299 "input.ml"

  | 3 ->
# 385 "input.mll"
    ( let s = Buffer.contents state.buf in
      add_token state (if comment then `Comment s else `String s);
      add_token state `Newline;
      Buffer.clear state.buf;
      string state comment lexbuf )
# 1308 "input.ml"

  | 4 ->
# 391 "input.mll"
    ( let s = Buffer.contents state.buf in
      add_token state (if comment then `Comment s else `String s);
      add_token state `Tab;
      Buffer.clear state.buf;
      string state comment lexbuf )
# 1317 "input.ml"

  | 5 ->
# 397 "input.mll"
    ( Buffer.add_string state.buf (lexeme lexbuf);
      string state comment lexbuf )
# 1323 "input.ml"

  | __ocaml_lex_state -> lexbuf.Lexing.refill_buff lexbuf; __ocaml_lex_string_rec state comment lexbuf __ocaml_lex_state

and quotation state lexbuf =
    __ocaml_lex_quotation_rec state lexbuf 88
and __ocaml_lex_quotation_rec state lexbuf __ocaml_lex_state =
  match Lexing.engine __ocaml_lex_tables __ocaml_lex_state lexbuf with
      | 0 ->
# 401 "input.mll"
            ( () )
# 1334 "input.ml"

  | 1 ->
# 402 "input.mll"
            ( Buffer.add_string state.buf "\\>>";
	      quotation state lexbuf )
# 1340 "input.ml"

  | 2 ->
# 405 "input.mll"
      ( let s = Buffer.contents state.buf in
	add_token state (`Quotation s);
	add_token state `Newline;
	Buffer.clear state.buf;
	quotation state lexbuf )
# 1349 "input.ml"

  | 3 ->
# 411 "input.mll"
      ( let s = Buffer.contents state.buf in
	add_token state (`Quotation s);
	add_token state `Tab;
	Buffer.clear state.buf;
	quotation state lexbuf )
# 1358 "input.ml"

  | 4 ->
# 416 "input.mll"
            ( Buffer.add_string state.buf (lexeme lexbuf);
	      quotation state lexbuf )
# 1364 "input.ml"

  | __ocaml_lex_state -> lexbuf.Lexing.refill_buff lexbuf; __ocaml_lex_quotation_rec state lexbuf __ocaml_lex_state

;;

# 419 "input.mll"
 
  let parse ?(annot = []) lexbuf =
    token (init_state annot lexbuf) lexbuf

  let string ?(filename = "") ?(annot = []) s = 
    let lexbuf = Lexing.from_string s in 
    init_lexbuf lexbuf filename;
    token (init_state annot lexbuf) lexbuf

  let channel ?(filename = "") ?(annot = []) ic = 
    let lexbuf = Lexing.from_channel ic in
    init_lexbuf lexbuf filename;
    token (init_state annot lexbuf) lexbuf

  let file ?annot s = 
    let ic = open_in s in
    let l = channel ~filename:s ?annot ic in
    close_in ic;
    l

# 1391 "input.ml"
