(* $Id: rpc_server.ml 199 2005-06-28 19:58:42Z gerd $
 * ----------------------------------------------------------------------
 *
 *)

open Rtypes
open Xdr
open Unixqueue
open Rpc_common
open Rpc

exception Connection_lost

class connection_id sock_name peer_name = 
object 
  method socket_name = (sock_name : Unix.sockaddr)
  method peer_name = (peer_name : Unix.sockaddr)
end ;;


class no_connection_id : connection_id = 
object 
  method socket_name = failwith "Unconnected"
  method peer_name = failwith "Unconnected"
end ;;


type rule =
    [ `Deny
    | `Drop
    | `Reject
    | `Accept
    | `Accept_limit_length of (int * rule)
    ]

type auth_result =
    Auth_positive of (string * string * string)
      (* (username, returned_verifier_flavour, returned_verifier_data) *)
  | Auth_negative of Rpc.server_error


type auth_peeker =
    Unix.file_descr -> string option


class type ['t] pre_auth_method =
object
  method name : string

  method flavors : string list

  method peek : auth_peeker option

  method authenticate :
    't ->
    connection_id ->
    Unix.file_descr ->
    Unix.sockaddr ->
    Unix.sockaddr ->
    string ->
    string ->
    string ->
    string ->
    (auth_result -> unit) ->
      unit

end


module Procnr = struct
  type t = uint4
  let compare (a:uint4) (b:uint4) =
    (* avoid calling Pervasives.compare *)
    let a' = logical_int32_of_uint4 a in
    let b' = logical_int32_of_uint4 b in
    if a' = b' then
      0
    else
      if a' < b' then
	-1
      else
	1
end


module ProcnrMap = Map.Make(Procnr)


let rec procnr_map_mk f l =
  match l with
      [] -> ProcnrMap.empty
    | x :: l' ->
	let (key,value) = f x in
	ProcnrMap.add key value (procnr_map_mk f l')
;;


type t =
      { mutable master : Unix.file_descr option;        (* master socket *)
	mutable slaves : (Unix.file_descr * connection) list;
	  (* Only two cases: [] and [d,conn] *)
	mutable prog : Rpc_program.t;
	mutable procs : binding ProcnrMap.t;
	mutable esys : event_system;
	mutable group : group;
	mutable prot : protocol;
	mutable mode : mode;
	mutable exception_handler : exn -> unit;
	mutable unmap_port : (unit -> unit);
	mutable onclose : (connection_id -> unit);
	mutable filter : (Unix.sockaddr -> connection_id -> rule);
	mutable auth_methods : (string, t pre_auth_method) Hashtbl.t;
	mutable auth_peekers : (auth_peeker * t pre_auth_method) list;
      }

and connection =
    (* For connected streams, but also used for datagram servers. *)
      { whole_server : t;
	conn_id : connection_id;                (* only used for prot = Tcp *)
	mutable ready : bool;
        mutable trans : Rpc_transport.t;
	mutable rule : rule option;
        (* TODO: The rule exists per incoming message, not per connection.
	 * Is it better to put it into Rpc_transport?
	 *)

	mutable next_call_id : int;

	(* replies to do: *)
	mutable replies : session Queue.t;

	(* RPC does not define how to check if replies are delivered,
	 * so there is no "re-reply" mechanism. The client has to call
	 * again; but the server cannot identify such repetitions.
	 * (The xid field cannot be used for this purpose!)
	 *)

	mutable has_output_resource : bool;

	mutable peeked :        bool;           (* whether already peeked *)
	mutable peeked_user :   string option;
	mutable peeked_method : t pre_auth_method;
      }

and session =
    (* intentionally immutable to make value sharing possible *)
      { server : connection;
	sess_conn_id : connection_id;
	sockaddr : Unix.sockaddr;   (* own address *)
	peeraddr : Unix.sockaddr;
	call_id : int;
	client_id : uint4;         (* xid *)
	procname : string;
	parameter : xdr_value;     (* XV_void if not used *)
	result : Rpc_packer.packed_value;
         (* complete result; "" if not used *)
	auth_method : t pre_auth_method;
	auth_user : string;
	auth_ret_flav : string;
	auth_ret_data : string;
      }

and connector =
      Localhost of int                     (* port, 0: automatically chosen *)
    | Portmapped
    | Internet of (Unix.inet_addr * int)   (* addr, port *)
    | Unix of string                       (* path to unix dom sock *)
    | Descriptor of Unix.file_descr
    | Dynamic_descriptor of (unit -> Unix.file_descr)

and binding_sync =
      { sync_name : string;
	sync_proc : xdr_value -> xdr_value
      }

and binding_async =
      { async_name : string;
	async_invoke : session -> xdr_value -> unit
                                            (* invocation of this procedure *)
      }

and binding =
      Sync of binding_sync
    | Async of binding_async

class type auth_method = [t] pre_auth_method ;;

class auth_none : auth_method =
object
  method name = "AUTH_NONE"
  method flavors = [ "AUTH_NONE" ]
  method peek = None
  method authenticate _ _ _ _ _ _ _ _ _ f = f(Auth_positive("","AUTH_NONE",""))
end

let auth_none = new auth_none

class auth_too_weak : auth_method =
object
  method name = "AUTH_TOO_WEAK"
  method flavors = []
  method peek = None
  method authenticate _ _ _ _ _ _ _ _ _ f = f(Auth_negative Auth_too_weak)
end

let auth_too_weak = new auth_too_weak

let debug = ref false

  (*****)

let null_packed_value =
  Rpc_packer.packed_value_of_string ""

let no_conn_id = new no_connection_id

  (*****)

type reaction =
    Execute_procedure
  | Reject_procedure of server_error

let process_incoming_message srv conn sockaddr peeraddr message reaction =
  let d = Rpc_transport.descriptor conn.trans in

  let make_immediate_answer xid procname result =
    { server = conn;
      sess_conn_id = if srv.prot = Rpc.Tcp then conn.conn_id
                     else new connection_id sockaddr peeraddr;
      sockaddr = sockaddr;
      peeraddr = peeraddr;
      call_id = (-1);          (* not applicable *)
      client_id = xid;
      procname = procname;
      parameter = XV_void;
      result = result;
      auth_method = auth_none;
      auth_user = "";
      auth_ret_flav = "AUTH_NONE";
      auth_ret_data = "";
    }
  in

  let schedule_answer answer =
    Queue.add answer conn.replies;
    if not conn.has_output_resource then begin
      add_resource srv.esys srv.group (Wait_out d, (-1.0));
      conn.has_output_resource <- true;
    end
  in

  let protect_protect f =
    try
      f()
    with
	any ->
	  (try srv.exception_handler any with _ -> ());
  in

  let protect ?(ret_flav="AUTH_NONE") ?(ret_data="") f =
    try
      f()
    with
	Rpc_server(Unavailable_program | Unavailable_version(_,_)|
                   Unavailable_procedure | Garbage | System_err
		   as condition) ->
	  protect_protect
	    (fun () ->
	       let xid = Rpc_packer.peek_xid message in
	       let reply = Rpc_packer.pack_accepting_reply xid
			     ret_flav ret_data condition in
	       let answer = make_immediate_answer xid "" reply in
	       schedule_answer answer
	    )
      | Xdr.Xdr_format _
      | Xdr.Xdr_format_message_too_long _ ->          (* Convert to Garbage *)
	  protect_protect
	    (fun () ->
	       let xid = Rpc_packer.peek_xid message in
	       let reply = Rpc_packer.pack_accepting_reply xid
			     ret_flav ret_data Garbage in
	       let answer = make_immediate_answer xid "" reply in
	       schedule_answer answer
	    )
      | Rpc_server condition ->
	  protect_protect
	    (fun () ->
	       let xid = Rpc_packer.peek_xid message in
	       let reply = Rpc_packer.pack_rejecting_reply xid condition in
	       let answer = make_immediate_answer xid "" reply in
	       schedule_answer answer
	    )
      | Abort(_,_) as x ->
	  raise x
      | any ->
	  (* Reply "System_err": *)
	  (try srv.exception_handler any with _ -> ());
	  protect_protect
	    (fun () ->
	       let xid = Rpc_packer.peek_xid message in
	       let reply = Rpc_packer.pack_accepting_reply xid
			     ret_flav ret_data System_err in
	       let answer = make_immediate_answer xid "" reply in
	       schedule_answer answer
	    )
  in

  protect
    (fun () ->
       match reaction with
	   Execute_procedure ->
	     let
	       xid, prog_nr, vers_nr, proc_nr,
	       flav_cred, data_cred, flav_verf, data_verf, frame_len
	       = Rpc_packer.unpack_call_frame_l message
	     in

	     let sess_conn_id =
	       if srv.prot = Rpc.Tcp then 
		 conn.conn_id
	       else 
		 new connection_id sockaddr peeraddr
	     in

	     (* First authenticate: *)
	     let auth_m =
	       try Hashtbl.find srv.auth_methods flav_cred
	       with Not_found -> auth_too_weak
	     in

	     (* The [authenticate] method will call the passed function
	      * when the authentication is done. This may be at any time
	      * in the future.
	      *)
	     auth_m # authenticate
	       srv sess_conn_id d sockaddr peeraddr
	       flav_cred data_cred flav_verf data_verf
	       (function Auth_positive(user,ret_flav,ret_data) ->
		  (* user: the username (method-dependent)
		   * ret_flav: flavour of verifier to return
		   * ret_data: data of verifier to return
		   *)
		  protect ~ret_flav ~ret_data
		    (fun () ->
		       if prog_nr <> Rpc_program.program_number srv.prog then
			 raise (Rpc_server Unavailable_program);

		       let my_vers = Rpc_program.version_number srv.prog in
		       if vers_nr <> my_vers then
			 raise (Rpc_server (Unavailable_version (my_vers, my_vers)));
		       (* TODO: report the correct range of versions *)

		       let proc =
			 try
			   ProcnrMap.find proc_nr srv.procs
			 with Not_found -> raise (Rpc_server Unavailable_procedure)
		       in

		       let procname =
			 match proc with
			     Sync p -> p.sync_name
			   | Async p -> p.async_name
		       in

		       let param =
			 Rpc_packer.unpack_call_body
			   srv.prog procname message frame_len in

		       begin match proc with
			   Sync p ->
			     let result_value =
			       p.sync_proc param
			     in
			     let reply = Rpc_packer.pack_successful_reply
					   srv.prog p.sync_name xid
					   ret_flav ret_data result_value in
			     let answer = make_immediate_answer
					    xid procname reply in
			     schedule_answer answer
			 | Async p ->
			     let u, m = match conn.peeked_user with
				 Some uid -> uid, conn.peeked_method
			       | None -> user, auth_m
			     in
			     let this_session =
			       { server = conn;
				 sess_conn_id = sess_conn_id;
				 sockaddr = sockaddr;
				 peeraddr = peeraddr;
				 call_id = conn.next_call_id;
				 client_id = xid;
				 procname = p.async_name;
				 parameter = param;
				 result = null_packed_value;
				 auth_method = m;
				 auth_user = u;
				 auth_ret_flav = ret_flav;
				 auth_ret_data = ret_data;
			       } in
			     conn.next_call_id <- conn.next_call_id + 1;
			     p.async_invoke this_session param
		       end
		    )
		  | Auth_negative code ->
		      protect (fun () -> raise(Rpc_server code))
	       )
	 | Reject_procedure reason ->
	     protect (fun () -> raise(Rpc_server reason))
    )
;;

  (*****)

exception Deferred_termination_event of connection_id ;;
  (* The event handler will be terminated in the future. The identifier
   * is used to distinguish between several handlers. This is normally only
   * used for connection-oriented handlers.
   *)

exception Deferred_server_termination_event of t
  (* Terminates the whole server *)


let schedule_termination_event srv conn_id =
  assert (conn_id <> no_conn_id);
  assert (srv.prot = Tcp);
  add_event
    srv.esys
    (Extra(Deferred_termination_event conn_id))
;;


let schedule_server_termination_event srv =
   add_event
    srv.esys
    (Extra(Deferred_server_termination_event srv))
;;


let connection_event_handler srv conn esys esys' ev =
  let conn_d = Rpc_transport.descriptor conn.trans in

  let terminate() =
    if !debug then prerr_endline "EH: terminate";
    if !debug then prerr_endline "EH: remove input";
    remove_resource srv.esys srv.group (Wait_in conn_d);
    if conn.has_output_resource then begin
      if !debug then prerr_endline "EH: remove output";
      remove_resource srv.esys srv.group (Wait_out conn_d);
    end;
    raise Equeue.Terminate
  in

  match ev with
      Extra(Deferred_termination_event id) when id = conn.conn_id ->
	if !debug then prerr_endline "EH: deferred termination";
	terminate()                           (* Will raise Equeue.Terminate *)

    | Extra(Deferred_server_termination_event s) when s == srv ->
	if !debug then prerr_endline "EH: deferred server termination";
	schedule_server_termination_event srv; (* shutdown other handlers too *)
	terminate()                           (* Will raise Equeue.Terminate *)

      (*** event: input data have arrived ***)

    | Input_arrived(_,d) ->
	if d <> conn_d then raise (Equeue.Reject);

	if Rpc_transport.at_eof conn.trans then begin
	  if !debug then prerr_endline "EH: eof";

	  if Rpc_transport.is_message_incomplete conn.trans then begin
	    (* Got EOF before the message has been completely received.
	     * This is an error.
	     * We clean the input side of the transporter such that
	     * this condition is only reported once. The descriptor
	     * remains at EOF such that the handler will be called
	     * again.
	     *)
	    if !debug then prerr_endline "EH: incomplete message";
	    Rpc_transport.clean_input conn.trans;
	    failwith "Rpc_server: EOF in the middle of a message";
	  end;

	  terminate()                         (* Will raise Equeue.Terminate *)

	end
	else begin

	  (* Peek credentials: *)
	  if not conn.peeked && srv.prot = Tcp then begin
	    let u = ref None in
	    let m = ref auth_none in
	    try
	      List.iter
		(fun (peeker, meth) ->
		   match peeker d with
		       Some uid -> u := Some uid; m := meth; raise Exit
		     | None -> ()
		)
		srv.auth_peekers;
	      conn.peeked <- true;
	    with
		Exit ->
		  conn.peeked <- true;
		  conn.peeked_user <- !u;
		  conn.peeked_method <- !m
	  end;

	  (* process all what is buffered: *)
	  let again = ref true in
	  while !again do

      	    (* Receive next part of the message *)

	    if !debug then prerr_endline "EH: receive_part";
	    begin try
	      ignore(Rpc_transport.receive_part conn.trans)
	    with
		err ->
		  (* On error: receive_part may raise an exception like EPIPE.
		   * The connection (if any) is closed, and the resources
		   * are removed.
		   *)
		  if !debug then
		    prerr_endline ("EH: read exception: " ^
				   Printexc.to_string err);
		  if srv.prot = Tcp then            (* Close TCP connection *)
		    schedule_termination_event srv conn.conn_id;
		  raise err
		    (* CHECK: We cannot terminate the handler immediately,
		     * because the exception [err] would be lost. So we
		     * add a special event causing that the handler will
		     * be terminated in the future.
		     * However, we don't know what happens until then.
		     * Check this.
		     *)
	    end;

	    (* Is the message complete? Yes: process it *)

	    let get_rule peer =
	      match conn.rule with
		  None ->
		    let r = srv.filter peer conn.conn_id in
		    conn.rule <- Some r;
		    r
		| Some r -> r
	    in

	    let rec unroll_rule r length =
	      match r with
		  `Accept_limit_length(limit,r') ->
		    if length > limit then unroll_rule r' length else `Accept
		| _ -> r
	    in

	    if Rpc_transport.is_message_incomplete conn.trans then begin
	      assert (srv.prot = Tcp);          (* ==> Unix.getpeername works *)
	      (* Apply filter rule, if any *)
	      match
		unroll_rule (get_rule (Unix.getpeername d))
		            (Rpc_transport.peek_length conn.trans)
	      with
		  `Accept -> ()
		| `Deny   -> if srv.prot = Tcp then terminate()
		| `Drop   -> Rpc_transport.drop conn.trans
		| `Reject -> ()
		| `Accept_limit_length(_,_) -> assert false
	    end
	    else
	      if Rpc_transport.is_message_complete conn.trans then begin
		if !debug then prerr_endline "EH: got message";
		let message = Rpc_transport.get conn.trans in
		Rpc_transport.clean_input conn.trans;
		let peeraddr =
	      	  if srv.prot = Udp then begin
		    (try Rpc_transport.get_sender conn.trans with
			 Not_found -> failwith "Rpc_server: cannot obtain socket address of peer")
	      	  end
		  else
		    (Unix.getpeername d)
		in
		let sockaddr = Unix.getsockname d in

		(* First check whether the message matches the filter rule: *)

		let rule =
		  unroll_rule (get_rule peeraddr)
		              (Rpc_packer.length_of_packed_value message)
		in
		conn.rule <- None;                  (* reset rule after usage *)

		match rule with
		    `Accept ->
		      process_incoming_message
			srv conn sockaddr peeraddr message
			Execute_procedure
		  | `Deny ->
		      if srv.prot = Tcp then terminate()
		  | `Drop ->
		      (* Simply forget the message *)
		      ()
		  | `Reject ->
		      process_incoming_message
			srv conn sockaddr peeraddr message
			(Reject_procedure Auth_too_weak)
		  | `Accept_limit_length(_,_) -> assert false

	      end;

	    again := not (Rpc_transport.is_buffer_empty conn.trans);
	  done;

          (* (don't) clean up:
	   *   Rpc_transport.clean_input conn.trans
           * -- is incorrect here because if the message is large, it is
           * possible that event has only processed the beginning of the
           * message.
           *)

	end

      (*** event: ready to output data ***)

    | Output_readiness(_,d) ->
	if d <> conn_d then raise (Equeue.Reject);

	if Rpc_transport.is_sending_complete conn.trans then begin
	  (* send next reply *)
	  try
	    let next_reply = Queue.take conn.replies in    (* or Queue.Empty *)
	    if !debug then prerr_endline "EH: next reply";
	    Rpc_transport.put conn.trans next_reply.result;
	    if srv.prot = Udp && srv.mode = Socket then
	      Rpc_transport.set_receiver conn.trans next_reply.peeraddr;
	  with
	      Queue.Empty ->
		(* this was the last reply in the queue *)
		if !debug then prerr_endline "EH: last reply";
		Rpc_transport.clean_output conn.trans;
		remove_resource srv.esys srv.group (Wait_out d);
		conn.has_output_resource <- false
	end;

	if not (Rpc_transport.is_sending_complete conn.trans) then begin
	  (* next part of call message *)
	  if !debug then prerr_endline "EH: send_part";
	  begin try
	    ignore(Rpc_transport.send_part conn.trans);
	  with
	      err ->
		(* On error: send_part may raise an exception like EPIPE.
		 * The connection (if any) is closed, and the resources
		 * are removed.
		 *)
		if !debug then
		  prerr_endline ("EH: write exception: " ^
				 Printexc.to_string err);
		if srv.prot = Tcp then            (* Close TCP connection *)
		  schedule_termination_event srv conn.conn_id;
		raise err
		  (* CHECK: We cannot terminate the handler immediately,
		   * because the exception [err] would be lost. So we
		   * add a special event causing that the handler will
		   * be terminated in the future.
		   * However, we don't know what happens until then.
		   * Check this.
		   *)
	  end;
	  ()
	end

    | _ ->
	raise (Equeue.Reject)

  (*****)

let close_conn conn descr =
  if conn.ready then begin
    conn.ready <- false;
    Unix.close descr
  end


let master_event_handler srv master esys esys' ev =
    match ev with
      (*** event: input data have been arrived ***)
      Input_arrived(_,d) ->
	if d <> master then raise (Equeue.Reject);
	(* Note: A master socket can never reach "end of file" *)
	let (slave_d, slave_addr) = Unix.accept d in
	let conn =
	    { whole_server = srv;
	      conn_id = new connection_id (Unix.getsockname slave_d) slave_addr;
	      ready = true;
	      trans = Rpc_transport.create slave_d srv.prot BiPipe;
	      rule = None;
	      next_call_id = 0;
	      replies = Queue.create();
	      has_output_resource = false;
	      peeked = false;
	      peeked_user = None;
	      peeked_method = auth_none;
	    } in

	add_handler srv.esys srv.group (connection_event_handler srv conn);
	add_resource srv.esys srv.group (Wait_in slave_d, (-1.0));
	add_close_action
	  srv.esys
	  srv.group
	  (slave_d, (fun d -> close_conn conn d; srv.onclose conn.conn_id; ))

    | Extra(Deferred_server_termination_event s) when s == srv ->
	if !debug then prerr_endline "EH: deferred server termination";
	schedule_server_termination_event srv; (* shutdown other handlers too *)
	remove_resource srv.esys srv.group (Wait_in master);
	raise Equeue.Terminate

    | _ ->
	raise (Equeue.Reject)

  (*****)

let create ?program_number ?version_number
           esys conn prot mode prog0 procs max_clients =

    let prog = Rpc_program.update ?program_number ?version_number prog0 in

    let default_exception_handler ex =
      prerr_endline ("Exception " ^ Printexc.to_string ex ^ " caught")
    in

    if prot = Udp && mode <> Socket then
      failwith "Rpc_server.create: UDP servers must use socket mode";

    let none = Hashtbl.create 3 in
    Hashtbl.add none "AUTH_NONE" auth_none;

    let srv =
      { master = None;
	slaves = [];
	prog = prog;
	procs =
	  procnr_map_mk
	    (fun b ->
	      let name =
		match b with
		  Sync b' -> b'.sync_name
		| Async b' -> b'.async_name
	      in
	      Rpc_program.procedure_number prog name, b)
	    procs;
	esys = esys;
	group = new_group esys;
	prot = prot;
	mode = mode;
	exception_handler = default_exception_handler;
	unmap_port = (fun () -> ());
	onclose = (fun _ -> ());
	filter = (fun _ _ -> `Accept);
	auth_methods = none;
	auth_peekers = [];
      }
    in

    (* make the socket listening or waiting, whatever is appropriate *)

    let establish_socket s =
      if mode = Socket && prot = Tcp then begin
	(*  make socket listening (assumes that it is a master socket) *)
	Unix.listen s max_clients;
	srv.master <- Some s
      end else begin
	  let c =
	    { whole_server = srv;
	      conn_id = 
		( if prot = Tcp then 
		    new connection_id (Unix.getsockname s) (Unix.getpeername s)
		  else 
		    no_conn_id);
	      ready = true;
	      rule = None;
	      trans = Rpc_transport.create s prot mode;
	      next_call_id = 0;
	      replies = Queue.create();
	      has_output_resource = false;
	      peeked = false;
	      peeked_user = None;
	      peeked_method = auth_none;
	    } in

	  srv.slaves <- [ s, c ]
      end
    in

    (* auxiliary function to open a socket on 127.0.0.1 *)

    let bind_to_internet addr port =
      let s =
      	Unix.socket
	  Unix.PF_INET
	  (if prot = Tcp then Unix.SOCK_STREAM else Unix.SOCK_DGRAM)
	  0
      in
      try
	Unix.setsockopt s Unix.SO_REUSEADDR (port <> 0);
	Unix.bind
	  s
	  (Unix.ADDR_INET (addr, port));
	s
      with
	any -> Unix.close s; raise any
    in
    let bind_to_localhost port =
      bind_to_internet (Unix.inet_addr_of_string "127.0.0.1") port
    in

    (* get the working descriptor, and a flag whether this descriptor should
     * be closed after all has been done
     *)

    let descr, shall_i_close =
      match conn with
	Localhost port ->
	  let s = bind_to_localhost port in
	  s, true
      | Internet (addr,port) ->
	  let s = bind_to_internet addr port in
	  s, true
      |	Portmapped ->
	  (* DEBUG *)
	  if !debug then prerr_endline "Portmapped!";
	  let s = bind_to_internet Unix.inet_addr_any 0 in
	  begin try
	    let Unix.ADDR_INET(_,port) = Unix.getsockname s in
	    (* DEBUG *)
	    if !debug then prerr_endline ("port: " ^ string_of_int port);
	    let pm = Rpc_portmapper.create_inet "127.0.0.1" in
	    let prog_nr = Rpc_program.program_number prog in
	    let vers_nr = Rpc_program.version_number prog in
	    let old_port = Rpc_portmapper.getport pm prog_nr vers_nr prot in
	    (* DEBUG *)
	    if !debug then
	      prerr_endline ("old port: " ^ string_of_int old_port);
	    if old_port > 0 then
	      (* remove old binding: *)
	      ignore(Rpc_portmapper.unset pm prog_nr vers_nr prot old_port);
	    (* set new binding: *)
	    let success = Rpc_portmapper.set pm prog_nr vers_nr prot port in
	    Rpc_portmapper.shut_down pm;
	    (* DEBUG *)
	    if !debug then
	      prerr_endline (if success then "success" else "failure");
	    if not success then
	      failwith "Rpc_server: could not register service";
	    srv.unmap_port <-
	      (fun () ->
		let pm = Rpc_portmapper.create_inet "127.0.0.1" in
		ignore(Rpc_portmapper.unset pm prog_nr vers_nr prot port);
		Rpc_portmapper.shut_down pm);
	    s, true
	  with
	    any -> Unix.close s; raise any
	  end
      |	Unix path ->
	  let s =
      	    Unix.socket
	      Unix.PF_UNIX
	      (if prot = Tcp then Unix.SOCK_STREAM else Unix.SOCK_DGRAM)
	      0
	  in
	  begin try
	    Unix.bind
	      s (Unix.ADDR_UNIX path);
	    s, true
	  with
	    any -> Unix.close s; raise any
	  end
      |	Descriptor d -> d, false
      |	Dynamic_descriptor f -> f(), true
    in

    (* protect the event handlers against exceptions: *)

    let protect_handler h esys esys' ev =
      try
	h esys esys' ev
     with
        Equeue.Reject as x -> raise x
      | Equeue.Terminate as x -> raise x
      | Abort(g,x') as x -> raise x
      | any ->
          (* DEBUG *)
          if !debug then
	    prerr_endline ("EXCEPTION: " ^ Printexc.to_string any);

          raise (Abort(srv.group, any))
    in

    (* the 'abort' handler shut everything down *)

    let abort g x =
      (* DEBUG *)
      if !debug then
	prerr_endline "aborting...";
      srv.unmap_port();
    in

    (* Now establish the server, add handlers and resources: *)

    try
      establish_socket descr;

      add_abort_action esys srv.group abort;

      begin
	match srv.master with
      	  None -> ()
	| Some m ->
	    add_handler esys srv.group
	                        (protect_handler (master_event_handler srv m));
	    add_resource esys srv.group (Wait_in m, (-1.0));
	    if shall_i_close then
	      add_close_action
	        esys srv.group (m, (fun descr -> Unix.close descr))
      end;

      List.iter
	(fun (d,conn) ->
	   (* CHECK: Is it really necessary to protect the handler here?
	    * The [connection_event_handler] has now some built-in
	    * protection, and aborting the server may be too much. So
	    * we try it here without.
	    *)
	   add_handler
	     esys srv.group
             ( (* protect_handler *)
	       (connection_event_handler srv conn));
	   add_resource esys srv.group (Wait_in (Rpc_transport.descriptor conn.trans), (-1.0));
	   add_close_action
	     srv.esys
	     srv.group
	     (d, (fun d' ->
		    if shall_i_close then close_conn conn d';
		    if srv.prot = Tcp then srv.onclose conn.conn_id;
		 ))
        )
	srv.slaves;

      srv
    with
      any ->
	(match conn with Descriptor _ -> () | _ -> Unix.close descr);
	raise any

  (*****)

let get_event_system a_session =
    a_session.server.whole_server.esys

let get_connection_id a_session =
    a_session.sess_conn_id

let get_xid a_session =
    a_session.client_id

let get_socket_name a_session =
    a_session.sockaddr

let get_peer_name a_session =
    a_session.peeraddr

let get_conn_socket_name conn_id = conn_id # socket_name

let get_conn_peer_name conn_id = conn_id # peer_name

let get_main_socket_name srv =
  match srv.master with
      None ->
	( match srv.slaves with
	      [ s, _ ] -> Unix.getsockname s
	    | _ -> failwith "Rpc_server.get_main_socket_name"
	)
    | Some s -> Unix.getsockname s


let get_server sess = sess.server.whole_server

let get_user sess = sess.auth_user

let get_auth_method sess = sess.auth_method

  (*****)

let reply a_session result_value =
    let conn = a_session.server in
    let srv = conn.whole_server in
    if not conn.ready then raise Connection_lost;

    let reply = Rpc_packer.pack_successful_reply
	srv.prog a_session.procname a_session.client_id
        a_session.auth_ret_flav a_session.auth_ret_data
        result_value
    in

    let reply_session =
      { a_session with
	  parameter = XV_void;
	  result = reply
      }
    in
    Queue.add reply_session conn.replies;

    let d = Rpc_transport.descriptor conn.trans in
    if not (exists_resource srv.esys (Wait_out d)) then begin
      add_resource srv.esys srv.group (Wait_out d, (-1.0));
      conn.has_output_resource <- true
    end


let reply_error a_session condition =
    let conn = a_session.server in
    let srv = conn.whole_server in
    if not conn.ready then raise Connection_lost;

    let reply =
      match condition with
	  Unavailable_program
	| Unavailable_version(_,_)
	| Unavailable_procedure
	| Garbage
	| System_err ->
	    Rpc_packer.pack_accepting_reply
	      a_session.client_id
	      a_session.auth_ret_flav a_session.auth_ret_data
              condition
	| _ ->
	    Rpc_packer.pack_rejecting_reply
	      a_session.client_id condition
    in

    let reply_session =
      { a_session with
	  parameter = XV_void;
	  result = reply
      }
    in
    Queue.add reply_session conn.replies;

    let d = Rpc_transport.descriptor conn.trans in
    if not (exists_resource srv.esys (Wait_out d)) then begin
      add_resource srv.esys srv.group (Wait_out d, (-1.0));
      conn.has_output_resource <- true
    end


let set_exception_handler srv eh =
  srv.exception_handler <- eh

let set_onclose_action srv a =
  srv.onclose <- a

let set_session_filter_2 srv f =
  srv.filter <- f

let set_session_filter srv f =
  srv.filter <- (fun addr conn_id -> f addr)

let set_auth_methods srv l =
  let h = Hashtbl.create 20 in
  let p = ref [] in
  List.iter
    (fun m ->
       List.iter (fun name -> Hashtbl.add h name m) m#flavors;
       match m # peek with
	   Some f -> p := !p @ [ f, m ];
	 | None -> ()
    )
    l;
  srv.auth_methods <- h;
  srv.auth_peekers <- !p

let stop_server srv =
  (* old implementation:  raise (Abort(srv.group, Stop_server)) *)
  schedule_server_termination_event srv;
  Unixqueue.once srv.esys srv.group 0.0 srv.unmap_port

let stop_connection srv conn_id =
  if srv.prot = Tcp then (
    schedule_termination_event srv conn_id;
  )

let verbose b =
  debug := b
