/**
  \class CDiffPanel
  \brief Show difference of two panels.

  This panel takes 2 panels as an argument, substracts panel2 from panel1
  and divides the result by 2, then adds 128.

*/

#if HAVE_CONFIG_H
#include "config.h"
#endif

#include "DiffPanel.h"

#ifdef HAVE_INTELCPU
#include "video_asm.h"
#endif


CDiffPanel::CDiffPanel(CCamPanel *p1, CCamPanel *p2, int scale,
                       const char *name, const char *desc)
	: CCamPanel(name, desc, YUV420)
{
   pPanel1 = p1;
   pPanel2 = p2;
   Scale = scale;
   if (Scale < 1) Scale = 1;

   if (pPanel1 == NULL || pPanel2 == NULL)
      return;
   if (pPanel1->GetPanelType() != YUV420 ||
       pPanel1->GetPanelType() != YUV420) {
     pPanel1 = NULL;
     pPanel2 = NULL;
     return; // Should be exception
   }  
     
   ConnectUsage(pPanel1);
   ConnectUsage(pPanel2);
   ConnectResizes(pPanel1);
   SetSize(pPanel1->GetImageSize());
}



// public

void CDiffPanel::UpdatePanel()
{
   void *dst, *src1, *src2;
   
   if (pPanel1 == NULL || pPanel2 == NULL)
     return;

   dst = ImgY.bits();
   src1 = pPanel1->GetImage(0).bits();
   src2 = pPanel2->GetImage(0).bits();
   
#ifdef HAVE_INTELCPU
   calc_diff128(Pixels, dst, src1, src2);
#else      
   // Non-assembly version
   int i;
   uchar *d, *s1, *s2;

   d = (uchar *)dst;
   s1 = (uchar *)src1;
   s2 = (uchar *)src2;

   for (i = 0; i < Pixels; i++) {
      *d = (128 + (*s1 >> 1) - (*s2 >> 1));
      d++;
      s1++;
      s2++;
   }
#endif

}


// public

/**
  \fn void CDiffPanel::SetSize(const QSize &new_size)
  \brief [overloaded] Set size & set palette.
  
  This function is overloaded because after the default SetSize, it
  creates a new palette that take the Scale into effect (in other words,
  the scale is only done for viewing, not the actual calculation).
*/

void CDiffPanel::SetSize(const QSize &new_size)
{
   CCamPanel::SetSize(new_size);
   Pixels = image_w * image_h;
   if (image_w && image_h) {
     // The scale is only in the viewing. not the actual calculation
     for (int c = 0; c < 256; c++) {
        int l;
        
        l = 128 + Scale * (c - 128);
        if (l < 0)   l = 0;
        if (l > 255) l = 255;
        ImgY.setColor(c, qRgb(l, l, l));
        ImgU.setColor(c, qRgb(l, l, l));
        ImgV.setColor(c, qRgb(l, l, l));
     }
   }
}
