/* the cantus project.
 * (c)2002 by Samuel Abels (sam@manicsadness.com)
 * This project's homepage is: http://software.manicsadness.com/cantus
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.

 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <gnome.h>
#include <dirent.h>
#include <sys/stat.h>
#include "memory.h"
#include "gui_dirtree.h"

/***************************************************************************************
 * BELOW FOLLOW THE STATICS
 ***************************************************************************************/

static PixMap*
pixmap_create(gchar *name)
{
	extern GtkWidget *cantus;
	PixMap *ppixmap = malloc(sizeof(PixMap));
	
	gchar *filename = NULL;
	GdkColormap *colormap = NULL;

	colormap = gtk_widget_get_colormap(cantus);

	if( strcmp(name, "opened") == 0 )
		filename = g_strconcat(PACKAGE_PIXMAPS_DIR, "/folder_opened.xpm", NULL);
	if( strcmp(name, "closed") == 0 )
		filename = g_strconcat(PACKAGE_PIXMAPS_DIR, "/folder_closed.xpm", NULL);
	if( strcmp(name, "locked") == 0 )
		filename = g_strconcat(PACKAGE_PIXMAPS_DIR, "/folder_locked.xpm", NULL);
	
	ppixmap->pixmap = gdk_pixmap_colormap_create_from_xpm(NULL, colormap, &(ppixmap->mask), NULL, filename);
	
	free(filename);
	return(ppixmap);	
}

// TRUE if directory, FALSE if not.
static gboolean
is_dir(gchar *filepathname)
{
	struct stat pstat;
	if(stat(filepathname, &pstat) < 0)
		return(FALSE);
	
	if( S_ISDIR(pstat.st_mode) != 0 )
		return(TRUE);
	
	return(FALSE);
}

/***************************************************************************************
 * END OF STATICS
 ***************************************************************************************/





/**********************************************************************
 * Add the rootdir to the directory-ctree
 * Arguments: ctree: The ctree to put this rootdir in.
 *            data: A glist with pointers to alloced data
 **********************************************************************/
void
ctree_init_dirtree(GtkCTree *ctree)
{
	GtkCTreeNode *parentnode = NULL;
	GtkCTreeNode *dummynode = NULL;
	gchar *column[1];
	PixMap *closed = pixmap_create("closed");
	PixMap *opened = pixmap_create("opened");
	

	gtk_ctree_set_indent(ctree, 10);
	setenv("PWD", "/", 1);

// Init parent node ("/")
	column[0] = strdup("/");
// Insert parentnode into ctree
	parentnode = gtk_ctree_insert_node(ctree, NULL, NULL, column, 5, closed->pixmap, closed->mask, opened->pixmap, opened->mask, FALSE, FALSE);
	gtk_ctree_node_set_row_data(ctree, parentnode, column[0]);
	free(closed);
	free(opened);

// and insert one dummy node, so the "+" appears.
	column[0] = strdup("/");
	dummynode = gtk_ctree_insert_node(ctree, parentnode, NULL, column, 0, NULL, NULL, NULL, NULL, FALSE, FALSE);
// append a "*" as dummy-marker
	sprintf(column[0], "*");
	gtk_ctree_node_set_row_data(ctree, dummynode, column[0]);
}






/**********************************************************************
 * Add a branch to the directory tree
 * Arguments: ctree: the ctree in which to expand the branch.
 *            row:   the row number of the item to expand.
 *            data:  A glist with pointers to alloced data
 **********************************************************************/
void
ctree_expand_dirtree(GtkCTree *ctree, gint row, gboolean showhidden)
{
	GtkCTreeNode *node = NULL;
	GtkCTreeNode *subnode = NULL;
	gchar *column[1];

	PixMap *closed = NULL;
	PixMap *opened = NULL;
	PixMap *locked = NULL;
	gboolean usedclosedpixmap = FALSE;
	gboolean usedopenedpixmap = FALSE;
	gboolean usedlockedpixmap = FALSE;
	
	struct dirent *dirstruct = NULL;
	DIR *dirstream = NULL, *subdirstream = NULL;

	gchar *directory = NULL;
	gchar *subdirectory = NULL;

// Delete dummy entry (if present). This will collapse the expanded branch!!
// if no dummy entry is present, then there are no subdirectories, so return.
	node = gtk_ctree_node_nth(ctree, row + 1);
	directory = (char *)gtk_ctree_node_get_row_data(ctree, node);
	if( directory && *directory != '*' )
		return;

	free(directory);
	gtk_ctree_remove_node (ctree, node);
	gtk_clist_freeze (GTK_CLIST(ctree));

// Then, get the directory name, which is attached to the node as data
 	node = gtk_ctree_node_nth(ctree, row);
	directory = (char *)gtk_ctree_node_get_row_data(ctree, node);

// open directory or fatal error.
	if ( (dirstream = opendir(directory)) == NULL ) 
	{
		printf("Error opening requested path ('%s')!\n", directory);
		exit(1);
	}

// Create pixmaps
	closed = pixmap_create("closed");
	opened = pixmap_create("opened");
	locked = pixmap_create("locked");
	
// Add each child-dir to the node.
	while ((dirstruct = readdir(dirstream)) != NULL)
	{
// Get next name from stream and create a fullpath.
		subdirectory = malloc(strlen(directory) + strlen(dirstruct->d_name) + 2);
		sprintf(subdirectory, "%s%s%c", directory, dirstruct->d_name, 0);
		
		if( is_dir(subdirectory)
		    && memcmp(dirstruct->d_name, "..\0", 3)
		    && memcmp(dirstruct->d_name, ".\0", 2)
			&& (strncmp(dirstruct->d_name, ".", 1) != 0 || showhidden) )
		{
// If this is a valid directory, we allocate mem for the tree entry and store the dirname there.
			column[0] = strdup(dirstruct->d_name);
// If we have no read access for the file, we must place an alternate pixmap there.
			if( access(subdirectory, R_OK) == 0 )
			{
				usedclosedpixmap = TRUE;
				usedopenedpixmap = TRUE;
				subnode = gtk_ctree_insert_node(GTK_CTREE (ctree), node, NULL, column, 5, closed->pixmap, closed->mask, opened->pixmap, opened->mask, FALSE, FALSE);
			}
			else
			{
				usedlockedpixmap = TRUE;
				subnode = gtk_ctree_insert_node(GTK_CTREE (ctree), node, NULL, column, 5, locked->pixmap, locked->mask, locked->pixmap, locked->mask, FALSE, FALSE);
			}
			free(column[0]);
			
// now we append the allocated mem containing the dirname to the newmade node (with an added slash).
			strncpy(subdirectory + strlen(subdirectory), "/", 1);
			gtk_ctree_node_set_row_data(ctree, subnode, subdirectory);
			
// so the childnode is inserted now, look for the childs subdirs. if any subdirs exist, enable the "+" box by adding a dummy-subnode.
			if( (subdirstream = opendir(subdirectory)) != NULL )
			{
				struct dirent *subdirstruct = NULL;
				GtkCTreeNode *dummynode = NULL;
					
				while ( (subdirstruct = readdir(subdirstream)) != NULL )
				{
					gchar fullfilename[strlen(subdirectory) + strlen(subdirstruct->d_name) + 1];
					sprintf(fullfilename ,"%s%s", subdirectory, subdirstruct->d_name);
					
					if ( is_dir(fullfilename)
					    && memcmp(subdirstruct->d_name, "..\0", 3)
		    			&& memcmp(subdirstruct->d_name, ".\0", 2) )
					{
// if the child has a subdir, append a dummynode.
						column[0] = strdup(dirstruct->d_name);
						dummynode = gtk_ctree_insert_node(GTK_CTREE (ctree), subnode, NULL, column, 0, NULL, NULL, NULL, NULL, FALSE, FALSE);
						free(column[0]);
						
// and append a "*" as dummy-marker.
						column[0] = strdup("*");
						gtk_ctree_node_set_row_data(ctree, dummynode, column[0]);
						
						break;
					}
				}
				closedir(subdirstream);
			}
		}
		else
// if the subdirectory was none, we must free the data!
			free(subdirectory);
	}
	closedir(dirstream);

// Cleanup...
	if(!usedopenedpixmap)
	{
		gdk_pixmap_unref (opened->pixmap);
		gdk_bitmap_unref (opened->mask);
	}
	if(!usedclosedpixmap)
	{
		gdk_pixmap_unref (closed->pixmap);
		gdk_bitmap_unref (closed->mask);
	}
	if(!usedlockedpixmap)
	{
		gdk_pixmap_unref (locked->pixmap);
		gdk_bitmap_unref (locked->mask);
	}
	free(closed);
	free(opened);
	free(locked);
	
	gtk_ctree_sort_recursive(ctree, node);
	gtk_ctree_expand(ctree, node);
	gtk_clist_thaw(GTK_CLIST(ctree));
}




void
remove_node (GtkCTree *ctree, GtkCTreeNode *node, void *null)
{
	gpointer data;
	
	g_return_if_fail (GTK_IS_CTREE(ctree));
	g_return_if_fail (node != NULL);
	
	data = gtk_ctree_node_get_row_data (ctree, node);
	if( data != NULL )
		free (data);
	
	gtk_ctree_remove_node (ctree, node);
}




/**********************************************************************
 * Remove a branch of the directory tree
 * Arguments: ctree: the ctree in which to collapse a branch.
 *            row:   the row number of the item to colapse.
 *            data:  A glist with pointers to alloced data
 **********************************************************************/
void
ctree_collapse_dirtree (GtkCTree *ctree, GtkCTreeNode *node)
{
	GtkCTreeNode *children = NULL;
	GtkCTreeNode *curchild = NULL;
	GtkCTreeNode *dummynode = NULL;
	gchar *column[1];

	g_return_if_fail (GTK_IS_CTREE(ctree));
	g_return_if_fail (node != NULL);
	
	gtk_clist_freeze (GTK_CLIST(ctree));
	
// get the first nodes children.
	children = GTK_CTREE_ROW(node)->children;
// Remove all children
	while( children != NULL )
	{
		curchild = children;
		children = GTK_CTREE_ROW(children)->sibling;
		gtk_ctree_post_recursive (ctree, curchild, (GtkCTreeFunc)remove_node, NULL);
	}
	
// and add a dummy node again.
	column[0] = strdup ("*");
	dummynode = gtk_ctree_insert_node (GTK_CTREE(ctree), node, NULL, column, 0, NULL, NULL, NULL, NULL, FALSE, FALSE);
	free(column[0]);
	column[0] = strdup ("*");
	gtk_ctree_node_set_row_data (ctree, dummynode, column[0]);
	
	gtk_clist_thaw(GTK_CLIST(ctree));
}





/**********************************************************************
 * Destroy the directory tree
 * Arguments: ctree: the ctree to destroy.
 **********************************************************************/
void
ctree_destroy_dirtree (GtkCTree *ctree, GtkCTreeNode *node)
{
	GtkCTreeNode *child = NULL;
	
	g_return_if_fail (GTK_IS_CTREE(ctree));
	g_return_if_fail (node != NULL);
	
// Remove all children
	ctree_collapse_dirtree (ctree, node);
	
// Collapse added a dummy node. Remove it.
	child = GTK_CTREE_ROW(node)->children;
	remove_node (ctree, child, NULL);

// Remove the node itselves.
	remove_node (ctree, node, NULL);
}


void
ctree_dirtree_open_directory(GtkCTree *ctree, gchar *directory)
{
	GtkCTreeNode *node = NULL;
	gchar *pdirectory = NULL;
	gchar subdir[2048] = "\0";
	gchar curdir[4096] = "\0";
	gchar curdircpy[4096] = "\0";
	gint i = -1;
	GList *dirs = NULL;
	
	if(!directory)
		return;
	
// append a slash, if there is none yet.
	if( *(directory + strlen(directory) - 1) != '/' )
		*(directory + strlen(directory)) = '/';
	
	gtk_clist_freeze(GTK_CLIST(ctree));
	pdirectory = directory;
// make a list of all directories ill have to open, e.g. for /home/sam it mus be a list of "/", "/home/" and "/home/sam/"
	while(strchr(pdirectory, '/'))
	{
		strncpy(subdir, pdirectory, 2047);
		*(strchr(subdir, '/') + 1) = '\0';
		pdirectory = strchr(pdirectory, '/') + 1;
		
		strncpy(curdircpy, curdir, 4096);
		snprintf(curdir, 4096, "%s%s", curdircpy, subdir);
		dirs = g_list_append(dirs, strdup(curdir));
	}
	
	while(dirs)
	{
		i = -1;
		while( (node = gtk_ctree_node_nth(ctree, ++i)) )
		{
			if(strcmp((gchar *)dirs->data, gtk_ctree_node_get_row_data(ctree, node)) == 0)
			{
				gtk_ctree_expand(ctree, node);
// if this is the last directory, we select it in the ctree
				if(!dirs->next)
				{
					gtk_ctree_select(ctree, node);
					gtk_ctree_node_moveto(ctree, node, 0, 0.3, 0);
				}
			}
		}
		free(dirs->data);
		dirs = dirs->next;
	}
	
	g_list_free(dirs);
	gtk_clist_thaw(GTK_CLIST(ctree));
}




