/* The Cantus project.
 * (c)2002, 2003, 2004 by Samuel Abels (spam debain org)
 * This project's homepage is: http://www.debain.org/cantus
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.

 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include "controller.h"

//#define _DEBUG_


/******************************************************************************
 * Constructor/Destructor
 ******************************************************************************/
/* Constructor.
 */
Controller::Controller(void)
{
  Glib::thread_init();
  
  pluginhandler   = new PluginHandler;
  fileinfomanager = new FileInfoManager;
  preferences     = new Preferences;
  configfile      = new ConfigFile;
  eventbus        = new EventBus;
  
  eventbus->add_listener_sigc(
      "GUI:Filelist:Selection:Changed",
      SigC::slot(*this, &Controller::on_filelist_selection_changed));
  eventbus->add_listener_sigc(
      "Filelist:Save:Request",
      SigC::slot(*this, &Controller::on_filelist_save_request));
  
  pluginhandler->signal_plugin_loaded.connect(
      SigC::slot(*this, &Controller::on_plugins_registered));
  
  fileinfomanager->signal_file_read_finished.connect(
      SigC::slot(*this, &Controller::emit_file_read_finished_event));
  fileinfomanager->signal_file_read_failed.connect(
      SigC::slot(*this, &Controller::emit_file_read_failed_event));
  fileinfomanager->signal_file_write_failed.connect(
      SigC::slot(*this, &Controller::emit_file_write_failed_event));
  fileinfomanager->signal_file_rename_finished.connect(
      SigC::slot(*this, &Controller::emit_file_rename_finished_event));
  fileinfomanager->signal_file_rename_failed.connect(
      SigC::slot(*this, &Controller::emit_file_rename_failed_event));
  fileinfomanager->signal_queue_read_finished.connect(
      SigC::slot(*this, &Controller::emit_filenames_read_finished_event));
  
  preferences->signal_changed.connect(
      SigC::slot(*this, &Controller::emit_preferences_changed_event));
}


/* Destructor.
 */
Controller::~Controller(void)
{
#ifdef _DEBUG_
  std::cout << "Controller::~Controller(): Called.\n";
#endif
  delete pluginhandler;
#ifdef _DEBUG_
  std::cout << "Controller::~Controller(): PluginHandler deleted.\n";
#endif
  delete fileinfomanager;
#ifdef _DEBUG_
  std::cout << "Controller::~Controller(): FileInfoManager deleted.\n";
#endif
  delete preferences;
#ifdef _DEBUG_
  std::cout << "Controller::~Controller(): Preferences deleted.\n";
#endif
  delete configfile;
#ifdef _DEBUG_
  std::cout << "Controller::~Controller(): ConfigFile deleted.\n";
#endif
  delete eventbus;
#ifdef _DEBUG_
  std::cout << "Controller::~Controller(): EventBus deleted.\n";
#endif
}


/******************************************************************************
 * Public
 ******************************************************************************/
/* Loads the preferences from the configfile.
 */
int Controller::configfile_load(void)
{
  int err = configfile->load();
  configfile->foreach_line(SigC::slot(*preferences, &Preferences::set));
  return err;
}


/* Loads the preferences from the configfile.
 */
int Controller::configfile_save(void)
{
  configfile->clear();
  preferences->foreach(SigC::slot(*configfile, &ConfigFile::append));
  return configfile->save();
}


/******************************************************************************
 * Private
 ******************************************************************************/
/* Triggers eventbus events and queues the files to the FileInfoManager.
 */
void Controller::on_filelist_selection_changed(void *filenames)
{
#ifdef _DEBUG_
  std::cout << "Controller::on_filelist_selection_changed(): Called.\n";
#endif
  // Get a list of all required plugins and emit it as an event.
  std::list<Plugin*> *resp = pluginhandler->get_responsible_plugins_from_list(
                                                             (GList*)filenames);
  eventbus->emit_event_with_pointer("Plugin:Responsibility:Changed",
                                   (gpointer)resp);
  // Destroy the list.
  std::list<Plugin*>::iterator iter;
  for (iter = resp->begin(); iter != resp->end(); iter++)
    (*iter)->unref();
  delete resp;
  
  eventbus->emit_event_with_pointer("Filelist:Read:Start", filenames);
  
#ifdef _DEBUG_
  std::cout << "Controller::on_filelist_selection_changed(): "
            << "Triggering read.\n";
#endif
  // Trigger the fileinfomanager to start reading all files.
  fileinfomanager->readqueue_set((GList*)filenames);
}


/* Whenever a new plugin has been registered, we need to update the
 * FileInfoManager.
 */
void Controller::on_plugins_registered(Plugin *plugin)
{
  fileinfomanager->register_plugins(pluginhandler->get_plugins());
}


/* This signal indicates that some file's data needs to be written to the
 * filesystem. May, for example, be triggered by a plugin when a "save" button
 * has been pressed.
 */
void Controller::on_filelist_save_request(void *list)
{
#ifdef _DEBUG_
  std::cout << "Controller::on_filelist_save_request(): Called.\n";
#endif
  fileinfomanager->writequeue_append((GList*)list);
}


/* Emits an eventbus event.
 */
void Controller::emit_file_read_finished_event(FileInfo *info)
{
#ifdef _DEBUG_
  std::cout << "Controller::emit_file_read_finished_event(): Called.\n";
#endif
  info->lock();
  eventbus->emit_event_with_pointer("File:Read:Finished",
                                    info->get_edited_hash());
  info->unlock();
}


/* Emits an eventbus event.
 */
void Controller::emit_file_read_failed_event(FileInfo *info)
{
#ifdef _DEBUG_
  std::cout << "Controller::emit_file_read_failed_event(): Called.\n";
#endif
  info->lock();
  eventbus->emit_event_with_pointer("File:Read:Failed",
                                    (void*)info->get_filename());
  info->unlock();
}


/* Emits an eventbus event.
 */
void Controller::emit_file_write_failed_event(FileInfo *info)
{
#ifdef _DEBUG_
  std::cout << "Controller::emit_file_write_failed_event(): Called.\n";
#endif
  info->lock();
  eventbus->emit_event_with_pointer("File:Write:Failed",
                                    (void*)info->get_filename());
  info->unlock();
}


/* Emits an eventbus event.
 */
void Controller::emit_file_rename_finished_event(FileInfo *info)
{
#ifdef _DEBUG_
  std::cout << "Controller::emit_file_renamed_event(): Called.\n";
#endif
  info->lock();
  eventbus->emit_event_with_pointer("File:Rename:Finished",
                                    (void*)info->get_filename());
  info->unlock();
}


/* Emits an eventbus event.
 */
void Controller::emit_file_rename_failed_event(FileInfo *info)
{
#ifdef _DEBUG_
  std::cout << "Controller::emit_file_rename_failed_event(): Called.\n";
#endif
  info->lock();
  eventbus->emit_event_with_pointer("File:Rename:Failed",
                                    (void*)info->get_filename());
  info->unlock();
}


/* Emits an eventbus event.
 */
void Controller::emit_filenames_read_finished_event(void)
{
#ifdef _DEBUG_
  std::cout << "Controller::emit_filenames_read_finished_event(): Called.\n";
#endif
  eventbus->emit_event_with_pointer("Filelist:Read:Finished", NULL);
}


/* Emits an eventbus event.
 */
void Controller::emit_preferences_changed_event(std::string key)
{
#ifdef _DEBUG_
  std::cout << "Controller::emit_preferences_changed_event(): Called.\n";
#endif
  eventbus->emit_event_with_pointer("Preferences:Changed", NULL);
}
