/* The id3v1 library.
 * (c)2002, 2003, 2004 by Samuel Abels (spam debain org)
 * This project's homepage is: http://www.debain.org/cantus
 *
 * This library is designed for easyest possible access to id3 V1 tags.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.

 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include "lib_id3genre.h"


/******************************************************************************
 * STATICS
 ******************************************************************************/
/*
 * Split a string that is delimited by any of the chars in "delimiters"
 * (max 1000) into an array.
 */
static char **explode(const char *string, const char *delimiters)
{
  char **strlist = (char**)calloc(1000, sizeof(char));
  char *stringcp = string ? strdup(string) : NULL;
  int i = 0;
  
  if (!string || *string == '\0')
    return(strlist);
  
  /* Init strtok. */
  strlist[0] = strdup(strtok(stringcp, delimiters));
  
  /* Split into pieces. */
  while ((strlist[++i] = strtok(NULL, delimiters)))
    strlist[i] = strdup(strlist[i]);
  
  free(stringcp);
  
  return(strlist);
}


/*
 * Free all elements of an array.
 */
static void free_array(char **array)
{
  int i = -1;
  
  while (array[++i])
    free(array[i]);
  
  free(array);
}


/*
 * Find the given string in any element of an array. Returns a pointer to the
 * element.
 */
static const char *array_element_case_insensitive(const char **array,
                                                  const char *needle)
{
  int i = -1;
  
  while (array[++i])
    if (strcasecmp(array[i], needle) == 0)
      return(array[i]);
  
  return(NULL);
}


/*
 * This function will strip leading/trailing/both spaces in "source".
 * Spaces are:
 * Single space, tab, vertical tab, form feed, carriage return, or newline.
 * loc: 0-start 1-end 2-both
 */ 
static void string_wipe_spaces(char *source, int loc) 
{
  char *psource = NULL;
  char *destination = (char*)calloc(2000, sizeof(char));
  char *temp = destination;
  
  if (source != NULL) {
    if (loc == 0 || loc == 2) {
      strcpy(destination, source);
      while (*destination == ' ')
        destination++;
      sprintf(source, "%s", destination);
    }
    if (loc == 1 || loc == 2) {
      for (psource = source + strlen(source) - 1; *psource != '\0'; psource--) {
        if (!isspace(*psource))
          break;
        *psource = '\0';
      }
    }
  }
  
  free(temp);
  return;
}


static char *str_append(char *str, char *append1, char *append2)
{
  char *newstr = (char*)malloc(strlen(str)
                                + strlen(append1)
                                + strlen(append2) + 1);
  sprintf(newstr, "%s%s%s", str, append1, append2);
  return(newstr);
}
/******************************************************************************
 * END STATICS
 ******************************************************************************/


int genre_string_2_v1genre(const char *pstring)
{
  char *string = strdup(pstring);
  char *genre_uc = NULL;
  int i = -1, f = 0;
  
  /* Uppercase the genre string. */
  f = -1;
  while (*(string + ++f) != '\0')
    *(string + f) = toupper(*(string + f));
  
  while (genre[++i]) {
    genre_uc = strdup(genre[i]);
    
    /* Uppercase the current genre. */
    f = -1;
    while (*(genre_uc + ++f) != '\0')
      *(genre_uc + f) = toupper(*(genre_uc + f));
    
    if (strstr(string, genre_uc) != NULL) {
      free(genre_uc);
      return(i);
    }
    
    free(genre_uc);
  }
  
  return(-1);
}


char *genre_string_2_v2genre(const char *string)
{
  char **genrelist = explode(string, ",");
  char *curgenre = NULL;
  char *pnewgenre = NULL;
  char *newgenre = NULL;
  char v1genres[2048];
  int i = -1;
  
  if (genrelist[0] == NULL) {
    free_array(genrelist);
    return(NULL);
  }
  
  memset(v1genres, 0, sizeof(v1genres));
  
  /* Parse through all the user's genres and create a string of all */
  /* V1 genres found. */
  while (genrelist[++i]) {
    string_wipe_spaces(genrelist[i], 2);
    
    /* Is the user genre a valid V1 genre? */
    if (!(curgenre = (char*)array_element_case_insensitive((const char**)genre,
                                                           genrelist[i])))
      continue;
    
    *genrelist[i] = '\0';
    snprintf(v1genres, 2047, "%s(%i)", v1genres, i);
  }
  
  newgenre = strdup(v1genres);
  
  /* Merge the array into one string, seperated by commas. */
  i = -1;
  while (genrelist[++i] && *genrelist[i] == '\0');
  if (!genrelist[i]) {
    free_array(genrelist);
    return(newgenre);
  }
  
  free(v1genres);
  
  newgenre = str_append(v1genres, ", ", genrelist[i]);
  while (genrelist[++i]) {
    if (*genrelist[i] == '\0')
      continue;
    
    pnewgenre = newgenre;
    newgenre = str_append(newgenre, ", ", genrelist[i]);
    free(pnewgenre);
  }
  
  free_array(genrelist);
  
  return(NULL);
}
