/*
	Chorus.cc
	
	Copyright 2004, 2005 Tim Goetze <tim@quitte.de>
	
	http://quitte.de/dsp/

	mono and mono-to-stereo chorus units.
	
*/
/*
	This program is free software; you can redistribute it and/or
	modify it under the terms of the GNU General Public License
	as published by the Free Software Foundation; either version 2
	of the License, or (at your option) any later version.

	This program is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.

	You should have received a copy of the GNU General Public License
	along with this program; if not, write to the Free Software
	Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
	02111-1307, USA or point your web browser to http://www.gnu.org.
*/

#include "basics.h"

#include "Chorus.h"
#include "Descriptor.h"

template <sample_func_t F>
void
Chorus::one_cycle (int frames)
{
	d_sample * s = ports[0];

	double one_over_n = 1 / (double) frames;
	double ms = .001 * fs;

	double t = time;
	time = *ports[1] * ms;
	double dt = (time - t) * one_over_n;

	double w = width;
	width = *ports[2] * ms;
	/* clamp, or we need future samples from the delay line */
	if (width >= t - 3) width = t - 3;
	double dw = (width - w) * one_over_n;

	if (rate != *ports[3]) 
		lfo.set_f (max (rate = *ports[3], .000001), fs, lfo.get_phase());
			
	double blend = *ports[4];
	double ff = *ports[5];
	double fb = *ports[6];

	d_sample * d = ports[7];

	/* flip 'renormal' addition constant */
	normal = -normal;

	DSP::FPTruncateMode truncate;

	for (int i = 0; i < frames; ++i)
	{
		d_sample x = s[i];

		/* truncate the feedback tap to integer, better quality for less
		 * cycles (just a bit of zipper when changing 't', but it does sound
		 * interesting) */
		int ti;
		fistp (t, ti);
		x -= fb * delay[ti];

		delay.put (x + normal);

#		if 0
		/* allpass delay sounds a little cleaner for a chorus
		 * but sucks big time when flanging. */
		x = blend * x + ff * tap.get (delay, t + w * lfo.get());
#		elif 0
		/* linear interpolation */
		x = blend * x + ff * delay.get_at (t + w * lfo.get());
#		else
		/* cubic interpolation */
		x = blend * x + ff * delay.get_cubic (t + w * lfo.get());
#		endif

		F (d, i, x, adding_gain);

		t += dt;
		w += dw;
	}
}

/* //////////////////////////////////////////////////////////////////////// */

PortInfo
Chorus::port_info [] =
{
	{
		"in",
		INPUT | AUDIO,
		{BOUNDED, -1, 1}
	}, {
		"t (ms)",
		INPUT | CONTROL,
		{BOUNDED | LOG | DEFAULT_LOW, 2.5, 40}
	}, {
		"width (ms)",
		INPUT | CONTROL,
		{BOUNDED | DEFAULT_1, .5, 10}
	}, {
		"rate (Hz)",
		INPUT | CONTROL,
		{BOUNDED | DEFAULT_LOW, 0, 5}
	}, {
		"blend",
		INPUT | CONTROL,
		{BOUNDED | DEFAULT_1, 0, 1}
	}, {
		"feedforward",
		INPUT | CONTROL,
		{BOUNDED | DEFAULT_LOW, 0, 1}
	}, {
		"feedback",
		INPUT | CONTROL,
		{BOUNDED | DEFAULT_MID, 0, 1}
	}, {
		"out",
		OUTPUT | AUDIO,
		{0}
	}
};

void
Descriptor<Chorus>::setup()
{
	UniqueID = 1767;
	Label = "Chorus";
	Properties = HARD_RT;

	Name = "Mono chorus/flanger";
	Maker = "Tim Goetze <tim@quitte.de>";
	Copyright = "GPL, 2004";

	/* fill port info and vtable */
	autogen();
}

/* //////////////////////////////////////////////////////////////////////// */

#define F0(fv) (((*(unsigned int*)&(fv))&0x7f800000)==0)?0.0f:(fv)

template <sample_func_t F>
void
StereoChorus::one_cycle (int frames)
{
	d_sample * s = ports[0];

	double one_over_n = 1 / (double) frames;
	double ms = .001 * fs;

	double t = time;
	time = *ports[1] * ms;
	double dt = (time - t) * one_over_n;

	double w = width;
	width = *ports[2] * ms;
	/* clamp, or we need future samples from the delay line */
	if (width >= t - 1) width = t - 1;
	double dw = (width - w) * one_over_n;

	if (rate != *ports[3] && phase != *ports[4]) 
	{
		rate = *ports[3];
		phase = *ports[4];
		double phi = left.lfo.get_phase();
		left.lfo.set_f (max (rate, .000001), fs, phi);
		right.lfo.set_f (max (rate, .000001), fs, phi + phase * M_PI);
	}

	double blend = *ports[5];
	double ff = *ports[6];
	double fb = *ports[7];

	d_sample * dl = ports[8];
	d_sample * dr = ports[9];

	/* flip 'renormal' addition constant */
	normal = -normal;

	/* to go sure (on i386) that the fistp instruction does the right thing 
	 * when looking up fractional sample indices */
	DSP::FPTruncateMode truncate;

	for (int i = 0; i < frames; ++i)
	{
		d_sample x = s[i];

		/* truncate the feedback tap to integer, better quality for less
		 * cycles (just a bit of zipper when changing 't', but it does sound
		 * interesting) */
		int ti;
		fistp (t, ti);
		x -= fb * delay[ti];

		delay.put (x + normal);

		d_sample l = blend * x + ff * delay.get_cubic (t + w * left.lfo.get());
		d_sample r = blend * x + ff * delay.get_cubic (t + w * right.lfo.get());

		F (dl, i, l, adding_gain);
		F (dr, i, r, adding_gain);

		t += dt;
		w += dw;
	}
}

/* //////////////////////////////////////////////////////////////////////// */

PortInfo
StereoChorus::port_info [] =
{
	{
		"in",
		INPUT | AUDIO,
		{BOUNDED, -1, 1}
	}, {
		"t (ms)",
		INPUT | CONTROL,
		{BOUNDED | DEFAULT_MIN, 2.5, 40}
	}, {
		"width (ms)",
		INPUT | CONTROL,
		{BOUNDED | DEFAULT_1, .5, 10}
	}, {
		"rate (Hz)",
		INPUT | CONTROL,
		{BOUNDED | DEFAULT_LOW, 0, 5}
	}, {
		"phase",
		INPUT | CONTROL,
		{BOUNDED | DEFAULT_MAX, 0, 1}
	}, {
		"blend",
		INPUT | CONTROL,
		{BOUNDED | DEFAULT_1, 0, 1}
	}, {
		"feedforward",
		INPUT | CONTROL,
		{BOUNDED | DEFAULT_LOW, 0, 1}
	}, {
		"feedback",
		INPUT | CONTROL,
		{BOUNDED | DEFAULT_MID, 0, 1}
	}, {
		"out:l",
		OUTPUT | AUDIO,
		{0}
	}, {
		"out:r",
		OUTPUT | AUDIO,
		{0}
	}
};

void
Descriptor<StereoChorus>::setup()
{
	UniqueID = 1768;
	Label = "StereoChorus";
	Properties = HARD_RT;

	Name = "Stereo chorus with feedback";
	Maker = "Tim Goetze <tim@quitte.de>";
	Copyright = "GPL, 2004";

	/* fill port info and vtable */
	autogen();
}

