/*
  This file is part of CDO. CDO is a collection of Operators to
  manipulate and analyse Climate model Data.

  Copyright (C) 2003-2019 Uwe Schulzweida, <uwe.schulzweida AT mpimet.mpg.de>
  See COPYING file for copying and redistribution conditions.

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; version 2 of the License.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
*/

/*
   This module contains the following operators:

      Timselstat    timselrange        Time selection range
      Timselstat    timselmin          Time selection minimum
      Timselstat    timselmax          Time selection maximum
      Timselstat    timselsum          Time selection sum
      Timselstat    timselmean         Time selection mean
      Timselstat    timselavg          Time selection average
      Timselstat    timselvar          Time selection variance
      Timselstat    timselvar1         Time selection variance [Normalize by (n-1)]
      Timselstat    timselstd          Time selection standard deviation
      Timselstat    timselstd1         Time selection standard deviation [Normalize by (n-1)]
*/

#include <cdi.h>

#include "cdo_int.h"
#include "param_conversion.h"

#include "datetime.h"

void *
Timselstat(void *process)
{
  TimeStat timestat_date = TimeStat::MEAN;
  int nrecs = 0;
  int varID, levelID;
  int tsID;
  int nsets;

  cdoInitialize(process);

  // clang-format off
  cdoOperatorAdd("timselrange", func_range, 0, nullptr);
  cdoOperatorAdd("timselmin",   func_min,   0, nullptr);
  cdoOperatorAdd("timselmax",   func_max,   0, nullptr);
  cdoOperatorAdd("timselsum",   func_sum,   0, nullptr);
  cdoOperatorAdd("timselmean",  func_mean,  0, nullptr);
  cdoOperatorAdd("timselavg",   func_avg,   0, nullptr);
  cdoOperatorAdd("timselvar",   func_var,   0, nullptr);
  cdoOperatorAdd("timselvar1",  func_var1,  0, nullptr);
  cdoOperatorAdd("timselstd",   func_std,   0, nullptr);
  cdoOperatorAdd("timselstd1",  func_std1,  0, nullptr);

  const int operatorID = cdoOperatorID();
  const int operfunc = cdoOperatorF1(operatorID);

  const bool lrange  = operfunc == func_range;
  const bool lmean   = operfunc == func_mean || operfunc == func_avg;
  const bool lstd    = operfunc == func_std || operfunc == func_std1;
  const bool lvarstd = operfunc == func_std || operfunc == func_var || operfunc == func_std1 || operfunc == func_var1;
  const int  divisor = operfunc == func_std1 || operfunc == func_var1;

  operatorInputArg("nsets <noffset <nskip>>");

  const int nargc  = operatorArgc();
  const int ndates = parameter2int(operatorArgv()[0]);
  const int noffset = (nargc > 1) ? parameter2int(operatorArgv()[1]) : 0;
  const int nskip   = (nargc > 2) ? parameter2int(operatorArgv()[2]) : 0;
  // clang-format on
  const bool lvars2 = lvarstd || lrange;

  if (Options::cdoVerbose) cdoPrint("nsets = %d, noffset = %d, nskip = %d", ndates, noffset, nskip);

  CdoStreamID streamID1 = cdoOpenRead(0);

  const int vlistID1 = cdoStreamInqVlist(streamID1);
  const int vlistID2 = vlistDuplicate(vlistID1);

  const int taxisID1 = vlistInqTaxis(vlistID1);
  const int taxisID2 = taxisDuplicate(taxisID1);
  taxisWithBounds(taxisID2);
  vlistDefTaxis(vlistID2, taxisID2);

  CdoStreamID streamID2 = cdoOpenWrite(1);
  cdoDefVlist(streamID2, vlistID2);

  const int maxrecs = vlistNrecs(vlistID1);
  std::vector<RecordInfo> recList(maxrecs);

  DateTimeList dtlist;
  dtlist.setStat(timestat_date);
  dtlist.setCalendar(taxisInqCalendar(taxisID1));

  const size_t gridsizemax = vlistGridsizeMax(vlistID1);

  Field field;
  field.resize(gridsizemax);

  FieldVector2D samp1, vars1, vars2;
  fieldsFromVlist(vlistID1, samp1, FIELD_NONE);
  fieldsFromVlist(vlistID1, vars1, FIELD_VEC);
  if (lvars2) fieldsFromVlist(vlistID1, vars2, FIELD_VEC);

  for (tsID = 0; tsID < noffset; tsID++)
    {
      nrecs = cdoStreamInqTimestep(streamID1, tsID);
      if (nrecs == 0) break;

      for (int recID = 0; recID < nrecs; recID++)
        {
          cdoInqRecord(streamID1, &varID, &levelID);

          if (tsID == 0)
            {
              recList[recID].varID = varID;
              recList[recID].levelID = levelID;
              recList[recID].lconst = vlistInqVarTimetype(vlistID1, varID) == TIME_CONSTANT;
            }
        }
    }

  int otsID = 0;
  if (tsID < noffset)
    {
      cdoWarning("noffset is larger than number of timesteps!");
      goto LABEL_END;
    }

  while (true)
    {
      for (nsets = 0; nsets < ndates; nsets++)
        {
          nrecs = cdoStreamInqTimestep(streamID1, tsID);
          if (nrecs == 0) break;

          dtlist.taxisInqTimestep(taxisID1, nsets);

          for (int recID = 0; recID < nrecs; recID++)
            {
              cdoInqRecord(streamID1, &varID, &levelID);

              if (tsID == 0)
                {
                  recList[recID].varID = varID;
                  recList[recID].levelID = levelID;
                  recList[recID].lconst = vlistInqVarTimetype(vlistID1, varID) == TIME_CONSTANT;
                }

              Field &rsamp1 = samp1[varID][levelID];
              Field &rvars1 = vars1[varID][levelID];

              const size_t fieldsize = rvars1.size;

              if (nsets == 0)
                {
                  cdoReadRecord(streamID1, rvars1.vec.data(), &rvars1.nmiss);
                  if (lrange)
                    {
                      vars2[varID][levelID].nmiss = rvars1.nmiss;
                      vars2[varID][levelID].vec = rvars1.vec;
                    }

                  if (rvars1.nmiss || !rsamp1.empty())
                    {
                      if (rsamp1.empty()) rsamp1.resize(fieldsize);

                      for (size_t i = 0; i < fieldsize; i++)
                        rsamp1.vec[i] = !DBL_IS_EQUAL(rvars1.vec[i], rvars1.missval);
                    }
                }
              else
                {
                  cdoReadRecord(streamID1, field.vec.data(), &field.nmiss);
                  field.grid = rvars1.grid;
                  field.missval = rvars1.missval;

                  if (field.nmiss || !rsamp1.empty())
                    {
                      if (rsamp1.empty()) rsamp1.resize(fieldsize, nsets);

                      for (size_t i = 0; i < fieldsize; i++)
                        if (!DBL_IS_EQUAL(field.vec[i], rvars1.missval)) rsamp1.vec[i]++;
                    }

                  if (lvarstd)
                    {
                      vfarsumq(vars2[varID][levelID], field);
                      vfarsum(rvars1, field);
                    }
                  else if (lrange)
                    {
                      vfarmin(vars2[varID][levelID], field);
                      vfarmax(rvars1, field);
                    }
                  else
                    {
                      vfarfun(rvars1, field, operfunc);
                    }
                }
            }

          if (nsets == 0 && lvarstd)
            for (int recID = 0; recID < maxrecs; recID++)
              {
                if (recList[recID].lconst) continue;

                const int varID = recList[recID].varID;
                const int levelID = recList[recID].levelID;

                vfarmoq(vars2[varID][levelID], vars1[varID][levelID]);
              }

          tsID++;
        }

      if (nrecs == 0 && nsets == 0) break;

      for (int recID = 0; recID < maxrecs; recID++)
        {
          if (recList[recID].lconst) continue;

          const int varID = recList[recID].varID;
          const int levelID = recList[recID].levelID;
          Field &rsamp1 = samp1[varID][levelID];
          Field &rvars1 = vars1[varID][levelID];

          if (lmean)
            {
              if (!rsamp1.empty())
                vfardiv(rvars1, rsamp1);
              else
                vfarcdiv(rvars1, (double) nsets);
            }
          else if (lvarstd)
            {
              Field &rvars2 = vars2[varID][levelID];
              if (!rsamp1.empty())
                {
                  if (lstd)
                    vfarstd(rvars1, rvars2, rsamp1, divisor);
                  else
                    vfarvar(rvars1, rvars2, rsamp1, divisor);
                }
              else
                {
                  Field &rvars2 = vars2[varID][levelID];
                  if (lstd)
                    vfarcstd(rvars1, rvars2, nsets, divisor);
                  else
                    vfarcvar(rvars1, rvars2, nsets, divisor);
                }
            }
          else if (lrange)
            {
              Field &rvars2 = vars2[varID][levelID];
              vfarsub(rvars1, rvars2);
            }
        }

      dtlist.statTaxisDefTimestep(taxisID2, nsets);
      cdoDefTimestep(streamID2, otsID);

      for (int recID = 0; recID < maxrecs; recID++)
        {
          if (otsID && recList[recID].lconst) continue;

          const int varID = recList[recID].varID;
          const int levelID = recList[recID].levelID;
          Field &rvars1 = vars1[varID][levelID];

          cdoDefRecord(streamID2, varID, levelID);
          cdoWriteRecord(streamID2, rvars1.vec.data(), rvars1.nmiss);
        }

      if (nrecs == 0) break;
      otsID++;

      for (int i = 0; i < nskip; i++)
        {
          nrecs = cdoStreamInqTimestep(streamID1, tsID);
          if (nrecs == 0) break;
          tsID++;
        }

      if (nrecs == 0) break;
    }

LABEL_END:

  cdoStreamClose(streamID2);
  cdoStreamClose(streamID1);

  cdoFinish();

  return nullptr;
}
