/*
  This file is part of CDO. CDO is a collection of Operators to
  manipulate and analyse Climate model Data.

  Copyright (C) 2003-2019 Uwe Schulzweida, <uwe.schulzweida AT mpimet.mpg.de>
  See COPYING file for copying and redistribution conditions.

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; version 2 of the License.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
*/

#include <cdi.h>

#include "cdo_int.h"

#include "cdo_defaultValues.h"  // Namespace CdoDefault

void *
Tocomplex(void *process)
{
  int nrecs;
  int varID, levelID;
  size_t nmiss;

  cdoInitialize(process);

  int RETOCOMPLEX = cdoOperatorAdd("retocomplex", 0, 0, nullptr);
  int IMTOCOMPLEX = cdoOperatorAdd("imtocomplex", 0, 0, nullptr);

  int operatorID = cdoOperatorID();

  CdoStreamID streamID1 = cdoOpenRead(0);

  int vlistID1 = cdoStreamInqVlist(streamID1);
  int vlistID2 = vlistDuplicate(vlistID1);

  int nvars = vlistNvars(vlistID2);
  for (varID = 0; varID < nvars; ++varID)
    {
      int datatype = vlistInqVarDatatype(vlistID2, varID);
      datatype = (datatype == CDI_DATATYPE_FLT64) ? CDI_DATATYPE_CPX64 : CDI_DATATYPE_CPX32;
      vlistDefVarDatatype(vlistID2, varID, datatype);
    }

  int taxisID1 = vlistInqTaxis(vlistID1);
  int taxisID2 = taxisDuplicate(taxisID1);
  vlistDefTaxis(vlistID2, taxisID2);

  // if (CdoDefault::FileType != CDI_FILETYPE_EXT) cdoAbort("Complex numbers need EXTRA format; used CDO option -f ext!");
  CdoStreamID streamID2 = cdoOpenWrite(1);
  cdoDefVlist(streamID2, vlistID2);

  size_t gridsize = vlistGridsizeMax(vlistID1);
  std::vector<double> array1(gridsize);
  std::vector<double> array2(2 * gridsize);

  int tsID = 0;
  int tsID2 = 0;
  while ((nrecs = cdoStreamInqTimestep(streamID1, tsID)))
    {
      taxisCopyTimestep(taxisID2, taxisID1);
      cdoDefTimestep(streamID2, tsID2++);

      for (int recID = 0; recID < nrecs; recID++)
        {
          cdoInqRecord(streamID1, &varID, &levelID);
          cdoDefRecord(streamID2, varID, levelID);

          gridsize = gridInqSize(vlistInqVarGrid(vlistID1, varID));

          cdoReadRecord(streamID1, array1.data(), &nmiss);

          if (operatorID == RETOCOMPLEX)
            {
              for (size_t i = 0; i < gridsize; ++i)
                {
                  array2[2 * i] = array1[i];
                  array2[2 * i + 1] = 0;
                }
            }
          else if (operatorID == IMTOCOMPLEX)
            {
              for (size_t i = 0; i < gridsize; ++i)
                {
                  array2[2 * i] = 0;
                  array2[2 * i + 1] = array1[i];
                }
            }

          cdoWriteRecord(streamID2, array2.data(), nmiss);
        }

      tsID++;
    }

  cdoStreamClose(streamID2);
  cdoStreamClose(streamID1);

  vlistDestroy(vlistID2);

  cdoFinish();

  return nullptr;
}
