/*
  This file is part of CDO. CDO is a collection of Operators to
  manipulate and analyse Climate model Data.

  Copyright (C) 2003-2019 Uwe Schulzweida, <uwe.schulzweida AT mpimet.mpg.de>
  See COPYING file for copying and redistribution conditions.

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; version 2 of the License.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
*/

/*
   This module contains the following operators:

      Ydayarith  ydayadd         Add multi-year daily time series
      Ydayarith  ydaysub         Subtract multi-year daily time series
      Ydayarith  ydaymul         Multiply multi-year daily time series
      Ydayarith  ydaydiv         Divide multi-year daily time series
*/

#include <cdi.h>

#include "cdo_int.h"

#define MAX_DOY 373

static int
getDayOfYearIndex(int64_t vdate)
{
  int year, month, day;
  cdiDecodeDate(vdate, &year, &month, &day);
  const int dayoy = (month >= 1 && month <= 12) ? (month - 1) * 31 + day : 0;
  if (dayoy < 0 || dayoy >= MAX_DOY) cdoAbort("Day of year %d out of range (date=%d)!", dayoy, vdate);
  return dayoy;
}

void *
Ydayarith(void *process)
{
  int nrecs;
  int varID, levelID;

  cdoInitialize(process);

  cdoOperatorAdd("ydayadd", func_add, 0, nullptr);
  cdoOperatorAdd("ydaysub", func_sub, 0, nullptr);
  cdoOperatorAdd("ydaymul", func_mul, 0, nullptr);
  cdoOperatorAdd("ydaydiv", func_div, 0, nullptr);

  const int operatorID = cdoOperatorID();
  const int operfunc = cdoOperatorF1(operatorID);

  CdoStreamID streamID1 = cdoOpenRead(0);
  CdoStreamID streamID2 = cdoOpenRead(1);

  const int vlistID1 = cdoStreamInqVlist(streamID1);
  const int vlistID2 = cdoStreamInqVlist(streamID2);
  const int vlistID3 = vlistDuplicate(vlistID1);

  vlistCompare(vlistID1, vlistID2, CMP_ALL);

  VarList varList1;
  varListInit(varList1, vlistID1);

  const size_t gridsizemax = vlistGridsizeMax(vlistID1);

  Field field1, field2;
  field1.resize(gridsizemax);
  field2.resize(gridsizemax);

  const int taxisID1 = vlistInqTaxis(vlistID1);
  const int taxisID2 = vlistInqTaxis(vlistID2);
  const int taxisID3 = taxisDuplicate(taxisID1);
  vlistDefTaxis(vlistID3, taxisID3);

  CdoStreamID streamID3 = cdoOpenWrite(2);
  cdoDefVlist(streamID3, vlistID3);

  FieldVector2D vars2[MAX_DOY];

  int tsID = 0;
  while ((nrecs = cdoStreamInqTimestep(streamID2, tsID)))
    {
      const int dayoy = getDayOfYearIndex(taxisInqVdate(taxisID2));
      if (vars2[dayoy].size() > 0) cdoAbort("Day of year index %d already allocated (date=%d)!", dayoy);

      fieldsFromVlist(vlistID2, vars2[dayoy], FIELD_VEC);

      for (int recID = 0; recID < nrecs; recID++)
        {
          size_t nmiss;
          cdoInqRecord(streamID2, &varID, &levelID);
          cdoReadRecord(streamID2, vars2[dayoy][varID][levelID].vec.data(), &nmiss);
          vars2[dayoy][varID][levelID].nmiss = nmiss;
        }

      tsID++;
    }

  tsID = 0;
  while ((nrecs = cdoStreamInqTimestep(streamID1, tsID)))
    {
      const int dayoy = getDayOfYearIndex(taxisInqVdate(taxisID1));
      if (vars2[dayoy].size() == 0) cdoAbort("Day of year index %d not found (date=%d)!", dayoy);

      taxisCopyTimestep(taxisID3, taxisID1);
      cdoDefTimestep(streamID3, tsID);

      for (int recID = 0; recID < nrecs; recID++)
        {
          cdoInqRecord(streamID1, &varID, &levelID);
          cdoReadRecord(streamID1, field1.vec.data(), &field1.nmiss);
          field1.grid = varList1[varID].gridID;
          field1.missval = varList1[varID].missval;

          vfarfun(field1, vars2[dayoy][varID][levelID], operfunc);

          cdoDefRecord(streamID3, varID, levelID);
          cdoWriteRecord(streamID3, field1.vec.data(), field1.nmiss);
        }
      tsID++;
    }

  cdoStreamClose(streamID3);
  cdoStreamClose(streamID2);
  cdoStreamClose(streamID1);

  cdoFinish();

  return nullptr;
}
