/*
  This file is part of CDO. CDO is a collection of Operators to
  manipulate and analyse Climate model Data.

  Copyright (C) 2003-2019 Uwe Schulzweida, <uwe.schulzweida AT mpimet.mpg.de>
  See COPYING file for copying and redistribution conditions.

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; version 2 of the License.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
*/

/*
   This module contains the following operators:

      Ymonarith  ymonadd         Add multi-year monthly time series
      Ymonarith  ymonsub         Subtract multi-year monthly time series
      Ymonarith  ymonmul         Multiply multi-year monthly time series
      Ymonarith  ymondiv         Divide multi-year monthly time series
      Ymonarith  yseasadd        Add multi-year seasonal time series
      Ymonarith  yseassub        Subtract multi-year seasonal time series
      Ymonarith  yseasmul        Multiply multi-year seasonal time series
      Ymonarith  yseasdiv        Divide multi-year seasonal time series
*/

#include <cdi.h>

#include "cdo_int.h"
#include "cdo_season.h"

#define MAX_MON 12

static int
getMonthIndex(const int64_t vdate)
{
  int year, mon, day;
  cdiDecodeDate(vdate, &year, &mon, &day);
  if (mon < 1 || mon > MAX_MON) cdoAbort("Month %d out of range!", mon);
  mon--;
  return mon;
}

static void
alreadyAllocated(bool isSeasonal, int mon, const char *seas_name[4])
{
  if (isSeasonal)
    cdoAbort("Season index %s already allocated!", seas_name[mon]);
  else
    cdoAbort("Month index %d already allocated!", mon);
}

static void
notFound(bool isSeasonal, int mon, const char *seas_name[4])
{
  if (isSeasonal)
    cdoAbort("Season index %s not found!", seas_name[mon]);
  else
    cdoAbort("Month index %d not found!", mon);
}

void *
Ymonarith(void *process)
{
  enum
  {
    MONTHLY,
    SEASONAL
  };
  int nrecs;
  int varID, levelID;
  const char *seas_name[4];

  cdoInitialize(process);

  // clang-format off
  cdoOperatorAdd("ymonadd",  func_add, MONTHLY, nullptr);
  cdoOperatorAdd("ymonsub",  func_sub, MONTHLY, nullptr);
  cdoOperatorAdd("ymonmul",  func_mul, MONTHLY, nullptr);
  cdoOperatorAdd("ymondiv",  func_div, MONTHLY, nullptr);
  cdoOperatorAdd("yseasadd", func_add, SEASONAL, nullptr);
  cdoOperatorAdd("yseassub", func_sub, SEASONAL, nullptr);
  cdoOperatorAdd("yseasmul", func_mul, SEASONAL, nullptr);
  cdoOperatorAdd("yseasdiv", func_div, SEASONAL, nullptr);
  // clang-format on

  const int operatorID = cdoOperatorID();
  const int operfunc = cdoOperatorF1(operatorID);
  const int opertype = cdoOperatorF2(operatorID);

  CdoStreamID streamID1 = cdoOpenRead(0);
  CdoStreamID streamID2 = cdoOpenRead(1);

  const int vlistID1 = cdoStreamInqVlist(streamID1);
  const int vlistID2 = cdoStreamInqVlist(streamID2);
  const int vlistID3 = vlistDuplicate(vlistID1);

  vlistCompare(vlistID1, vlistID2, CMP_ALL);

  VarList varList1;
  varListInit(varList1, vlistID1);

  const size_t gridsizemax = vlistGridsizeMax(vlistID1);

  Field field1, field2;
  field1.resize(gridsizemax);
  field2.resize(gridsizemax);

  const int taxisID1 = vlistInqTaxis(vlistID1);
  const int taxisID2 = vlistInqTaxis(vlistID2);
  const int taxisID3 = taxisDuplicate(taxisID1);
  vlistDefTaxis(vlistID3, taxisID3);

  CdoStreamID streamID3 = cdoOpenWrite(2);
  cdoDefVlist(streamID3, vlistID3);

  if (opertype == SEASONAL) get_season_name(seas_name);

  FieldVector2D vars2[MAX_MON];
  
  int tsID = 0;
  while ((nrecs = cdoStreamInqTimestep(streamID2, tsID)))
    {
      int mon = getMonthIndex(taxisInqVdate(taxisID2));
      if (opertype == SEASONAL) mon = month_to_season(mon + 1);
      if (vars2[mon].size()) alreadyAllocated(opertype == SEASONAL, mon, seas_name);

      fieldsFromVlist(vlistID2, vars2[mon], FIELD_VEC);

      for (int recID = 0; recID < nrecs; recID++)
        {
          size_t nmiss;
          cdoInqRecord(streamID2, &varID, &levelID);
          cdoReadRecord(streamID2, vars2[mon][varID][levelID].vec.data(), &nmiss);
          vars2[mon][varID][levelID].nmiss = nmiss;
        }

      tsID++;
    }

  tsID = 0;
  while ((nrecs = cdoStreamInqTimestep(streamID1, tsID)))
    {
      int mon = getMonthIndex(taxisInqVdate(taxisID1));
      if (opertype == SEASONAL) mon = month_to_season(mon + 1);
      if (vars2[mon].size() == 0) notFound(opertype == SEASONAL, mon, seas_name);

      taxisCopyTimestep(taxisID3, taxisID1);
      cdoDefTimestep(streamID3, tsID);

      for (int recID = 0; recID < nrecs; recID++)
        {
          cdoInqRecord(streamID1, &varID, &levelID);
          cdoReadRecord(streamID1, field1.vec.data(), &field1.nmiss);
          field1.grid = varList1[varID].gridID;
          field1.missval = varList1[varID].missval;

          vfarfun(field1, vars2[mon][varID][levelID], operfunc);

          cdoDefRecord(streamID3, varID, levelID);
          cdoWriteRecord(streamID3, field1.vec.data(), field1.nmiss);
        }

      tsID++;
    }

  cdoStreamClose(streamID3);
  cdoStreamClose(streamID2);
  cdoStreamClose(streamID1);

  cdoFinish();

  return nullptr;
}
