/*
  This file is part of CDO. CDO is a collection of Operators to
  manipulate and analyse Climate model Data.

  Copyright (C) 2003-2019 Uwe Schulzweida, <uwe.schulzweida AT mpimet.mpg.de>
  See COPYING file for copying and redistribution conditions.

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; version 2 of the License.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
*/

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <string>
#include <cassert>

#include "cdoStream.h"
#include "cdo_options.h"

static int nextCdoStreamID = 1;
static int createdCdoStreams = 0;

CdoStream::~CdoStream() {}
CdoStream::CdoStream() : m_cdoStreamID(nextCdoStreamID++)
{
  isopen = false;
  m_vlistID = -1;
  m_tsID = -1;
  m_filetype = CDI_UNDEFID;
  m_name = "";
  m_varID = -1;
  m_nvals = 0;
}

int
CdoStream::getID()
{
  return m_cdoStreamID;
}

int
CdoStream::getTsID()
{
  return m_tsID;
}
int
CdoStream::getVarID()
{
  return m_varID;
}
int
CdoStream::getVlistID()
{
  return m_vlistID;
}

void
CdoStream::defVarList(int p_vlistID)
{
  int filetype = m_filetype;

  if (m_vlistID != -1) cdoAbort("Internal problem, vlist already defined!");

  if (m_varlist.size() != 0) cdoAbort("Internal problem, varlist already allocated!");

  const int nvars = vlistNvars(p_vlistID);
  assert(nvars > 0);

  m_varlist.resize(nvars);

  for (int varID = 0; varID < nvars; ++varID)
    {
      m_varlist[varID].gridsize = gridInqSize(vlistInqVarGrid(p_vlistID, varID));
      m_varlist[varID].datatype = vlistInqVarDatatype(p_vlistID, varID);
      m_varlist[varID].missval = vlistInqVarMissval(p_vlistID, varID);
      m_varlist[varID].addoffset = vlistInqVarAddoffset(p_vlistID, varID);
      m_varlist[varID].scalefactor = vlistInqVarScalefactor(p_vlistID, varID);

      m_varlist[varID].check_datarange = false;

      const bool laddoffset = IS_NOT_EQUAL(m_varlist[varID].addoffset, 0);
      const bool lscalefactor = IS_NOT_EQUAL(m_varlist[varID].scalefactor, 1);

      int datatype = m_varlist[varID].datatype;

      if (filetype == CDI_FILETYPE_NC || filetype == CDI_FILETYPE_NC2 || filetype == CDI_FILETYPE_NC4
          || filetype == CDI_FILETYPE_NC4C || filetype == CDI_FILETYPE_NC5)
        {
          if (datatype == CDI_DATATYPE_UINT8
              && (filetype == CDI_FILETYPE_NC || filetype == CDI_FILETYPE_NC2 || filetype == CDI_FILETYPE_NC5))
            {
              datatype = CDI_DATATYPE_INT16;
              m_varlist[varID].datatype = datatype;
            }

          if (datatype == CDI_DATATYPE_UINT16
              && (filetype == CDI_FILETYPE_NC || filetype == CDI_FILETYPE_NC2 || filetype == CDI_FILETYPE_NC5))
            {
              datatype = CDI_DATATYPE_INT32;
              m_varlist[varID].datatype = datatype;
            }

          if (laddoffset || lscalefactor)
            {
              if (datatype == CDI_DATATYPE_INT8 || datatype == CDI_DATATYPE_UINT8 || datatype == CDI_DATATYPE_INT16
                  || datatype == CDI_DATATYPE_UINT16)
                m_varlist[varID].check_datarange = true;
            }
          else if (Options::CheckDatarange)
            {
              m_varlist[varID].check_datarange = true;
            }
        }
    }

  m_vlistID = p_vlistID; /* used for -r/-a */
}
