/*
  This file is part of CDO. CDO is a collection of Operators to
  manipulate and analyse Climate model Data.

  Copyright (C) 2003-2021 Uwe Schulzweida, <uwe.schulzweida AT mpimet.mpg.de>
  See COPYING file for copying and redistribution conditions.

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; version 2 of the License.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
*/

#ifndef PROCESS_H
#define PROCESS_H

#include "cdoStream.h"
#include "modules.h"

#include <vector>
#include <iostream>
#include <string>
#include <set>

constexpr int MAX_PROCESS = 65536;
constexpr int MAX_OPERATOR = 128;

enum class ProcessStatus
{
  Ok = 0,
  UnlimitedIOCounts = -1,
  MissInput = -2,
  MissOutput = -3,
  TooManyStreams = -4,
  TooFewStreams = -5,
};

class oper_t
{
public:
  int f1 = 0;
  int f2 = 0;
  const char *name = nullptr;
  const char *enter = nullptr;

  oper_t() {}
  oper_t(int _f1, int _f2, const char *_name, const char *_enter) : f1(_f1), f2(_f2), name(_name), enter(_enter) {}
};

class Process
{
public:
  int m_ID;
  int m_posInParent;
#ifdef HAVE_LIBPTHREAD
  pthread_t threadID;
#endif
  short nchild = 0;
  std::vector<Process *> childProcesses;
  std::vector<Process *> parentProcesses;
  std::vector<CdoStreamID> inputStreams;
  std::vector<CdoStreamID> outputStreams;
  int nChildActive = 0;

  double startTime;

  int nvars = 0;

  int ntimesteps = 0;
  int m_streamCnt = 0;
  char const *m_operatorCommand = "UNINITALIZED";
  char const *operatorName;
  const char *m_obase;
  char prompt[64];

  short m_noper = 0;
  bool m_isActive;

  module_t m_module;
  std::vector<std::string> m_oargv;
  oper_t oper[MAX_OPERATOR];

  Process(int p_ID, const char *p_operatorNamme, const char *operatorCommand);

  pthread_t run();
  ProcessStatus m_status = ProcessStatus::Ok;

  /**
   * returns the number of in streams this process currently has.
   **/
  int get_stream_cnt_in();
  /**
   * returns the number of out streams this process currently has.
   */
  int get_stream_cnt_out();

  /**
   * returns full command through which this process was created including operator name and operator arguments
   */
  const char *get_command();

  /**
   * Adds a Process as child and creates and adds a new pipe stream.
   */
  void add_child(Process *child_process);
  /**
   * Adds a Process as parent and adds the parents input stream to out streams.
   */
  void add_parent(Process *parent_process);
  /**
   * Compares the wanted and current stream counts.
   * @return if the wanted count is -1 this function always returns false.
   * Are the current and wanted stream counts equal 1 and if they differ false.
   */
  bool has_hall_inputs();
  /**
   * Adds and creates a new file pstream to the in streams
   */
  void add_file_in_stream(std::string file);
  /**
   * Adds and creates a new file pstream to the out streams
   */
  void add_file_out_stream(std::string file);
  /**
   * Adds and creates a new pipe pstream to the in streams
   */
  void add_pipe_in_stream();
  /**
   * Adds and creates a new file pstream to the out streams
   */
  void add_pipe_out_stream();
  /**
   * Adds an operator to the process
   */
  int operator_add(const char *name, int f1, int f2, const char *enter);
  /**
   * returns the operatorID of the currently in use operator
   */
  int get_operator_id();
  void set_inactive();
  const char *inq_promt() const;
  void print_processed_values();
  void print_benchmarks(double p_walltime, const char *p_memstring);
  void check_stream_cnt();
  void validate();
  void handle_process_err();

  bool has_out_stream(const CdoStreamID p_streamPtr);
  bool has_in_stream(const CdoStreamID p_streamPtr);
  bool has_no_pipes();
  size_t inq_nvals();
  const char *get_out_stream_name();
  size_t get_oper_argc();
  std::string get_argv(int idx);
  std::string get_obase();
  bool input_is_variable();

  void init_process(const char *p_operatorName, const char *operatorCommand);

private:
  Process();
  void def_prompt();
  /**
   * Splits the operator arguments stores them.
   * Operator arguments are stored as char arrays and appended with a zero termination.
   */
  void set_operator_argv(const char *operatorArguments);
};

int get_process_num();
void set_process_num(int p_num);

#endif /* PROCESS_H */
