/* cdw
 * Copyright (C) 2002 Varkonyi Balazs
 * Copyright (C) 2007 Kamil Ignacak
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */
#include <linux/cdrom.h>
#include <sys/ioctl.h>

#include <stdlib.h>
#include <unistd.h>
#include <libintl.h>
#include <stdio.h>
#include <sys/types.h>
#include <fcntl.h>

#include <ncursesw/ncurses.h>

#include "options.h"
#include "gettext.h"
#include "cdw_widgets.h" /* dialogobox() */



extern struct conf config;


/**
 * Check for cdrom drive status
 *
 * Check for cdrom drive status using ioctl (with parameter CDROM_DRIVE_STATUS).
 * Documentation on ioctl can be found on linux source code documentation in
 * Documentation/ioctl/cdrom.txt.
 *
 * This function tries to open cdrom device described by config.cdrwdevice.
 *
 * Return codes are mapped to new names in cdrom_ctrl.h, so there is no need
 * to include linux/cdrom.h again in *c files.
 *
 * \returns normal ioctl return codes:
 * 	            CDS_NO_INFO         Information not available.
 *                  CDS_NO_DISC
 *                  CDS_TRAY_OPEN
 *                  CDS_DRIVE_NOT_READY
 *                  CDS_DISC_OK
 *                  -1                  error
 *
 *                  or -2 on cdrom_status() error (cannot open cdrom file)
 */
int cdrom_status(void)
{
	int cddev;

	if ( (cddev=open(config.cdrwdevice, O_RDONLY | O_NONBLOCK)) != -1 ){
		int rv = ioctl(cddev, CDROM_DRIVE_STATUS, 0);
		close(cddev);
		return rv;
 	} else {
		; // FIXME - some error message, check path to CD device
	}

	return -2;
}




void eject_tray(char *device)
{
    int cddev;

    if ( (cddev=open(device, O_RDONLY | O_NONBLOCK)) != -1 ){
	ioctl(cddev, CDROMEJECT, 0);
	close(cddev);
    }
}




void close_tray(char *device)
{
    int cddev;

    if ( (cddev=open(device, O_RDONLY | O_NONBLOCK)) != -1 ){
	ioctl(cddev, CDROMCLOSETRAY, 0);
	close(cddev);
    }
}





/*
 * Ensure that disc is in drive and drive tray is closed
 *
 * This function prevents program from attempting to access CD
 * when there is no disc in drive.
 * Uses ioctl() to check drive status and ensure that disc is in drive.
 * Uses dialog box to communicate with user.
 *
 * \returns true if disc is in tray and tray is closed, false if user
 * 	decided to not put disc in drive or drive doesnt't answer
 *
 */
bool disc_in_drive(void)
{
	int k = 0;
	int i;

	int rvd;

	for (i = 0; i < 10; i++) {
		int rv = cdrom_status();
		switch (rv) {
			case CDS_DISC_OK:
				return true;
			case CDS_NO_DISC:
				/* 2TRANS: this is title of dialog window:
				   some error occured while trying to
				   access disc */
				rvd = dialogbox(_("CD drive error"),
						/* 2TRANS: this is message in dialog window: no optical disc
						   found in drive; user can select OK or CANCEL button */
						_("No disc in drive. Please insert disc."), DIALOG_OK_CANCEL);
				if (rvd == BUTTON_OK) {
					break;
				} else {
					return false;
				}
			case CDS_TRAY_OPEN:
				/* 2TRANS: this is title of dialog window:
				   some error occured while trying to access
				   disc */
				rvd = dialogbox(_("CD drive error"),
						/* 2TRANS: this is message in dialog window; user
						   can select OK or CANCEL button */
						_("Please close CDROM tray."), DIALOG_OK_CANCEL);
				if (rvd == BUTTON_OK) {
					break;
				} else {
					return false;
				}
			case CDS_DRIVE_NOT_READY:
				sleep(2);
				k++;
				if (k == 5) { /* FIXME: why try 5 times before informing about timeout? */
					/* 2TRANS: this is title of dialog window:
					   some error occured while trying to
					   access disc */
					rvd = dialogbox(_("CD drive error"),
							/* 2TRANS: this is message in dialog window;
							   user can select OK or CANCEL button */
							_("CDROM doesn't respond (timeout). Try again?"), DIALOG_OK_CANCEL);
					if (rvd == BUTTON_OK) {
						k = 0; /* start asking cdrom for response - again */
						break;
					} else { /* cancel, don't try anymore */
						return false; /* timeout, failed */
					}
				} else {
					break; /* silenty try again */
				}
			case CDS_NO_INFO:
				/* 2TRANS: this is title of dialog window:
				   some error occured while trying to access
				   disc */
				rvd = dialogbox(_("CD drive error"),
						/* 2TRANS: this is message in dialog window;
						   cannot read some meta-data from cd; user
						   can select OK or CANCEL button */
						_("Can't get CDROM info. Continue?"), DIALOG_OK_CANCEL);
				if (rvd == BUTTON_OK) { /* user might want to try perform action anyway; let him try, because we are not 100% sure that the action will fail */
					return true;
				} else {
					return false;
				}
			default: /* FIXME - is it really our last option? */
				/* 2TRANS: this is title of dialog window:
				   some error occured while trying to access
				   disc */
				rvd = dialogbox(_("CD drive error"),
						/* 2TRANS: this is message in dialog window: no optical disc found in drive; user can select OK or CANCEL button */
						_("No disc in drive. Please insert disc."), DIALOG_OK_CANCEL);
				if (rvd == BUTTON_OK) {
					break;
				} else {
					return false;
				}
				break;
		} /* switch */
	} /* for */
	/* 2TRANS: this is message in dialog window:
	   no optical disc found in drive */
	dialogbox(_("CD drive message"),
		  /* 2TRANS: this is message in dialog window: some errors
		     occured while accessing cd */
		  _("Something went wrong.\nPlease check hardware settings in cdw options and try once more."), DIALOG_OK);
	return false; /* user couldn't decide what to do or hardware settings in cdw optins are invalid */
}






