/* cdw
 * Copyright (C) 2002 Varkonyi Balazs
 * Copyright (C) 2007 Kamil Ignacak
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */
#include <stdio.h>
#include <sys/types.h>
#include <math.h>
#include <unistd.h>
#include <stdlib.h>
#include <string.h>
#include "errno.h"

#include "gettext.h"
#include <libintl.h>

#include <linux/fs.h>

#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <sys/ioctl.h>
#include <linux/cdrom.h>

#include <cdio/cdio.h>
#include <cdio/mmc.h>

#include "cdw_cdio.h"
#include "options.h"
#include "main.h"
#include "cdw_widgets.h"
#include "processwin.h"


/* parts of code in this file are heavily influenced by online documentation
 * of cdio library: FIXME: provide link
 *
 * some technical details about CD sectors are from various online resources:
 * - http://www-is.informatik.uni-oldenburg.de/~dibo/teaching/mm98/buch/node113.html
 * - FIXME: provide other links
 */


/* typedef driver_return_code_t (*sector_reader_fun)(const CdIo_t *p_cdio, void *p_buf, lsn_t i_lsn, bool b_form2, uint32_t i_blocks); */
int raw_read_sector(lsn_t sector, char *buffer); /* just a debug tool, normally not used */
int raw_read_sector2(lsn_t sector); /* just a debug tool, normally not used */

/* there is only one disc that can be opened at given time;
   this is the variable representing it */
static struct disc_information disc;
extern struct conf config;

int cdw_cdio_recognize_track(struct disc_information *disc, track_t t);
bool cdw_get_track_format_with_cdio_mmc(lsn_t sector, cdw_track_format_t *track_mode);




/**
 * Open disc, collect some useful metadata
 *
 * Open optical disc using cdio_open(), check disc mode, type,
 * check tracks number and formats, check total number of sectors on disc.
 * Mark disc as open.
 *
 * \param device - full path to device file
 *
 * \returns pointer to disc structure if disc was opened succesfully and is of supported type, or was already opened, -1 if disc type is unsupported, -2 if cannot open disc
 */
struct disc_information *cdw_cdio_disc_open(char *device)
{
	if (disc.open) { /* device already opened */
		/* fprintf(stderr, "ERROR: cdw_cdio_disc_open(): disc already opened\n"); */
		return &disc;
	}

	disc.p_cdio = cdio_open(device, DRIVER_LINUX);
	if (disc.p_cdio == NULL) {
		/* fprintf(stderr, "ERROR: cdw_cdio_disc_open(): cannot open cd drive\n"); */
		return -2;
	}

	/* retrieve some basic information about disc */
	disc.mode = cdio_get_discmode(disc.p_cdio);
	if (disc.mode == CDIO_DISC_MODE_CD_DATA) {
		/* libcdio 0.78 calls mmc command to get disc type, but the
		call returns CD_DATA for mixed mode CD; let's do
		something about it */
		int fd = open(config.cdrom, O_RDONLY);
		if (fd != -1) {
			disc.mode2 = ioctl(fd, CDROM_DISC_STATUS, 0);
			close(fd);
			if (disc.mode2 != -1) {
				if (disc.mode2 == CDS_MIXED) {
					disc.mode = CDIO_DISC_MODE_CD_MIXED;
				}
			}
		}
	}
	if ( ! (  (disc.mode == CDIO_DISC_MODE_CD_DATA)
	       || (disc.mode == CDIO_DISC_MODE_CD_XA)
	       || (disc.mode == CDIO_DISC_MODE_CD_DA)
	    /* || (disc.mode == CDIO_DISC_MODE_CD_MIXED)
	       || (disc.mode == CDIO_DISC_MODE_DVD_ROM)         // DVD ROM (e.g. movies)
	       || (disc.mode == CDIO_DISC_MODE_DVD_RAM)         // DVD-RAM
	       || (disc.mode == CDIO_DISC_MODE_DVD_R)           // DVD-R
	       || (disc.mode == CDIO_DISC_MODE_DVD_RW)          // DVD-RW
	       || (disc.mode == CDIO_DISC_MODE_DVD_PR)          // DVD+R
	       || (disc.mode == CDIO_DISC_MODE_DVD_PRW)         // DVD+RW
	       || (disc.mode == CDIO_DISC_MODE_DVD_OTHER) 	// Unknown/unclassified DVD type
	        */ )) {   /* if unsupported disc type */
		return (struct disc_information *) -1; /* I know this is invalid */
	}

	/* debug code */
	/*
	if (disc.mode == CDIO_DISC_MODE_CD_DATA) {
		fprintf(stderr, "INFO: cdw_cdio_disc_open(): disc.mode == CDIO_DISC_MODE_CD_DATA\n");
	} else if (disc.mode == CDIO_DISC_MODE_CD_XA) {
		fprintf(stderr, "INFO: cdw_cdio_disc_open(): disc.mode == CDIO_DISC_MODE_CD_XA\n");
	} else if (disc.mode == CDIO_DISC_MODE_CD_DA) {
		fprintf(stderr, "INFO: cdw_cdio_disc_open(): disc.mode == CDIO_DISC_MODE_CD_DA\n");
	} else {
		fprintf(stderr, "INFO: cdw_cdio_disc_open(): disc.mode == UNKNOWN\n");
	}
	*/


	disc.first_track = cdio_get_first_track_num(disc.p_cdio);
	disc.last_track = cdio_get_last_track_num(disc.p_cdio);
	disc.total_tracks = disc.last_track - disc.first_track + 1;

	disc.total_sectors = cdio_get_track_last_lsn(disc.p_cdio, disc.total_tracks);

	/*
	fprintf(stderr, "INFO: cdw_cdio_disc_open(): first track = %d\n", disc.first_track);
	fprintf(stderr, "INFO: cdw_cdio_disc_open(): last track = %d\n", disc.last_track);
	fprintf(stderr, "INFO: cdw_cdio_disc_open(): total tracks = %d\n", disc.total_tracks);
	fprintf(stderr, "INFO: cdw_cdio_disc_open(): total sectors = %d\n", disc.total_sectors);
	*/

	track_t t = 1;
	for (t = disc.first_track; t <= disc.last_track; t++) {
		/* get mode, form and sector-related values of each data track on Compact Disc */
		cdw_cdio_recognize_track(&disc, t);
	}

	/* mark disc as open to avoid opening it for a second time */
	disc.open = true;
	return &disc;
}




/**
 * Close disc that was previously opened with cdw_cdio_disc_open()
 */
void cdw_cdio_disc_close(void)
{
	if (disc.open == true) {
		cdio_destroy(disc.p_cdio);
		disc.open = false;
	}

	return;
}



/**
 * Try to guess track mode (Audio/Mode1/Mode2) and
 * (if applicable) form (Form1/Form2)
 *
 * Try to check data size in sectors from given track. Checking performed in
 * this function (and functions called by this function) is still a bit
 * errorneous. It also may fail if your optical drive does not support
 * MMC commands (if mmc_read_cd() fails).
 *
 * This function depends disc.mode, so be sure that it has valid value
 * before calling the function - use cdio_get_discmode(disc.p_cdio).
 *
 * \param disc - cdio obiect to read form
 * \param t - track to check
 *
 * \return -1 if track format is unknown, 0 in any other situation (including other errors)
 */
int cdw_cdio_recognize_track(struct disc_information *disc, track_t t)
{
	/* fprintf(stderr, "\n\nINFO: Recognizing track t = %d\n", (int)t); */
	bool mmc_success = false; /* id mmc command checking track type succeeded? */

	/* these two values will be used later when reading sectors */
	disc->tracks[t].first_sector = cdio_get_track_lsn(disc->p_cdio, t);
	disc->tracks[t].last_sector = cdio_get_track_last_lsn(disc->p_cdio, t);
	/* fprintf(stderr, "INFO: cdw_cdio_recognize_track(): first sector: %d, last sector: %d, sectors: %d\n", disc->tracks[t].first_sector, disc->tracks[t].last_sector, disc->tracks[t].last_sector - disc->tracks[t].first_sector + 1); */

	/* cdio track format, value that we set here will be used as 'backup'
	 * value if our own method of checking track type will fail */
	disc->tracks[t].format = cdio_get_track_format(disc->p_cdio, t);

	if (disc->tracks[t].format == TRACK_FORMAT_AUDIO) { /* Red Book CD Digital Audio */
		disc->tracks[t].cdw_format = CDW_RED_BOOK_AUDIO;
		disc->tracks[t].sector_size = 2352;
		/* fprintf(stderr, "INFO: cdw_cdio_recognize_track(): cdio recognizes TRACK_FORMAT_AUDIO track\n"); */
		/* that's all we need in case of audio format: format and sector_size */
		return 0;
	} else if ( (disc->tracks[t].format == TRACK_FORMAT_DATA)
			|| (disc->tracks[t].format == TRACK_FORMAT_XA)) { /* Data of some sort */

		/* debug code */
		/*
		if (disc->tracks[t].format == TRACK_FORMAT_DATA) {
			fprintf(stderr, "INFO: cdw_cdio_recognize_track(): cdio recognizes TRACK_FORMAT_DATA track\n");
		} else if (disc->tracks[t].format == TRACK_FORMAT_XA) {
			fprintf(stderr, "INFO: cdw_cdio_recognize_track(): cdio recognizes TRACK_FORMAT_XA track\n");
		} else {
			fprintf(stderr, "INFO: cdw_cdio_recognize_track(): cdio: unknown TRACK_FORMAT track\n");
		}
		*/
		cdw_track_format_t _cdw_track_format;
		lsn_t sector = disc->tracks[t].first_sector + 190;
		mmc_success = cdw_get_track_format_with_cdio_mmc(/* int */ sector, /* out */ &_cdw_track_format);
		if (!mmc_success) {
			disc->tracks[t].cdw_format = CDW_BLACK_BOOK_UNKNOWN;
			/* fprintf(stderr, "ERROR: cdw_cdio_recognize_track(): returning -1 (unrecognized track format (!mmc_success))\n"); */
			return -1; /* FIXME - find another way of checking track format */
		} else {
			disc->tracks[t].cdw_format = _cdw_track_format; /* value set by mmc function */
		}
	} else { /* TRACK_FORMAT_ERROR || TRACK_FORMAT_PSX || TRACK_FORMAT_CDI  - these are not handled at this moment */
		if (disc->tracks[t].format == TRACK_FORMAT_PSX) {
			/* fprintf(stderr, "INFO: cdw_cdio_recognize_track(): TRACK_FORMAT_PSX track\n"); */
		} else if (disc->tracks[t].format == TRACK_FORMAT_CDI) {
			/* fprintf(stderr, "INFO: cdw_cdio_recognize_track(): TRACK_FORMAT_CDI track\n"); */
		} else if (disc->tracks[t].format == TRACK_FORMAT_ERROR) {
			/* fprintf(stderr, "INFO: cdw_cdio_recognize_track(): TRACK_FORMAT_ERROR track\n"); */
		} else {
			disc->tracks[t].cdw_format = CDW_BLACK_BOOK_UNKNOWN;
			/* fprintf(stderr, "INFO: cdw_cdio_recognize_track(): CDW_BLACK_BOOK_UNKNOWN track\n"); */
		}
		/* fprintf(stderr, "ERROR: cdw_cdio_recognize_track(): returning -1 (unsupported/unrecognized disc)\n"); */
		return -1;
	}

	if (mmc_success) { /* whe have information about data size extracted
		              directly from sector header(s) */
		if (disc->tracks[t].cdw_format == CDW_RED_BOOK_AUDIO) {
			disc->tracks[t].sector_size = 2352;
			/* fprintf(stderr, "INFO: cdw_cdio_recognize_track(): CDW_RED_BOOK_AUDIO track\n"); */
		} else if (disc->tracks[t].cdw_format == CDW_YELLOW_BOOK_MODE1) {
			disc->tracks[t].sector_size = 2048;
			/* fprintf(stderr, "INFO: cdw_cdio_recognize_track(): CDW_YELLOW_BOOK_MODE1 track\n"); */
		} else if (disc->tracks[t].cdw_format == CDW_YELLOW_BOOK_MODE2) {
			disc->tracks[t].sector_size = 2336;
			/* fprintf(stderr, "INFO: cdw_cdio_recognize_track():  CDW_YELLOW_BOOK_MODE2 track\n"); */
		} else if (disc->tracks[t].cdw_format == CDW_GREEN_BOOK_MODE2_FORM1) {
			disc->tracks[t].sector_size = 2048;
			/* fprintf(stderr, "INFO: cdw_cdio_recognize_track(): CDW_GREEN_BOOK_MODE2_FORM1 track\n"); */
		} else if (disc->tracks[t].cdw_format == CDW_GREEN_BOOK_MODE2_FORM2) {
			disc->tracks[t].sector_size = 2324;
			/* fprintf(stderr, "INFO: cdw_cdio_recognize_track(): CDW_GREEN_BOOK_MODE2_FORM2 track\n"); */
		} else { /* CDW_BLACK_BOOK_UNKNOWN */
			disc->tracks[t].sector_size = 2048; /* some default value */
			/* fprintf(stderr, "INFO: cdw_cdio_recognize_track(): CDW_BLACK_BOOK_UNKNOWN track\n"); */
		}
	} else { /* our method failed, use 'format' value set by cdio function */
		/* FIXME - check these values of sector sizes again */
		/* fprintf(stderr, "WARN: cdw_cdio_recognize_track(): assigning arbitrary track size\n"); */
		if (disc->tracks[t].format == TRACK_FORMAT_AUDIO) {
			disc->tracks[t].sector_size = 2352;
		} else if (disc->tracks[t].format == TRACK_FORMAT_CDI) { /* mode 2 form unknown */
			disc->tracks[t].sector_size = 2324;
		} else if (disc->tracks[t].format == TRACK_FORMAT_XA) { /* mode 2 form unknown */
			disc->tracks[t].sector_size = 2324;
		} else if (disc->tracks[t].format == TRACK_FORMAT_DATA) { /* mode 1 form unknown */
			disc->tracks[t].sector_size = 2048;
		} else if (disc->tracks[t].format == TRACK_FORMAT_PSX) { /* mode 1 form unknown */
			disc->tracks[t].sector_size = 2336;
		} else {
			disc->tracks[t].sector_size = 2048; /* some default value */
		}
	}

	return 0;
}







/**
 * Read raw CD sector using Linux ioctl(2) - just a non-portable tool
 *
 * \param sector - valid sector number of sector to be read
 * \param buffer - char table of size 2352
 *
 * \return 0
 */
int raw_read_sector(lsn_t sector, char *_buffer)
{
	int fd = open(config.cdrom, O_RDONLY | O_NONBLOCK);
	if (fd == -1) {
		/* fprintf(stderr, "\n\n raw_read failed\n\n\n"); */
		return -1;
	}


	union {
		struct cdrom_msf msf;   /* input WARNING: make sure which data structure is used as input, read comments for CDROMREAD* in ioctl.txt */
		char buffer[2352];      /* return */
	} arg;

	msf_t p_msf;
	cdio_lsn_to_msf(3000, &p_msf);
	arg.msf.cdmsf_min0 = p_msf.m;     /* start minute */
	arg.msf.cdmsf_sec0 = p_msf.s;     /* start second */
	arg.msf.cdmsf_frame0 = p_msf.f;   /* start frame */

	int rv0 = ioctl(fd, CDROMREADRAW, &arg);
	/* perror("1 -- ");
	fprintf(stderr, "rv0 = %d\n", rv0);

	fprintf(stderr, ">>RAW1>>");
	int i = 0;
	for (i =  0; i< 40; i++) {
		fprintf(stderr, " %x ", arg.buffer[i]);
	}
	fprintf(stderr, "<<\n"); */


	/* ************************************* */


	char internal_buffer[2352];
	struct cdrom_read arg2;
	arg2.cdread_lba = 3000; /* FIXME: LBA != lsn */
	arg2.cdread_bufaddr = (unsigned char *) malloc(2352);
	arg2.cdread_buflen = 2352;

	int rv = ioctl(fd, CDROMREADRAW, &arg2);
	/* perror("2 -- ");
	fprintf(stderr, "rv = %d\n", rv); */


	/*
	fprintf(stderr, ">>RAW2>>");
	for (i =  0; i< 40; i++) {
		fprintf(stderr, " %x ", internal_buffer[i]);
	}
	fprintf(stderr, "<<\n");
	*/

	close(fd);


	// int disc_status = ioctl(fd, CDROM_DISC_STATUS, 0);
/*
	if (disc_status == CDS_NO_INFO) {
		fprintf(stderr, ">>>>>>>>> CDS_NO_INFO <<<<<<<<<<<\n");
	} else if (disc_status == CDS_AUDIO) {
		fprintf(stderr, ">>>>>>>>> CDS_AUDIO <<<<<<<<<<<\n");
	} else if (disc_status == CDS_MIXED) {
		fprintf(stderr, ">>>>>>>>> CDS_MIXED <<<<<<<<<<<\n");
	} else if (disc_status == CDS_XA_2_2) {
		fprintf(stderr, ">>>>>>>>> CDS_XA_2_2 <<<<<<<<<<<\n");
	} else if (disc_status == CDS_XA_2_1) {
		fprintf(stderr, ">>>>>>>>> CDS_XA_2_1 <<<<<<<<<<<\n");
	} else if (disc_status == CDS_DATA_1) {
		fprintf(stderr, ">>>>>>>>> CDS_DATA_1 <<<<<<<<<<<\n");
	} else {
		fprintf(stderr, ">>>>>>>>> dupa <<<<<<<<<<<\n");
	}

*/
	return 0;
}









int raw_read_sector2(lsn_t sector)
{
	struct cdrom_msf msf;
	char buffer[2352];

	msf_t p_msf;

	int fd = open(config.cdrom, O_RDWR | O_NONBLOCK);
	if (fd == -1) {
		return -1;
		/* fprintf(stderr, "bledny desk. pliku\n"); */
	}

	cdio_lsn_to_msf(sector, &p_msf);
	msf.cdmsf_min0 = p_msf.m;     /* start minute */
	msf.cdmsf_sec0 = p_msf.s;     /* start second */
	msf.cdmsf_frame0 = p_msf.f;   /* start frame */

	msf.cdmsf_min0 = 1;     /* start minute */
	msf.cdmsf_sec0 = 1;     /* start second */
	msf.cdmsf_frame0 = 1;   /* start frame */

	memcpy(buffer, &msf, sizeof(msf));
	int rv = ioctl(fd, CDROMREADRAW, buffer);
	/* perror("UNO: -- ");
	fprintf(stderr, ": %d\n", rv); */
	close(fd);

	return 0;
}




/**
 * Use MMC command to obtain sector's (and its parent track) format
 *
 * Read full sector (all 232 bytes) using mmc command and extract track
 * mode data from its header(s). Call this function only for non-CDDA
 * sectors.
 *
 * \param track_format - this variable will be set accordingly to track format
 * \param sector - sector number of sector belonging to track, which format we want to check
 *
 * \return true if mmc command returned with success, false otherwise
 */
bool cdw_get_track_format_with_cdio_mmc(lsn_t sector, cdw_track_format_t *_cdw_track_format)
{
	unsigned char sector_buffer[2352];

	/* note that CDIO_DISC_MODE_CD_CDDA should be ruled out before
	 * calling this function, but we will check it anyway */
	/* this is just a preparation of one of arguments of mmc_read_cd() */
	int header_code = 0; /* which sector header should be returned? */
	if (disc.mode == CDIO_DISC_MODE_CD_DA) { /* Red Book disc, no sector header and no sector subheader */
		header_code = 0; /* do not get any header, since it does not exists */
		/* fprintf(stderr, "cdw_get_track_format_with_cdio_mmc header code = 0\n"); */
	} else if (disc.mode == CDIO_DISC_MODE_CD_DATA) { /* Yellow Book disc (Data disc), only 4-bit header exists */
		header_code = 1;
		/* fprintf(stderr, "cdw_get_track_format_with_cdio_mmc header code = 1\n"); */
	} else if (disc.mode == CDIO_DISC_MODE_CD_XA) { /* Green Book disc (eXtended Attributes disc), both 4-bit header and 8-bit subheader exist */
		header_code = 3; /* get both header and subheader */
		/* fprintf(stderr, "cdw_get_track_format_with_cdio_mmc header code = 3\n"); */
	} else { /* FIXME: CD-I or MIXED maybe? */
		/* fprintf(stderr, "cdw_get_track_format_with_cdio_mmc header code = 0 by default\n"); */
		header_code = 0; /* just in case: don't get any headers because mmc_read_cd() may fail */
	}

	/* FIXME - this will fail if drive does not have MMC capabilities or this fnctionality is not supported */
	driver_return_code_t drv = mmc_read_cd(
		disc.p_cdio,   /* object to read from */
		sector_buffer, /* place to store data */
		sector,        /* sector number to read */
		0,             /* expected sector type; 0 - all sector types */
		false,         /* b_digital_audio_play - audio related stuff, doesn't matter */
		true,          /* b_sync_header - return synchronization header */
		3,             /* uint8_t header_codes - data header and subheader; 0: none, 1: 4-bit header, 2: 8-bit subheader. 3: both */
		false,         /* bool b_user_data - don't return user data */
		false,         /* bool b_edc_ecc - error detection/correction data */
		0,             /* uint8_t c2_error_information, yet another error-related field (?) */
		0,             /* uint8_t subchannel_selection - subchannel selection bits (?)*/
		2352,          /* uint16_t i_blocksize - size of the block expected to be returned, 2352 is full sector size, including sync bytes */
		1              /* uint32_t i_blocks - number of blocks expected to be returned */
		);

	if (drv < 0) {
		/* fprintf(stderr, "mmc_read_cd returned error value = %d (sector %d)\n", (int) drv, (int) sector);
		fprintf(stderr, "mmc_read_cd returned error value: retry with XA header\n"); */
		if (header_code == 3) {
			header_code = 1;
		} else {
			header_code = 3;
		}

		drv = mmc_read_cd(
			disc.p_cdio,   /* object to read from */
			sector_buffer, /* place to store data */
			sector,        /* sector number to read */
			0,             /* expected sector type; 0 - all sector types */
			false,         /* b_digital_audio_play - audio related stuff, doesn't matter */
			true,          /* b_sync_header - return synchronization header */
			header_code,   /* uint8_t header_codes - data header and subheader; 0: none, 1: 4-bit header, 2: 8-bit subheader. 3: both */
			false,         /* bool b_user_data - return user data */
			false,         /* bool b_edc_ecc - error detection/correction data */
			0,             /* uint8_t c2_error_information, yet another error-related field (?) */
			0,             /* uint8_t subchannel_selection - subchannel selection bits (?)*/
			2352,          /* uint16_t i_blocksize - size of the block expected to be returned, 2352 is full sector size, including sync bytes */
			1              /* uint32_t i_blocks - number of blocks expected to be returned */
			);
		if (drv < 0) {
			/* fprintf(stderr, "mmc_read_cd returned XA error value = %d (sector %d)\n", (int) drv, (int) sector); */
			/* raw_read_sector(sector, sector_buffer); */
			// *_cdw_track_format = CDW_BLACK_BOOK_UNKNOWN;
			return false; /* mmc not supported or failed */
		}
	}

	int i = 0;
	/* first 12 bytes are bytes of synchronization: 00 FF FF FF FF FF FF FF FF FF FF 00;
	 * we collect them just for debug purposes - it doesn't cost much in terms of computer resources */
	unsigned char sector_sync[12];
	for (i = 0; i < 12; i++) {
		sector_sync[i] = sector_buffer[i];
	}

	/* then 4-byte header - present in Data CD sectors of all formats */
	unsigned char sector_header[4];
	for (i = 0; i < 4; i++) {
		sector_header[i] = sector_buffer[12 + i];
	}

	/* then (possibly) 8-byte subheader - it exists only in XA (eXtended Attributes Green Book) disc sectors */
	unsigned char sector_subheader[8];
	if (disc.mode == CDIO_DISC_MODE_CD_XA) {
		for (i = 0; i < 8; i++) {
			sector_subheader[i] = sector_buffer[12 + 4 + i];
		}
	}

	/* fprintf(stderr, "sector: %d, driver return value: %d\n", (int) sector, (int) drv); */

	/* mmc function returned valid data, analyse it */

	/*
	fprintf(stderr, "sync: >>");
	for (i =  0; i< 12; i++) {
		fprintf(stderr, " %x ", sector_sync[i]);
	}
	fprintf(stderr, "<< header: >>");
	for (i =  0; i< 4; i++) {
		fprintf(stderr, " %x ", sector_header[i]);
	}
	fprintf(stderr, "<<");

	if (disc.mode == CDIO_DISC_MODE_CD_XA) {
		fprintf(stderr, " subheader: >>");
		for (i =  0; i< 8; i++) {
			fprintf(stderr, " %x ", sector_subheader[i]);
		}
	}
	fprintf(stderr, "<<\n");

	fprintf(stderr, "\nAnalysing header:\n");
	fprintf(stderr, "header byte 0 = %u (minutes)\n", sector_header[0]);
	fprintf(stderr, "header byte 1 = %u (seconds)\n", sector_header[1]);
	fprintf(stderr, "header byte 2 = %u (sector number within one second (1-75))\n", sector_header[2]);
	fprintf(stderr, "header byte 3 = %u (sector mode (0/1/2))\n", sector_header[3]);
	*/

	/* we have header, let's now check sector format */
	/* CDW_RED_BOOK_AUDIO is not checked, because
	* cdw_get_track_format_with_cdio_mmc() should not be
	* called for audio track; there are no headers for audio sector,
	* so this code would produce "garbage out"  */

	if (sector_header[3] == 2) {
		*_cdw_track_format = CDW_YELLOW_BOOK_MODE2; /* or Green Book Mode2 Formless */
	} else if (sector_header[3] == 1) {
		*_cdw_track_format = CDW_YELLOW_BOOK_MODE1;
	} else if (sector_header[3] == 0) {
		*_cdw_track_format = CDW_BLACK_BOOK_UNKNOWN;
	} else {
		*_cdw_track_format = CDW_BLACK_BOOK_UNKNOWN;
	}
	/* _cdw_track_format may still be modified if the disc is XA */

	// if (disc.mode == CDIO_DISC_MODE_CD_XA) { /* FIXME: would it make sence to check that sector_header[3] == 2 ? */
	if (sector_header[3] == 2) { /* MODE 2 DISC */
		/* fprintf(stderr, "\nAnalysing subheader, some bytes may not displayed (this is XA disc, so the header should be valid):\n"); */
		if ( (sector_subheader[0] == sector_subheader[4])
					&& (sector_subheader[1] == sector_subheader[5])
					&& (sector_subheader[2] == sector_subheader[6])
				&& (sector_subheader[3] == sector_subheader[7])) {
			/* fprintf(stderr, " ** subheader is valid ** \n"); */
		} else {
			/* fprintf(stderr, " ** subheader is invalid ** \n"); */
		}
		/*
		fprintf(stderr, "subheader byte 0 = %u (?? (00 for not interleaved))\n", sector_subheader[0]);
		fprintf(stderr, "subheader byte 1 = %u (channel number)\n",              sector_subheader[1]);
		fprintf(stderr, "subheader byte 2 = %u (submode byte)\n",                sector_subheader[2]);
		fprintf(stderr, "   subheader bit 2.0 = %u (EOR - End Of Record (?))\n",    (sector_subheader[2] & 0x01));
		fprintf(stderr, "   subheader bit 2.1 = %u (Video)\n",                      (sector_subheader[2] & 0x02) >> 1);
		fprintf(stderr, "   subheader bit 2.2 = %u (Audio)\n",                      (sector_subheader[2] & 0x04) >> 2);
		fprintf(stderr, "   subheader bit 2.3 = %u (Data)\n",                       (sector_subheader[2] & 0x08) >> 3);
		fprintf(stderr, "   subheader bit 2.4 = %u (?)\n",                          (sector_subheader[2] & 0x10) >> 4);
		fprintf(stderr, "   subheader bit 2.5 = %u (Form 1/2)\n",                   (sector_subheader[2] & 0x20) >> 5);
		fprintf(stderr, "   subheader bit 2.6 = %u (Real-Time-Sector (?))\n",       (sector_subheader[2] & 0x40) >> 6);
		fprintf(stderr, "   subheader bit 2.7 = %u (EOF)\n",                        (sector_subheader[2] & 0x80) >> 7);
		fprintf(stderr, "subheader byte 3 = %u (Audio/Video encoding (0 for Data))\n", sector_subheader[3]);
		fprintf(stderr, "subheader byte 4 = %u (should be equal to byte 1)\n",    sector_subheader[4]);
		fprintf(stderr, "subheader byte 5 = %u (should be equal to byte 2)\n",    sector_subheader[5]);
		fprintf(stderr, "subheader byte 6 = %u (should be equal to byte 3)\n",    sector_subheader[6]);
		fprintf(stderr, "subheader byte 7 = %u (should be equal to byte 4)\n",    sector_subheader[7]);
		*/

		/* enough printing, let's check track format */
		/* CDW_RED_BOOK_AUDIO is not checked, because
		 * cdw_get_track_format_with_cdio_mmc() should not be
		 * called for audio track */
		if ( ((sector_subheader[2] & 0x20) >> 5) == 0x01) { /* if bit 5 is set - form 2 */
			*_cdw_track_format = CDW_GREEN_BOOK_MODE2_FORM2;
		} else {
			*_cdw_track_format = CDW_GREEN_BOOK_MODE2_FORM1;
		}
	}

	return true; /* sucess */
}



/**
 * Some cdio functions return value of driver_return_code_t type.
 * If this value is not DRIVER_OP_SUCCESS or greater than zero, then
 * the value should be analyzed and acted upon. Do this here.
 */
void cdw_deal_with_cdio_driver_error(driver_return_code_t error_num)
{

	if (error_num == DRIVER_OP_ERROR) {
		fprintf(stderr, "cdio driver error: DRIVER_OP_ERROR (%d)\n", error_num);
	} else if (error_num == DRIVER_OP_UNSUPPORTED) {
		fprintf(stderr, "cdio driver error: DRIVER_OP_UNSUPPORTED (%d)\n", error_num);
	} else if (error_num == DRIVER_OP_UNINIT) {
		fprintf(stderr, "cdio driver error: DRIVER_OP_UNINIT (%d)\n", error_num);
	} else if (error_num == DRIVER_OP_NOT_PERMITTED) {
		fprintf(stderr, "cdio driver error: DRIVER_OP_NOT_PERMITTED (%d)\n", error_num);
	} else if (error_num == DRIVER_OP_BAD_PARAMETER) {
		fprintf(stderr, "cdio driver error: DRIVER_OP_BAD_PARAMETER (%d)\n", error_num);
	} else if (error_num == DRIVER_OP_BAD_POINTER) {
		fprintf(stderr, "cdio driver error: DRIVER_OP_BAD_POINTER (%d)\n", error_num);
	} else if (error_num == DRIVER_OP_NO_DRIVER) {
		fprintf(stderr, "cdio driver error: DRIVER_OP_NO_DRIVER (%d)\n", error_num);
	} else {
		fprintf(stderr, "cdio unknown driver error: %d\n", error_num);
	}

	return;
}




/**
 * This function (pretends that it) reads track from given disc and writes
 * them to given file descriptor
 *
 * The truth is that
 *     cdw_cdio_read_write_tracks_from_disc()
 * pushes down all the hard work to
 *     cdw_cdio_read_write_sectors_from_track()
 * This is just simple wrapper for one 'for' loop (iterate on tracks)
 * and one function call inside loop. It may however be extended in future.
 *
 * \param disc - data structure describing disc
 * \param output_file_image - file descriptor of already opened file.
 *                            The file must have correct permissions.
 *
 * \returns true if no errors occured during reading or writing, false otherwise
 */
bool cdw_cdio_read_write_tracks_from_disc(struct disc_information *disc, int output_image_file)
{
	track_t t = 1; /* tracks start from one */

	/* fprintf(stderr, "INFO: cdw_cdio_read_write_tracks_from_disc(): total tracks to read: %d\n", (int)disc->total_tracks); */

	/* track parameters were recognized after opening
	the disc, so let's read tracks one by one */
	for (t = disc->first_track; t <= disc->last_track; t++) {
		cdw_rv_t rv = cdw_cdio_read_write_sectors_from_track(disc, t, output_image_file);
		/* fprintf(stderr, "------------INFO: cdw_cdio_read_write_tracks_from_disc(): track %d finished rv = %d\n", (int) t, (int) rv); */
		if (rv != CDW_OK) {
			/* fprintf(stderr, "ERROR: cdw_cdio_read_write_tracks_from_disc(): returning false in cdw_cdio_read_write_tracks_from_disc() (rv != 0 for track t = %d)\n", (int) t); */
			return false;
		}
	}

	return true;
}





/**
 * Read sectors from given track of disc, write them to given file descriptor
 *
 * \param struct disc_information *disc - dat structure with some information about optical disc
 * \param track_t t - track number of track to read sectors from
 * \param int imagefile - file descriptor to write the data to (already opened and with proper permissions)
 *
 * \returns CDW_OK on success, CDW_CANCEL on failure
 */
cdw_rv_t cdw_cdio_read_write_sectors_from_track(struct disc_information *disc, track_t t, int output_image_file)
{
	lsn_t sector = disc->tracks[t].first_sector;; /* iterator - sector number */
	unsigned char sector_buf[CDIO_CD_FRAMESIZE_RAWER];   /* buffer for data read from disc, CDIO_CD_FRAMESIZE_RAWER is maximal size of sector defined in cdio header files */

	bool success = false;

	/* debug code */
	/*
	fprintf(stderr, "##                                          track number t = %8d \n", (int) t);
	fprintf(stderr, "##    first sector on track (disc->tracks[t].first_sector) = %8d \n", (int) disc->tracks[t].first_sector);
	fprintf(stderr, "##      last sector on track (disc->tracks[t].last_sector) = %8d \n", (int) disc->tracks[t].last_sector);
	fprintf(stderr, "##        total sectors on disc ((int)disc->total_sectors) = %8d \n", (int) disc->total_sectors);
	fprintf(stderr, "## sector size in this track (disc->tracks[t].sector_size) = %8d \n", (int) disc->tracks[t].sector_size);
	*/

	for (sector = disc->tracks[t].first_sector; sector <= disc->tracks[t].last_sector; sector++) {
		driver_return_code_t rvt = 0;
		/* fprintf(stderr, "reading track t = %d, sector s = %d out of %d sectors (total %d sectors on disc), current sector size is %d\n", (int)t, (int) sector, disc->tracks[t].last_sector, (int)disc->total_sectors, disc->tracks[t].sector_size); */
		if (disc->tracks[t].cdw_format != CDW_BLACK_BOOK_UNKNOWN) {
			if (disc->tracks[t].cdw_format == CDW_RED_BOOK_AUDIO) {
				rvt = cdio_read_audio_sector(disc->p_cdio, sector_buf, sector);
				/* fprintf(stderr, "rvt = %d, reading track t = %d, sector s = %d out of %d sectors (total %d sectors on disc), current sector size is %d\n", (int) rvt, (int)t, (int) sector, disc->tracks[t].last_sector, (int)disc->total_sectors, disc->tracks[t].sector_size); */
			} else { /* having 'disc->tracks[t].sector_size' set
				* at some time before reaching this point AND
				* having cdio_read_data_sectors() function
				 * helps us to avoid multiple ifs here */
				rvt = cdio_read_data_sectors(disc->p_cdio, sector_buf, sector, disc->tracks[t].sector_size, 1); /* 1 = number of blocks */
			}
		} else {
			/* fprintf(stderr, "cdw_cdio_read_write_sectors_from_track(): no cdw format info available for track t = %d, exiting!\n", (int) t); */ /* FIXME: shouldn't quit so quickly */
			success = false;
			break;
		}

		if (rvt == DRIVER_OP_SUCCESS || rvt > 0) { /* sometimes cdio functions return valid 'enum driver_return_code_t' value, but sometimes it's some positive value; see comment on driver_return_code_t in cdio/driver.h */
			int wrv = write(output_image_file, sector_buf, disc->tracks[t].sector_size);
			if (wrv != ((int) disc->tracks[t].sector_size)) {
				success = false;
				/* fprintf(stderr, "ERROR: cdw_cdio_read_write_sectors_from_track(): write error for track t = %d (wrv = %d, but sector size = %d), exiting!\n", (int) t, wrv, ((int) disc->tracks[t].sector_size));*/ /* FIXME: shouldn't quit so quickly */
				break;
			} else {
				long current = (long)(((long)sector) * disc->tracks[t].sector_size) / 1024 / 1024;
				long total = (long) ((disc->total_sectors * disc->tracks[t].sector_size) / 1024 / 1024);
				char current_value_string[PROCESSWIN_MAX_TEXT_LEN];
				/* 2TRANS: this string will be displayed as message in progress window;
				   first %ld is amount of data already read from cd,
				   second %ld is total amount of data to be read */
				sprintf(current_value_string, _("%ld/%ld MB"), current, total);
				conditional_processwin_display_progress(current, total, current_value_string);
				success = true; /* this is the only place in this function where we say that read/write was successful */
			}
		} else {
			/*
			fprintf(stderr, "cdw_cdio_read_write_sectors_from_track(): driver return without success when reading %d bytes, track t = %d, exiting!\n", (int) disc->tracks[t].sector_size, (int) t); */ /* FIXME: shouldn't quit so quickly */
			/*
			fprintf(stderr, "errno = %d\n", errno);
			perror("ERROR: "); */
			/* cdw_deal_with_cdio_driver_error(rvt); */ /* some debug output */
			success = false; /* cdio read function returned with some error value */
			break;
		}
	}

	if (success) {
		return CDW_OK;
	} else {
		return CDW_CANCEL;
	}
}






/* ******************************* */
/* ****** unused code below ****** */
/* ******************************* */






/*
long int cdw_cdio_get_total_disc_sectors(void)
{
	return (int) disc.total_sectors;
}
*/





#if 0 /* not used because now we have per-track method */

/*
 * Return useful data size in sector based on disc type
*/
int cdw_cdio_sector_size(void)
{
	switch (disc.mode) {
		case CDIO_DISC_MODE_CD_DA:     /**< CD-DA */
			return 2352;
		case CDIO_DISC_MODE_CD_DATA:   /**< CD-ROM mode 1 form 1 */
			return 2048;
		case CDIO_DISC_MODE_CD_XA:     /**< CD-ROM XA mode 2 form 2 */
			return 2324;
		case CDIO_DISC_MODE_CD_MIXED:  /**< Some combo of above. */
			return -1; /* FIXME - handle data and audio frames separately */
		case CDIO_DISC_MODE_DVD_ROM:   // DVD ROM (e.g. movies)
		case CDIO_DISC_MODE_DVD_RAM:   // DVD-RAM
		case CDIO_DISC_MODE_DVD_R:     // DVD-R
		case CDIO_DISC_MODE_DVD_RW:    // DVD-RW
		case CDIO_DISC_MODE_DVD_PR:    // DVD+R
		case CDIO_DISC_MODE_DVD_PRW:   // DVD+RW
		case CDIO_DISC_MODE_DVD_OTHER: // Unknown/unclassified DVD type
			return 2048;
		/* FIXME - figure out what are correct values for these cases */
	case CDIO_DISC_MODE_NO_INFO:
	case CDIO_DISC_MODE_ERROR:
		case CDIO_DISC_MODE_CD_I:      // CD-i
			return -1;
	default:
			return -1;
}
}
#endif






#if 0
sector_reader_fun cdw_cdio_sector_reader(void)
{
	switch (disc.mode) {
		case CDIO_DISC_MODE_CD_DA:     /**< CD-DA */
			return &cdio_read_audio_sector;
		case CDIO_DISC_MODE_CD_DATA:   /**< CD-ROM form 1 */
			return NULL;
		case CDIO_DISC_MODE_CD_XA:     /**< CD-ROM XA form2 */
			return NULL;
		case CDIO_DISC_MODE_CD_MIXED:  /**< Some combo of above. */
			return NULL; /* FIXME - handle data and audio frames separately */

	}
}

#endif
#if 0

int main(void)
{
	char buf[2048];

	disc.p_cdio = cdio_open("/dev/hdd", DRIVER_DEVICE);
	if (disc.p_cdio == NULL) {
		fprintf(stderr, "error: cannot open cd drive\n");
		return -1;
	}

	track_t track_num = cdio_get_first_track_num(disc.p_cdio);
	disc.total_tracks = cdio_get_num_tracks(disc.p_cdio);
	printf("disc contains %d track(s)\n", disc.total_tracks);

	int i = 0;

	lsn_t sector_number;
	for (i = 0; i < disc.total_tracks; i++, track_num++) {
		sector_number = cdio_get_track_lsn(disc.p_cdio, track_num);
		if (sector_number != CDIO_INVALID_LSN) {
			printf("track number %d starts at sector %d\n", (int) track_num, (int) sector_number);
		}

	}

	sector_number = cdio_get_track_lsn(disc.p_cdio, CDIO_CDROM_LEADOUT_TRACK);
	printf("leadout starts at sector %d\n", sector_number);

	lsn_t sectors_to_read = cdio_get_track_last_lsn(disc.p_cdio, disc.total_tracks);
	sectors_to_read = cdio_get_track_last_lsn(disc.p_cdio, 1);
	printf("sectors to read: %d\n", sectors_to_read);

	for (sector_number = sectors_to_read; sector_number >= 0; sector_number--) {
		driver_return_code_t rv = cdio_read_mode1_sector(disc.p_cdio, &buf, sector_number, false);
		if (rv != 0){
			printf("return value for sector %d is %d\n", (int) sector_number, (int) rv);
		}
	}

	cdio_destroy(disc.p_cdio);

	return 0;
}

#endif

